/**********************************************************************
 * Copyright (c) 2003, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceUtils.java,v 1.23 2007/08/21 10:07:55 ayaffe Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * 
 * $Id: TraceUtils.java,v 1.23 2007/08/21 10:07:55 ayaffe Exp $
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.loaders.hierarchy.Constants;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCConfiguration;
import org.eclipse.hyades.models.hierarchy.TRCOption;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCFullMethodInvocation;
import org.eclipse.hyades.models.trace.TRCFullTraceObject;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCMethodProperties;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCSourceInfo;
import org.eclipse.hyades.models.trace.TraceFactory;
import org.eclipse.hyades.models.trace.impl.TRCPackageImpl;


/**
 * @author slavescu
 *
 */
public final class TraceUtils {
    //~ Static fields/initializers -----------------------------------------------------------------

    private static StringBuffer strBuf = new StringBuffer();
	public static final int CLASS_CLASS_DEFAULT_ID = -(Integer.MAX_VALUE/2);
	public static final String UNKNOWN = "unknown";
	public static final String DEFAULT = "";

    //~ Methods ------------------------------------------------------------------------------------

    public static boolean isConstructor(String msg) {
        return (msg.equals("<clinit>") || msg.equals("-clinit-") || msg.equals("-init-"));
    }

    public static String getForwardInvocationsId(InvocationContext invocationContext) {
        return getForwardInvocationsId(invocationContext.getInvocationAgentIdRef());
    }

    public static String getForwardInvocationsId(String agentRuntimeID) {
        //        return "inv_" + agentRuntimeID;
        return agentRuntimeID;
    }

    public static String getMethodInvocationId(InvocationContext invocationContext) {
        return getMethodInvocationId(invocationContext.getInvocationThreadIdRef(), invocationContext.getInvocationTicket());
    }

    public static String getMethodInvocationId(int threadId, long ticket) {
        return threadId + "_" + ticket;
    }

    public static String getMethodName(TRCMethod method, String originalMethodName) {
        if (originalMethodName.equals("-init-")) {
            return method.getDefiningClass().getName();
        }

        return originalMethodName;
    }

    public static TRCProcess getProcess(TRCAgent agent) {
        if ((agent != null) && agent.getProcess() instanceof TRCProcess) {
//			System.out.println("getInputOutputContainer="+((TRCProcess)agent.getProcess()).getInputOutputContainer());
//
//    		for (Iterator iter = ((TRCProcess)agent.getProcess()).getInputOutputContainer().getEntries().iterator(); iter.hasNext();) {
//    			TRCInputOutputEntryImpl element = (TRCInputOutputEntryImpl) iter.next();
//    			System.out.println("key="+element.getKey());
//    			for (int i=0;i<element.getTypedValue().size(); i++) {
//    				EObject value = (EObject) element.getTypedValue().get(i);
//    				System.out.println("	value["+i+"]="+value);
//    			}
//    		}
//			System.out.println("getInputOutputContainer.getInputOutputValues.size="+((TRCProcess)agent.getProcess()).getInputOutputContainer().getInputOutputValues().size());
//        	try {
//        		((TRCProcess)agent.getProcess()).getInputOutputContainer().eResource().save(Collections.EMPTY_MAP);
//			} catch (IOException e) {
//				// TODO Auto-generated catch block
//				e.printStackTrace();
//			}
            return (TRCProcess) agent.getProcess();
        } else {
            return null;
        }
    }
    
    public static String getOptionValue(TRCAgentProxy agentProxy, String optionName) {
    	if (agentProxy == null ||
    		agentProxy.getConfigurations() == null) {
    		return null;
    	}
		Object[] conflist = agentProxy.getConfigurations().toArray();
		
		for(int idx = 0; idx < conflist.length; idx++){
			TRCConfiguration c = (TRCConfiguration) conflist[idx];
			if(!c.isActive())continue;
    		Object[] optlist = c.getOptions().toArray();
    		for(int jdx = 0; jdx < optlist.length; jdx++){
    			TRCOption o = (TRCOption)optlist[jdx];
    			if(o.getKey().equals(optionName))
    				return o.getValue();
    		}
		}
    		
    	return null;
    }
    
	public static boolean isBooleanOptionEnabled(HierarchyContext context,String name) {
		boolean optionSet = context.getCustomData().containsKey(name);
		boolean optionValue;
		if (!optionSet) {
			String value = getOptionValue(context.getAgentProxy(), name);
			if (value != null && Boolean.valueOf(value).booleanValue()) {
				context.getCustomData().put(name,Boolean.TRUE);
				optionValue = true;
			} else {
				context.getCustomData().put(name,Boolean.FALSE);
				optionValue = false;
			}
		} else {
			optionValue = ((Boolean)context.getCustomData().get(name)).booleanValue();
		}
		return optionValue;
	}

	public static void setSourceInfo(HierarchyContext context, TRCProcess p, TRCClass theClass, String sourceName, String language) {
	      
  	 	  TRCSourceInfo sourceInfo = (TRCSourceInfo) LookupServiceExtensions.getInstance().locate(context, TRCSourceInfo.class, LoadersUtils.getHashCode(sourceName));
	      if (sourceInfo == null) {
	      	if(isBooleanOptionEnabled(context,Constants.LLDATA_ENABLED))
	        		sourceInfo = TraceFactory.eINSTANCE.createTRCSourceInfoWithLLData();
	        	else
	        		sourceInfo = TraceFactory.eINSTANCE.createTRCSourceInfo();
	            sourceInfo.setLocation(sourceName);
	            sourceInfo.setLanguage(language);
	            p.getSourceInfos().add(sourceInfo);
	      }

	      if (theClass.getSourceInfo() == null) {
	            theClass.setSourceInfo(sourceInfo);
	      } else {
	            if (!theClass.getSourceInfo().getLocation().equals(sourceName)) {
	                theClass.setSourceInfo(sourceInfo);
	            }
	      }
	  }


    public static void addClassObjectToClassClass(HierarchyContext context, TRCProcess process, TRCObject theObject) {
        TRCClass theClassClass = process.getClassClass();

        if (theClassClass == null) {
            //creates the fake Class class
            theClassClass = TraceFactory.eINSTANCE.createTRCClass();
            theClassClass.setId(CLASS_CLASS_DEFAULT_ID);//(-1); // fake ID

            if(process.getLanguages().isEmpty() || process.getLanguages().indexOf("Java")!=-1)
            {
                theClassClass.setName("Class"); //fake Class class
            	addClassToPackage(context, theClassClass, "java.lang.Class", process);
                addLanguageIfRequired("Java",process.getLanguages());
            }
            else
            {
                theClassClass.setName("ClassClass"); //fake Class class
            	addClassToPackage(context, theClassClass, "ClassClass", process);
            }

       
            
            LookupServiceExtensions.getInstance().register(context, theClassClass);

            process.setClassClass(theClassClass);
        }

        if (theObject!= null && theObject.getIsA() != theClassClass) {
            theObject.setIsA(theClassClass);
        }
    }

    public static void addClassToPackage(HierarchyContext context, TRCClass theClass, String classFullyQualifiedName, TRCProcess process) {
        if (theClass.getPackage() != null) {
            return;
        }

        String packageName = packageName(classFullyQualifiedName);

        if(packageName.length()==0)
        {
        	packageName = DEFAULT;
        }
        
        TRCPackage thePackage = (TRCPackage) LookupServiceExtensions.getInstance().locate(context, TRCPackageImpl.class, packageName);

        if (thePackage != null) {
            theClass.setPackage(thePackage);

            return;
        }

        thePackage = TraceFactory.eINSTANCE.createTRCPackage();
        thePackage.setName(packageName);
        thePackage.setProcess(process);
        theClass.setPackage(thePackage);

        //        LookupServiceExtensions.getInstance().register(context, thePackage);
        createSuperPackages(context, thePackage);
    }

    public static synchronized String className(String fullclassname) {
    	if(fullclassname==null)
    		return DEFAULT;
    	
        if (fullclassname.startsWith("[")) {
            strBuf.setLength(0);
            expandArrayType(fullclassname, 0, strBuf);

            fullclassname = strBuf.toString();

            strBuf.setLength(0);

        }

        int lastslash = Math.max(fullclassname.lastIndexOf('/'), fullclassname.lastIndexOf('\\'));

        int lastdot = fullclassname.lastIndexOf('.');

        lastslash = Math.max(lastslash, lastdot);

        if (lastslash < 0) {
            return fullclassname;
        } else {
            return fullclassname.substring(lastslash + 1);
        }
    }

    public static String convertMethodSignature(TRCMethod method, String originalMethodSignature) {
    	return convertMethodSignature(method.getModifier(),originalMethodSignature);
    }
    // converts the JVM method signature to a more readable signature 
    public static String convertMethodSignature(int modifiers, String originalMethodSignature) {
        if (originalMethodSignature == null) {
            return "()";
        }

        StringBuffer newMethodSignature = new StringBuffer();

        // Find indices of '(' and ')'
        //----------------------------
        int i1 = originalMethodSignature.indexOf('(');

        // Return just the message name if there are no parameters
        //--------------------------------------------------------
        if (i1 >= 0) {
            i1++;

            // Parse the message parameters
            //-----------------------------
            int i2 = originalMethodSignature.indexOf(')');

            // If no parameters, do not parse
            //-------------------------------
            if (i2 == i1) {
                newMethodSignature.append("()");
            }
            // Parse parameters
            //-----------------
            else {
                // Expand message parameters
                //--------------------------
                newMethodSignature.append("(");

                while (i1 < i2) {
                    i1 = expandMessageFieldType(originalMethodSignature, i1, newMethodSignature);

                    if ((i1 < i2)) {
                        newMethodSignature.append(", ");
                    }
                }

                newMethodSignature.append(")");
            }

            // Expand return type if not constructor
            //--------------------------------------
            if ((modifiers & TRCMethodProperties.JAVA_CONSTRUCTOR) == 0) {
                newMethodSignature.append(" ");
                expandMessageFieldType(originalMethodSignature, ++i1, newMethodSignature);
            }
        }

        return newMethodSignature.toString();
    }

    public static void createSuperPackages(HierarchyContext context, TRCPackage thePackage) {
        TRCProcess p = thePackage.getProcess();
        String packageName = thePackage.getName();

        TRCPackage sPkg;
        int pos = 0;
        int pPos = packageName.length();

        while (pPos > 0) {
            pos = packageName.lastIndexOf('.', pPos);

            if (pos < 0) {
                break;
            }

            String pName = packageName.substring(0, pos);

            sPkg = (TRCPackage) LookupServiceExtensions.getInstance().locate(context, TRCPackageImpl.class, pName);

            if (sPkg == null) {
                sPkg = TraceFactory.eINSTANCE.createTRCPackage();
                sPkg.setName(pName);
                sPkg.setProcess(p);
            }

            thePackage.setParent(sPkg);
            thePackage = sPkg;
            pPos = pos - 1;
        }
    }

    //    // the size of one element of the array.
    //    public static int elementSize(int type) {
    //        switch (type) {
    //        case TRCArrayType.JVMPI_INT:
    //            return 4;
    //
    //        case TRCArrayType.JVMPI_LONG:
    //            return 8;
    //
    //        case TRCArrayType.JVMPI_FLOAT:
    //            return 4;
    //
    //        case TRCArrayType.JVMPI_DOUBLE:
    //            return 8;
    //
    //        case TRCArrayType.JVMPI_BOOLEAN:
    //            return 1;
    //
    //        case TRCArrayType.JVMPI_BYTE:
    //            return 1;
    //
    //        case TRCArrayType.JVMPI_CHAR:
    //            return 2;
    //
    //        case TRCArrayType.JVMPI_SHORT:
    //            return 2;
    //
    //        case TRCArrayType.JVMPI_CLASS: // assuming pointers of 4 bytes long
    //            return 4;
    //
    //        default:
    //            return 0;
    //        }
    //    }
    //---------------------------------------------------------------------------
    //
    // MessageSignature::expandMessageFieldType
    //
    // From          To
    // ------------- -------
    // B byte        byte    signed byte
    // C             char    character
    // D             double  double-precision IEEE 754 float
    // F             float   single-precision IEEE 754 float
    // I             int     integer
    // J             long    long integer
    // L<classname>; ...     an instance of the class
    // S             short   signed short
    // Z             boolean true or false
    // [             ...     one array dimension
    //
    //---------------------------------------------------------------------------
    public static int expandMessageFieldType(String strInput, int iIndex, StringBuffer strbufExpanded) {
    	if(iIndex>=strInput.length())
    	{
    		strbufExpanded.append(strInput);
    		return iIndex;
    	}
        int iArray = 0; // Array dimension
        char cFieldType = strInput.charAt(iIndex);

        // Determine the array dimension if any
        //-------------------------------------
        while (cFieldType == '[') {
            ++iArray;
            cFieldType = strInput.charAt(++iIndex);
        }

        // Expand the field type character to a string
        //--------------------------------------------
        switch (cFieldType) {
        case 'B':
            strbufExpanded.append("byte");

            break;

        case 'C':
            strbufExpanded.append("char");

            break;

        case 'D':
            strbufExpanded.append("double");

            break;

        case 'F':
            strbufExpanded.append("float");

            break;

        case 'I':
            strbufExpanded.append("int");

            break;

        case 'J':
            strbufExpanded.append("long");

            break;

        case 'S':
            strbufExpanded.append("short");

            break;

        case 'Z':
            strbufExpanded.append("boolean");

            break;

        case 'V':
            strbufExpanded.append("void");

            break;

        case 'L':

            // Convert 'Ljava/lang/String;' to 'java.lang.String'
            //------------------------------------------
            int iSemiColon = strInput.indexOf(';', iIndex);

            strbufExpanded.append(strInput.substring(iIndex + 1, iSemiColon).replace('/', '.'));
            iIndex = iSemiColon;

            break;
        }

        while (iArray > 0) {
            strbufExpanded.append("[]");
            --iArray;
        }

        return ++iIndex;
    }

    public static String fullClassName(String fullclassname) {
        String name = TString.change(fullclassname, "/", ".");

        name = TString.change(name, "\\", ".");

        return name;
    }

    public static String packageName(String fullclassname) {
    	if(fullclassname==null)
    		return DEFAULT;

        if (fullclassname.startsWith("[")) {
        	int index=fullclassname.lastIndexOf('[')+1;
        	
            switch (fullclassname.charAt(index)) {
            case 'B':
            case 'C':
            case 'D':
            case 'F':
            case 'I':
            case 'J':
            case 'S':
            case 'Z':
            case 'V':
            case 'L':
            	index++;
            default:
            }
           fullclassname=fullclassname.substring(index);
        }

        int lastslash = Math.max(fullclassname.lastIndexOf('/'), fullclassname.lastIndexOf('\\'));

        int lastdot = fullclassname.lastIndexOf('.');

        lastslash = Math.max(lastslash, lastdot);

        if (lastslash < 0) {
            return DEFAULT;
        } else {
            String packageName = fullclassname.substring(0, lastslash);

            packageName = TString.change(packageName, "/", ".");
            packageName = TString.change(packageName, "\\", ".");

            return packageName;
        }
    }

    public static void removeSnapshot(TRCProcess process) {
    	if (process == null) return;
    	
        process.removeSnapshot();

        for (Iterator iter1 = process.getPackages().iterator(); iter1.hasNext();) {
            TRCPackage thePackage = (TRCPackage) iter1.next();

            thePackage.removeSnapshot();

            for (Iterator iter2 = thePackage.getClasses().iterator(); iter2.hasNext();) {
                TRCClass theClass = (TRCClass) iter2.next();

                theClass.removeSnapshot();

                for (Iterator iter = theClass.getMethods().iterator(); iter.hasNext();) {
                    TRCMethod theMethod = (TRCMethod) iter.next();

                    theMethod.removeSnapshot();
                }
            }
        }
    }

	public static void removeSnapshot(TRCAgentProxy agentProxy) {
    	TraceUtils.removeSnapshot(TraceUtils.getProcess(agentProxy.getAgent()));
	}
    
    public static void removeVirtualObject(HierarchyContext context, long objIdRef) {
        LookupServiceExtensions.getInstance().deregister(context, VirtualObjectInfo.class, objIdRef);
    }

    public static void takeSnapshot(TRCProcess process) {
        process.takeSnapshot();

        for (Iterator iter1 = process.getPackages().iterator(); iter1.hasNext();) {
            TRCPackage thePackage = (TRCPackage) iter1.next();

            thePackage.takeSnapshot();

            for (Iterator iter2 = thePackage.getClasses().iterator(); iter2.hasNext();) {
                TRCClass theClass = (TRCClass) iter2.next();

                theClass.takeSnapshot();

                for (Iterator iter = theClass.getMethods().iterator(); iter.hasNext();) {
                    TRCMethod theMethod = (TRCMethod) iter.next();

                    theMethod.takeSnapshot();
                }
            }
        }
    }

    public static int expandArrayType(String strInput, int iIndex, StringBuffer strbufExpanded) {
        int iArray = 0; // Array dimension

        char cFieldType = strInput.charAt(iIndex);

        // Determine the array dimension if any
        //-------------------------------------
        while (cFieldType == '[') {
            ++iArray;
            cFieldType = strInput.charAt(++iIndex);
        }

        // Expand the field type character to a string
        //--------------------------------------------
        switch (cFieldType) {
        case 'B':
            strbufExpanded.append("byte");

            break;

        case 'C':
            strbufExpanded.append("char");

            break;

        case 'D':
            strbufExpanded.append("double");

            break;

        case 'F':
            strbufExpanded.append("float");

            break;

        case 'I':
            strbufExpanded.append("int");

            break;

        case 'J':
            strbufExpanded.append("long");

            break;

        case 'S':
            strbufExpanded.append("short");

            break;

        case 'Z':
            strbufExpanded.append("boolean");

            break;

        case 'V':
            strbufExpanded.append("void");

            break;

        case 'L':

            // Convert 'Ljava/lang/String;' to 'java.lang.String'
            //------------------------------------------
            int iSemiColon = strInput.indexOf(';', iIndex);

            strbufExpanded.append(strInput.substring(++iIndex, iSemiColon).replace('/', '.'));
            iIndex = iSemiColon;

            break;

        default:
            strbufExpanded.append(strInput.substring(iIndex).replace('/', '.'));

            break;
        }

        for (int i = iArray; i-- >0;) {
        	strbufExpanded.append("[]");
		}
        	
        	
        return ++iIndex;
    }

    //~ Inner Classes ------------------------------------------------------------------------------

    public static final class InvocationInfo {
        private CallStackPerThread callStackPerThread;
        private TRCClass theClass;
        private TRCClass theObjectClass; //this represent the method class
        private TRCFullMethodInvocation methodInvocation;
        private TRCFullTraceObject theObject;
        private TRCMethod theMethod;

        //        private TRCProcess process;
        private boolean classLocked = false;
        private boolean packageLocked = false;
        private boolean methodLocked = false;
        private boolean objectClassLocked = false;
        private boolean objectLocked = false;
        private double entryTime = -1;
        private double lastChildExitTime = -1;
	    private double lastChildOverhead = -1; 

        public InvocationInfo(TRCFullMethodInvocation inv, CallStackPerThread callStackPerThread) {
            //            this.process = inv.getProcess();
            this.callStackPerThread = callStackPerThread;
            update(inv);
        }

        public InvocationInfo(CallStackPerThread callStackPerThread, TRCFullTraceObject object, TRCClass theClass, TRCMethod method, double entryTime) {
            //			this.process = theObject.getProcess();
            this.callStackPerThread = callStackPerThread;
            update(object, theClass, method, entryTime);
        }

        /**
         * Returns the classLocked.
         * @return boolean
         */
        public boolean isClassLocked() {
            return classLocked;
        }

        /**
         * Returns the classLocked.
         * @return boolean
         */
        public boolean isPackageLocked() {
            return packageLocked;
        }

        /**
         * Returns the entryTime.
         * @return double
         */
        public double getEntryTime() {
            return entryTime;
        }

        /**
         * Sets the lastChildExitTime.
         * @param lastChildExitTime The lastChildExitTime to set
         */
        public void setLastChildExitTime(double lastChildExitTime) {
            this.lastChildExitTime = lastChildExitTime;
        }

        /**
         * Returns the lastChildExitTime.
         * @return double
         */
        public double getLastChildExitTime() {
            return lastChildExitTime;
        }

        /**
         * Method getMethod.
         */
        public TRCMethod getMethod() {
            return theMethod;
        }

        /**
         * Returns the methodInvocation.
         * @return TRCMethodInvocation
         */
        public TRCFullMethodInvocation getMethodInvocation() {
            return methodInvocation;
        }

        /**
         * Returns the methodLocked.
         * @return boolean
         */
        public boolean isMethodLocked() {
            return methodLocked;
        }

        /**
         * Method getObject.
         * @return TRCObject
         */
        public TRCFullTraceObject getObject() {
            return theObject;
        }

        public TRCClass getObjectClass() {
            return theObjectClass;
        }

        public boolean isObjectClassLocked() {
            return objectClassLocked;
        }

        /**
         * Returns the objectLocked.
         * @return boolean
         */
        public boolean isObjectLocked() {
            return objectLocked;
        }

        //        public void setProcess(TRCProcess process) {
        //            this.process = process;
        //        }
        //
        //        public TRCProcess getProcess() {
        //            return process;
        //        }

        /**
         * Returns the theClass.
         * @return TRCClass
         */
        public TRCClass getTheClass() {
            return theClass;
        }

        public void initInvocationInfo(double entryTime) {
            theObjectClass = null;

            if (theObject!= null && theObject.getIsA() != theClass) {
                if (theObject.getIsA() != theObject.getProcess().getClassClass()) {
                    theObjectClass = theObject.getIsA();
                }
            }

            this.entryTime = entryTime;

            if (callStackPerThread.locks.containsKey(theObject)) {
                objectLocked = true;
            } else {
                callStackPerThread.locks.put(theObject, null);
            }

            if (callStackPerThread.locks.containsKey(theMethod)) {
                methodLocked = true;
            } else {
                callStackPerThread.locks.put(theMethod, null);
            }

            if (callStackPerThread.locks.containsKey(theClass)) {
                classLocked = true;
            } else {
                callStackPerThread.locks.put(theClass, null);
            }

			if (callStackPerThread.locks.containsKey(theClass.getPackage())) {
                packageLocked = true;
            } else {
                callStackPerThread.locks.put(theClass.getPackage(), null);
            }

            if (theObjectClass != null) {
                if (callStackPerThread.locksInherited.containsKey(theObjectClass)) {
                    objectClassLocked = true;
                } else {
                    callStackPerThread.locksInherited.put(theObjectClass, null);
                }
//				if (callStackPerThread.locks.containsKey(theObjectClass.getPackage())) {
//	                packageLocked = true;
//	            } else {
//	                callStackPerThread.locks.put(theObjectClass.getPackage(), null);
//	            }
//				
            }

        }

        /**
         * Method removeLocks.
         */
        public void removeLocks() {
            if (!methodLocked) {
                callStackPerThread.locks.remove(theMethod);
            }

            if (!objectLocked) {
                callStackPerThread.locks.remove(theObject);
            }

            if (!classLocked) {
                callStackPerThread.locks.remove(theClass);
            }

			if (!packageLocked) {
                callStackPerThread.locks.remove(theClass.getPackage());
            }

            if (!objectClassLocked) {
                callStackPerThread.locksInherited.remove(theObjectClass);
            }
        }

        /**
         * Method reset.
         */
        public void reset() {
            removeLocks();
            methodLocked = false;
            objectLocked = false;
            objectClassLocked = false;
            classLocked = false;
			packageLocked = false;
            entryTime = -1;
            lastChildExitTime = -1;
            theClass = null;
            theObject = null;
            theObjectClass = null;
            theMethod = null;
            methodInvocation = null;

            //            process=null;
            callStackPerThread = null;
        }

        /**
         * Method update.
         * @param object
         * @param theClass
         * @param method
         * @param entryTime
         */
        public void update(TRCFullTraceObject object, TRCClass theClass, TRCMethod method, double entryTime) {
            theMethod = method;
            theObject = object;
            this.theClass = theClass;
            initInvocationInfo(entryTime);
        }

        //        private TRCObject getClassObject(TRCClass theClass, Class class1) {
        //        	for (Iterator iter = theClass.getClassObjects().iterator(); iter.hasNext();) {
        //				TRCObject element = (TRCObject) iter.next();
        //				if(element.getClass() == class1)
        //					return element;
        //			}
        //			return null;
        //		}

        /**
         * Method update.
         * @param inv
         */
        public void update(TRCFullMethodInvocation inv) {
            this.methodInvocation = inv;
            theObject = (TRCFullTraceObject) inv.getOwningObject();
            theMethod = inv.getMethod();
            theClass = theMethod.getDefiningClass();
            initInvocationInfo(inv.getEntryTime());
        }

        // bugzilla 123628
        void setLastChildOverhead(double lastChildOverhead) {
        	this.lastChildOverhead = lastChildOverhead;
        }
        
        double getLastChildOverhead() {
        	return lastChildOverhead;
        }
    	// end bugzilla 123628
    }


	public static final class InvocationPool {
        private ArrayList invocations = new ArrayList(20);
        private int currentEntryIndex = -1;

        public InvocationPool() {
            super();
        }

        public InvocationInfo allocInvocation(TRCFullMethodInvocation inv, CallStackPerThread callStackPerThread) {
            InvocationInfo invocationInfo;

            if (invocations.size() == (currentEntryIndex + 1)) {
                invocationInfo = new InvocationInfo(inv, callStackPerThread);
                invocations.add(invocationInfo);
                currentEntryIndex++;
            } else {
                currentEntryIndex++;
                invocationInfo = (InvocationInfo) invocations.get(currentEntryIndex);
                invocationInfo.callStackPerThread = callStackPerThread;
                invocationInfo.update(inv);
            }

            return invocationInfo;
        }

        public InvocationInfo allocInvocation(CallStackPerThread callStackPerThread, TRCFullTraceObject object, TRCClass theClass, TRCMethod method, double entryTime) {
            InvocationInfo invocationInfo;

            if (invocations.size() == (currentEntryIndex + 1)) {
                invocationInfo = new InvocationInfo(callStackPerThread, object, theClass, method, entryTime);
                invocations.add(invocationInfo);
                currentEntryIndex++;
            } else {
                currentEntryIndex++;
                invocationInfo = (InvocationInfo) invocations.get(currentEntryIndex);
                invocationInfo.callStackPerThread = callStackPerThread;
                invocationInfo.update(object, theClass, method, entryTime);
            }

            return invocationInfo;
        }

        public void release(InvocationInfo invocationInfo) {
            invocationInfo.reset();
            currentEntryIndex--;
        }
    }

	public static void setSourceInfo(HierarchyContext context, TRCProcess p, TRCMethod theMethod, String sourceName, String language) {
	 	  TRCSourceInfo sourceInfo = (TRCSourceInfo) LookupServiceExtensions.getInstance().locate(context, TRCSourceInfo.class, LoadersUtils.getHashCode(sourceName));
	      if (sourceInfo == null) {
	      	if(isBooleanOptionEnabled(context,Constants.LLDATA_ENABLED))
	        		sourceInfo = TraceFactory.eINSTANCE.createTRCSourceInfoWithLLData();
	        	else
	        		sourceInfo = TraceFactory.eINSTANCE.createTRCSourceInfo();
	            sourceInfo.setLocation(sourceName);
	            sourceInfo.setLanguage(language);
	            p.getSourceInfos().add(sourceInfo);
	      }

	      if (theMethod.getSourceInfo() == null) {
	            theMethod.setSourceInfo(sourceInfo);
	      } else {
	            if (!theMethod.getSourceInfo().getLocation().equals(sourceName)) {
	                theMethod.setSourceInfo(sourceInfo);
	            }
	      }
	}
	
	public static boolean addLanguageIfRequired(String language, List languages) {
		if(languages == null || language == null)
			return false;
		for (int i = languages.size()-1; i >=0; i--) {
			if(languages.get(i).equals(languages))
				return false;
		}
		languages.add(language);
		return true;
	}

	public static TRCClass findClass(TRCProcess theProcess, String className, String packageName) {
		for (int i = theProcess.getPackages().size(); i-->0; ) {
			
			TRCPackage thePackage = (TRCPackage)theProcess.getPackages().get(i);
			if(thePackage.getName().equals(packageName))
			{
				for (int j = thePackage.getClasses().size(); j-->0; ) {
					TRCClass theClass = (TRCClass)thePackage.getClasses().get(j);
					if(theClass.getName().equals(className))
						return theClass;
				}
			}
		}
		return null;
	}
}
