/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLmonWaitedLoader.java,v 1.10 2010/05/05 04:12:04 jwest Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * Intel - loader is extended for contention analysis support  
 * 
 * $Id: XMLmonWaitedLoader.java,v 1.10 2010/05/05 04:12:04 jwest Exp $
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadDeadAndNotifyJoinedEvent;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadInterruptThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadNotifyAllEvent;
import org.eclipse.hyades.models.trace.TRCThreadNotifyEvent;
import org.eclipse.hyades.models.trace.TRCThreadRunningEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitTimeoutExceedEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForJoinEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForObjectEvent;
import org.eclipse.hyades.models.trace.TraceFactory;
import org.eclipse.hyades.models.trace.TracePackage;

/** XMLmonWaitedLoader handles the Monitor Waited event, which is sent when a thread finishes waiting 
 * on a monitor (e.g. the thread is now executing the code after the wait() call).
 * This can either be because it was notified, or because it timed out. */
public class XMLmonWaitedLoader extends XMLmonWaitLoader {
	
	protected static final String IS_TIMED_OUT = "isTimedOut";
	
	protected boolean isTimedOut = false;
	
	public void addAttribute(String name, String value) {
		switch (LoadersUtils.getHashCode(name)) {
			case TraceConstants.isTimedOut_int :
				//TODO: verify hash value
				isTimedOut = value.equals("1");
				break;
			default :
				super.addAttribute(name, value);
				break;
		}
	}

	
	public void addYourselfInContext() {
		if(context.isFilterOn())
		{
			if(context.getFilterEngine().isFiltered(TracePackage.eINSTANCE.getTRCThread_Name(),LoadersUtils.getLookUpKey(threadIdRef)))
			{
				return;
			}
		}
		
		// TODO MS - timeout is not used and has a different meaning here
		theProcess = getProcess();
		theThread = getThreadByIdRef(theProcess);
		dispatchProcessMode(ProcessSteps.ALL);
		
		// This event is what is added _after_ (in the list) the other thread event
		TRCThreadRunningEvent runningEvent = TraceFactory.eINSTANCE.createTRCThreadRunningEvent();
		runningEvent.setTime(createDeltaTime());//time);
		addAnnotationsIfRequired(runningEvent);

		if (isTimedOut) {
			// If it timed out, create the timeout event, then add the running event after the timeout
			TRCThreadWaitTimeoutExceedEvent timeoutEvent = TraceFactory.eINSTANCE.createTRCThreadWaitTimeoutExceedEvent();
			timeoutEvent.setTime(createDeltaTime());//time);
			theThread.getThreadEvents().add(timeoutEvent);
			theThread.getThreadEvents().add(runningEvent);
		}
		else {
			// This event did not timeout, so therefore we received another thread's notify/notifyAll (or, less likely, we were interrupted)
			
			theThread.getThreadEvents().add(runningEvent);

			ThreadEventsContext threadEventsContext = getThreadEventsContext();
			
			// Check for an interuption; if one exists in the map, then create a running event and add it
			TRCThreadInterruptThreadEvent interruptEvent = (TRCThreadInterruptThreadEvent)
				threadEventsContext.getThread2InterruptionMap().get(theThread);
			if (interruptEvent != null && canEventsBeBound(runningEvent, interruptEvent)) {
				interruptEvent.getRunningEvents().add(runningEvent);
				threadEventsContext.getThread2InterruptionMap().remove(theThread);
				return;
			}
			
			// Check for join/terminate (theObject is the monitor that the current thread was waiting on) 
			TRCThread threadObject = (TRCThread)threadEventsContext.getObject2ThreadMap().get(theObject);
			
			// If this thread was waiting on a monitor that corresponds to another thread's object, then
			// this thread was waiting on another thread e.g. it is a join scenario.
			
			// Does the current thread have an associated object?
			if (threadObject != null) {
				
				// Does the thread have an associated ThreadDeadAndNotifyJoinedEvent?
				TRCThreadDeadAndNotifyJoinedEvent deadNotifyEvent = (TRCThreadDeadAndNotifyJoinedEvent)threadEventsContext
					.getThread2ThreadDeadNotifyEventMap().get(threadObject);
				
				if (deadNotifyEvent != null) {
					if (canEventsBeBound(runningEvent, deadNotifyEvent)) {
						deadNotifyEvent.getRunningEvents().add(runningEvent);
						
						// Here we replace the existing TRCThreadWaitingForObjectEvent element, with 
						// a new TRCThreadWaitingForJoinEvent, and copy the values from the old one 
						// to the new one.

						// Find the last TRCThreadWaitingForObjectEvent in the list of events (starting at end of list)
						TRCThreadWaitingForObjectEvent lastWaitForObjEvent = null;
						int objectPos = -1;
						for(int x = theThread.getThreadEvents().size()-1; x >= 0 && (x >= (theThread.getThreadEvents().size()-3)); x--) {
							// We only search the last 3 events in the list, as we expect it to be in one of these places
							
							TRCThreadEvent te = theThread.getThreadEvents().get(x);
							if(te instanceof TRCThreadWaitingForObjectEvent) {
								lastWaitForObjEvent = (TRCThreadWaitingForObjectEvent)te;
								objectPos = x;
								break;
							}
						}
						
						// We replace the WaitingForObjectEvent with a WaitingForJoinEvent, because we know that 
						// this thread was in fact waiting for a join 
						
						if(objectPos != -1 && lastWaitForObjEvent != null &&
								theThread.getThreadEvents().size() - objectPos <= 3) {
							
							// We will only go back a maximum of 3 places in the thread event list to locate the TRCThreadWaitingForObjectEvent event
							TRCThreadWaitingForJoinEvent waitJoinEvent = TraceFactory.eINSTANCE.createTRCThreadWaitingForJoinEvent(); 

							TRCThreadWaitingForObjectEvent waitNotifyEvent = (TRCThreadWaitingForObjectEvent)theThread
								.getThreadEvents().set(objectPos, waitJoinEvent);
											
							waitJoinEvent.setObjectWaitingFor(waitNotifyEvent.getObjectWaitingFor());
							waitJoinEvent.setThread(waitNotifyEvent.getThread());
							waitJoinEvent.setTime(waitNotifyEvent.getTime());
							waitJoinEvent.setTimeout(waitNotifyEvent.getTimeout());
							
							waitJoinEvent.getAnnotations().addAll(waitNotifyEvent.getAnnotations());
							
						}
						
						return;
					}
				} 
			}
			
			
			//check for single notify
			List notifyEventList = (List)threadEventsContext
				.getMonitor2NotifyMap().get(theObject);
			
			if (notifyEventList != null) {
				// choose nearest event
				for(Iterator it = notifyEventList.iterator(); it.hasNext();) {
					TRCThreadNotifyEvent notifyEvent = (TRCThreadNotifyEvent) it.next();
					if (canEventsBeBound(runningEvent, notifyEvent)) {
						notifyEvent.getRunningEvents().add(runningEvent);
						notifyEventList.remove(notifyEvent);
						if (notifyEventList.isEmpty()) {
							threadEventsContext.getMonitor2NotifyMap().remove(theObject);
						}
						return;
					}			
				}
			}
			
			//check for notify all
			// iteration via all already bound running events required (no resume for thread two times)
			// previous wait (from which waited exit) event should happen before notifyAll with which we try to bind 
			TRCThreadNotifyAllEvent notifyAllEvent = (TRCThreadNotifyAllEvent)threadEventsContext
				.getMonitor2NotifyAllMap().get(theObject);
		
			if (notifyAllEvent != null) {
				double waitEventTime = ((TRCThreadEvent)theThread.getThreadEvents()
						.get(theThread.getThreadEvents().size() - 2)).getTime();
				
				if (waitEventTime < notifyAllEvent.getTime()) {
					for (Iterator it = notifyAllEvent.getRunningEvents().iterator(); it.hasNext();) {
						
						//bug 295395
						TRCThreadRunningEvent threadRE = (TRCThreadRunningEvent)it.next();
						TRCThread reThread = threadRE.getThread();
						if (reThread == null) continue;
						if (reThread.equals(theObject)) {
							// This thread already has been resumed by its notifyAll
							return;
						}
					}
				}
				if (canEventsBeBound(runningEvent, notifyAllEvent)) {
					notifyAllEvent.getRunningEvents().add(runningEvent);
				}
			}

		}

	}
}