/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TPFTestSuiteSerializer.java,v 1.12 2006/08/28 04:25:01 sleeloy Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.export.util.impl;

import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.Stack;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.common.common.CMNDefaultProperty;
import org.eclipse.hyades.models.common.common.CMNExtendedProperty;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionType;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFLoopEvent;
import org.eclipse.hyades.models.common.testprofile.TPFMessageEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTimedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTypedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictReason;
import org.eclipse.hyades.models.common.testprofile.TPFWaitEvent;
import org.eclipse.hyades.models.common.util.ExecutionUtil;
import org.w3c.dom.Element;

public class TPFTestSuiteSerializer extends XMLSerializer {

	public static final String RESPONSE_URL = "ResponseURL:"; //$NON-NLS-1$
	public static final String TIME = "Time:"; //$NON-NLS-1$
	public static final String LF = "\n"; //$NON-NLS-1$
    protected boolean onlyConsiderMostRecentResult = false;
    protected boolean constrainDates = true;
	
	protected int[] suiteVerdict;
	protected int[] overallVerdict;
	protected Stack containsVerdicts;

	protected long startTime = -1;
	protected long endTime = -1;
	
	/**
	 * Constructor that takes in a list of EMF objects (the source model) and an outputstream.
	 * @param content a list of EMF objects to serialize
	 * @param os the outputstream to serialize to
	 */
	public TPFTestSuiteSerializer(EList content, OutputStream os){
		super(content, os);
	}

	/**
	 * Constructor that takes in a list of EMF objects (the source model) and an outputstream.
	 * @param content a list of EMF objects to serialize
	 */
	public TPFTestSuiteSerializer(EList content){
		super(content);
	}
	
	/**
	 * Initialize serializer.
	 */
	protected void init(){
		try {
			dom = createDocument("testSuites"); //$NON-NLS-1$
			parent = dom.getDocumentElement();
			parentStack.push(parent);
			suiteVerdict = new int[4];
			overallVerdict = new int[4];
			containsVerdicts = new Stack();
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}		
	}
	
	protected void cleanup(){
			dom = null;
			parent = null;
			parentStack.clear();
			parentStack=null;
			suiteVerdict = new int[4];
			overallVerdict = new int[4];
			containsVerdicts.clear();
			containsVerdicts=null;
	}
	
	
	protected boolean isContainsVerdict(){
		return ((Boolean)containsVerdicts.peek()).booleanValue();
	}
	protected void setParentContainsVerdict(boolean flag){
		Object currentVerdict = containsVerdicts.pop();
		//pop parent verdict
		if (!containsVerdicts.empty()){
			containsVerdicts.pop();
			containsVerdicts.push(Boolean.valueOf(flag));
		}
		//push back previous verdict
		containsVerdicts.push(currentVerdict);
	}
	
	
	protected void setVerdict(boolean flag){
		containsVerdicts.push(Boolean.valueOf(flag));
	}
	
	protected void resetVerdict(){
		containsVerdicts.pop();
	}
	
    public boolean isOnlyConsiderMostRecentResult() {
		return onlyConsiderMostRecentResult;
	}

	public void setOnlyConsiderMostRecentResult(boolean onlyConsiderMostRecentResult) {
		this.onlyConsiderMostRecentResult = onlyConsiderMostRecentResult;
	}
	
	/**
	 * This method filters the map of TestSuites to ExecutionResults by 
	 * removing all but the last execution result that falls within the 
	 * stard and end ranges of the report window.
	 */
	private List filterOldExecutionResults(List results) {
			Iterator resultIter = results.iterator();
			TPFExecutionResult latestResult = null;
			long latestTimestamp = 0;
			while (resultIter.hasNext())
			{
				TPFExecutionResult result = (TPFExecutionResult) resultIter.next();
				TPFExecutionHistory history = result.getExecutionHistory();
				if ( history != null && !history.getExecutionEvents().isEmpty())
				{
					TPFExecutionEvent event = (TPFExecutionEvent) history.getExecutionEvents().get(0);
					long timestamp = event.getTimestamp();

					if (timestamp > latestTimestamp && dateWithinRange(timestamp))
					{
						latestResult = result;
						latestTimestamp = timestamp;
					}
				}
			}
			if (latestResult != null)
			{
				results.clear();
				results.add(latestResult);
			}
			return results;
		
	}	
	
	/**
	 * This method will serialize the contents of the EMF test model to a DOM representation.  After this method 
	 * is processed the DOM reprentation can be extracted using the getDOM method.
	 * 
	 * @see org.eclipse.hyades.test.tools.ui.common.internal.report.ISerializer#serialize()
	 */
	public void serialize(OutputStream out){
		serialize(out, null);	
	}
	public void serialize(OutputStream out, Properties properties){
		this.os = out;
		EList allTestSuites = buildTestSuiteList();
		List[] allExecutionResults = new ArrayList[allTestSuites.size()];
		// Prebuild an array of lists containing the execution results for each test suite
		for (int x = 0; x <allTestSuites.size(); x++){
			TPFTestSuite testSuite = (TPFTestSuite)allTestSuites.get(x);

			List executionResults = ExecutionUtil.findExecutionResults(testSuite, startTime, endTime);
            if ( isOnlyConsiderMostRecentResult() )
            	executionResults = filterOldExecutionResults(executionResults);

            allExecutionResults[x] = executionResults;
		}
		
		serialize(allTestSuites, allExecutionResults ,properties);
	
	}
	
	protected EList buildTestSuiteList(){
    	EList results = new BasicEList();
        Iterator iterator = content.iterator();
        while (iterator.hasNext()) {
            //- Find the all children testsuites given a list of test suites
                TPFTestSuite testSuite = (TPFTestSuite)iterator.next();
            	
                Iterator subIterator = findAllReferencedTestSuites(testSuite).iterator();
                while (subIterator.hasNext()) {
                    TPFTestSuite currentTestSuite = ((TPFTestSuite) (subIterator.next()));
                    
                    if ((currentTestSuite.getName()==null) || (currentTestSuite.getName().length()==0))
                       continue;
                    
                    if (!results.contains(currentTestSuite)) {
                    	results.add(currentTestSuite);
                    }
                }
        }
        return results;
		
	}


    /**
     * Recursively resolves all referenced test suites from the parameter root
     * test suite.
     * <p>
     * 
     * @param testSuite
     *            The root test suite.
     * @return The list of all recursively referenced test suites, including the
     *         root grandparent test suite.
     */
    public static List findAllReferencedTestSuites(TPFTestSuite testSuite) {
        List referencedTestSuites = new LinkedList();
        //Step 1. Add the parameter test suite:
        referencedTestSuites.add(testSuite);
        //Step 2. Add all the referenced test suite(s):
        Iterator referencedSuitesIterator = testSuite.getReferencedSuites().iterator();
        while (referencedSuitesIterator.hasNext()) {
            referencedTestSuites.addAll(findAllReferencedTestSuites(((TPFTestSuite) (referencedSuitesIterator.next()))));
        }
        return referencedTestSuites;
    }
    
	/**
	 * This method will serialize the contents of the EMF test model to a DOM representation.  After this method 
	 * is processed the DOM reprentation can be extracted using the getDOM method.
	 * @param constrainDates 
	 * 
	 * @see org.eclipse.hyades.test.tools.ui.common.internal.report.ISerializer#serialize()
	 */
	public void serialize(EList content, List[] allExecutionResults){
		serialize(content, allExecutionResults, null);
	}
	public void serialize(EList content, List[] allExecutionResults, Properties properties){
		
		init();

		for (int i = 0; i< overallVerdict.length ; i++)
			overallVerdict[i] = 0;
		int overallTests = 0;
		for (int x = 0; x <content.size(); x++){
			TPFTestSuite testSuite = (TPFTestSuite)content.get(x);
			beginTag(testSuite, true);
			
			int totalTests = ExecutionUtil.resolveTestCaseCount(testSuite);


			for (int i = 0; i< suiteVerdict.length ; i++)
				suiteVerdict[i] = 0;
			
			List executionResults = allExecutionResults[x];
			
	        TPFExecutionResult currentExecutionResult = null;

	        Iterator executionResultsIterator = executionResults.iterator();
	        while (executionResultsIterator.hasNext()) {
	            currentExecutionResult = ((TPFExecutionResult) (executionResultsIterator.next()));
	            EList executionEvents = currentExecutionResult.getExecutionHistory().getExecutionEvents();
	            if (executionEvents.size()>0) {
		        	setVerdict(false);
	                long currentTimeStamp = ((TPFExecutionEvent) executionEvents.get(0)).getTimestamp();
	                if (dateWithinRange(currentTimeStamp)) {
	                    currentTimeStamp = ((TPFExecutionEvent) (executionEvents.get(executionEvents.size() - 1))).getTimestamp();
	                    if (dateWithinRange(currentTimeStamp)) {

	                    	beginTag(currentExecutionResult, true);

	                    	recurseGenerate(executionEvents, true);
	            	            
	        				parent.setAttribute("timestamp", Long.toString(currentTimeStamp));
	        				parent.setAttribute("totalError", Integer.toString(suiteVerdict[0]));
	        				parent.setAttribute("totalFail", Integer.toString(suiteVerdict[1]));
	        				parent.setAttribute("totalInconclusive", Integer.toString(suiteVerdict[2]));
	        				parent.setAttribute("totalPass", Integer.toString(suiteVerdict[3]));
                            int totalAttempted = (suiteVerdict[0] + suiteVerdict[1] + suiteVerdict[2] + suiteVerdict[3]);
                            if("AllJUnitTests".equals(testSuite.getName()) || (totalAttempted>totalTests)) {
                            	// Can't calculate the total from this test suite directly because 
                            	// in this case the total should include the total of all of the 
                            	// sub-testsuites as well. 
                            	totalTests = totalAttempted;
                            }	        				
	        				parent.setAttribute("totalTests", Integer.toString(totalTests));
	        				overallTests += totalTests;
	                    	endTag(currentExecutionResult, true);
	                    }
	                }
		        	resetVerdict();	                
					for (int i = 0; i< overallVerdict.length ; i++){
						overallVerdict[i] += suiteVerdict[i];
					}
	            }
	        }

			if (overallTests == 0){
				overallTests = totalTests;
			}
			parent.setAttribute("totalTests", Integer.toString(totalTests));
			
			endTag(testSuite, true);
		}		
		parent.setAttribute("startTime", Long.toString(startTime));
		parent.setAttribute("endTime", Long.toString(endTime));
		parent.setAttribute("totalError", Integer.toString(overallVerdict[0]));
		parent.setAttribute("totalFail", Integer.toString(overallVerdict[1]));
		parent.setAttribute("totalInconclusive", Integer.toString(overallVerdict[2]));
		parent.setAttribute("totalPass", Integer.toString(overallVerdict[3]));
		parent.setAttribute("totalTests", Integer.toString(overallTests));
		
		//need to add this information in a element since BIRT needs it formatted this way.
		Element verdictProperty = dom.createElement("verdictProperty"); //$NON-NLS-1$
		if (properties == null)
			verdictProperty.setAttribute("name", "inconclusive"); //$NON-NLS-1$ //$NON-NLS-2$
		else
			verdictProperty.setAttribute("name", properties.getProperty("inconclusive")); //$NON-NLS-1$ //$NON-NLS-2$
		verdictProperty.setAttribute("value", Integer.toString(overallVerdict[2]));
		parent.appendChild(verdictProperty);
		verdictProperty = dom.createElement("verdictProperty"); //$NON-NLS-1$
		if (properties == null)
			verdictProperty.setAttribute("name", "pass");//$NON-NLS-1$ //$NON-NLS-2$
		else
			verdictProperty.setAttribute("name", properties.getProperty("pass"));//$NON-NLS-1$ //$NON-NLS-2$
		verdictProperty.setAttribute("value", Integer.toString(overallVerdict[3]));
		parent.appendChild(verdictProperty);
		verdictProperty = dom.createElement("verdictProperty"); //$NON-NLS-1$
		if (properties == null)
			verdictProperty.setAttribute("name", "fail");//$NON-NLS-1$ //$NON-NLS-2$
		else
			verdictProperty.setAttribute("name", properties.getProperty("fail"));//$NON-NLS-1$ //$NON-NLS-2$
		verdictProperty.setAttribute("value", Integer.toString(overallVerdict[1]));
		parent.appendChild(verdictProperty);
		verdictProperty = dom.createElement("verdictProperty"); //$NON-NLS-1$
		if (properties == null)
			verdictProperty.setAttribute("name", "error");//$NON-NLS-1$ //$NON-NLS-2$
		else
			verdictProperty.setAttribute("name", properties.getProperty("error"));//$NON-NLS-1$ //$NON-NLS-2$
		verdictProperty.setAttribute("value", Integer.toString(overallVerdict[0]));
		parent.appendChild(verdictProperty);
		
		int verdictValue = TPFVerdict.PASS;
        if (overallVerdict[3] > 0) {
        	verdictValue = TPFVerdict.ERROR;
        }
        if (overallVerdict[2] > 0) {
        	verdictValue = TPFVerdict.FAIL;
        }
        if (overallVerdict[1] > 0) {
        	verdictValue=  TPFVerdict.INCONCLUSIVE;
        }
		parent.setAttribute("verdictValue", Integer.toString(verdictValue));
        
        
		try {
			serializeGeneratedDocument();
//			System.out.println(serializeGeneratedDocumentToString());
		} catch (Exception e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		cleanup();
		
	}

	private boolean dateWithinRange(long currentTimeStamp) {
		// A date is within the start and end range if either
		// we are not constraining the date range or
		// the date is within the start and end range 
		return (!isConstrainDates() || (currentTimeStamp >= startTime) && (currentTimeStamp <= endTime));
	}
	
	/**
	 * Helper function that processes the list of elements during serialization.
	 * @param list
	 */
	protected void recurseGenerate(EList list, boolean root){
        Iterator executionEventsIterator = list.iterator();
        while (executionEventsIterator.hasNext()) {
        	TPFExecutionEvent executionEvent = (TPFExecutionEvent) executionEventsIterator.next();
        	
			//search for child objects			
			if (executionEvent instanceof TPFInvocationEvent)
			{
	    		TPFExecutionResult invokedExecutionResult = ((TPFInvocationEvent)executionEvent).getInvokedExecutionResult();
				
	            EList executionEvents = invokedExecutionResult.getExecutionHistory().getExecutionEvents();
	            if (executionEvents.size()>0) {
	            	setVerdict(false);
	                long currentTimeStamp = ((TPFExecutionEvent) executionEvents.get(0)).getTimestamp();
	                if (dateWithinRange(currentTimeStamp)) {
	                    currentTimeStamp = ((TPFExecutionEvent) (executionEvents.get(executionEvents.size() - 1))).getTimestamp();
	                    if (dateWithinRange(currentTimeStamp)) {
	        				EList rList = ((TPFInvocationEvent)executionEvent).getInvokedExecutionResult()
							.getExecutionHistory().getExecutionEvents();
	        				beginTag(executionEvent, root);
	        				if (!rList.isEmpty()){
	        					recurseGenerate(rList, false);
	        				}
	        				endTag(executionEvent, root);
	                    }
	                }
		        	resetVerdict();	                
	            }
	    		
			}
			else if (executionEvent instanceof TPFExecutionEvent)
			{
				EList rList = ((TPFExecutionEvent)executionEvent).getChildren();
	        	beginTag(executionEvent, root);
				if (!rList.isEmpty()){
					recurseGenerate(rList, false);
				}
            	endTag(executionEvent,root);					
			}			
			else{
            	beginTag(executionEvent, root);
            	endTag(executionEvent, root);					
			}
        }		
	}

	/**
	 * Constructs an XML element based on an object
	 * @param element object that is used to construct the XML element
	 * @return XML element representation of the object provided
	 */	
	protected Element createElement(Object element, boolean root){
		Element child = null;
		if (element instanceof TPFTestSuite){
			TPFTestSuite testSuite = ((TPFTestSuite)element);
			child = dom.createElement("testsuite"); //$NON-NLS-1$
			setAttribute(child, "name", testSuite.getName()); //$NON-NLS-1$
			setAttribute(child, "resource", testSuite.getResource()); //$NON-NLS-1$
			setAttribute(child, "location", testSuite.getLocation()); //$NON-NLS-1$
			setAttribute(child, "type", testSuite.getType()); //$NON-NLS-1$
			setAttribute(child, "id", testSuite.getId()); //$NON-NLS-1$
			
			if (testSuite.getDescription() != null){
				Element description = dom.createElement("description"); //$NON-NLS-1$
				child.appendChild(description);
				appendTextNode(description, testSuite.getDescription());
			}
			
		}
		else if (element instanceof TPFTypedEvent){
			TPFTypedEvent event = ((TPFTypedEvent)element);
			child = dom.createElement("typedEvent"); //$NON-NLS-1$
			if (event.getType() != null)
				setAttribute(child, "type", Integer.toString(event.getType().getValue())); //$NON-NLS-1$
			if(event.getType()==TPFExecutionType.STOP_LITERAL)
			{
				setAttribute(child, "stopTime", Long.toString(event.getTimestamp())); //$NON-NLS-1$
			}
			else if(startTime==-1)
			{
				setAttribute(child, "startTime", Long.toString(event.getTimestamp())); //$NON-NLS-1$
			}			
		}		
		else if (element instanceof TPFLoopEvent){
			TPFLoopEvent event = ((TPFLoopEvent)element);
			child = dom.createElement("loopEvent"); //$NON-NLS-1$
			setAttribute(child, "iterations", Long.toString(event.getIterations())); //$NON-NLS-1$
			setAttribute(child, "asynchronous", Boolean.toString(event.isAsynchronous())); //$NON-NLS-1$
		}		
		else if (element instanceof TPFVerdictEvent){
			TPFVerdictEvent event = ((TPFVerdictEvent)element);
			child = dom.createElement("verdictEvent"); //$NON-NLS-1$
			TPFVerdict verdict = event.getVerdict();
			setAttribute(child, "verdictName", verdict.getName()); //$NON-NLS-1$
			setAttribute(child, "verdictValue", Integer.toString(verdict.getValue())); //$NON-NLS-1$			
			TPFVerdictReason verdictReason= event.getReason();
			setAttribute(child, "verdictReasonName", verdictReason.getName()); //$NON-NLS-1$
			setAttribute(child, "verdictReasonValue", Integer.toString(verdictReason.getValue())); //$NON-NLS-1$	

			//should not count the root verdict in the count
			if (!root){
				setParentContainsVerdict(true);
				if (!isContainsVerdict()){
					switch (verdict.getValue()) {
			            case TPFVerdict.ERROR:
			            	suiteVerdict[0]++;
			            	break;
			            case TPFVerdict.FAIL:
			            	suiteVerdict[1]++;
			                break;
			            case TPFVerdict.INCONCLUSIVE:
			            	suiteVerdict[2]++;
			                break;
			
			            case TPFVerdict.PASS:
			            	suiteVerdict[3]++;
			                break;
		            }
				}
			}
			else{
				parent.setAttribute("verdictValue", Integer.toString(verdict.getValue()));
			}
			
		}		
		else if (element instanceof TPFTimedEvent){
			TPFTimedEvent event = ((TPFTimedEvent)element);
			if (event instanceof TPFWaitEvent)
				child = dom.createElement("waitEvent"); //$NON-NLS-1$
			else
				child = dom.createElement("timedEvent"); //$NON-NLS-1$
			setAttribute(child, "endTimestamp", Long.toString(event.getEndTimestamp())); //$NON-NLS-1$
		}		
		else if (element instanceof TPFMessageEvent){
			TPFMessageEvent event = ((TPFMessageEvent)element);
			child = dom.createElement("messageEvent"); //$NON-NLS-1$
			setAttribute(child, "severity", Integer.toString(event.getSeverity().getValue())); //$NON-NLS-1$
			setAttribute(child, "severityLabel", event.getSeverity().getLabel()); //$NON-NLS-1$
			String message = event.getText();
			if (message != null){
			
				int responseTimeIndex = message.indexOf(TIME); 
				if(responseTimeIndex!=-1)
				{
					String pageName = message.substring(message.indexOf(RESPONSE_URL)+RESPONSE_URL.length(),responseTimeIndex).trim();
					String responseTime = message.substring(responseTimeIndex+TIME.length(),message.length());
					responseTime=responseTime.substring(0,responseTime.indexOf(LF)).trim();
					setAttribute(child, "pageName", pageName); //$NON-NLS-1$
					setAttribute(child, "responseTime", responseTime);					 //$NON-NLS-1$
				}
			}
		}
		else if (element instanceof TPFInvocationEvent){
			TPFInvocationEvent event = ((TPFInvocationEvent)element);
    		TPFExecutionResult invokedExecutionResult = ((TPFInvocationEvent)event).getInvokedExecutionResult();
            // use the verdict directly from execution result of a invocation event.
			
			child = dom.createElement("invocationEvent"); //$NON-NLS-1$
			
			TPFVerdict verdict = invokedExecutionResult.getVerdict();
			setAttribute(child, "verdictName", verdict.getName()); //$NON-NLS-1$
			setAttribute(child, "verdictValue", Integer.toString(verdict.getValue())); //$NON-NLS-1$			
			setAttribute(child, "statusName", event.getStatus().getName()); //$NON-NLS-1$
			setAttribute(child, "statusValue", Integer.toString(event.getStatus().getValue())); //$NON-NLS-1$			
		}		
		else if (element instanceof TPFExecutionResult){
			TPFExecutionResult results = ((TPFExecutionResult)element);
			child = dom.createElement("testExecutionResults"); //$NON-NLS-1$
			setAttribute(child, "name", results.getName()); //$NON-NLS-1$
			setAttribute(child, "version", results.getTestVersion()); //$NON-NLS-1$
			setAttribute(child, "type", results.getType()); //$NON-NLS-1$
			if (results.getDescription() != null){
				Element description = dom.createElement("description"); //$NON-NLS-1$
				child.appendChild(description);
				appendTextNode(description, results.getDescription());
			}
//			setAttribute(child, "name", ((TPFTestSuite)element).getName());
		}
		
		if (element instanceof TPFExecutionEvent){
			TPFExecutionEvent executionEvent = (TPFExecutionEvent)element;
			if (child == null)
				child = dom.createElement("testExecutionEvents"); //$NON-NLS-1$
			
			setAttribute(child, "name", executionEvent.getName()); //$NON-NLS-1$
			setAttribute(child, "eventType", executionEvent.getEventType()); //$NON-NLS-1$
			setAttribute(child, "timestamp", Long.toString(executionEvent.getTimestamp())); //$NON-NLS-1$
			if (executionEvent.getDescription() != null){
				Element description = dom.createElement("description"); //$NON-NLS-1$
				child.appendChild(description);
				appendTextNode(description, executionEvent.getDescription());
			}
			String message = executionEvent.getText();
			if (message != null){
				Element description = dom.createElement("text"); //$NON-NLS-1$
				child.appendChild(description);
				appendTextNode(description, executionEvent.getText());
			}
			Iterator iter = executionEvent.getProperties().iterator();
			Element propertiesElement = dom.createElement("properties"); //$NON-NLS-1$
			child.appendChild(propertiesElement);
			while (iter.hasNext()){				
				CMNDefaultProperty listElement = (CMNDefaultProperty)iter.next();
				//currently only support extended properties
				if (listElement instanceof CMNExtendedProperty){
					CMNExtendedProperty property = (CMNExtendedProperty)element;
					Element popertyElement = dom.createElement("property"); //$NON-NLS-1$
					propertiesElement.appendChild(popertyElement);
					setAttribute(popertyElement, "name", property.getName()); //$NON-NLS-1$
					setAttribute(popertyElement, "type", property.getType()); //$NON-NLS-1$
					appendTextNode(popertyElement, property.getValue());					
				}
				
			}
		}
		return child;
	}

	public long getEndTime() {
		return endTime;
	}

	public void setEndTime(long endTime) {
		this.endTime = endTime;
	}

	public long getStartTime() {
		return startTime;
	}

	public void setStartTime(long startTime) {
		this.startTime = startTime;
	}

	public boolean isConstrainDates() {
		return constrainDates;
	}

	public void setConstrainDates(boolean constrainDates) {
		this.constrainDates = constrainDates;
	}
	
}
