/**********************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLSerializer.java,v 1.7 2009/01/09 12:33:16 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.common.export.util.impl;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Stack;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.common.export.util.ISerializer;
import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

public abstract class XMLSerializer implements ISerializer {

	/**
	 * Outputstream used to serialize the model
	 */
	protected OutputStream os;
	/**
	 * List of EMF objects to serialize
	 */
	protected EList content;
	/**
	 * DOM object
	 */
	protected Document dom;
	/**
	 * Holds the parent element of the element that is being serialized
	 */
	protected Element parent = null;
	/**
	 * Holds the element parent stack
	 */
	protected Stack parentStack = new Stack();
	
	/**
	 * Used indent value
	 */
	protected int indent = 4;
	
	/**
	 * Determines if the xml output should be formatted
	 */
	protected boolean format = false;

	/**
	 * Constructor that takes a list of EMF objects and the outputstream to serialize the list of EMF objects.
	 * 
	 * @param content list of EMF object to serialize
	 * @param os output stream used to serialize the list of objects
	 */
	public XMLSerializer(EList content, OutputStream os){
		this.content = content;
		this.os = os;		
	}

	/**
	 * Constructor that takes a list of EMF objects and the outputstream to serialize the list of EMF objects.
	 * 
	 * @param content list of EMF object to serialize
	 */
	public XMLSerializer(EList content){
		this.content = content;
	}
		
		
	/**
	 * Constructs the XML begin tag given an object.
	 * 
	 * @param element the object to process
	 */
	protected void beginTag(Object element, boolean root){
		Element child = createElement(element ,root);
		if (child != null){
			parent.appendChild(child);
			parent = child;
			parentStack.push(parent);
		}
	}
	
	/**
	 * Constructs an XML element based on an object
	 * @param element object that is used to construct the XML element
	 * @return XML element representation of the object provided
	 */
	protected abstract Element createElement(Object element, boolean root);
	

	/**
	 * Processes the element after serialization.
	 * 
	 * @param element the element to process
	 */
	protected void endTag(Object element, boolean root){
		
		parentStack.pop();
		parent = (Element)parentStack.peek();
	}
	
	/**
	 * Creates an XML document and creates a root element with the specified tag name..
	 * 
	 * @param tag the name of the root element
	 * @return Document the XML document
	 * @throws Exception
	 */
	protected Document createDocument(String tag) throws Exception {
		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		DocumentBuilder builder;
		builder = factory.newDocumentBuilder();
		DOMImplementation domImpl = builder.getDOMImplementation();
		Document document = domImpl.createDocument(null, tag, null);
		return document;
	}
	
	/**
	 * Serializes the DOM to a String.
	 * @return the string representation of the DOM
	 * @throws Exception an error occurs during serialization
	 */
	public String serializeGeneratedDocumentToString() throws Exception {
		if (dom == null) {
			return null;
		}
		OutputStreamWriter writer = null;
		ByteArrayOutputStream stream = new ByteArrayOutputStream();

		writer = new OutputStreamWriter(stream, "UTF-8");
		DOMSource source = new DOMSource(dom);
		StreamResult result = new StreamResult(writer);
		

		TransformerFactory transFactory = TransformerFactory.newInstance();
		Transformer transformer = transFactory.newTransformer();
		
		//Determine if we should format the output
		if (format){
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");  
			try {
				transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
			} catch (IllegalArgumentException e) {
				//Don't indent if the parser does not support this property
			}
		}

		transformer.transform(source, result);

		return stream.toString();

	}   	
	/**
	 * Serializes the DOM to a Stream.
	 * @throws Exception an error occurs during serialization
	 */
	public void serializeGeneratedDocument() throws Exception {
		if (dom == null) {
			return;
		}
		OutputStreamWriter writer = null;

		writer = new OutputStreamWriter(os, "UTF-8");
		DOMSource source = new DOMSource(dom);
		StreamResult result = new StreamResult(writer);
		

		TransformerFactory transFactory = TransformerFactory.newInstance();
		Transformer transformer = transFactory.newTransformer();
		
		//Determine if we should format the output
		if (format){
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");  
			try {
				transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
			} catch (IllegalArgumentException e) {
				//Don't indent if the parser does not support this property
			}
		}
		transformer.transform(source, result);


	}   	

	/**
	 * Constructs an XML attribute on the element provided.
	 * 
	 * @param owner element to add the attribute to
	 * @param attName the name of the attribute
	 * @param attValue the value of the attribute
	 */
	protected void setAttribute(Element owner, String attName, String attValue){
		if (attValue != null)
			owner.setAttribute(attName, attValue);
	}
	
	/**
	 * Constructs a text node on the element provided.
	 * @param owner the element to add the text node
	 * @param text the content of the text node
	 */
	protected void appendTextNode(Element owner, String text){
		if (text != null){
			Text textNode = dom.createTextNode(decode(text));
			owner.appendChild(textNode);
		}
	}
	
	/**
     * Replace few xml entity '&xxx;' to their char value in string.
     * @return modified string, or s directly.
	 */
	public static String decode(String s)
	{
		if (s == null) return s;
		s = s.replaceAll("&apos;", "'");
		s = s.replaceAll("&lt;", "<");
		s = s.replaceAll("&gt;", ">");
		s = s.replaceAll("&quot;", "\"");
		s = s.replaceAll("&amp;","&");
		return s;
	}   

	/**
	 * Contructs an XML element and adds the element to the current parent element.
	 * @param tag the tag name of the element
	 */
	protected void beginTag(String tag, boolean root){
		
		Element child = dom.createElement(tag);		
		if (child != null){
			parent.appendChild(child);
			parent = child;
			parentStack.push(parent);
		}
	}
	
	/**
	 * Processes the element after serialization.
	 * 
	 * @param tag the tag name of the element
	 */
	protected void endTag(String tag, boolean root){
		parentStack.pop();
		parent = (Element)parentStack.peek();		
	}

	/**
	 * Returns the dom that contains the XML model.
	 * 
	 * @return the xml DOM model
	 */
	public Document getDom() {
		return dom;
	}

	public boolean isFormat() {
		return format;
	}

	public void setFormat(boolean format) {
		this.format = format;
	}

	public int getIndent() {
		return indent;
	}

	public void setIndent(int indent) {
		this.indent = indent;
	}
	
}
