/*******************************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FacadeResourceImpl.java,v 1.17 2010/03/17 14:45:47 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.facades.behavioral.impl;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.URIConverter;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.hyades.models.common.common.CMNAnnotation;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.facades.behavioral.IAnnotatableResource;
import org.eclipse.hyades.models.common.fragments.Common_Behavior_FragmentsPackage;
import org.eclipse.hyades.models.common.interactions.Common_Behavior_InteractionsPackage;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.util.FileUtil;
import org.eclipse.hyades.models.hierarchy.util.HierarchyURIConverterImpl;

/**
 * <p>FacadeResourceImpl.java</p>
 * 
 * 
 * @author  Joseph P. Toomey
 * @author  Paul E. Slauenwhite
 * @version March 17, 2010
 * @since   January 25, 2005
 */
public class FacadeResourceImpl extends XMIResourceImpl implements IAnnotatableResource {

	private File annotationDir = null;

	// These three static members are used to manage the temporary directories and
	// files created in support of annotation based attachments.  Previous attempts
	// to clean up these temporary files and directories using finalization, explicit
	// cleanup calls by consumers and File.deleteOnExit() mechanisms have all proved 
	// inadequate.  We now register a single static shutdown hook (registered by the
	// first constructor call to this class and synchronized on the class object) 
	// and we maintain a list of annotation directories to be cleaned up.  If the
	// user unloads the model or calls the cleanup() method explicitly, we will
	// clean up that model's annotation directory and remove it from the list
	// (in an effort to reduce the work done at shutdown time.)
	private static List annotationDirectories = new ArrayList();
	private static boolean shutdownHookRegistered = false;
	private static Thread shutdownThread = new Thread() {
		public void run() {
			Iterator it = annotationDirectories.iterator();
			while (it.hasNext()) {
				File dir = (File) it.next();
				if (dir != null && dir.exists()) {
					dir = dir.getParentFile();
					if (dir != null && dir.exists()) {
						FileUtil.deltree(dir);
					}
				}
			}
		}
	};
	
	/**
	 * 
	 */
	public FacadeResourceImpl() {
		super();
		registerCleanupHook();
	}

	/**
	 * @param arg0
	 */
	public FacadeResourceImpl(URI arg0) {
		super(arg0);
		registerCleanupHook();
	}

	private void registerCleanupHook() {
		// Synchronize on the class to ensure that we don't register multiple
		// shutdown hooks.
		synchronized (FacadeResourceImpl.class) {
			if (!shutdownHookRegistered) {
				Runtime.getRuntime().addShutdownHook(shutdownThread);
				shutdownHookRegistered = true;
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#attached(org.eclipse.emf.ecore.EObject)
	 */
	public void attached(EObject arg0) {
		super.attached(arg0);

		// Map this object's ID to the object in the ID to object map.
		String id = EcoreUtil.getID(arg0);
    	if (id != null)
    	{
    		getIDToEObjectMap().put(id, arg0);
    	}
				
		if(arg0.eClass().equals(Common_TestprofilePackage.eINSTANCE.getTPFTestSuite()))
		{
			HyadesCommon_TestprofileAdapterFactory adaptorFactory = new HyadesCommon_TestprofileAdapterFactory();
			adaptorFactory.adapt(arg0, Common_TestprofilePackage.eINSTANCE.getTPFTestSuite()); 
		}
		if(arg0.eClass().equals(Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteraction()))
		{
			HyadesCommon_Behavior_FragmentsAdapterFactory adaptorFactory = new HyadesCommon_Behavior_FragmentsAdapterFactory();
			adaptorFactory.adapt(arg0, Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteraction()); 
		}
		if(arg0.eClass().equals(Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteractionOperand()))
		{
			HyadesCommon_Behavior_FragmentsAdapterFactory adaptorFactory = new HyadesCommon_Behavior_FragmentsAdapterFactory();
			adaptorFactory.adapt(arg0, Common_Behavior_FragmentsPackage.eINSTANCE.getBVRInteractionOperand());
		}
		if(arg0.eClass().equals(Common_TestprofilePackage.eINSTANCE.getTPFTestComponent()))
		{
			HyadesCommon_TestprofileAdapterFactory adaptorFactory = new HyadesCommon_TestprofileAdapterFactory();
			adaptorFactory.adapt(arg0, Common_TestprofilePackage.eINSTANCE.getTPFTestComponent()); 
		}
		
		if(arg0.eClass().equals(Common_Behavior_InteractionsPackage.eINSTANCE.getBVRExecutionOccurrence()))
		{
			TptpCommon_BVRExecutionOccurrenceAdapterFactory adaptorFactory = new TptpCommon_BVRExecutionOccurrenceAdapterFactory();
			adaptorFactory.adapt(arg0, Common_Behavior_InteractionsPackage.eINSTANCE.getBVRExecutionOccurrence());
		}
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#detached(org.eclipse.emf.ecore.EObject)
	 */
	public void detached(EObject arg0) {
		super.detached(arg0);
		arg0.eAdapters().clear();
		
		// Remove this object from the ID to object map.
		String id = EcoreUtil.getID(arg0);
    	if (id != null)
    	{
    		getIDToEObjectMap().remove(id);
    	}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doLoad(java.io.InputStream, java.util.Map)
	 */
	public void doLoad(InputStream inputStream, Map options) throws IOException {

		//bugzilla 130574
		if(options == null)
			options = new HashMap();
		options.put(XMLResource.OPTION_RECORD_UNKNOWN_FEATURE, Boolean.TRUE);
		
/*		try {
				if ( inputStream.available() == 0 )
					return;
			}
			catch (IOException ex)
			{
				return;
			}
*/	
		if ( inputStream instanceof ZipInputStream )
		{
			ZipInputStream zipIn = (ZipInputStream) inputStream;
			
			// Read the ResourceContents into a temporary file so we can call the 
			// super.doLoad() after we read out any annotations
			File tempResourceContents = File.createTempFile("tempResource", "");
			tempResourceContents.deleteOnExit();
			try {
				FileOutputStream file = new FileOutputStream(tempResourceContents);
				BufferedOutputStream bufFile = new BufferedOutputStream(file);

				int bufSize = 65536;
				byte[] readBuffer = new byte[bufSize];
				int count;
				while ((count = zipIn.read(readBuffer, 0, bufSize)) != -1) {
					bufFile.write(readBuffer, 0, count);
				}
				bufFile.close();

				ZipEntry entry = zipIn.getNextEntry();
				if ( entry != null )
				{
					// We have at least one annotation file.
					URI baseAnnotationDirURI = URI.createFileURI(getAnnotationDir().getAbsolutePath());
					while ( entry != null )
					{
						String entryName = entry.getName();
						URI relativeURI = URI.createURI(entryName);
						URI fileURI = relativeURI.resolve(baseAnnotationDirURI);
						File dest = new File(fileURI.toFileString()); 
						if ( !(dest.getParentFile().exists()))
						{
							dest.getParentFile().mkdirs();
						}
						file = new FileOutputStream(fileURI.toFileString());
						bufFile = new BufferedOutputStream(file);
						
						while ((count = zipIn.read(readBuffer, 0, bufSize)) != -1) {
							bufFile.write(readBuffer, 0, count);
						}
						bufFile.close();
						entry = zipIn.getNextEntry();
					}
				}
				FileInputStream fileIn = new FileInputStream(tempResourceContents);
				BufferedInputStream bufIn = new BufferedInputStream(fileIn);
				super.doLoad(bufIn, options);
				bufIn.close();
			}
			finally {
				tempResourceContents.delete();
			}
		}
		else
			super.doLoad(inputStream, options);
		
	}
	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doSave(java.io.OutputStream, java.util.Map)
	 */
	public void doSave(OutputStream outputStream, Map options)
			throws IOException {
		
		super.doSave(outputStream, options);
		
		if ( hasAnnotations() && outputStream instanceof ZipOutputStream)
		{
			ZipOutputStream zipOut = (ZipOutputStream)outputStream;
			File annotationDir = getAnnotationDir();
			URI baseAnnotationDirURI = URI.createFileURI(annotationDir.getAbsolutePath());			
			File[] annotations = FileUtil.listAllFiles(annotationDir);
			if ( annotations == null )
				return;
			
			for ( int i=0; i<annotations.length; i++)
			{
				URI currentFile = URI.createFileURI(annotations[i].getAbsolutePath());
				URI relativeURI = currentFile.deresolve(baseAnnotationDirURI);
				FileInputStream in = new FileInputStream(annotations[i]);
				ZipEntry entry = new ZipEntry(relativeURI.toString());
				entry.setMethod(ZipEntry.STORED);
				entry.setSize(annotations[i].length());
				
				// This stinks, but in order to copy a file into a zipoutputstream
				// using the STORED method (meaning, please don't try to compress this),
				// you need to precalculate the CRC, which means you need to read the
				// entire file twice (unless you want to hold the whole thing in memory,
				// which we don't want to do...)
				CRC32 crc = new CRC32();
				BufferedInputStream contentStream = new BufferedInputStream(in);

				// 512k buffer for reading and writing
				int bufSize = 1024 * 512;
				byte[] readBuffer = new byte[bufSize];
				int count;
				while ((count = contentStream.read(readBuffer, 0, bufSize)) != -1) {
					crc.update(readBuffer, 0, count);
				}
				contentStream.close();
				entry.setCrc(crc.getValue());
				zipOut.putNextEntry(entry);

				in = new FileInputStream(annotations[i]);
				contentStream = new BufferedInputStream(in);
				while ((count = contentStream.read(readBuffer, 0, bufSize)) != -1) {
					zipOut.write(readBuffer, 0, count);
				}
				zipOut.closeEntry();
				contentStream.close();
			}
		}
	}
	
	public boolean hasAnnotations()
	{
		return (annotationDir != null);
	}
	

	public synchronized File getAnnotationDir() throws IOException
	{
		// This method is synchronized to provide atomicity of the test & set
		// of the anotationDir
		if ( annotationDir != null )
		{
			return annotationDir;
		}
		
		File systemTempDir = FileUtil.getTempDir();
		File temp = File.createTempFile("Hyades.TestModel.AnnotationDir", "", systemTempDir);
		if ( temp.delete() && temp.mkdir() )
		{
			annotationDir = new File(temp, "annotations");
			if ( !annotationDir.mkdir() )
				throw new IOException("Unable to create new annotation Dir " + annotationDir.getAbsolutePath());
		}
		else
		{
			throw new IOException("Unable to create new annotation Dir " + temp.getAbsolutePath());
		}
		
		annotationDirectories.add(annotationDir);
		return annotationDir;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#finalize()
	 */
	protected void finalize() throws Throwable {
		super.finalize();
	}
	
	/**
	 * This method simply calls cleanupAnnotationDir today, but may do
	 * more cleanup in the future
	 */
	public void cleanup()
	{
		cleanupAnnotationDir();
	}
	
	private void cleanupAnnotationDir()
	{
		if ( hasAnnotations() )
		{
			// only remove this directory from the list of cleanup directories
			// if the delete succeeds (consumers may not have closed the necessary files.)
			boolean deleted = FileUtil.deltree(annotationDir.getParentFile()); 
			if (deleted) {
				annotationDirectories.remove(annotationDir);
			}

			annotationDir = null;
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#doUnload()
	 */
	protected void doUnload() {
		super.doUnload();
		cleanupAnnotationDir();
		annotationDir = null;
	}
	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#getEObjectByID(java.lang.String)
	 */
	protected EObject getEObjectByID(String id) {
	    if (idToEObjectMap != null)
	    {
	      EObject eObject = (EObject) idToEObjectMap.get(id);
	      if (eObject != null)
	      {
	        return eObject;
	      }
	    }
	    return null;
	}


	/* (non-Javadoc)
	 * @see org.eclipse.emf.ecore.resource.impl.ResourceImpl#getURIConverter()
	 * We need to replace the default URIConverter, because EMF's URIConverter
	 * uses a ByteArrayOutputStream when saving the model, which requires that
	 * the entire model will fit into a single byte array.  The JDK limitation
	 * on the numner of elements in an array thus limits us to at most 256MB,
	 * which is not an acceptable limitation.
	 */
	protected URIConverter getURIConverter() {
		return new HierarchyURIConverterImpl();
	}

	public URI getFileAnnotation(CMNAnnotation annotationElement) throws IOException {
		String uriString = annotationElement.getURI();

		// Get the directory for all annotations on this entire resource 
		File annotationDir = getAnnotationDir();
		
		URI baseAnnotationDirURI = URI.createFileURI(annotationDir.getAbsolutePath());
		URI annotation = URI.createURI(uriString);
		annotation = queryURItoRelativeURI(annotation);
		URI resolvedURI = annotation.resolve(baseAnnotationDirURI);
		
		return resolvedURI;
	}

	public boolean isFileAnnotation(CMNAnnotation annotationElement) {
		String uriString = annotationElement.getURI();
		if ( uriString == null || uriString.length() == 0 )
			return false;
		
		URI uri = URI.createURI(uriString);
		uri = queryURItoRelativeURI(uri);
		if ( uri.isFile() )
			return true;
		else
			return false;
	}

	public void putFileAnnotation(URI file, CMNAnnotation annotationElement) throws IOException{
		// Get the model GUID of the container object. 
		String containerID = getContainerID(annotationElement);
		
		// Get the directory for all annotations on this entire resource 
		File annotationDir = getAnnotationDir();
		
		// Either create or verify the existence of a sub-directory for
		// storing all annotations of this particular model element
		File objectAnnotationDir = new File(annotationDir, ((containerID != null) ? containerID : (String.valueOf(System.currentTimeMillis()))));
		if ( !objectAnnotationDir.exists() )
		{
			if (!objectAnnotationDir.mkdir())
			{
				throw new IOException("Collision making directory " + objectAnnotationDir.getAbsolutePath());
			}
		}
		else if ( !objectAnnotationDir.isDirectory())
		{
			throw new IOException("Object annotation dir name exists as file " + objectAnnotationDir.getAbsolutePath());
		}
		
		// we now know that objectAnnotationDir exists and is a directory.
		File srcFile = new File(file.toFileString());
		if ( !srcFile.exists() )
		{
			throw new IOException("Specified file does not exist " + file.toFileString());
		}

		File destFile = new File(objectAnnotationDir, srcFile.getName());
		if ( destFile.exists() )
		{
			throw new IOException("FileAnnotation of same name (" + srcFile.getName() + ") already exists on object " + annotationElement.eContainer().toString());
		}
		
		// Copy the file annotation into our container object's annotation directory
		FileUtil.copyFileSafe(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
		
		URI baseAnnotationDirURI = URI.createFileURI(annotationDir.getAbsolutePath());
		URI annotation = URI.createFileURI(destFile.getAbsolutePath());
		URI relativeURI = annotation.deresolve(baseAnnotationDirURI);
		
		annotationElement.setURI(relativeURItoQueryURI(relativeURI).toString());		
	}

	public void removeFileAnnotation(CMNAnnotation annotationImpl) throws IOException{
		URI resolvedURI = annotationImpl.getFileAnnotation();
		File file = new File(resolvedURI.toFileString());
		try {
			if (file.exists())
			{
				file.delete();
				File parentDir = file.getParentFile();
				if ( parentDir.exists())
				{
					File[] siblings = parentDir.listFiles();
					if ( siblings.length == 0 )
					{
						// Removed the only annotation of the model element. 
						// Delete the subdirectory
						parentDir.delete();
					}
				}
			}
		}
		catch(Throwable t) {
			
		}
	}
	
	public static final String ANNOTATION_QUERY = "annotationID=/";
	
	/**
	 * @param uri2
	 */
	private URI queryURItoRelativeURI(URI uri) {
		
		if ( uri.hasQuery() )
		{
			String query = uri.query();
			String uriString = query.replaceFirst(ANNOTATION_QUERY, "");
			uri = URI.createURI(uriString);
		}
		return uri;
	}
	
	private URI relativeURItoQueryURI(URI uri) {
		
		if ( !uri.hasQuery() )
		{
			String queryString = ANNOTATION_QUERY + uri.toString();
			uri = URI.createHierarchicalURI(null, null, null, queryString, null);			
		}
		return uri;
	}

	private String getContainerID(CMNAnnotation annotationElement) throws UnsupportedOperationException {
		String containerID;
		EObject container = annotationElement.eContainer();
		
		if ( container == null )
		{
			throw new UnsupportedOperationException("FileAnnotation must be contained in a model element");
		}
		
		if ( container instanceof CMNNamedElement )
		{
			CMNNamedElement con = (CMNNamedElement) container;
			containerID = con.getId();
		}
		else if ( container instanceof TPFExecutionEvent )
		{
			TPFExecutionEvent con = (TPFExecutionEvent) container;
			containerID = con.getId();
		}
		else
		{
			throw new UnsupportedOperationException("Annotations not supported for container of type " + container);
		}
		return containerID;
	}
	
	
}
