/*******************************************************************************
 * Copyright (c) 2003, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IDatapoolFactory.java,v 1.3 2006/02/08 19:59:53 slavescu Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.runtime.datapool;

/**
 * A datapool factory manages the construction and sharing of 
 * {@link IDatapool} and {@link IDatapoolIterator} objects.
 * Test scripts by default use managed datapools that share the
 * datapool and iterator instance across multiple scripts.
 */
public interface IDatapoolFactory
{
	
	/**
	 * Load the datapool from the specified file.  If a
	 * <code>sharedInstance</code> is requested then the cache of
	 * other datapools that have been opened is checked for an 
	 * existing copy of the datapool.  Note that modifying a shared
	 * instance of the datapool may effect other uses of the datapool
	 * and should only be done in a confined execution environment.
	 * 
	 * @param	dpFile			The datapool file name.
	 * @param	sharedInstance	Available for sharing instances of
	 *						the datapool, which can save significantly 
	 *						on datapool loading overhead.
	 * 
	 * @see	#unload(IDatapool)
	 */
	public IDatapool load(java.io.File dpFile, boolean sharedInstance);
	
	/**
	 * Removes a reference to an instance of a datapool.  If the
	 * datapool is shared the reference count is reduced.  Once 
	 * all references to a datapool have been unloaded the datapool
	 * instance may be made available to garbage collection.
	 * 
	 * @param	datapool	The datapool instance being unloaded.
	 * 
	 * @see	#load(java.io.File,boolean)
	 */
	public void unload(IDatapool datapool);
	
	/**
	 * Return an iterator instance of the supplied iterator class.
	 * Depending on attributes associated with the iterator class it 
	 * may be shared.  The attributes of an iterator are negotiated 
	 * between the factory implementation and the iterator base class.
	 * 
	 * @param	datapool	The datapool instance being iterated over.
	 * 
	 * @see	#close(IDatapoolIterator)
	 */
	public IDatapoolIterator open(IDatapool datapool,
								  String iteratorClassName);
		
	/**
	 * Return the iterator to the factory so that it can be properly 
	 * destroyed.  Note that if shared the iterator will not be destructed 
	 * until all uses have been accounted for.
	 * 
	 * @param	iterator	The iterator being freed.
	 * 
	 * @see	#open(IDatapool,String)
	 */
	public void close(IDatapoolIterator iterator);
	
	
}
