/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionContext.java,v 1.21 2010/04/12 12:38:40 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.common;

import java.util.HashMap;
import java.util.Hashtable;
import java.util.LinkedList;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.common.provisional.TestlogFilenameExtensionManager;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.fragments.BVRCombinedFragment;
import org.eclipse.hyades.models.common.interactions.BVRExecutionOccurrence;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTypedEvent;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.common.util.ResourceCache;
import org.eclipse.hyades.models.common.util.SaveManager;
import org.eclipse.hyades.models.hierarchy.util.internal.EMFWorkspaceUtil;
import org.eclipse.hyades.models.util.ModelDebugger;

/**
 * <p>This class serves two purposes:</p>
 * 
 * <ol>
 * <li>As a lookup mechanism for the execution history/result for a given event.</li>
 * <li>Log and save an execution related event.<li>
 * </ol>
 *  
 * 
 * @author  Ashish Mathur
 * @author  Paul E. Slauenwhite
 * @author	Jerome Bozier
 * @version April 12, 2010
 * @since   January 25, 2005
 */
public class ExecutionContext {
	
	// Static variables
	public static final boolean debug = true;
	public static char   separator = '.';
	public static String root = "ROOT";
	
	
	// Instance variables
	private TPFTestSuite 		  testSuite = null;	  /* Root test suite being executed */
	private ResourceSet			  resourceSet;
	private URI					  testSuiteURI;
	private boolean				  databaseResource = false; /* specifies if this execution result should be persisted in a database */
	
	protected ExecutionResultData rootResult = null;	  /* Root ExecutionResult */
	protected HashMap 			  resultMap = new HashMap(); /* Map for all execution results */
	private Resource 			  resource = null;	  /* Resource for the execution result */
	private String 				  executionResultLocation;  /* Location for the execution result */
	private String 				  executionResultName;  /* Location for the execution result */
	private URI					  executionURI; /* URI used to create the execution result */
	protected EObjectCache 		  cache = null;

	
	private int agentCount=0;
	private boolean testLogFile = false;
	
	/**
	 * This class handles the creation of execution results for all tests 
	 * run within this test suite, logging events and also saving the resource.
	 * 
	 * @param suite - TPFTestSuite being run
	 * @param id - ID of the test suite
	 * @param executionResultLocation - folder where the results resource needs to be created
	 * @param executionResultName - name of the execution result specified by the user
	 * 
	 */
	public ExecutionContext(TPFTestSuite suite, String id, String executionResultLocation, String executionResultName) {
		this(suite, id, executionResultLocation, executionResultName, false, null);
	}
	
	/**
	 * This class handles the creation of execution results for all tests 
	 * run within this test suite, logging events and also saving the resource.
	 * 
	 * @param suite - TPFTestSuite being run
	 * @param id - ID of the test suite
	 * @param location - location where the execution will take place.
	 * @param executionResultLocation - folder where the results resource needs to be created
	 * @param executionResultName - name of the execution result specified by the user
	 * 
	 */
	public ExecutionContext(TPFTestSuite suite, String id, CFGLocation location, String executionResultLocation, String executionResultName) {
		this(suite, id, executionResultLocation, executionResultName);
		TPFDeployment deployment = location.getDeployment();
		if (deployment != null)
		{
			// if not the default deployment, populate the execution result with the deployment
			if(!ConfigurationUtil.isDefaultDeployment(deployment))
			rootResult.getResult().setDeployment(deployment);
		}
	}

	public ExecutionContext(TPFTestSuite suite, String id, 
		String executionResultLocation, String executionResultName, boolean 
		overrideExisting, TPFDeployment deployment, boolean databaseResource, 
		boolean testLogFile) 
	{
		this.testSuite = suite;
		this.databaseResource = databaseResource;
		this.testLogFile = testLogFile;
		
		resourceSet = suite.eResource().getResourceSet();
		testSuiteURI = EcoreUtil.getURI(testSuite);
		
		this.executionResultLocation = executionResultLocation;

		//Note: Use the setExecutionResultName(String) method to set the executionResultName to remove a possible trailing '.execution':
		if(executionResultName == null){
			setExecutionResultName(suite.getName());
		}
		else{
			setExecutionResultName(executionResultName);
		}
		
		rootResult = new ExecutionResultData(getTestSuite(), suite.getId());
		
		// Only do this set up if we are loading a model.
		if (!testLogFile) {
			resource = createExecutionResultResource(overrideExisting);
			resource.getContents().add(rootResult.getResult());		
			rootResult.getResult().setName(executionResultName);
			SaveManager.saveResource(resource);

			if (deployment != null)
			{
				// if not the default deployment, populate the execution result with the deployment
				if(!ConfigurationUtil.isDefaultDeployment(deployment))
					rootResult.getResult().setDeployment(deployment);
			}
			cache = new EObjectCache(0);
		}

		resultMap.put(getTestSuite().getId() + "(1)", rootResult);	
		
	}

	public ExecutionContext(TPFTestSuite suite, String id, String 
			executionResultLocation, String executionResultName, boolean 
			overrideExisting, TPFDeployment deployment) 
	{
		this(suite, id, executionResultLocation, executionResultName, false, null, false, false);
	}

	/**
	 * <p>Creates a new execution result resource.</p> 
	 * 
	 * <p>If <code>overwrite</code> is <code>true</code>, the existing execution result resource (if any) is deleted.  
	 * Otherwise, the new execution result resource is renamed to:</p>
	 * 
	 * <p><code>&lt;new execution result file name&gt;_&lt;current time stamp (milliseconds)&gt;.&lt;new execution result file extension&gt;</code></p>
	 * 
	 * @param timestamp Ignored.
	 * @return The new execution result resource.
	 * @deprecated As of TPTP 4.7.0, use {@link #createExecutionResultResource(boolean)} passing in <code>false</code>.
	 * @see #createExecutionResultResource(boolean)
	 */
	protected Resource createExecutionResultResource(long timestamp){
		return (createExecutionResultResource(false));
	}

	/**
	 * <p>Creates a new execution result resource.</p> 
	 * 
	 * <p>If <code>overwrite</code> is <code>true</code>, the existing execution result resource (if any) is deleted.  
	 * Otherwise, the new execution result resource is renamed to:</p>
	 * 
	 * <p><code>&lt;new execution result file name&gt;_&lt;current time stamp (milliseconds)&gt;.&lt;new execution result file extension&gt;</code></p>
	 * 
	 * @param overwrite <code>true</code> if the existing execution result resource is overwritten, otherwise <code>false</code>.
	 * @return The new execution result resource.
	 */
	protected Resource createExecutionResultResource(boolean overwrite){

		//Create a filename for the new execution result resource:
		if(!overwrite){
			setExecutionResultName(executionResultName + '_' + System.currentTimeMillis());
		}

		//Create a URI for the new execution result resource:
		executionURI = createURI(getExecutionResultLocation() + '/' + getExecutionResultName(databaseResource));

		//Confirm if the new execution result resource can be created:
		if(getTestSuite().eResource().getResourceSet() == null){

			ModelDebugger.log("Unable to create execution result resource: " + formatURI(executionURI)); //$NON-NLS-1$

			return null;
		}

		//Overwrite the file of the existing execution result resource (if exists):
		if((overwrite) && (EMFWorkspaceUtil.exists(executionURI))){

			if (EMFWorkspaceUtil.getFileFromURI(executionURI).delete()) {

				//Refresh the workspace to remove the deleted file:
				if (RegistryReader.isWorkspaceMode()) {

					try {
						EMFWorkspaceUtil.refreshLocal(executionURI);
					} 
					catch (Exception e) {
						ModelDebugger.log(e);
					}
				}
			}
			else {
				ModelDebugger.log("Unable to delete existing execution result file: " + formatURI(executionURI)); //$NON-NLS-1$
			}
		}	

		return (ResourceCache.getInstance().createSharedResource(executionURI, resourceSet));
	}

	//Note: Any changes to the type of the returned URI MUST also be changed in #formatURI(URI).
	private URI createURI(String filePath)
	{
		if ( RegistryReader.isPlatformMode() && RegistryReader.isWorkspaceMode() )
			return URI.createPlatformResourceURI(filePath, false);
		else
			return URI.createFileURI(filePath);
	}

	//Note: Any changes to the type of the returned from #formatURI(URI) URI MUST also be changed in this method.
	private String formatURI(URI uri){

		if(uri != null){

			if(uri.isPlatform()){
				return (uri.toPlatformString(true));
			}
			else if(uri.isFile()){
				return (uri.toFileString());
			}
			else{
				return (uri.toString());
			}
		}

		return null;
	}
	
	/**
	 * @return
	 */
	public TPFTestSuite getTestSuite() {
		if(testSuite.eResource().getResourceSet() == null)
		{
			EObject eObject = resourceSet.getEObject(testSuiteURI, true);
			if(eObject instanceof TPFTestSuite)
				testSuite = (TPFTestSuite)eObject;
		}
			
		return testSuite;
	}

	/**
	 * @return
	 */
	public Resource getResource()
	{
		return resource;
	}

	/**
	 * @return
	 */
	public String getExecutionResultLocation()
	{
		return executionResultLocation;
	}
	
	protected String getExecutionResultName()
	{
		return getExecutionResultName(false);
	}
	
	protected String getExecutionResultName(boolean databaseResource) {
		// JPT: We need to remove the use of ModelDebugger to calculate the
		// right suffix for using a database resource and key off of the user's
		// specification.  This code is in flux, but right now if
		// debugDatabaseResourcePostfix is not defined, we'll throw an NPE
		if( databaseResource )
			//|| (ModelDebugger.INSTANCE.debugDatabaseResourcePostfix!=null && 
			//	ModelDebugger.INSTANCE.debugDatabaseResourcePostfix.length()>0))
		{
			String s = ModelDebugger.INSTANCE.debugDatabaseResourcePostfix;
			return (executionResultName + s.substring(0,s.indexOf('.'))+".executiondb");
		}
		else
		{
			return (executionResultName + "." + getFileExtension());
		}		
	}

	protected void setExecutionResultName(String name)
	{
		
		if((name != null) && (name.trim().endsWith("." + getFileExtension()))){
			this.executionResultName = name.substring(0, (name.lastIndexOf('.')));			
		}
		else{
			this.executionResultName = name;
		}
	}

	/**
	 * This method logs a given execution event against the execution result it belongs
	 * to and saves the resource.
	 * 
	 * @param event - TPFExecutionEvent
	 */
	public void logEvent(TPFExecutionEvent event) {
		
		// Parse the ID
		ExecutionMapData mapData = new ExecutionMapData(event.getOwnerId());
		ExecutionResultData data = mapData.getData();
		
		if (data == null) {
			reportErrorInEvent(event);
			return;
		}
		
		// Get the history to add the event to
		TPFExecutionHistory history = data.getHistory();
		event.setExecutionHistory(history);
		
		// Set the BVRInteractionFragment, if its behavior 
		// Or the test if its that
		EObject eObject = data.getTest().eResource().getEObject(mapData.getOwnerId());

		if (event instanceof TPFInvocationEvent) {
			TPFTest test = getTestSuite();
			
			if (eObject instanceof BVRInteractionFragment) {
				event.setInteractionFragment((BVRInteractionFragment) eObject);
				
				// Also load the suite or test that this is referencing for
				// future lookup! Calling getOtherBehavior() apparently loads it!
				TPFBehavior behavior = ((BVRExecutionOccurrence)eObject).getOtherBehavior();
				if(behavior.getTest() != null)
					test = behavior.getTest();
			}
			
			// Create the ExecutionResultData for this invocation event if needed
			ExecutionResultData child = new ExecutionResultData(test, event.getOwnerId());
			child.setInvocationEvent((TPFInvocationEvent) event);
		
			// Put this new data set in the result map for later lookup
			resultMap.put(child.getId(), child);
			((TPFInvocationEvent)event).setExecutionHistory(data.getResult().getExecutionHistory());
		}
		else if (event instanceof TPFTypedEvent)
		{
//			TPFTypedEvent type = (TPFTypedEvent)event;
			// set the test for the execution result that this event belongs to
			if (eObject instanceof TPFTest) {
				data.getResult().setTest((TPFTest)eObject);
			}
			else if (eObject instanceof BVRCombinedFragment) {
				event.setInteractionFragment((BVRInteractionFragment)eObject);
			}
			// JPT: If the eObject is a Loop, address hierarchy here.
		}
		else if (event instanceof TPFVerdictEvent)
		{
			data.getResult().setVerdict(((TPFVerdictEvent) event).getVerdict());
		}
	}
	
	private void reportErrorInEvent(TPFExecutionEvent event) {
		if (debug) {
			System.err.println("Ignoring the following event due to invalid ownerId: ");
			System.err.println("Event    : " + event.getClass().getName());
			System.err.println("ownerId  : " + event.getOwnerId());
			System.err.println("text     : " + event.getText());
			System.err.println("timestamp: " + event.getTimestamp());
		}
	}
		
	public void cleanUp() {
		
		if (!testLogFile) {
			SaveManager.saveResource(resource);
			ResourceCache.getInstance().releaseSharedResource(executionURI);
			cache.clear();
		}
		
		resultMap.clear();
	}

	public EObject getEObjectByID(String id)
	{
		Object obj = null;

		if (resource instanceof ResourceImpl && ((ResourceImpl) resource).getIntrinsicIDToEObjectMap() != null) {
			obj = ((ResourceImpl) resource).getIntrinsicIDToEObjectMap().get(id);
		} else {
			obj = cache.get(id);
			if ( obj == null )
			{
				// Find the object and put it into the cache
				EObject newObj;
				newObj = resource.getEObject(id);
				//changed to explicit throwing RuntimeException from allowing NullPointerException to occur when putting a null
				//value in the cache.  I believe it better to avoid NullPointers at all cost and never to use them for 
				//program failure
				//Ernest Jessee 1/8/2008
				if(newObj!=null)
				{
					cache.put(id, newObj);
					return newObj;
				}
				else
					throw new RuntimeException("unable to identify object by id");
				
			}
		}

		return (EObject) obj;
	}

	public void putEObjectByID(String id, EObject obj)
	{
		if (resource instanceof ResourceImpl && ((ResourceImpl) resource).getIntrinsicIDToEObjectMap() != null) {
			((ResourceImpl) resource).getIntrinsicIDToEObjectMap().put(id, obj);
		} else {
			cache.put(id, obj);
		}
	}

	/**
	 * @provisional
	 */
	public void removeEObjectByID(String id)
	{
		if (resource instanceof ResourceImpl && ((ResourceImpl) resource).getIntrinsicIDToEObjectMap() != null) {
			((ResourceImpl) resource).getIntrinsicIDToEObjectMap().remove(id);
		} else {
			cache.remove(id);
		}
	}
	
	protected class EObjectCache {
		private LinkedList order = null;
		private Hashtable map = null;
		private int capacity = 0;

		/**
		 * Constructs an empty MRU cache of the specified maximum capacity.
		 * Specifying 0 or Integer.MAX_VALUE as the capacity means that the
		 * cache should have an unbounded capacity (limited to Integer.MAX_VALUE)
		 *  
		 * @param size the maximum number of elements to allow the cache to
		 * store.  The actual size of the cache will grow until it reaches
		 * this number.  If this number is reached, then it will discard 
		 * least recently used entries as new entries are added.  If the
		 * cache size is unbounded (see above), then no notion of MRU is 
		 * tracked, and all added elements are kept in the cache until it 
		 * is cleared.
		 */
		public EObjectCache(int size) {
			map = new Hashtable(89);
			order = new LinkedList();
			if ( size == 0 )
				capacity = Integer.MAX_VALUE;
			else
				capacity = size;
		}
		
		/**
		 * Clears the contents of the cache
		 */
		public void clear() {
			map.clear();
			order.clear();
		}
		
		/**
		 * @provisional
		 */
		public void remove(String id) {
			map.remove(id);
		}

		
		/**
		 * @param id the key to retrieve in the cache
		 * @return the value associate with the specified key, or null if the
		 * key is not present in the cache
		 */
		public EObject get(String id)
		{
			Object obj = map.get(id);
			if ( obj != null )
			{
				// move the element to the front of the MRU list and return
				moveToFront(id);
				return (EObject)obj;
			}
			return null;
		}

		
		/**
		 * @param id the key for the object being inserted into the cache
		 * @param obj the value being inserted into the cache
		 */
		public void put(String id, EObject obj)
		{
			Object temp = map.get(id);
			if ( temp != null)
			{
				// Already in the Cache.  Move to front and return
				moveToFront(id);
			}
			else
			{
				if ( atCapacity() )
				{
					// Throw out the oldest element
					removeLast();
				}
				// Add the new element
				insert(id, obj);
			}
		}

		// These private methods have special handling for a cache of unbounded
		// capacity (set to Integer.MAX_VALUE).  In this case, the order LinkedList
		// is not populated or updated (since we will not discard
		private void insert(String id, EObject obj) {
			map.put(id, obj);
			if ( capacity != Integer.MAX_VALUE )
				order.add(0, id);
		}
		private void moveToFront(String id)
		{
			if ( capacity != Integer.MAX_VALUE )
			{
				int index = order.indexOf(id);
				if ( index > 0 )
				{
					Object obj = order.get(index);
					order.remove(index);
					order.addFirst(obj);
				}
			}
		}
		private void removeLast() {
			if ( capacity != Integer.MAX_VALUE )
			{
				String deadID = (String)order.getLast();
				map.remove(deadID);
				order.removeLast();
			}
		}
		private boolean atCapacity() {
			// TODO Auto-generated method stub
			if ( map.size() >= capacity )
				return true;
			else
				return false;
		}

	}

	
	/**
	 * @author amathur
	 *
	 * This class parses an ID into the owning execution result ID and the 
	 * behavior ID of the event.
	 */
	public class ExecutionMapData {
	
		String parentId = null;
		String ownerId = null;		
		ExecutionResultData data = null;	

		/**
		 * 
		 */
		public ExecutionMapData(String id) {
			// If the ID is null, then we cant really locate the execution 
			// result it belongs to, so just quit
			if ((id == null) || (id.equals(""))) {
				return;
			}
			
			// Parse this id into 2 parts
			// 1. ID of the ExecutionOccurence/TPFTest/CombinedFragment(loop) 
			//		- this is the trailing part of the string
			// 2. Result ID - ID of the test result that this event belongs to
			int index = id.lastIndexOf(separator);
			if (index < 0) {
				// This is a root event, which means the map ID needs to
				// be the test suite ID
				// Dont do anything, the mapId & ownerId are already set
				parentId = id;
				ownerId = stripParanthesis(id);
				
				data = (ExecutionResultData) resultMap.get(parentId);
				
				if (data == null) {
					// The parent ID was badly formed. Try to see if the UUID is still
					// correct which means the iteration count was simply missing
					String testSuiteId = getTestSuite().getId();
					String testSuiteBehaviorId = getTestSuite().getBehavior().getId();
					if (parentId.compareTo(testSuiteId) == 0 ||
						parentId.compareTo(testSuiteBehaviorId) == 0 ) {
						data = rootResult;
					}
				}
			}
			else {
				// This is not a root event - does not belong to the root
				// execution result
				
				// Stip trailing ID (which will be the behavior ID)
				ownerId = id.substring(index+1);
				ownerId = stripParanthesis(ownerId);
				
				// Now get the ID of the parent in the hierarchy
				parentId = id.substring(0, index);
				
				// Make sure the execution result for the parent exists
				// If not, then this was an async event and needs to be dealt 
				// accordingly
				data = (ExecutionResultData) resultMap.get(parentId);
				
				// If the immediate parent ID does not have an execution result associated, then 
				// hunt for the execution result up the chain. If one does not exist, then
				// this event came out of order. Create the missing execution results as we 
				// traverse up the chain
				if (data == null) {
					// Try to recover for events that come out of order
					// First parse this parent ID using the data map
					ExecutionMapData map = new ExecutionMapData(parentId);
					ExecutionResultData tempData = map.getData();
					if (tempData != null) {
						// Found a parent result data
						// If the map.getOwnerId() is a loop, then jus tre-use the same execution result data
						// First get the test 
						EObject eObject = tempData.getObjectInTest(map.getOwnerId());
						if (eObject instanceof BVRCombinedFragment) {
							// Its a loop so just use it as is
							data = tempData;
						}
						else if (eObject instanceof TPFTest) {
							data = tempData;
						}
						else if (eObject instanceof BVRExecutionOccurrence) {
							TPFInvocationEvent invocationEvent = Common_TestprofileFactoryImpl.eINSTANCE.createTPFInvocationEvent();
							invocationEvent.setOwnerId(parentId);
							logEvent(invocationEvent);
							
							data = (ExecutionResultData) resultMap.get(parentId);
						}
					}
				}				
			}	
		}
		
		private String stripParanthesis(String id) {
			String str = id;
			int index = str.lastIndexOf('(');
			if (index >= 0) {
				str = str.substring(0, index);
			}
			
			return str;

		}

		/**
		 * @return
		 */
		public String getOwnerId() {
			return ownerId;
		}

		/**
		 * @return
		 */
		public String getParentId() {
			return parentId;
		}

		/**
		 * @return
		 */
		public ExecutionResultData getData() {
			return data;
		}
	}
	/**
	 * returns the file extension to be used by this ExecutionContext
	 * will search for extension points declaring a custom extension
	 * if no extension point found, will use the default ".execution"
	 * @provisional
	 * @return
	 */
	private String getFileExtension()
	{
		String customExtension = TestlogFilenameExtensionManager.getInstance().getExtensionForTestType(testSuite.getType());
		return customExtension!=null ? customExtension : ICommonConstants.EXECUTION_FILE_EXTENSION;
	}
		
	/**
	 * @return Returns the context's agentCount decremented by 1.
	 */
	public int decrementAgentCount() {
		return --agentCount;
	}
	
	/**
	 * increments the context's agentCount;
	 */
	public void incrementAgentCount() {
		agentCount++;
	}
}
