/*******************************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Messages.java,v 1.5 2009/03/04 17:50:49 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.datapool.impl;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import com.ibm.icu.text.MessageFormat;

/**
 * <p>Datapool model resource bundle.</p>
 * 
 * 
 * @author      Andrew Hoppe
 * @author      Paul E. Slauenwhite
 * @version     March 4, 2009
 * @since       January 1, 2005
 */
public class Messages {
	
	/**
     * Static constant representing the name 
     * (<code>org.eclipse.hyades.models.common.datapool.impl.messages</code>) 
     * of the resource bundle for the datapool model.
     */
	private static final String BUNDLE_NAME = "org.eclipse.hyades.models.common.datapool.impl.messages";//$NON-NLS-1$
	
    /**
     * Static reference to the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code>
     * resource bundle for the datapool model.
     */
	private static final ResourceBundle RESOURCE_BUNDLE = ResourceBundle.getBundle(BUNDLE_NAME);
	
	/**
	 * Private constructor.
	 */
	private Messages() {
		//No-operation.
	}
	
	/**
     * Resolves the localized message associated with the parameter <code>key</code>
     * based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model, the <code>key</code> is 
     * returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     */
	public static String getString(String key) {

		try {
			return (RESOURCE_BUNDLE.getString(key));
		} 
		catch (MissingResourceException e) {
			return key;
		}
	}
	
	/**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameter based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model.
     * <p>
	 * Message formatting is done by the {@link com.ibm.icu.text.MessageFormat} class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argument The first message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see com.ibm.icu.text.MessageFormat
	 * @see #getString(String, Object[])
     */
    public static String getString(String key, Object argument) {
        return (getString(key, new Object[] { argument}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model.
     * <p>
	 * Message formatting is done by the {@link com.ibm.icu.text.MessageFormat} class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see com.ibm.icu.text.MessageFormat
	 * @see #getString(String, Object[])
     */
    public static String getString(String key, Object argumentA, Object argumentB) {
        return (getString(key, new Object[] { argumentA, argumentB}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model.
     * <p>
	 * Message formatting is done by the {@link com.ibm.icu.text.MessageFormat} class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the datapool model, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see com.ibm.icu.text.MessageFormat
	 * @see #getString(String)
     */
    public static String getString(String key, Object[] arguments) {
		return (MessageFormat.format(getString(key), arguments));
	}
}