/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolSuggestedTypeChecker.java,v 1.1 2009/04/20 13:21:39 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.datapool.internal.util;

import org.eclipse.hyades.edit.datapool.IDatapoolSuggestedType;

/**
 * <p>DatapoolSuggestedTypeChecker.java</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Paul Slauenwhite
 * @version April 20, 2009
 * @since   April 20, 2009
 */
public class DatapoolSuggestedTypeChecker {

	private int languageType = 0;

	private static DatapoolSuggestedTypeChecker instance = null;

	public static final String JAVA_TYPE_VOID = "void"; //$NON-NLS-1$
	public static final String JAVA_TYPE_BOOLEAN = "boolean"; //$NON-NLS-1$
	public static final String JAVA_TYPE_BYTE = "byte"; //$NON-NLS-1$
	public static final String JAVA_TYPE_SHORT = "short"; //$NON-NLS-1$
	public static final String JAVA_TYPE_INT = "int"; //$NON-NLS-1$
	public static final String JAVA_TYPE_LONG = "long"; //$NON-NLS-1$
	public static final String JAVA_TYPE_FLOAT = "float"; //$NON-NLS-1$
	public static final String JAVA_TYPE_DOUBLE = "double"; //$NON-NLS-1$
	public static final String JAVA_TYPE_CHAR = "char"; //$NON-NLS-1$

	public static final String JAVA_OBJ_TYPE_BOOLEAN = "java.lang.Boolean"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_BYTE = "java.lang.Byte"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_SHORT = "java.lang.Short"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_INTEGER = "java.lang.Integer"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_LONG = "java.lang.Long"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_FLOAT = "java.lang.Float"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_DOUBLE = "java.lang.Double"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_CHAR = "java.lang.Character";	 //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_STRING = "java.lang.String"; //$NON-NLS-1$

	public static final String JAVA_OBJ_TYPE_BOOLEAN_SN = "Boolean"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_BYTE_SN = "Byte"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_SHORT_SN = "Short"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_INTEGER_SN = "Integer"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_LONG_SN = "Long"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_FLOAT_SN = "Float"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_DOUBLE_SN = "Double"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_CHAR_SN = "Character"; //$NON-NLS-1$
	public static final String JAVA_OBJ_TYPE_STRING_SN = "String"; //$NON-NLS-1$
	
	public static final String JAVA_INTERFACE_TYPE_ENUMERATION = "java.lang.Enumeration"; //$NON-NLS-1$
	public static final String JAVA_INTERFACE_TYPE_ENUMERATION_SN = "Enumeration"; //$NON-NLS-1$

	public static final String CSHARP_TYPE_BOOL = "bool"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_SBYTE = "sbyte"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_BYTE = "byte"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_USHORT = "ushort"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_SHORT = "short"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_UINT = "uint"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_INT = "int"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_ULONG = "ulong"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_LONG = "long"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_FLOAT = "float"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_DOUBLE = "double"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_DECIMAL = "decimal";	 //$NON-NLS-1$
	public static final String CSHARP_TYPE_CHAR = "char"; //$NON-NLS-1$
	public static final String CSHARP_TYPE_STRING = "string"; //$NON-NLS-1$

	public static final String CSHARP_OBJ_TYPE_BOOL = "System.Boolean"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_SBYTE = "System.SByte"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_BYTE = "System.Byte"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_USHORT = "System.UInt16"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_SHORT = "System.Int16"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_UINT = "System.UInt32"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_INT = "System.Int32"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_ULONG = "System.UInt64"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_LONG = "System.Int64"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_FLOAT = "System.Single"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_DOUBLE = "System.Double"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_DECIMAL = "System.Decimal";	 //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_CHAR = "System.Char"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_STRING = "System.String"; //$NON-NLS-1$

	public static final String CSHARP_OBJ_TYPE_BOOL_SN = "Boolean"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_SBYTE_SN = "SByte"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_BYTE_SN = "Byte"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_USHORT_SN = "UInt16"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_SHORT_SN = "Int16"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_UINT_SN = "UInt32"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_INT_SN = "Int32"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_ULONG_SN = "UInt64"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_LONG_SN = "Int64"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_FLOAT_SN = "Single"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_DOUBLE_SN = "Double"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_DECIMAL_SN = "Decimal";	 //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_CHAR_SN = "Char"; //$NON-NLS-1$
	public static final String CSHARP_OBJ_TYPE_STRING_SN = "String"; //$NON-NLS-1$
	
	public static final String CPP_TYPE_VOID = "void"; //$NON-NLS-1$
	public static final String CPP_TYPE_CHAR = "char"; //$NON-NLS-1$
	public static final String CPP_TYPE_SINT = "short int"; //$NON-NLS-1$
	public static final String CPP_TYPE_INT = "int"; //$NON-NLS-1$
	public static final String CPP_TYPE_LINT = "long int"; //$NON-NLS-1$
	public static final String CPP_TYPE_FLOAT = "float"; //$NON-NLS-1$
	public static final String CPP_TYPE_DOUBLE = "double"; //$NON-NLS-1$
	public static final String CPP_TYPE_LDOUBLE = "long double";	 //$NON-NLS-1$
	public static final String CPP_TYPE_S_CHAR = "signed char"; //$NON-NLS-1$
	public static final String CPP_TYPE_S_SINT = "signed short int"; //$NON-NLS-1$
	public static final String CPP_TYPE_S_INT = "signed int"; //$NON-NLS-1$
	public static final String CPP_TYPE_S_LINT = "signed long int"; //$NON-NLS-1$
	public static final String CPP_TYPE_U_CHAR = "unsigned char"; //$NON-NLS-1$
	public static final String CPP_TYPE_U_SINT = "unsigned short int"; //$NON-NLS-1$
	public static final String CPP_TYPE_U_INT = "unsigned int"; //$NON-NLS-1$
	public static final String CPP_TYPE_U_LINT = "unsigned long int"; //$NON-NLS-1$
	
	public static final int LANG_JAVA = 1;
	public static final int LANG_CSHARP = 2;
	public static final int LANG_CPP = 3;
		

	/**
	 * Resolves a singleton instance of the {@link DatapoolSuggestedTypeChecker}.
	 * 
	 * @return Singleton instance of the {@link DatapoolSuggestedTypeChecker}.
	 */
	public static DatapoolSuggestedTypeChecker getInstance(){
		
		if(instance == null){
			instance = new DatapoolSuggestedTypeChecker();
		}
		
		return instance;
	}
	
	/**
	 * Private constructor.
	 */
	private DatapoolSuggestedTypeChecker(){
		//No-operation.
	}
	
	/**
	 * Set the language type ({@link #LANG_JAVA}, {@link #LANG_CPP}, or {@link #LANG_CSHARP}) 
	 * of the {@link DatapoolSuggestedTypeChecker}.
	 * 
	 * @param type The language type ({@link #LANG_JAVA}, {@link #LANG_CPP}, or {@link #LANG_CSHARP}) of the {@link DatapoolSuggestedTypeChecker}.
	 */
	public void setLanguageType(int type){
		
		switch (type) {
		case LANG_JAVA:
		case LANG_CPP:
		case LANG_CSHARP:
			languageType = type;
			break;
		}
	}
	
	/**
	 * Determines if the suggested type is a number, based on the language type.
	 * 
	 * @param type The suggested type.
	 * @return True if the suggested type is a number, otherwise false.
	 */
	public boolean isNumber(String type)
	{
		if(type == null)
			return false;
		if(languageType == 0 || languageType == LANG_JAVA)
		{
			if(type.equals(JAVA_TYPE_INT) ||
				type.equals(JAVA_TYPE_FLOAT) ||
				type.equals(JAVA_TYPE_DOUBLE) ||
				type.equals(JAVA_TYPE_LONG) ||
				type.equals(JAVA_TYPE_SHORT) ||
				type.equals(JAVA_OBJ_TYPE_INTEGER) ||
				type.equals(JAVA_OBJ_TYPE_INTEGER_SN) ||
				type.equals(JAVA_OBJ_TYPE_FLOAT) ||
				type.equals(JAVA_OBJ_TYPE_FLOAT_SN)||
				type.equals(JAVA_OBJ_TYPE_DOUBLE) ||
				type.equals(JAVA_OBJ_TYPE_DOUBLE_SN) ||
				type.equals(JAVA_OBJ_TYPE_LONG) ||
				type.equals(JAVA_OBJ_TYPE_LONG_SN) ||
				type.equals(JAVA_OBJ_TYPE_SHORT) ||
				type.equals(JAVA_OBJ_TYPE_SHORT_SN))
				return true;
		}
		if(languageType == 0 || languageType == LANG_CSHARP)
		{
			if(type.equals(CSHARP_OBJ_TYPE_DECIMAL) ||
				type.equals(CSHARP_OBJ_TYPE_DECIMAL_SN) ||
				type.equals(CSHARP_OBJ_TYPE_DOUBLE) ||
				type.equals(CSHARP_OBJ_TYPE_DOUBLE_SN) ||
				type.equals(CSHARP_OBJ_TYPE_FLOAT) ||
				type.equals(CSHARP_OBJ_TYPE_FLOAT_SN) ||
				type.equals(CSHARP_OBJ_TYPE_INT) ||
				type.equals(CSHARP_OBJ_TYPE_INT_SN) ||
				type.equals(CSHARP_OBJ_TYPE_LONG) ||
				type.equals(CSHARP_OBJ_TYPE_LONG_SN) ||
				type.equals(CSHARP_OBJ_TYPE_SHORT) ||
				type.equals(CSHARP_OBJ_TYPE_SHORT_SN) ||
				type.equals(CSHARP_OBJ_TYPE_UINT) ||
				type.equals(CSHARP_OBJ_TYPE_UINT_SN) ||
				type.equals(CSHARP_OBJ_TYPE_ULONG) ||
				type.equals(CSHARP_OBJ_TYPE_ULONG_SN) ||
				type.equals(CSHARP_OBJ_TYPE_USHORT) ||
				type.equals(CSHARP_OBJ_TYPE_USHORT_SN) ||
				type.equals(CSHARP_TYPE_DECIMAL) ||
				type.equals(CSHARP_TYPE_FLOAT) ||
				type.equals(CSHARP_TYPE_INT) ||
				type.equals(CSHARP_TYPE_LONG) ||
				type.equals(CSHARP_TYPE_SHORT) ||
				type.equals(CSHARP_TYPE_UINT) ||
				type.equals(CSHARP_TYPE_ULONG) ||
				type.equals(CSHARP_TYPE_USHORT))
				return true;
		}
		if(languageType == 0 || languageType == LANG_CPP)
		{
			if(type.equals(CPP_TYPE_DOUBLE) ||
				type.equals(CPP_TYPE_FLOAT) ||
				type.equals(CPP_TYPE_SINT) ||
				type.equals(CPP_TYPE_INT) ||
				type.equals(CPP_TYPE_LINT) ||
				type.equals(CPP_TYPE_LDOUBLE) ||
				type.equals(CPP_TYPE_S_INT) ||
				type.equals(CPP_TYPE_S_LINT) ||
				type.equals(CPP_TYPE_S_SINT) ||
				type.equals(CPP_TYPE_U_INT) ||
				type.equals(CPP_TYPE_U_LINT) ||
				type.equals(CPP_TYPE_U_SINT))
				return true;
		}
		return false;
	}
	
	/**
	 * Determines if the suggested type is a boolean, based on the language type.
	 * 
	 * @param type The suggested type.
	 * @return True if the suggested type is a boolean, otherwise false.
	 */
	public boolean isBoolean(String type){
		
		if(type == null)
			return false;
		if(languageType == 0 || languageType == LANG_JAVA)
		{
			if(type.equals(JAVA_TYPE_BOOLEAN) ||
				type.equals(JAVA_OBJ_TYPE_BOOLEAN) ||
				type.equals(JAVA_OBJ_TYPE_BOOLEAN_SN))
				return true;
		}
		if(languageType == 0 || languageType == LANG_CSHARP)
		{
			if(type.equals(CSHARP_TYPE_BOOL) ||
				type.equals(CSHARP_OBJ_TYPE_BOOL) ||
				type.equals(CSHARP_OBJ_TYPE_BOOL_SN))
				return true;
		}
		if(languageType == 0 || languageType == LANG_CPP)
		{
		}
		return false;
	}
	
	/**
	 * Determines if the suggested type is a string, based on the language type.
	 * 
	 * @param type The suggested type.
	 * @return True if the suggested type is a string, otherwise false.
	 */
	public boolean isString(String type){
		
		if(type == null)
			return false;		
		if(languageType == 0 || languageType == LANG_JAVA)
		{
			if(type.equals(JAVA_OBJ_TYPE_STRING) ||
				type.equals(JAVA_OBJ_TYPE_STRING_SN))
				return true;
		}
		if(languageType == 0 || languageType == LANG_CSHARP)
		{
			if(type.equals(CSHARP_OBJ_TYPE_STRING) ||
				type.equals(CSHARP_OBJ_TYPE_STRING_SN) ||
				type.equals(CSHARP_TYPE_STRING))
				return true;
		}
		if(languageType == 0 || languageType == LANG_CPP)
		{
		}
		return false;
		
	}
	
	/**
	 * Determines if the suggested type is an enumeration, based on the language type.
	 * 
	 * @param type The suggested type.
	 * @return True if the suggested type is an enumeration, otherwise false.
	 */
	public boolean isEnumeration(String type){
		
		if(type == null)
			return false;		
		if(languageType == 0 || languageType == LANG_JAVA)
		{
		
			if(type.equals(JAVA_INTERFACE_TYPE_ENUMERATION) ||
					type.equals(JAVA_INTERFACE_TYPE_ENUMERATION_SN)){
					return true;
			}

			try
			{
				Class cls = Class.forName(type);
				while(cls != null)
				{
					Class[] interfaces = cls.getInterfaces();
					for(int i = 0; i < interfaces.length; i++)
					{
						Class inter = interfaces[i];
						if(inter.getName().equals(JAVA_INTERFACE_TYPE_ENUMERATION) || inter.getName().equals(JAVA_INTERFACE_TYPE_ENUMERATION_SN)) //$NON-NLS-1$
							return true;
					}
					cls = cls.getSuperclass();
				}
			}
			catch(Exception e)
			{
			}
		}
		return false;
	}
	
	/**
	 * Determines if the suggested type is a primitive, based on the language type.
	 * 
	 * @param type The suggested type.
	 * @param includeVoid True if the suggested type should include the void primitive , otherwise false.
	 * @return True if the suggested type is a primitive, otherwise false.
	 */
	public boolean isPrimitive(String type, boolean includeVoid){
		if(type == null)
			return false;
		if(languageType == 0 || languageType == LANG_JAVA)
		{
			if(type.equals(JAVA_TYPE_BOOLEAN) ||
				type.equals(JAVA_TYPE_BYTE) ||
				type.equals(JAVA_TYPE_CHAR) ||
				type.equals(JAVA_TYPE_DOUBLE) ||
				type.equals(JAVA_TYPE_FLOAT) ||
				type.equals(JAVA_TYPE_INT) ||
				type.equals(JAVA_TYPE_LONG) ||
				type.equals(JAVA_TYPE_SHORT))
				return true;
			if(includeVoid)
			{
				if(type.equals(JAVA_TYPE_VOID))
					return true;
			}
		}
		if(languageType == 0 || languageType == LANG_CSHARP)
		{
			return false;
		}
		if(languageType == 0 || languageType == LANG_CPP)
		{
			if(type.equals(CPP_TYPE_DOUBLE) ||
				type.equals(CPP_TYPE_FLOAT) ||
				type.equals(CPP_TYPE_SINT) ||
				type.equals(CPP_TYPE_INT) ||
				type.equals(CPP_TYPE_LINT) ||
				type.equals(CPP_TYPE_LDOUBLE) ||
				type.equals(CPP_TYPE_CHAR) ||
				type.equals(CPP_TYPE_U_CHAR) ||
				type.equals(CPP_TYPE_S_CHAR) ||
				type.equals(CPP_TYPE_S_INT) ||
				type.equals(CPP_TYPE_S_LINT) ||
				type.equals(CPP_TYPE_S_SINT) ||
				type.equals(CPP_TYPE_U_INT) ||
				type.equals(CPP_TYPE_U_LINT) ||
				type.equals(CPP_TYPE_U_SINT))
				return true;
			if(includeVoid)
			{
				if(type.equals(CPP_TYPE_VOID))
					return true;
			}			
		}
		
		return false;		
	}

	/**
	 * Sets the suggested type and suggested class named derived from the suggested type
	 * on the suggested type object.
	 * 
	 * @param suggestedType The suggested type object.
	 * @param type The suggested type.
	 */
	public void setVariableType(IDatapoolSuggestedType suggestedType, String type){
		
		if(suggestedType == null)
			return;
		if(DatapoolSuggestedTypeChecker.getInstance().isString(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_STRING);
		else if(DatapoolSuggestedTypeChecker.getInstance().isBoolean(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_BOOLEAN);
		else if(DatapoolSuggestedTypeChecker.getInstance().isNumber(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_NUMBER);
		else if(DatapoolSuggestedTypeChecker.getInstance().isEnumeration(type))
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_ENUMERATION);
		else
			suggestedType.setSuggestedType(IDatapoolSuggestedType.TYPE_COMPLEX);
		
		suggestedType.setSuggestedClassName(type);	
	}
	
	/**
	 * Determines if the suggested type is the same as either the suggested type and suggested class named in the suggested type object.
	 * 
	 * @param type The suggested type.
	 * @param suggestedType The suggested type object.
	 * @return True if the suggested type is the same as either the suggested type and suggested class named in the suggested type object, otherwise false.
	 */
	public boolean isTypeMatch(String type, org.eclipse.hyades.execution.runtime.datapool.IDatapoolSuggestedType suggestedType){
		
		if(type == null || suggestedType == null)
			return false;
		
		if(type.equals(suggestedType.getSuggestedClassName()))
			return true;
		
		int sugType = suggestedType.getSuggestedType();
		if(sugType == IDatapoolSuggestedType.TYPE_STRING && this.isString(type))
			return true;
		else if(sugType == IDatapoolSuggestedType.TYPE_BOOLEAN && this.isBoolean(type))
			return true;
		else if(sugType == IDatapoolSuggestedType.TYPE_NUMBER && this.isNumber(type))
			return true;
		else if(sugType == IDatapoolSuggestedType.TYPE_ENUMERATION && this.isEnumeration(type))
			return true;
		
		return false;
	}
}
