/********************************************************************** 
 * Copyright (c) 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: DatapoolEncryptManager.java,v 1.3 2008/05/09 16:45:19 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.models.common.datapool.util;

import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolCell;
import org.eclipse.hyades.edit.datapool.IDatapoolRecord;
import org.eclipse.hyades.execution.runtime.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.execution.runtime.datapool.IDatapoolVariable;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.DPLVariable;
import org.eclipse.hyades.models.common.datapool.impl.DPLVariableImpl;
import org.eclipse.hyades.models.common.util.EncryptionManager;
import org.eclipse.hyades.models.plugin.ModelsPlugin;

/**
 * Utility APIs for encrypted datapools.
 * <p/>
 * 
 * 
 * @author      Huang Xin Ying 
 * @author      Paul E. Slauenwhite
 * @version     May 9, 2008
 * @since       March 20, 2008
 * @provisional Provisional API as of TPTP 4.5.0.
 */
public class DatapoolEncryptManager {
	
    /**
     * Used to encrypt the variable in the datapool using password
     * @param varible
     * @param key
     * @param datapool
     */
	public static void encryptedCellInVarible(IDatapoolVariable varible,
			String key, IDatapool datapool) {

		int index = datapool.getDefaultEquivalenceClassIndex();
		if (index == -1) {

			return;
		}
		IDatapoolEquivalenceClass equivalenceClass = datapool
				.getEquivalenceClass(index);
		
		for (int i = 0; i < equivalenceClass.getRecordCount(); i++) {
			IDatapoolRecord record = (IDatapoolRecord) equivalenceClass
					.getRecord(i);			
			for (int k = 0; k < record.getCellCount(); k++) {
				IDatapoolCell cell = (IDatapoolCell) record.getCell(k);
				if (cell.getCellVariable().getId().equals(varible.getId())) {
					cell.setCellValue(EncryptionManager.encrypt(cell.getStringValue(),
							key));
				}
			}
		}
	}

	/**
	 * Decrypt selected variable in the datapool using key
	 * @param varible
	 * @param key
	 * @param datapool
	 */
	public static void decryptedCellInVarible(IDatapoolVariable variable,
			String key, IDatapool datapool) {
		int index = datapool.getDefaultEquivalenceClassIndex();
		if (index == -1) {
			return;
		}
		IDatapoolEquivalenceClass equivalenceClass = datapool
				.getEquivalenceClass(index);

		for (int i = 0; i < equivalenceClass.getRecordCount(); i++) {
			IDatapoolRecord record = (IDatapoolRecord) equivalenceClass
					.getRecord(i);			
			for (int k = 0; k < record.getCellCount(); k++) {
				IDatapoolCell cell = (IDatapoolCell) record.getCell(k);
				if (cell.getCellVariable().getId().equals(variable.getId())) {
					cell.setCellValue(EncryptionManager.decrypt(cell.getStringValue(),
							key));
				}
			}
		}
	}

	/**
	 * Removed the key for an encrypted datapool.
	 * <p/>
	 * 
	 * @param datapool The encrypted datapool. 
	 */
	public static void removeKey(IDatapool datapool) {
		changeKey(null, datapool);
	}
	
	/**
	 * Change key for an encrypted datapool.
	 * <p/>
	 * 
	 * @param newKey The new key.
	 * @param datapool The encrypted datapool. 
	 */
	public static void changeKey(String newKey, IDatapool datapool) {
		
		if (datapool instanceof DPLDatapool){

			if ((newKey != null) && (newKey.trim().length() > 0)){
				
				try{
					((DPLDatapool) datapool).setChallenge(EncryptionManager.EncoderByMd5(newKey));
				}
				catch(Exception e){						
					ModelsPlugin.getPlugin().log(e);
				}
			}
			else{
				((DPLDatapool) datapool).setChallenge(null);
			}
		}
	}

	/**
	 * Update encrypted variables using new key 
	 * @param datapool
	 * @param oldKey
	 * @param newKey
	 */
	public static void changeKeyOfEncryptedCell(IDatapool datapool, String oldKey,
			String newKey) {
		int variableCount = datapool.getVariableCount();
		IDatapoolVariable variable = null;
		for (int i = 0; i < variableCount; i++) {
			variable = datapool.getVariable(i);
			if (isVariableEncrypted(variable)) {
				decryptedCellInVarible(variable, oldKey, datapool);
				encryptedCellInVarible(variable, newKey, datapool);
			}
		}
	}
	
	/**
	 * Decrypt datapool's variables into unencrypted 
	 * @param datapool
	 * @param key
	 */
	public static void decryptDatapool(IDatapool datapool, String key){
		int variableCount = datapool.getVariableCount();
		IDatapoolVariable variable = null;
		for (int i = 0; i < variableCount; i++) {
			variable = datapool.getVariable(i);
			if (isVariableEncrypted(variable)) {
				decryptedCellInVarible(variable, key, datapool);				
			}
		}
	}
	
	/**
	 * Encrypt datapool using assigned key 
	 * @param datapool
	 * @param key
	 */
	public static void encryptDatapool(IDatapool datapool, String key){
		int variableCount = datapool.getVariableCount();
		IDatapoolVariable variable = null;
		for (int i = 0; i < variableCount; i++) {
			variable = datapool.getVariable(i);
			if (isVariableEncrypted(variable)) {
				encryptedCellInVarible(variable, key, datapool);			
			}
		}
	}

	public static boolean isDatapoolEncrypted(IDatapool datapool) {
		if (!(datapool instanceof DPLDatapool))
			return false;

		if (((DPLDatapool) datapool).getChallenge() == null
				|| ((DPLDatapool) datapool).getChallenge().equals(""))
			return false;
		return true;
	}
	
	public static boolean containsEncryptedVariable(IDatapool datapool){
        boolean containsEncryptedVariable = false;
        int variableCount = datapool.getVariableCount();
		for(int i=0;i<variableCount;i++){			
			if(DatapoolEncryptManager.isVariableEncrypted(((DPLVariableImpl)datapool.getVariable(i)))){
				containsEncryptedVariable = true;
				break;
			}
		}
		return containsEncryptedVariable;
	}

	public static boolean isKeyCorrect(IDatapool datapool, String key) {
		if (!(datapool instanceof DPLDatapool))
			return false;
		if (key == null)
			return false;
		boolean flag = false;
		try {
			flag = EncryptionManager.EncoderByMd5(key).equals(
					((DPLDatapool) datapool).getChallenge());
		} catch (Exception e) {
			ModelsPlugin.getPlugin().log("invalid key exception occurs");
			ModelsPlugin.getPlugin().log(e);
		}
		return flag;

	}

	public static boolean isVariableEncrypted(IDatapoolVariable varible) {
		return (varible instanceof DPLVariable) ? ((DPLVariable) varible)
				.isEncrypted() : false;
	}

	public static String encrypt(String content, String pass) {
		return EncryptionManager.encrypt(content, pass);
	}

	public static String decrypt(String content, String pass) {
		return EncryptionManager.decrypt(content, pass);
	}

}
