/**********************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolUtil.java,v 1.15 2009/03/04 17:50:49 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.common.util;

import java.util.HashMap;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.datapool.evaluator.DatapoolEvaluatorLiteral;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.execution.runtime.datapool.IDatapoolEvaluatorExt;
import org.eclipse.hyades.models.common.datapool.Common_DatapoolFactory;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.DPLDatapoolSpec;
import org.eclipse.hyades.models.common.datapool.impl.DPLLogImpl;
import org.eclipse.hyades.models.common.datapool.impl.Messages;

/**
 * Datapool utilities.
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @author      Marius Slavescu
 * @version     March 4, 2009
 * @since       January 25, 2005
 */
public class DatapoolUtil {
	private static IExtension[] dpEvaluatorExts;

	private static HashMap dpEvaluatorsMap = new HashMap();

	private static IExtension[] getEvaluatorExts() {
		if (dpEvaluatorExts == null) {
			// get extension point
			IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.tptp.platform.models.datapool_evaluator"); //$NON-NLS-1$  
			if (extPoint != null) {
				DPLLogImpl.log("ext point found!!!"); //$NON-NLS-1$
				dpEvaluatorExts = extPoint.getExtensions();
				DPLLogImpl.log("extensions.length: " + dpEvaluatorExts.length); //$NON-NLS-1$
			} else
				// if no extension point found, initialize dpEvaluatorExts to
				// prevent query again.
				dpEvaluatorExts = new IExtension[0];
		}

		return dpEvaluatorExts;
	}

	public static IDatapoolEvaluatorExt findDatapoolEvaluator(String tag) {
		try {
			IDatapoolEvaluatorExt evaluator = (IDatapoolEvaluatorExt) dpEvaluatorsMap.get(tag);
			if (evaluator != null)
				return evaluator;

			// check for DatapoolEvaluatorLiteral first because its less
			// expensive than query extension point.
			if (tag != null && tag.equals("Literal")) //$NON-NLS-1$
			{
				Class cls = Class.forName("org.eclipse.hyades.datapool.evaluator.DatapoolEvaluatorLiteral"); //$NON-NLS-1$
				evaluator = (DatapoolEvaluatorLiteral) cls.newInstance();
				dpEvaluatorsMap.put(tag, evaluator);
				return evaluator;
			}

			IExtension[] extensions = getEvaluatorExts();
			for (int i = 0; i < extensions.length; i++) {
				IExtension currentExtension = extensions[i];
				DPLLogImpl.log("   extension:" + currentExtension.getUniqueIdentifier()); //$NON-NLS-1$
				IConfigurationElement[] configElements = currentExtension.getConfigurationElements();
				DPLLogImpl.log("configElements.length : " + configElements.length); //$NON-NLS-1$				
				for (int j = 0; j < configElements.length; j++) {
					IConfigurationElement currentConfigElement = configElements[j];
					String currentTag = currentConfigElement.getAttribute("tag"); //$NON-NLS-1$
					DPLLogImpl.log("Tag: " + currentTag); //$NON-NLS-1$
					DPLLogImpl.log("configElement: " + currentConfigElement.toString()); //$NON-NLS-1$
					if (tag.equals(currentTag)) {
						evaluator = (IDatapoolEvaluatorExt) currentConfigElement.createExecutableExtension("class"); //$NON-NLS-1$
						dpEvaluatorsMap.put(tag, evaluator);
						DPLLogImpl.log("evaluator: " + evaluator.toString()); //$NON-NLS-1$
						return evaluator;
					}
				}
			}
		} catch (Throwable t) {
			DPLLogImpl.log(t.getMessage());
		}

		return null;
	}

	// Utility methods to manipulate datapool object

	public static IDatapool createNewDatapool(String name, String description) {
		DPLDatapool datapool = Common_DatapoolFactory.eINSTANCE.createDPLDatapool();
		datapool.setName(name == null ? "" : name);
		if ((description != null) && (description.trim().length() > 0))
			datapool.setDescription(description);

		DPLDatapoolSpec datapoolSpec = Common_DatapoolFactory.eINSTANCE.createDPLDatapoolSpec();
		datapool.setDatapoolSpec(datapoolSpec);
		return datapool;
	}

	public static String createUniqueECName(IDatapool datapool) {

		//For performance reasons (e.g. importing a datapool with numerous equivalence classes), start 
		//the counter at the count of the next/new equivalence classes (e.g. one more than the number of 
		//existing equivalence classes):
		int suffix = (datapool.getEquivalenceClassCount() + 1);

		String name = Messages.getString("DATAPOOL_EC_NAME", String.valueOf(suffix)); //$NON-NLS-1$
		
		while (!isECNameUnique(name, datapool)) {
			name = Messages.getString("DATAPOOL_EC_NAME", String.valueOf(++suffix)); //$NON-NLS-1$
		}

		return name;
	}

	public static boolean isECNameUnique(String name, IDatapool datapool) {
		if (name == null)
			throw new IllegalArgumentException();
		for (int i = 0; i < datapool.getEquivalenceClassCount(); i++) {
			if (name.equals(datapool.getEquivalenceClass(i).getName()))
				return false;
		}
		return true;
	}

	public static String createUniqueVariableName(IDatapool datapool) {

		//For performance reasons (e.g. importing a datapool with numerous variables), start 
		//the counter at the count of the next/new variable (e.g. one more than the number of 
		//existing variables):
		int counter = (datapool.getVariableCount() + 1);
		
		String uniqueVariableName = Messages.getString("DATAPOOL_VARIABLE_NAME", (String.valueOf(counter))); //$NON-NLS-1$
		
		while(!isVariableNameUnique(uniqueVariableName, datapool)) {
			uniqueVariableName = Messages.getString("DATAPOOL_VARIABLE_NAME", (String.valueOf(++counter))); //$NON-NLS-1$
		}
		
		return uniqueVariableName;
	}

	public static boolean isVariableNameUnique(String name, IDatapool datapool) {
		if (name == null)
			throw new IllegalArgumentException();
		for (int i = 0; i < datapool.getVariableCount(); i++) {
			if (name.equals(datapool.getVariable(i).getName()))
				return false;
		}
		return true;
	}

	/**
	 * This method adds a new Evaluator class for the sepcified class name.
	 * @param tag
	 * @param newEvalClassName
	 * @provisional
	 */
	public static  void addEvaluator(String tag, String newEvalClassName) { 
        try {
           Class cls  = Class.forName(newEvalClassName); 
           IDatapoolEvaluatorExt newEval = (IDatapoolEvaluatorExt)cls.
           		newInstance();
           dpEvaluatorsMap.put(tag, newEval);
           DPLLogImpl.log("Evaluator added to map: " + tag + " " + 
        		   newEvalClassName);
        }
        catch (Throwable t) 
        {
                DPLLogImpl.log("addEvaluator: " + t.getMessage());
        }
	}
}
