/*******************************************************************************
 * Copyright (c) 2003, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionUtil.java,v 1.23 2009/12/16 20:29:33 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.common.provisional.TestlogFilenameExtensionManager;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.models.common.facades.behavioral.IAction;
import org.eclipse.hyades.models.common.facades.behavioral.IBlock;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.QueryResult;
import org.eclipse.hyades.models.hierarchy.extensions.ResultEntry;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;

import com.ibm.icu.util.Calendar;

/**
 * <p>ExecutionUtil.java</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version December 16, 2009
 * @since   January 25, 2005
 */
public class ExecutionUtil
{
	/**
	 * Returns a list with the execution results of a given 
	 * {@link TPFTest test}.  The execution results are queried from the 
	 * workspace based on matching the test suite's name.
	 * @param test
	 * @return a not <code>null</code> list
	 */
	public static List findExecutionResults(final TPFTest test)
	{
		if((test != null) && (test.eResource() != null))
		{
			URI uri = test.eResource().getURI().trimFileExtension();
			final String executionPrefix = uri.lastSegment().toString();
			ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					if(file.getName().startsWith(executionPrefix))
					{
						EObject[] eObjects = ResourceUtil.load(new ResourceSetImpl(), file);
						if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
						{
							TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
							if(executionResult.getTest().getId().equals(test.getId())){
								return executionResult;
							}
						}
					}
					return null;
				}
			};
			
			List fileList = buildsLogFileList(test, visitor);
			
			return fileList;

		}
		
		return Collections.EMPTY_LIST;
	}
	
	private static List buildsLogFileList(final TPFTest test, ResourceUtil.IGetFilesVisitor visitor){
		
		List fileList = ((List)(ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), new String[]{ICommonConstants.EXECUTION_FILE_EXTENSION}, visitor).get(ICommonConstants.EXECUTION_FILE_EXTENSION)));
		
		String customFileNameExtension = TestlogFilenameExtensionManager.getInstance().getExtensionForTestType(test.getType());
		
		if(customFileNameExtension != null){
			
			List customList = ((List)(ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), new String[]{customFileNameExtension}, visitor).get(customFileNameExtension)));
			
			if((customList != null) && (customList.size() > 0)){
				fileList.addAll(customList);
			}
		}
		
		return fileList;		
	}

 	/**
	 * <p>Resolves the list of execution results for a {@link TPFTest test} created in date and 
	 * time window (end date and time - start date and time).</p>
	 * 
	 * <p><b>NOTE:</b> This method opens all the execution results in the workspace to resolve the 
	 * creation time stamp.  As such, this method is extremely processor, disk, and memory 
	 * intensive.</p>
	 * 
	 * <p><b>NOTE:</b> If the start date and time is 0 or negative, it defaults to 12:00.00 AM of 
	 * the current date.  Similarly, if the end date and time is 0 or negative, it defaults to current 
	 * date and time</p>
	 * 
	 * @param test The test to to resolve the list of execution results.
	 * @param startDate The start date and time of the date and time window.
	 * @param endDate The start date and time of the date and time window.
	 * @return A non-<code>null</code> list of execution results for the {@link TPFTest test} created in date and time window (end date and time - start date and time).
	 */
	public static List findExecutionResults(final TPFTest test, long startDate, long endDate){
		
		if((test != null) && (test.eResource() != null)){
			
			//Check the date and time window:
			if ((startDate <= 0) || (startDate <= 0)){

		        //Resolve the current date and time:
		        Calendar currentDateTimeCalendar = Calendar.getInstance();
		        currentDateTimeCalendar.set(Calendar.MILLISECOND, 0);

		        //Initialize the end date and time to the current date and time:
		        if(endDate <= 0){
		        	endDate = currentDateTimeCalendar.getTimeInMillis();
		        }

		        //Initialize the start date and time to 12:00.00 AM of the current date:
		        if(startDate <= 0){
		        	
		        	currentDateTimeCalendar.set(Calendar.HOUR_OF_DAY, 0);
		        	currentDateTimeCalendar.set(Calendar.MINUTE, 0);
		        	currentDateTimeCalendar.set(Calendar.SECOND, 0);

		        	startDate = currentDateTimeCalendar.getTimeInMillis();
		        }
			}
			
			final long fStartTime = startDate;
			
			final long fEndTime = endDate;
				
			ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					//Open the execution result file only if modified after the start time stamp:
					if(file.getLocalTimeStamp() >= fStartTime){
						EObject[] eObjects = ResourceUtil.load(new ResourceSetImpl(), file);
						if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
						{
							TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
							TPFExecutionHistory history = executionResult.getExecutionHistory();
							if ( history != null && !history.getExecutionEvents().isEmpty())
							{
								TPFExecutionEvent event = (TPFExecutionEvent) history.getExecutionEvents().get(0);
								long timestamp = event.getTimestamp();
								if ((timestamp>=fStartTime && timestamp <=fEndTime)){
									if(executionResult.getTest().getId().equals(test.getId()))
										return executionResult;									
								}
							}
						}
					}
					return null;
				}
			};
			
			return (buildsLogFileList(test, visitor));
		}
		
		return Collections.EMPTY_LIST;
	}
	
 	/**
	 * <p>Resolves the list of execution results for a list of {@link TPFTest tests} created in date and 
	 * time window (end date and time - start date and time).</p>
	 * 
	 * <p><b>NOTE:</b> This method opens all the execution results in the workspace to resolve the 
	 * creation time stamp.  As such, this method is extremely processor, disk, and memory 
	 * intensive.</p>
	 * 
	 * <p><b>NOTE:</b> If the start date and time is 0 or negative, it defaults to 12:00.00 AM of 
	 * the current date.  Similarly, if the end date and time is 0 or negative, it defaults to current 
	 * date and time</p>
	 * 
	 * @param tests The list of test to to resolve the list of execution results.
	 * @param startTime The start date and time of the date and time window.
	 * @param endTime The start date and time of the date and time window.
	 * @return A non-<code>null</code> list of execution results for the list of {@link TPFTest tests} created in date and time window (end date and time - start date and time).
	 */
	public static Map findExecutionResults(final List tests, long startDate, long endDate)
	{
		if((tests != null))
		{
			final Map results = new HashMap();
			//initialize results
			TPFTest test = null;
			for (int x = 0; x < tests.size(); x++){
				test = (TPFTest)tests.get(x);
				results.put(test.getId(), new ArrayList());
			}
			if (test != null){

				//Check the report window:
				if ((startDate <= 0) || (endDate <= 0)){

			        //Resolve the current date and time:
			        Calendar currentDateTimeCalendar = Calendar.getInstance();
			        currentDateTimeCalendar.set(Calendar.MILLISECOND, 0);

			        //Initialize the end date and time to the current date and time:
			        if(endDate <= 0){
			        	endDate = currentDateTimeCalendar.getTimeInMillis();
			        }

			        //Initialize the start date and time to 12:00.00 AM of the current date:
			        if(startDate <= 0){
			        	
			        	currentDateTimeCalendar.set(Calendar.HOUR_OF_DAY, 0);
			        	currentDateTimeCalendar.set(Calendar.MINUTE, 0);
			        	currentDateTimeCalendar.set(Calendar.SECOND, 0);

			        	startDate = currentDateTimeCalendar.getTimeInMillis();
			        }
				}
				
				final Map timeStampLookup = new HashMap();
				final long fStartTime = startDate;
				
				final long fEndTime = endDate;
					
				ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
				{
					public Object visit(IFile file)
					{	

						//Open the execution result file only if modified after the start time stamp:
						if(file.getLocalTimeStamp() >= fStartTime){
							
							EObject[] eObjects = ResourceUtil.load(new ResourceSetImpl(), file);							
							if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
							{
								TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
								TPFExecutionHistory history = executionResult.getExecutionHistory();
								if ( history != null && !history.getExecutionEvents().isEmpty())
								{
									TPFExecutionEvent event = (TPFExecutionEvent) history.getExecutionEvents().get(0);
									long timestamp = event.getTimestamp();
									if ((timestamp>=fStartTime && timestamp <=fEndTime)){
										List list = (List)results.get(executionResult.getTest().getId());
										

										if (list != null){
												long oldTimeStamp = 0;
												//There are cases where junit tests have the same id with different names.
												if ((executionResult.getTest().getType().equals("org.eclipse.hyades.test.java.junit.testSuite"))){								        									        
													boolean found = false;
													TPFExecutionResult oldResult = null;
													for (int y = 0; y < list.size(); y++){
														Object elem = list.get(y);
														if (elem instanceof TPFExecutionResult){
															if ((((TPFExecutionResult)elem).getName()).equals(executionResult.getName())){
																oldResult = (TPFExecutionResult)elem;
																found = true;
																break;
															}
														}
													}
													if (!found){
														timeStampLookup.put(executionResult, new Long(timestamp));
														list.add(executionResult);											
													}
													else if (oldResult != null){
														//need to check for current time.
														Object timeStampObj = timeStampLookup.get(oldResult);													
														if (timeStampObj != null){
															oldTimeStamp = ((Long)timeStampObj).longValue();
															//if timestamp is more current replace execution result
															if (timestamp > oldTimeStamp){
																list.remove(oldResult);
																list.add(executionResult);
																timeStampLookup.remove(oldResult);
																timeStampLookup.put(executionResult, new Long(timestamp));
															}
														}
													}
								        
										        }									
										        else{
													for (int y = 0; y < list.size(); y++){
														Object elem = list.get(y);
														if (elem instanceof Long){
															oldTimeStamp = ((Long)elem).longValue();
															break;
														}
													}
														if (timestamp > oldTimeStamp)
														{
															list.clear();
															list.add(new Long(timestamp));
															list.add(executionResult);
														}														
												}
											}
										}
								}
								return null;
							}
						}
						return null;
					}
				};
				
				String[] customExtensions = TestlogFilenameExtensionManager.getInstance().getAllRegisteredLogExtensions();

				String[] extensionList = new String[(customExtensions != null ? (customExtensions.length + 1) : 1)];
				extensionList[extensionList.length - 1] = ICommonConstants.EXECUTION_FILE_EXTENSION;
				
				ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), extensionList, visitor);				
				
				return results; 
			}
		}
		
		return null;
	}	
	
	/**
	 * Returns a list with the execution results of a given 
	 * {@link TPFTest test}.  The execution results are queried from the 
	 * workspace based on opening <b>every</b> execution result and comparing the
	 * referenced test to the specified test
	 * 
	 * Warning -- this is an extremely processor, disk & memory intensive 
	 * operation.
	 * @param test
	 * @return a not <code>null</code> list
	 */
	public static List findAllExecutionResults(final TPFTest test)
	{
		if((test != null) && (test.eResource() != null))
		{
			
			ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
			{
				public Object visit(IFile file)
				{
					EObject[] eObjects = ResourceUtil.load(new ResourceSetImpl(), file);
					if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
					{
						TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
						if(executionResult.getTest().equals(test))
							return executionResult;
					}
					return null;
				}
			};
			
			return buildsLogFileList(test, visitor);
		}
		
		return Collections.EMPTY_LIST;
	}
	
	
	/**
	 * Returns a map with the list of execution results for each 
	 * {@link TPFTest test} in the argument.  The execution results are queried 
	 * from the workspace.
	 * 
	 * <p>The tests are supposed to be located in the same <code>ResourceSet</code>. 
	 * 
	 * @param test
	 * @return a not <code>null</code> list
	 */
	public static Map findExecutionResults(List tests)
	{
		ArrayList customLogExtensions = new ArrayList();
		if((tests != null) && (!tests.isEmpty()))
		{
			ResourceSet resourceSet = null;
			Set resourcePrefixes = new HashSet(tests.size()+1);
			for (Iterator i = tests.iterator(); i.hasNext();)
			{
				Object object = i.next();
				if(object instanceof TPFTest)
				{
					TPFTest test = (TPFTest)object;
					String customExtension = TestlogFilenameExtensionManager.getInstance().getExtensionForTestType(test.getType()); 
					if(customExtension!=null && !customLogExtensions.contains(customExtension))
						customLogExtensions.add(customExtension);
					
					if(test.eResource() != null)
					{
						if(resourceSet == null)
							resourceSet = test.eResource().getResourceSet();
						
						URI uri = test.eResource().getURI().trimFileExtension();
						resourcePrefixes.add(uri.lastSegment().toString());
					}					
				}
			}
			
			if(!resourcePrefixes.isEmpty())
			{
				final ResourceSet finalResourceSet = resourceSet;
				final String[] prefixes = (String[])resourcePrefixes.toArray(new String[resourcePrefixes.size()]); 
				final Map ret = new HashMap(resourcePrefixes.size()+1,1);
				ResourceUtil.IGetFilesVisitor visitor = new ResourceUtil.IGetFilesVisitor()
				{
					public Object visit(IFile file)
					{
						for (int i = 0, maxi = prefixes.length; i < maxi; i++)
						{
							if(file.getName().startsWith(prefixes[i]))
							{
								EObject[] eObjects = ResourceUtil.load(finalResourceSet, file);
								if((eObjects.length == 1) && (eObjects[0] instanceof TPFExecutionResult))
								{
									TPFExecutionResult executionResult = (TPFExecutionResult)eObjects[0];
									List executionResults = (List)ret.get(executionResult.getTest());
									if(executionResults == null)
									{
										executionResults = new ArrayList();
										ret.put(executionResult.getTest(), executionResults);
									}
									executionResults.add(executionResult);
								}
							}
						}
						return null;
					}
				};
			
				String[] customExtensions = (String[]) customLogExtensions.toArray(new String[0]);
				String[] fileExtensionList = new String[customExtensions.length+1];
				fileExtensionList[fileExtensionList.length-1]=ICommonConstants.EXECUTION_FILE_EXTENSION;
				ResourceUtil.getFiles(ResourcesPlugin.getWorkspace().getRoot(), fileExtensionList, visitor);
				
				return ret;
			}
		}
		
		return Collections.EMPTY_MAP;
	}
	
	/**
	 * Recursively adds all referenced test suites from and including the 
	 * parameter root test suite.
	 * <p>
	 * Referenced test suites are defined as invocations of external test suites.
	 * <p>
	 * Only referenced test suites not in the parameter list ({@link TPFTestSuite}) 
	 * are added.  As such, the parameter list ({@link TPFTestSuite}) does not 
	 * contain duplicate test suites.
	 * <p> 
	 * 
	 * @param testSuite The root test suite.
	 * @param referencedTestSuites The list ({@link TPFTestSuite}) of referenced test suites.
	 */
	public static void addReferencedTestSuites(TPFTestSuite testSuite, List referencedTestSuites) {

		//Step 1. Add the parameter test suite:
		referencedTestSuites.add(testSuite);
		
		//Step 2. Add all the referenced test suite(s):   
		IImplementor implementor = testSuite.getImplementor();

		if (implementor != null) {

			IBlock block = implementor.getBlock();

			if (block != null) {	 			
				addReferencedTestSuites(block, referencedTestSuites);
			}            
		}
	}	

	/**
	 * Recursively adds all referenced test suites from the parameter root 
	 * block to the parameter list ({@link TPFTestSuite}).
	 * <p>
	 * Referenced test suites are defined as invocations of external test suites.
	 * <p>
	 * Only referenced test suites not in the parameter list ({@link TPFTestSuite}) 
	 * are added.  As such, the parameter list ({@link TPFTestSuite}) does not 
	 * contain duplicate test suites.
	 * <p> 
	 * 
	 * @param block The root block.
	 * @param referencedTestSuites The list ({@link TPFTestSuite}) of referenced test suites.
	 */
	private static void addReferencedTestSuites(IBlock block, List referencedTestSuites) {

		Iterator actionsIterator = block.getActions().iterator();

		while(actionsIterator.hasNext()) {

			IAction action = ((IAction)(actionsIterator.next()));

			if (action instanceof ITestInvocation) {

				ITest test = ((ITestInvocation)(action)).getInvokedTest();

				//Only invocations of external test suites currently not in the list are added:
				if (test instanceof TPFTestSuite) {

					if(!referencedTestSuites.contains(test)) {
						addReferencedTestSuites(((TPFTestSuite)(test)), referencedTestSuites);
					}
				} 
				else if (test instanceof TPFTestCase) {

					TPFTestSuite testSuite = ((TPFTestCase) (test)).getTestSuite();

					if(!referencedTestSuites.contains(testSuite)) {
						addReferencedTestSuites(testSuite, referencedTestSuites);
					}
				}
			} 
			else if (action instanceof ILoop) {
				addReferencedTestSuites(((ILoop)(action)).getBlock(), referencedTestSuites);
			}
		}
	}
	
    /**
     * Determines if the parameter test suite contains at least one test case invocation.
     * <p/>
     * A test case invocation constitutes one of the following:
     * <p/>
     * <ul>
     * <li>An invocation of a test case in the test suite.</li> 
     * <li>An invocation of a test case in a referenced test suite.</li> 
     * <li>One or more test cases with an external implementor (e.g. test suite behavior 
     * implemented as code).</li> 
     * </ul>
     * <p/>
     * Note, an invocation of a referenced test suite does not constitute a test case 
     * invocation.
     * <p/>
     * 
     * @param testSuite The test suite.
     * @return True if the parameter test suite contains at least one test case invocation, otherwise false.
     */
    public static boolean containsTestCaseInvocations(TPFTestSuite testSuite){
		
		IImplementor implementor = testSuite.getImplementor();

		if (implementor != null) {

			IBlock block = implementor.getBlock();

			if (block != null) {	 			
				return (containsTestCaseInvocations(block));
			}   
			
			//Handle the case where the test suite has an external implementor and test cases:
			else{				
				return (testSuite.getTestCases().size() > 0); 
			}
		}
		
		return false;
	}
	
    /**
     * Determines if the parameter root block contains at least one test case invocation.
     * <p/>
     * A test case invocation constitutes one of the following:
     * <p/>
     * <ul>
     * <li>An invocation of a test case in the test suite.</li> 
     * <li>An invocation of a test case in a referenced test suite.</li> 
     * <li>One or more test cases with an external implementor (e.g. test suite behavior 
     * implemented as code).</li> 
     * </ul>
     * <p/>
     * Note, an invocation of a referenced test suite does not constitute a test case 
     * invocation.
     * <p/>
     * 
     * @param block The root block.
     * @return True if the parameter root block contains at least one test case invocation, otherwise false.
     */
    private static boolean containsTestCaseInvocations(IBlock block) {

		Iterator actionsIterator = block.getActions().iterator();

		while(actionsIterator.hasNext()) {

			IAction action = ((IAction)(actionsIterator.next()));

			if (action instanceof ITestInvocation) {

				if(((ITestInvocation)(action)).getInvokedTest() instanceof TPFTestCase){
					return true;
				}
			} 
			else if (action instanceof ILoop) {
				
				if(containsTestCaseInvocations(((ILoop)(action)).getBlock())){
					return true;
				}
			}
		}
		
		return false;
	}
    
	/**
     * Recursively resolves the count of all test invocations for the parameter 
     * test suite.
     * <p/>
     * A test case invocation constitutes one of the following:
     * <p/>
     * <ul>
     * <li>An invocation of a test case in the test suite.</li> 
     * <li>An invocation of a test case in a referenced test suite.</li> 
     * <li>An invocation of a referenced test suite.*</li> 
     * <li>One or more test cases with an external implementor (e.g. test suite behavior 
     * implemented as code).</li> 
     * </ul>
     * <p/>
     * Note, an invocation of a referenced test suite recursively resolves the count of all 
     * test invocations for the referenced test suite.  The referenced test suite is not counted.
     * <p/>
     * 
     * @param testSuite The test suite.
     * @return The count of all test invocations for the parameter test suite.
     */
    public static int resolveTestCaseCount(TPFTestSuite testSuite){
		
    	int count = 0;
		IImplementor implementor = testSuite.getImplementor();

		if (implementor != null) {

			IBlock block = implementor.getBlock();

			if (block != null) {	 			
				count = (resolveTestCaseCount(block));
			}   
			
			//Handle the case where the test suite has an external implementor and test cases:
			else{				
				count = (testSuite.getTestCases().size()); 
			}
		}
		
		return count;
	}
	
    /**
    * Recursively resolves the count of all test invocations for the parameter 
     * root block.
     * <p/>
     * A test case invocation constitutes one of the following:
     * <p/>
     * <ul>
     * <li>An invocation of a test case in the test suite.</li> 
     * <li>An invocation of a test case in a referenced test suite.</li> 
     * <li>An invocation of a referenced test suite.*</li> 
     * <li>One or more test cases with an external implementor (e.g. test suite behavior 
     * implemented as code).</li> 
     * </ul>
     * <p/>
     * Note, an invocation of a referenced test suite recursively resolves the count of all 
     * test invocations for the referenced test suite.  The referenced test suite is not counted.
     * <p/>
     * 
     * @param block The root block.
     * @return The count of all test invocations for the parameter root block.
     */
    private static int resolveTestCaseCount(IBlock block) {

    	int count = 0;
		Iterator actionsIterator = block.getActions().iterator();

		while(actionsIterator.hasNext()) {

			IAction action = ((IAction)(actionsIterator.next()));

			if (action instanceof ITestInvocation) {

				ITest test = ((ITestInvocation)(action)).getInvokedTest();

				if (test instanceof TPFTestSuite) {
					count += resolveTestCaseCount(((TPFTestSuite)(test)));
				} 
				else if (test instanceof TPFTestCase) {
					count += 1;
				}
			} 
			else if (action instanceof ILoop) {
				count += resolveTestCaseCount(((ILoop)(action)).getBlock());
			}
		}
		
		return count;
	}
    
    /**
	 * Resolves the list ({@link TPFVerdictEvent}) of verdict events  from the 
	 * parameter root execution result.
	 * <p>
	 * 
	 * @param testExecutionResult The root execution result.
	 * @return The list ({@link TPFVerdictEvent}) of verdict events.
	 */
	public static List getVerdictEvents(TPFExecutionResult testExecutionResult) {

		SimpleSearchQuery verdictQuery = ExtensionsFactory.eINSTANCE.createSimpleSearchQuery();
		verdictQuery.getSources().add(EcoreUtil.getURI(testExecutionResult).toString());

		//Minimum paths required to traverse the complete execution history:
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionResult_ExecutionHistory());
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionHistory_ExecutionEvents());
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFExecutionEvent_Children());
		verdictQuery.getRequiredPaths().add(Common_TestprofilePackage.eINSTANCE.getTPFInvocationEvent_InvokedExecutionResult());

		SimpleOperand operand = ExtensionsFactory.eINSTANCE.createSimpleOperand();
		operand.setType(Common_TestprofilePackage.eINSTANCE.getTPFVerdictEvent());
		verdictQuery.getOutputElements().add(operand);
		
		LogicalExpression whereExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
		whereExpression.setOperator(LogicalOperators.OR_LITERAL);
							
		verdictQuery.setWhereExpression(whereExpression);
		
		QueryResult queryResult = null;
		IHyadesResourceExtension hyadesResourceExtension = (IHyadesResourceExtension) HyadesResourceExtensions.getInstance().get(LoadersUtils.getPostfix(testExecutionResult.eResource().getURI().toString()));
		if (hyadesResourceExtension != null) {
			List notLoadedTypes = new ArrayList();
			ResourceSet resourceSet = testExecutionResult.eResource().getResourceSet();
			queryResult = hyadesResourceExtension.executeQuery(verdictQuery, resourceSet, notLoadedTypes);
		}
		
		List result = queryResult.getResultEntries();
		
		if (!result.isEmpty()) {
			return (((List)(((ResultEntry) result.get(0)).getValue())));
		}
		
		return (Collections.EMPTY_LIST);
	}	
}
