/*******************************************************************************
 * Copyright (c) 2003, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileUtil.java,v 1.7 2007/07/23 13:20:37 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.models.common.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.Vector;

/**
 * This class implements some utility methods for dealing with java.io.File
 * Some of the methods require JDK 1.4
 * 
 * @author jtoomey
 *
 */
public class FileUtil {

	static File tempDir=null;
	/**
  	 * Returns the system temp directory.
  	 * @return File
  	 */
    public static synchronized File getTempDir() {
		if (tempDir == null) {
			try {
				File tempFile = File.createTempFile("test", "FileUtil");
				tempDir = tempFile.getParentFile();
				tempFile.delete();
			} catch (IOException e) {
				ResourceUtil.logError(e);
			}
		} else if (!tempDir.exists()) {
			tempDir.mkdirs();
		}
		return tempDir;
	}
    
    /**
	 * This method uses JDK 1.4 nio channels to perform fast file copy. Use
	 * copyFileSafe instead to receive exception if copy fails
	 * 
	 * @param srcFile
	 * @param destFile
	 * @return true if the file copy was successful, false otherwise
	 * 
	 */
    public static boolean copyFile( String srcFile, String destFile )
    {
    	try {
    		copyFileSafe(srcFile, destFile);
		} 
    	catch (IOException e) {
    		return false;
    	}
    	return true;
    }
    
    public static void copyFileSafe(String srcFile, String destFile) 
    throws IOException {
    	final int MB = 1024 * 1024;
    	
		FileChannel srcChannel = new FileInputStream(srcFile).getChannel();
		FileChannel destChannel = new FileOutputStream(destFile).getChannel();
		// Transfer at most 32MB at a time, to workaround several windows 
		// specific JVM defects that are not planned to be fixed.
		// http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4643189
		// http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4938442
		long fileSize = srcChannel.size();
		long maxBuf = 32 * MB;
		long bytesTransferred = 0;
		for (long l=0; l < fileSize / maxBuf; l++) {
    		long startPos = bytesTransferred;
    		destChannel.transferFrom(srcChannel, startPos, maxBuf);
			bytesTransferred += maxBuf;
		}
		if (fileSize % maxBuf != 0) {
    		long startPos = bytesTransferred;
    		destChannel.transferFrom(srcChannel, startPos, fileSize % maxBuf);
			bytesTransferred += fileSize % maxBuf;
		}
		srcChannel.close();
		destChannel.close();
    }

    
    /**
     * This method returns an array of File objects that are children of the
     * specified srcDir, and that return <code>true</code> to the .isFile() method.
     * In other words, the array is the result of a recursive search of the
     * specified directory for all files (but not directories.) 
     * @param srcDir
     * @return File[] All files that are children of the specified directory, 
     * 		   or null if srcDir does not contain any files.
     */
    public static File[] listAllFiles(File srcDir)
    {
    	Vector files = new Vector();
    	File[] children = srcDir.listFiles();
    	if ( children == null )
    	{
    		return new File[]{};
    	}
    	
    	for ( int i=0; i<children.length; i++ )
    	{
    		if ( children[i].isFile())
    		{
    			files.add(children[i]);
    		}
    		else
    		{
    			File[] sub = listAllFiles(children[i]);
    			for ( int j=0; j<sub.length; j++ )
    			{
    				files.add(sub[j]);
    			}
    		}
    	}
    	
    	return (File[])files.toArray(new File[files.size()]);
    }
    
    /**
     * This method attempts to delete the specified root file.  If root is a directory
     * with children, this method attempts for first recursively delete all children 
     * of root.  If this methods returns true, then root has been deleted.  If this
     * method returns false, root has not been deleted, however if root is a 
     * directory, it is possible that some of its children were successfully 
     * deleted. 
     * @param root
     * @return true if root has bene deleted, false otherwise.
     */
    public static boolean deltree(File root)
    {
    	if ( root.exists() )
    	{
    		if ( root.isFile() )
    		{
    			return root.delete();
    		}
    		else
    		{
    			File[] children = root.listFiles();
    			if ( children != null )
    			{
	    			for (int i=0; i<children.length; i++)
	    			{
	    				deltree(children[i]);
	    			}
    			}
    			return root.delete();
    		}
    	}
    	return true;
    }
	public static void main(String[] args) {
		
		System.out.println("FileUtil.getTempDir()="+FileUtil.getTempDir());
		long time = System.currentTimeMillis();
		int i;
		for (i = 0; i < 1000000; i++) {
			FileUtil.getTempDir();
		}
		System.out.println("i="+i+", elapsedTime="+(System.currentTimeMillis()-time));
	}
}
