/********************************************************************** 
 * Copyright (c) 2003, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ResourceCache.java,v 1.7 2010/04/16 17:48:04 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.models.common.util;

import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;

/**
 * <p>Cache for sharing in-memory {@link Resource}s.</p>
 * 
 * <p>To create a new sharable {@link Resource} for a given {@link URI} and {@link ResourceSet}, 
 * call {@link #createSharedResource(URI, ResourceSet)}.  The cache will create a new instance 
 * of the {@link Resource} in the {@link ResourceSet}, loaded into memory, and set the reference 
 * counter to 1.</p>
 * 
 * <p>To resolve a shared {@link Resource} for a given {@link URI}, call {@link #getSharedResource(URI)}.  
 * If the {@link Resource} has not been opened by the {@link ResourceCache}, the cache will 
 * create a new instance of the {@link Resource} in a new {@link ResourceSet} and set the 
 * reference counter to 1.  If the {@link Resource} has been opened by the {@link ResourceCache}, 
 * the cache will increment the reference counter and return a previously created instance of the 
 * {@link Resource}.</p>
 *  
 * <p>To release a shared {@link Resource} for a given {@link URI}, call {@link #releaseSharedResource(URI)}.  
 * The cache will decrement the reference counter.  When the reference counter is 0, the {@link Resource}
 * is removed from the cache and unloaded from memory.</p>
 * 
 * 
 * @author  Joseph Toomey
 * @author  Paul Slauenwhite
 * @version April 16, 2010
 * @since   March 17, 2005
 */
public class ResourceCache {
	
	//TODO: 
	//1. Make sure that createSharedResource's {@link URI} parameter is not already in the
	//cache (must be at more one element in the cache for a given {@link URI}).
	//2. Is it reasonable to take URI as parameter for releaseSharedResource, or
	//should we take the Resource as the parameter and compute the URI for the caller. 
	//3. Absolutize all URIs before adding to cache?
	//4. Synchronize access to the pairs in the table by locking on the table.

	private class ResourcePair
	{
		private Resource resource;
		private int refCount;
		
		/**
		 * @param resource
		 */
		public ResourcePair(Resource resource) {
			super();
			this.resource = resource;
			this.refCount = 1;
		}
		public void incrementRefCount()
		{
			refCount++;
		}

		public void decrementRefCount()
		{
			refCount--;
		}

		/**
		 * @return Returns the refCount.
		 */
		public int getRefCount() {
			return refCount;
		}

		/**
		 * @return Returns the resource.
		 */
		public Resource getResource() {
			return resource;
		}
	}
	
	private Hashtable resourceMap;
	private static ResourceCache instance = null;

	public static ResourceCache getInstance() {
		if ( instance == null )
		{
			instance = new ResourceCache();
			instance.resourceMap = new Hashtable();		
		}
		return instance;
	}

	private ResourceCache() {
		super();
		// TODO Auto-generated constructor stub
	}
	
	/**
	 * <p>If a client wants a resource for a given, existing URI, he may
	 * call getSharedResource.  This will either open a new instance of that resource
	 * in a new ResourceSet and set the refCount to 1 (if the resource has not yet been
	 * opened by the ResourceCache), or it will increment the refCount and return a
	 * previously opened instance of the Resource if one has already been opened.</p>
	 * @param resourceURI the URI of the resource requested by the caller
	 * @return a shared instance of the resource specified by the resourceURI
	 * @see releaseSharedResource(URI)
	 */
	public Resource getSharedResource(URI resourceURI){
		
		ResourcePair pair = ((ResourcePair)(resourceMap.get(resourceURI)));
		
		if (pair != null){
			
			pair.incrementRefCount();
			
			return (pair.getResource());
		}
		else{
			
			ResourceSet resourceSet = createResourceSet();
			Resource resource = resourceSet.getResource(resourceURI, true);
		
			if (resource != null){
				resourceMap.put(resourceURI, new ResourcePair(resource));
			}
			
			return resource;
		}
	}

	public ResourceSet createResourceSet(){
		
		Adapter adapter = new AdapterImpl(){
			
			/* (non-Javadoc)
			 * @see org.eclipse.emf.common.notify.impl.AdapterImpl#notifyChanged(org.eclipse.emf.common.notify.Notification)
			 */
			public void notifyChanged(Notification msg){
				
				switch(msg.getEventType()){
				case Notification.ADD:
					if(msg.getNewValue() instanceof Resource){
						((Resource)msg.getNewValue()).setTrackingModification(true);
					}
					break;

				case Notification.ADD_MANY:
					
					Collection collection = (Collection)msg.getNewValue();
					for (Iterator i = collection.iterator(); i.hasNext();)
					{
						Object element = i.next();
						
						if(element instanceof Resource){
							((Resource)element).setTrackingModification(true);
						}
					}
					break;	
				}
			}
		};
		
		ResourceSet resourceSet = new ResourceSetImpl();
		resourceSet.eAdapters().add(adapter);
		
		return resourceSet;
	}
	
	/**
	 * If a client wants to create a new resource that will be sharable, he may call
	 * createSharedResource and specify the URI for the resource to be created and
	 * the resource set in which the resource will be created.  This results in 
	 * creating the requested resource and registering it with a refCount of 1 in the
	 * cache.
	 * @param resourceURI the URI of the resource that the caller is requesting to create and share
	 * @param resourceSet the resource set in which the new resource will be created 
	 * @return a shared instance of a new resource specified by the resourceURI in 
	 * the specified resourceSet
	 * @see releaseSharedResource(URI)
	 */
	public Resource createSharedResource(URI resourceURI, ResourceSet resourceSet)
	{
		Resource newResource = resourceSet.createResource(resourceURI);
		if ( newResource != null )
		{
			ResourcePair pair = new ResourcePair(newResource);
			resourceMap.put(resourceURI, pair);
		}
		return newResource;
	}
	
	public int getSharedResourceCount(URI resourceURI)
	{
//		int count = 0;
		ResourcePair pair = (ResourcePair)resourceMap.get(resourceURI);
		if ( pair != null )
			return pair.getRefCount();
		else
			return 0;
	}
	
	/**
	 * <p>When a caller is finished using a given resource, he should call releaseSharedResource
	 * with the Resource's URI.  This will decrement the resource's refCount, and when the
	 * refCount reaches zero, the resource will be removed from the cache and unloaded
	 * from memory.</p>
	 * @param resourceURI the URI of the shared resource with which the caller is finished
	 */
	public void releaseSharedResource(URI resourceURI)
	{
//		int count = 0;
		ResourcePair pair = (ResourcePair)resourceMap.get(resourceURI);
		if ( pair != null )
		{
			pair.decrementRefCount();
			if ( pair.getRefCount() < 1 )
			{
				// No more references to this resource.
				// Remove it from the map, and unload the model.
				resourceMap.remove(resourceURI);
				Resource resource = pair.getResource();
				ResourceSet rs = resource.getResourceSet();
				if (rs != null) {
					rs.eAdapters().clear();
				}
				resource.unload();
			}
		}
	}
}
