/**********************************************************************
 * Copyright (c) 2003, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SaveManager.java,v 1.7 2006/08/22 18:55:00 jptoomey Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.models.common.util;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.hyades.loaders.util.RegistryReader;
import org.eclipse.hyades.models.plugin.ModelsPlugin;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class SaveManager
extends TimerTask
{
	public static final Map RESOURCE_OPTIONS = new HashMap();
	static {
		RESOURCE_OPTIONS.put(XMIResource.OPTION_DECLARE_XML, Boolean.TRUE);
		RESOURCE_OPTIONS.put(XMIResource.OPTION_SKIP_ESCAPE, Boolean.FALSE);
		RESOURCE_OPTIONS.put(XMIResource.OPTION_SKIP_ESCAPE_URI, Boolean.FALSE);
		RESOURCE_OPTIONS.put(XMLResource.OPTION_RECORD_UNKNOWN_FEATURE, Boolean.TRUE);
	}

	private static SaveManager instance;

	private Timer timer;
	private List resources;
	private ArrayList saveState;
	private boolean running;
	
	private SaveManager()
	{
		running = false;
	}
	
	public synchronized void dispose()
	{
		stop();
		
		if(resources != null)
			resources.clear();

		instance = null;
	}
	
	public static SaveManager getInstance()
	{
		if(instance == null)
			instance = new SaveManager();
		return instance;
	}

	public static void saveResource(EObject eObject)
	{
		if(eObject != null)
			saveResource(eObject.eResource());
	}
	
	public static void saveResource(Resource resource)
	{
		final Resource theResource = resource;
		if(theResource != null)
		{
			try
			{
				// If we're running inside the workbench, and if we're in 
				// "workspace mode" (which is an artificial designation 
				// we introduced to support RCP apps that do not use the
				// workspace, but which still have the core.resources plugin)
				// we need to wrap this save action in an IWorkspaceRunnable
				// to avoid notifying resource listeners while the save is
				// in progress.  This is run in the current thread, but the
				// action is atomic to all workspace aware resource listeners.
				if (Platform.isRunning() && RegistryReader.isWorkspaceMode()) {
					IWorkspaceRunnable runnable = new IWorkspaceRunnable(){
						public void run(IProgressMonitor arg0) throws CoreException {
							try {
								theResource.save(RESOURCE_OPTIONS);
							} catch (IOException e) {
								IStatus status = new Status(IStatus.ERROR, ResourcesPlugin.getPlugin().getBundle().getSymbolicName(), 0, e.getMessage(), e);
								throw new CoreException(status);
							}
						}
					};
					IWorkspace workspace = ResourcesPlugin.getWorkspace();
					workspace.run(runnable, new NullProgressMonitor());
				}
				else
					theResource.save(RESOURCE_OPTIONS);
			}catch (CoreException e) {
				ModelsPlugin.INSTANCE.log(e.getStatus().getException());
			}
			catch (IOException e)
			{
				ModelsPlugin.INSTANCE.log(e);
			} 
		}
	}
	
	public void addResource(Resource resource)
	{
		if(resource == null)
			return;
			
		resource.setTrackingModification(true);
		if(resources == null)
			resources = new ArrayList();
		if(saveState == null)
			saveState = new ArrayList();
		resources.add(resource);
		saveState.add(Boolean.FALSE);
	}

	public void removeResource(Resource resource)
	{
		if(resources != null)
		{
			int index = resources.indexOf(resource);
			if ( index != -1 )
			{
				saveState.remove(index);
				resources.remove(index);
			}
		}
	}
	
	public void removeAllResources()
	{
		if(resources != null)
			resources.clear();
		if(saveState != null)
			saveState.clear();
	}
	
	public Resource[] getResources()
	{
		if(resources == null)
			return new Resource[0];
		return (Resource[])resources.toArray(new Resource[resources.size()]);
	}
	
	public Boolean[] getSaveState()
	{
		if(saveState == null)
			return new Boolean[0];
		return (Boolean[])saveState.toArray(new Boolean[saveState.size()]);
	}

	public void saveResources()
	{	
		if((resources == null) || (resources.isEmpty()))
			return;

		Resource[] resourceArray = getResources();
		Boolean resourceSaveState = Boolean.FALSE;
		Resource resource = null;
		for(int i = 0, maxi = resourceArray.length; i < maxi; i++)
		{
			resource = resourceArray[i];
			if (resource.isModified())
			{
				// if the resource is currently being saved by another
				// SaveManager thread, then we don't want to save it again.
				// Just continue.  Otherwise, mark the fact that we are
				// saving it, and then save it.
				synchronized(resource) {
					resourceSaveState = (Boolean)saveState.get(i);
					if (resourceSaveState == Boolean.TRUE)
						continue;
					else
						saveState.set(i, Boolean.TRUE);
				}
				saveResource(resource);

				// If the test has not stopped (and removed this 
				// resource from our set while we were saving),
				// update the save state to indicate that we are finished.
				if (resources.contains(resource))
					saveState.set(i, Boolean.FALSE);
			}
		}
	}
	
	public synchronized void start(long saveInterval)
	{
		if((!isRunning()) && (saveInterval > 0))
		{
			timer = new Timer(true);
			timer.schedule(this, saveInterval, saveInterval);
			running = true;
		}
	}
	
	public synchronized void stop()
	{
		running = false;
		if(timer != null)
		{
			timer.cancel();
			timer = null;
		}
	}
	
	public boolean isRunning()
	{
		return running;
	}
		
	public void run()
	{
		if((resources == null) || (resources.isEmpty()))
			return;
		
		Thread thread = new Thread()
		{
			public void run()
			{
				if(isRunning())
					saveResources();
			}
		};
		thread.setName(getClass().getName());
		thread.start();
	}	
}
