/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * IBM - Initial API and implementation
 * 
 * $Id$
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Stack;

import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.trace.TRCAggregatedMethodInvocation;
import org.eclipse.hyades.models.trace.TRCClass;
import org.eclipse.hyades.models.trace.TRCMethod;
import org.eclipse.hyades.models.trace.TRCPackage;
import org.eclipse.hyades.models.trace.TRCProcess;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TraceFactory;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.models.util.ModelDebugger;


/**
 * @author slavescu
 *
 */
public class XMLagMethodEntryLoader extends TraceMethodBaseLoader {
	protected double baseTime;
	protected double minTime;
	protected double maxTime;
//	protected int callerIdRef;
	protected int numCalls;
	
	protected static final String AG_BASE_TIME = "baseTime";
//	protected static final String MIN_TIME = "minTime";
//	protected static final String MAX_TIME = "maxTime";
	protected static final String NUM_CALLS = "numcalls";
//	protected static final String CALLER_ID_REF = "callerIdRef";
	protected double cpuTime;
	protected IdentityHashMap visitedNodes;
	
	public void addAttribute(String name, String value) {
		switch (LoadersUtils.getHashCode(name)) {
		case TraceConstants.AG_BASE_TIME_int : 
		case TraceConstants.BASE_TIME_int : 
						baseTime = Double.parseDouble(value);
						break;
/*		case TraceConstants.callerIdRefHashCode : 
						callerIdRef = Integer.parseInt(value);
						break;
*/
		case TraceConstants.MIN_TIME_int: 
						minTime = Double.parseDouble(value);
						break;
		case TraceConstants.MAX_TIME_int :
						maxTime = Double.parseDouble(value);
						break;
		case TraceConstants.COUNT_int: // this is the supported attribute
		case TraceConstants.NUM_CALLS_int :
						numCalls = Integer.parseInt(value);
						break;
		case TraceConstants.CPU_TIME_int :
						cpuTime = LoadersUtils.parseTraceTime(value);
						break;
		default:		
				super.addAttribute(name, value);
 
		}
					
	}
	

	protected void processEF(int step) {

	
		if (ModelDebugger.INSTANCE.debug) {
			System.out.println("push");
		}
		
		// TODO: we don't really need a Stack here, just a reference to
		// a TRCAggregtatedMethodInvocation. The invokedBy field serves
		// as a "next frame down the stack" pointer.

		cs = (CallStackPerThread) LookupServiceExtensions.getInstance().locate(context, CallStackPerThread.class, threadIdRef);

		if (cs == null) {
			Object csKey = LoadersUtils.getLookUpKey(threadIdRef);
			cs = new CallStackPerThread();
			LoadersUtils.registerGenericLookUpEntry(context, csKey, cs);
		}
        
        TRCAggregatedMethodInvocation tos = null;
        TRCAggregatedMethodInvocation invocation = null;
        
        if( cs.size() > 0 ) { 
        	tos = (TRCAggregatedMethodInvocation) cs.peek();
		
        	List l = tos.getInvokes();
			for (int i = l.size(); --i>=0;) {
				Object in = l.get(i);
				if( in instanceof TRCAggregatedMethodInvocation ) {
					TRCAggregatedMethodInvocation aggIn = (TRCAggregatedMethodInvocation) in;
					if( aggIn.getMethod() == theMethod ) {
						if (ModelDebugger.INSTANCE.debug) {
							System.out.println("re-use existing agMI for this frame");
						}
						invocation = aggIn;
						break;
					}
				}
			}
		}
        else
        {
        	// look for a top of the stack invocation of this method in the same thread
        	List l = theMethod.getInvocations();
			for (int i = l.size(); --i>=0;) {
				Object in = l.get(i);
				if( in instanceof TRCAggregatedMethodInvocation ) {
					TRCAggregatedMethodInvocation aggIn = (TRCAggregatedMethodInvocation) in;
					if( aggIn.getThread() == theThread && aggIn.getInvokedBy() == null) {
						if (ModelDebugger.INSTANCE.debug) {
							System.out.println("re-use existing agMI for this frame");
						}
						invocation = aggIn;
						break;
					}
				}
			}
        }
        // else leave tos == null, meaning this stack was empty


		if (invocation == null) {
			invocation = TraceFactory.eINSTANCE.createTRCAggregatedMethodInvocation();
			invocation.setMethod(theMethod);
//			invocation.setOwningObject(null);
			invocation.setProcess(theProcess);
			invocation.setThread(theThread);
			if (tos == null) {
				// This stack was empty. Add to the thread's initial invocations.
				theThread.getInitialInvocations().add(invocation);
			}
			else {
				invocation.setInvokedBy(tos);
			}
		}
		
		cs.push(invocation);
		
		
		// Add deltas to existing counts and times.
		// The common case is where we just created a new TRCAgMI object,
		// so all the times are zero and the deltas are equal to what's in
		// this XML element. But the other case is where we
		// are getting a second snapshot after some initial data was received.
		// In this case we want to add only the delta, the
		// amount that wasn't already there.

		int deltaNumCalls;
		double deltaBaseTime;
		
		if (theProcess.getAgent().getName().equals("org.eclipse.tptp.jvmti")) {
			deltaNumCalls = numCalls;
			deltaBaseTime = baseTime;
			invocation.setCount(invocation.getCount() + numCalls);
			invocation.setBaseTime(invocation.getBaseTime() + baseTime);			
		}
		else /*JVM PI*/ {
			deltaBaseTime = baseTime - invocation.getBaseTime();
			deltaNumCalls = numCalls - invocation.getCount();
			invocation.setCount(numCalls);
			invocation.setBaseTime(baseTime);
		}
			
			
			
		if (invocation.getMinTime() == 0 || invocation.getMinTime() > minTime)
			invocation.setMinTime(minTime);
		if (invocation.getMaxTime() < maxTime)
			invocation.setMaxTime(maxTime);
		
		
		if(numCalls==0 || (deltaNumCalls==0 && deltaBaseTime==0))
			return;

		updateMethodStatisticalInfo(cs, deltaBaseTime, deltaNumCalls);
		
		// System.out.println("Pushed " + invocation);
	}
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.trace.TraceXMLFragmentLoader#processENI(int)
	 */
	protected void processENI(int step) {
		super.processENI(step);
	}
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.trace.TraceXMLFragmentLoader#processES(int)
	 */
	protected void processES(int step) {
		processEF(step);
	}

	
	public void addYourselfInContext() {
		if(!loadToModel)
			return;
		if (context.isFilterOn()) {
			if (context.getFilterEngine().isFiltered(TracePackage.eINSTANCE.getTRCMethod_Name(), LoadersUtils.getLookUpKey(methodIdRef))) {
				return;
			}
		}		
        theProcess = getProcess();
        theThread = getThreadByIdRef(theProcess);

        theMethod = getMethodByIdRef(null);

		dispatchProcessMode(ProcessSteps.ALL);
	}
	
//	public void processHS(int step) {
////		processES(step);
//	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.loaders.util.XMLFragmentLoader#cleanUp()
	 */
	public void cleanUp() {
		if (ModelDebugger.INSTANCE.debug) {
			// Debugging: dump all the method IDs in a structured way
			System.out.println("Cleaning up aggrmethodcallload");
			dump( new CallBack() {
				public void call(TRCAggregatedMethodInvocation o, int level) {
					for( int i=0; i < level; i++) System.out.print("  ");
					System.out.println(o.getMethod().getId());
				}
				
			}, theProcess.getThreads().iterator(),0);
		}

		// TODO: roll up cumulative times here?
		// This is called after the data has been loaded; we have to
		// zero out the base and cumulative times in the process,
		// thread, package, class, and method objects, and re-compute
		// them based on what's in the model now.
		//
		// The other way to do this - the way it's done now - is to roll 
		// things up as the data comes in, being careful to always add 
		// only the deltas from the stack chain base and cumulative 
		// time that were already there.

		// Empty out the stacks we've been keeping for each thread
		if(cs!=null)
		{
			cs.clear();
			LookupServiceExtensions.getInstance().deregister(context, CallStackPerThread.class, threadIdRef);
		}
	}

	public interface CallBack {
		public void call(TRCAggregatedMethodInvocation o, int level);
	}
	
	protected void dump(CallBack cb, Iterator it, int level) {
		while (it.hasNext()) {
			Object o = it.next();
			if (o instanceof TRCThread) {
				TRCThread t = (TRCThread) o;
				dump( cb, t.getInitialInvocations().iterator(), level+1);
			}
			else if (o instanceof TRCAggregatedMethodInvocation) {
				TRCAggregatedMethodInvocation in = (TRCAggregatedMethodInvocation) o;
				cb.call((TRCAggregatedMethodInvocation) o, level);
				dump(cb, in.getInvokes().iterator(), level+1);
			}
		}
	}
	
	protected void updateMethodStatisticalInfo(Stack cs, double deltaBaseTime, int deltaNumCalls) {
		// Update base times and call counts for the method, class, package, and process.

		if (theMethod != null) {
			theMethod.setBaseTime(theMethod.getBaseTime() + deltaBaseTime);
			theMethod.setCalls(theMethod.getCalls() + deltaNumCalls);
			if(cpuTime!=0)
				theMethod.setTotalCpuTime(cpuTime);
			TRCClass theClass = theMethod.getDefiningClass();  
			if (theClass != null) {
				theClass.setBaseTime(theClass.getBaseTime() + deltaBaseTime);
				theClass.setCalls(theClass.getCalls() + deltaNumCalls);
				if(cpuTime!=0)
					theClass.setTotalCpuTime(theClass.getTotalCpuTime()+cpuTime);
				TRCPackage thePackage = theClass.getPackage();
				if (thePackage != null) {
					thePackage.setBaseTime(thePackage.getBaseTime() + deltaBaseTime);
					thePackage.setCalls(thePackage.getCalls() + deltaNumCalls);
					if(cpuTime!=0)
						thePackage.setTotalCpuTime(thePackage.getTotalCpuTime()+cpuTime);
				}
			}
		}
		if (theProcess != null) {
			theProcess.setBaseTime(theProcess.getBaseTime() + deltaBaseTime);
			theProcess.setCalls(theProcess.getCalls() + deltaNumCalls);
			if(cpuTime!=0)
				theProcess.setTotalCpuTime(theProcess.getTotalCpuTime()+cpuTime);
		}

		// Now chase up the stack: for each frame, add deltaBaseTime to the cumulative time in
		// that frame's method, that method's class, that class's package, and the package's process.
		// But only once per method/class/package/process - after adding this delta base time
		// and delta calls to one of those, add it to a list of ones we've already hit
		// so we don't add this delta to that one again.
		//
		// TODO: revisit this. Is it efficient? Can we do this "at the end" by walking
		// up the stack just once, after all data is loaded, rolling cumulative time up to 
		// methods, classes, etc.?
		//
		// One way to do this is to do it in "exit" instead of "enter," and in particular
		// only when exit results in an empty shadow stack. When that happens you can
		// roll up the stack into the classes etc. all at once, walking the stack just once.
		// But it might not work for multiple snapshots - only here in "entry" can we
		// compute "deltaBaseTime," the base time that has happened since the last snapshot.
		//
		// Note: the "visitedNodes" structure is an IdentityHashMap because it's faster
		// to use == than to use equals(). If there were an IdentityHashSet I'd use it.
		// It could be slightly faster to use separate hashes for methods, classes, 
		// packages, and processes since the hash tables would thus be smaller, and 
		// searches possibly faster.

		if(visitedNodes==null)
			visitedNodes = new IdentityHashMap();	// note: java 1.4 dependency
		else
			visitedNodes.clear();
		for (int i = cs.size() ; --i >= 0;) {
			TRCAggregatedMethodInvocation frame = (TRCAggregatedMethodInvocation)(cs.get(i));
			TRCMethod theMethod = frame.getMethod();
			if (!visitedNodes.containsKey(theMethod)) {
				visitedNodes.put(theMethod, null);
				theMethod.setCumulativeTime(theMethod.getCumulativeTime() + deltaBaseTime);
				if(cpuTime!=0)
					theMethod.setTotalCpuTime(theMethod.getTotalCpuTime()+cpuTime);
				TRCClass theClass = theMethod.getDefiningClass();
				if (theClass != null && !visitedNodes.containsKey(theClass)) {
					visitedNodes.put(theClass, null);
					theClass.setCumulativeTime(theClass.getCumulativeTime() + deltaBaseTime);
					if(cpuTime!=0)
						theClass.setTotalCpuTime(theClass.getTotalCpuTime()+cpuTime);
					TRCPackage thePackage = theClass.getPackage();
					if (thePackage != null && !visitedNodes.containsKey(thePackage)) {
						visitedNodes.put(thePackage, null);
						thePackage.setCumulativeTime(thePackage.getCumulativeTime() + deltaBaseTime);
						if(cpuTime!=0)
							thePackage.setTotalCpuTime(thePackage.getTotalCpuTime()+cpuTime);
						TRCProcess theProcess = thePackage.getProcess();
						if (theProcess != null && !visitedNodes.containsKey(theProcess)) {
							visitedNodes.put(theProcess, null);
							theProcess.setCumulativeTime(theProcess.getCumulativeTime() + deltaBaseTime);
							if(cpuTime!=0)
								theProcess.setTotalCpuTime(theProcess.getTotalCpuTime()+cpuTime);
						}
					}
				}
			}
		}
		visitedNodes.clear();
	}
	
	public void initialize(HierarchyContext context, String name) {
		loadToModel = context.isLoadToModel();
		super.initialize(context, name);
		maxTime=0;
		minTime=0;
		numCalls=0;
		baseTime=0;
		cpuTime=0;
	}
}
