/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLmonContendedEnterLoader.java,v 1.16 2010/09/08 17:42:01 jwest Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 * Intel - loader is extended for contention analysis support  
 * 
 * $Id: XMLmonContendedEnterLoader.java,v 1.16 2010/09/08 17:42:01 jwest Exp $
 **********************************************************************/
package org.eclipse.hyades.loaders.trace;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.loaders.hierarchy.Constants;
import org.eclipse.hyades.loaders.util.HierarchyContext;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.loaders.util.LookupServiceExtensions;
import org.eclipse.hyades.models.trace.TRCHeapObject;
import org.eclipse.hyades.models.trace.TRCObject;
import org.eclipse.hyades.models.trace.TRCThread;
import org.eclipse.hyades.models.trace.TRCThreadDeadLockEvent;
import org.eclipse.hyades.models.trace.TRCThreadEvent;
import org.eclipse.hyades.models.trace.TRCThreadWaitingForLockEvent;
import org.eclipse.hyades.models.trace.TraceFactory;
import org.eclipse.hyades.models.trace.TracePackage;
import org.eclipse.hyades.models.trace.impl.TRCFullHeapObjectImpl;
import org.eclipse.hyades.models.trace.impl.TRCHeapObjectImpl;
import org.eclipse.hyades.models.trace.impl.TRCThreadImpl;
public class XMLmonContendedEnterLoader extends TraceXMLThreadEventsLoader {
	
	private static final boolean DEADLOCK_DETECTION = false;
	
	//~ Static fields/initializers
	// -----------------------------------------------------------------
	private static final String THREAD_OWNER = "threadOwner";
	//~ Instance fields
	// ----------------------------------------------------------------------------
	private int threadOwner;
	private HashMap processesMap;
	//~ Methods
	// ------------------------------------------------------------------------------------
	public void addAttribute(String name, String value) {	  
		switch (LoadersUtils.getHashCode(name)) {
			case TraceConstants.THREAD_OWNER_int :
				threadOwner = Integer.parseInt(value);
				break;
			default :
				super.addAttribute(name, value);
				break;
		}
	}
	public void addYourselfInContext() {
		if(context.isFilterOn())
		{
			if(context.getFilterEngine().isFiltered(TracePackage.eINSTANCE.getTRCThread_Name(),LoadersUtils.getLookUpKey(threadIdRef)))
			{
				return;
			}
			if(context.getFilterEngine().isFiltered(TracePackage.eINSTANCE.getTRCThread_Name(),LoadersUtils.getLookUpKey(threadOwner)))
			{
				return;
			}
		}
		theProcess = getProcess();
		theThread = getThreadByIdRef(theProcess);
		dispatchProcessMode(ProcessSteps.ALL);
		TRCThread lockingThread = null;
		
		//WFL event is systematically created ...
		{
		  TRCThreadWaitingForLockEvent threadWaitingForLockEvent = TraceFactory.eINSTANCE.createTRCThreadWaitingForLockEvent();
		  threadWaitingForLockEvent.setLockedObject(theObject);
		  threadWaitingForLockEvent.setThread(theThread);
		  threadWaitingForLockEvent.setTime(createDeltaTime());//time);
		  
		  if (threadOwner > 0) {
		    lockingThread = (TRCThread) LookupServiceExtensions.getInstance().locate(context, TRCThreadImpl.class, threadOwner);
		    threadWaitingForLockEvent.setLockingThread( lockingThread );
			getThreadEventsContext().getMonitor2OwnerMap().put(theObject, lockingThread);
		  }
		  addAnnotationsIfRequired(threadWaitingForLockEvent);
		  theThread.getThreadEvents().add( threadWaitingForLockEvent );
		  
		  //memorize still locked for next coming waiting for lock
		  ArrayList stillLocked = XMLmonStillOwnedLoader.getStillLocked( theProcess, theThread );
		  setStillLocked( threadWaitingForLockEvent, stillLocked );
  
		  //but as this wfl could be processed after a newer wfl, the locking thread could
		  //not be the right one (as another wfl have stillOwner which designate the correct one)
		  if( theProcess.getLastEventTime() > threadWaitingForLockEvent.getTime() )
		  {
            updateNewestOwnerOfLock( threadWaitingForLockEvent, stillLocked );
		  }
		  
          updateWaitingForLockEvents( stillLocked );
		}

        // ... deadlock detection can be processed now (due to asynchronous incoming event's time in traces) 
		if( threadOwner > 0 && theThread!=lockingThread ) 
		{
			//time to check dead lock:
			ArrayList wfls = new ArrayList();
			boolean dl_by_contagion = false;
			
			if(DEADLOCK_DETECTION) {
				dl_by_contagion = getCircularity( theThread, wfls );
			}
			
			boolean new_deadlock = false;
			//deadlocked by contagion:
			
			if( dl_by_contagion )
			{
				  //wfl event mut be removed before deadlock event is created:
				  EList events = theThread.getThreadEvents();
				  events.remove( events.size()-1 );
				  //replace wfl by a deadlock:
				  TRCThreadDeadLockEvent evt = TraceFactory.eINSTANCE.createTRCThreadDeadLockEvent();
				  evt.setLockedObject(theObject);
				  evt.setThread(theThread);
				  evt.setTime( createDeltaTime());
				  evt.setLockingThread( lockingThread );
				  if (wfls.get(0) instanceof TRCThreadDeadLockEvent) {
					  TRCThreadDeadLockEvent contagious = (TRCThreadDeadLockEvent)wfls.get(0);
					  evt.setNextDeadLockEvent( contagious );
				  }
				  addCopyOfAnnotationsIfRequired(evt);
				  events.add( evt );
				  new_deadlock = true;
			}
			//have found a circularity ?
			else if ( wfls.size() > 0 )
			{
               //get the real latest wfl event (as current one could not the the latest of circularity)
               double reftime = ((TRCThreadWaitingForLockEvent)wfls.get(0)).getTime();
               int    index= 0;
               for( int i=1; i<wfls.size(); i++ )
               {
                 double evttime = ((TRCThreadWaitingForLockEvent)wfls.get(i)).getTime();
                 if( evttime > reftime )
                 {
                   reftime = evttime;
                   index = i;
                 }
               }
               boolean deadlock = false;
               if(DEADLOCK_DETECTION) {
            	   //now time to check if this is a real circularity checking monStillOwner
            	   deadlock= checkCircularLock( wfls, index ) ;
               }
               
               if( deadlock )  
			   {		
                 TRCThreadDeadLockEvent last_dl = null;
                 TRCThreadDeadLockEvent first_dl = null;
                 double deadlock_time = ((TRCThreadWaitingForLockEvent)wfls.get(index)).getTime();
                 //foreach wfl in list, add deadlock, except for lastest: it must be replaced
                 for( int i=0; i<wfls.size(); i++ )
                 {
                   //original wfl:
                   TRCThreadWaitingForLockEvent wfl = (TRCThreadWaitingForLockEvent)wfls.get(i);
                   EList wfl_events = wfl.getThread().getThreadEvents();
                   //replace or add ?
                   //must be removed before create deadlock event as setThread() add in list !
                   if( i==index || wfl.getTime() == deadlock_time )
                   {
                     //replace:
//                     Object o =wfl_events.remove( wfl_events.size()-1 );
                   }
   					
                   //create a deadlock event:
                   TRCThreadDeadLockEvent evt = TraceFactory.eINSTANCE.createTRCThreadDeadLockEvent();
                   evt.setLockedObject( wfl.getLockedObject() );
                   evt.setThread( wfl.getThread() );
                   evt.setTime( deadlock_time );
                   evt.setLockingThread( wfl.getLockingThread() );
                   
                   if( last_dl!=null ) last_dl.setNextDeadLockEvent( evt );
       			   addCopyOfAnnotationsIfRequired(evt);
       			   
     			   wfl_events.add( evt );
                   
                   if( first_dl==null ) first_dl = evt;
                   last_dl = evt;
                  }
                 //complete circularity:
                 last_dl.setNextDeadLockEvent( first_dl );
                 new_deadlock = true;
			   }
			}
			if( DEADLOCK_DETECTION && new_deadlock )
			{
			  //as a waitingforlock event with time > new_deadlock time could be already processed
			  //we must check if other waitingForLock event must be switched to DeadLock (by contagion)
			  updateDeadLockByContagion();
			}
		}
	}
    //fill wfls with circularity, return true in case of deadlock-by-contagion, false otherwise
    private boolean getCircularity( TRCThread startThread, ArrayList wfls )
	{
	  if( startThread==null ) return false;
	  
	  TRCThread currThread = startThread;
	  HashMap thread_checked = new HashMap();
	  while( true )
	  {
	    EList evts = currThread.getThreadEvents();
	    if( evts.size()==0 ) { wfls.clear(); return false; }
	    TRCThreadEvent last_event = (TRCThreadEvent)evts.get( evts.size()-1 );
	    if( last_event instanceof TRCThreadDeadLockEvent ) 
	    {
	      wfls.clear();
	      wfls.add( last_event );
	      return true;
	    }
	    if( !(last_event instanceof TRCThreadWaitingForLockEvent) ) { wfls.clear(); return false; }
	    
	    TRCThreadWaitingForLockEvent wfl = (TRCThreadWaitingForLockEvent)last_event;
	    wfls.add( wfl );
	    thread_checked.put( currThread, null );	    
	    currThread = wfl.getLockingThread();
	    if( currThread==null ) //unknown locking thread: no circularity
	    {
	      wfls.clear();
	      return false;
	    }
	    //stop condition of circularity I'm searching for.
	    if( currThread==startThread ) break;
	    //end of another circularity, but not the one which can make dead locks
	    if( thread_checked.containsKey(currThread) ) 
	    {
	      wfls.clear();
	      return false;
	    }
	  }
	  return true;
	}    
	private boolean checkCircularLock( ArrayList wfls, int start_index )
	{
	  int index = start_index;
	  for( int i=0; i<wfls.size(); i++)
	  {
	    TRCThreadWaitingForLockEvent wfl      = (TRCThreadWaitingForLockEvent)wfls.get(index);
        TRCThreadWaitingForLockEvent next_wfl = (TRCThreadWaitingForLockEvent)wfls.get( (index+1)%wfls.size() );
        boolean mso_required = next_wfl.getTime() > wfl.getTime();
	    //should I check fo monStillOwner ?
	    if( mso_required )
	    {
	      TRCObject lock = wfl.getLockedObject();
	      ArrayList so = getStillLocked( next_wfl );
	      if( so==null || !so.contains( lock ) ) return false;
	    }
	    index++;
	    if( index>=wfls.size() ) index=0;
	  }
	  return true;
	}
	private void updateWaitingForLockEvents( ArrayList stillOwner )
	{
	  if( stillOwner==null || stillOwner.size()==0 ) return ;
	  //for deadlock detection all invalid waitingForLockEvents must be updated
	  //invalid means wfl's lockingThread is 'theThread' as we trust stillOwner lock array.
	  EList threads = theProcess.getThreads();
	  for( Iterator T=threads.iterator(); T.hasNext(); )
	  {
	    TRCThread thread = (TRCThread)T.next();
	    EList events = thread.getThreadEvents();
	    if( events==null || events.size()==0 ) continue;
	    TRCThreadEvent last_event = (TRCThreadEvent)events.get( events.size()-1 );
	    if( last_event instanceof TRCThreadWaitingForLockEvent )
	    {
	      TRCThreadWaitingForLockEvent wfl = (TRCThreadWaitingForLockEvent)last_event;
	      boolean is_still_owned_lock = stillOwner.contains( wfl.getLockedObject() ); 
	      //should this event is still valid ?
	      if( is_still_owned_lock && wfl.getLockingThread() != theThread )
	      {
 	        //we are sure this wfl event's locking thread is no more valid
	        //... add a new wfl event which point out to the right (new) locking thread
	        TRCThreadWaitingForLockEvent new_wfl = TraceFactory.eINSTANCE.createTRCThreadWaitingForLockEvent();
		    new_wfl.setLockedObject( wfl.getLockedObject() );
		    new_wfl.setThread( thread );
		    new_wfl.setTime(createDeltaTime());
		    new_wfl.setLockingThread( theThread );
			addCopyOfAnnotationsIfRequired(new_wfl);
		    events.add( new_wfl );	
		    //stillOwned lock must be associated to this new WFL:
		    setStillLocked( new_wfl, getStillLocked( wfl ) );		    
	      }
	    }
	  }	 
	}
	private ArrayList getStillLocked( TRCThreadWaitingForLockEvent wfl )
	{
      if( processesMap==null ) return null;
	  HashMap mp = (HashMap)processesMap.get( theProcess );
	  if( mp == null ) return null;
	  return (ArrayList)mp.get( wfl );
	}
	private void setStillLocked( TRCThreadWaitingForLockEvent wfl, ArrayList stillLocked )
	{
	  if( stillLocked==null ) return ;
		  
	  if( processesMap==null ) {
	    processesMap = new HashMap();
	  }
	  HashMap mp = (HashMap)processesMap.get( theProcess );
	  if( mp == null )
	  {
	    mp = new HashMap();
	    processesMap.put( theProcess, mp );
	  }
	  
	  mp.put( wfl, stillLocked );
	}
	private void updateDeadLockByContagion()
	{
	  for( boolean dl_added=true; dl_added; )
	  {
	    dl_added=false;
	    for( Iterator T=theProcess.getThreads().iterator(); T.hasNext(); )
	    {
	      TRCThread thread = (TRCThread)T.next();
	      EList events = thread.getThreadEvents();
	      if( events==null || events.size()==0 ) continue;
	      Object last_event = events.get( events.size()-1);
	      if( last_event instanceof TRCThreadDeadLockEvent ) continue;
	      if( !( last_event instanceof TRCThreadWaitingForLockEvent )) continue;
	      TRCThreadWaitingForLockEvent wfl = (TRCThreadWaitingForLockEvent)last_event;
	      TRCThread lockingThread = wfl.getLockingThread();
	      if( lockingThread==null ) continue;
	      EList lt_events = lockingThread.getThreadEvents();
	      if( lt_events==null || lt_events.size()==0 ) continue;
	      Object lt_last_event = lt_events.get( lt_events.size()-1);
	      if( !(lt_last_event instanceof TRCThreadDeadLockEvent )) continue;
	      TRCThreadDeadLockEvent dl = (TRCThreadDeadLockEvent)lt_last_event;

	      //this wfl must be replaced by a deadlock !
	      events.remove( events.size()-1 );
		  //replace wfl by a deadlock:
			  TRCThreadDeadLockEvent evt = TraceFactory.eINSTANCE.createTRCThreadDeadLockEvent();
			  evt.setLockedObject( wfl.getLockedObject() );
			  evt.setThread( thread );
			  evt.setTime( wfl.getTime() );
			  evt.setLockingThread( lockingThread );
			  evt.setNextDeadLockEvent( dl );
			  addCopyOfAnnotationsIfRequired(evt);
			  events.add( evt );
			  dl_added = true;
			  //as another loop is required: do it right now!
			  break;
		    }
		  }
		}

	private void updateNewestOwnerOfLock( TRCThreadWaitingForLockEvent ref_wfl, ArrayList stillLocked )
	{
	  TRCThread owner = null;
	  double time = 0.0;
	  for( Iterator T=theProcess.getThreads().iterator(); T.hasNext(); )
	  {
	    TRCThread thread = (TRCThread)T.next();
	    EList events = thread.getThreadEvents();
	    if( events==null || events.size()==0 ) continue;
	    Object last_event = events.get( events.size()-1 );
	    if( last_event instanceof TRCThreadWaitingForLockEvent )
	    {
	      TRCThreadWaitingForLockEvent wfl = (TRCThreadWaitingForLockEvent)last_event;
	      ArrayList so = getStillLocked( wfl );
	      if( so!=null && so.contains( ref_wfl.getLockedObject()) )
	      {
	        if( owner==null || wfl.getTime() > time )
	        {
	          owner = wfl.getThread();
	          time  = wfl.getTime();
	        }
	      }
	    }
	  }
	  if( owner!=null && owner!=ref_wfl.getLockingThread() )
	  {
	    //ok need to create a new WFL event to reflect the new lockingThread
        TRCThreadWaitingForLockEvent wfl = TraceFactory.eINSTANCE.createTRCThreadWaitingForLockEvent();
		wfl.setLockedObject( ref_wfl.getLockedObject() );
		wfl.setThread( ref_wfl.getThread() );
		wfl.setTime( time );//time);
		wfl.setLockingThread( owner );
		addCopyOfAnnotationsIfRequired(wfl);  
		ref_wfl.getThread().getThreadEvents().add( wfl );
		  
		setStillLocked( wfl, stillLocked );  
	  }
	}
	public void initialize(HierarchyContext context, String name) {
		loadToModel = context.isLoadToModel();
		super.initialize(context, name);
		threadOwner = 0;
		annotations.clear();
		currentAnnotationValueEntryActive=false;

	}
	public void cleanUp() {
       super.cleanUp();
       threadOwner = 0;
       processesMap = null;
    }

	protected void processHF(int step) {
		super.processHF(step);
		Class clazz = (TraceUtils.isBooleanOptionEnabled(context,Constants.MULTIPLE_HEAP_DUMPS) ? TRCFullHeapObjectImpl.class : TRCHeapObjectImpl.class);
		if ((theObject == null) || !(theObject instanceof TRCHeapObject)) {
			theObject = (TRCHeapObject) LookupServiceExtensions.getInstance().locate(context, clazz, objIdRef);
		}
	}
}