/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ApplicationServerListener.java,v 1.8 2009/10/30 15:54:56 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.profile.server.wst.internal;

import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.profile.server.core.internal.IProfileServerConstants;
import org.eclipse.tptp.platform.profile.server.wst.internal.reflect.EclipseClass;
import org.eclipse.tptp.platform.profile.server.wst.internal.reflect.LinkageException;
import org.eclipse.wst.server.core.IServer;
import org.eclipse.wst.server.core.IServerLifecycleListener;
import org.eclipse.wst.server.core.IServerListener;
import org.eclipse.wst.server.core.ServerCore;
import org.eclipse.wst.server.core.ServerEvent;
import org.eclipse.wst.server.core.internal.IStartup;

/**
 * Add listeners for servers starting in profiling mode.
 */
public class ApplicationServerListener implements IStartup {

    /*
     * @see org.eclipse.wst.server.core.internal.IStartup#startup()
     */
    public void startup() {

        /* Attach a listener to each of the existing servers. */
        final IServerListener serverListener = new ServerListener();
        IServer[] servers = ServerCore.getServers();
        for (int i = 0; i < servers.length; i++) {
            servers[i].addServerListener(serverListener,ServerEvent.STATE_CHANGE | ServerEvent.SERVER_CHANGE);
        }

        /* If servers are added and removed, the listener will be added and removed accordingly. */
        ServerCore.addServerLifecycleListener(new ServerResourceAdapter(serverListener));
    }

    /*
     * Ask the user for profiling options, then attach the
     * profiler to the server.
     */
    static void attachToAgents(final IServer server) {
        try {
            ILaunchConfiguration conf = server.getLaunchConfiguration(false, null);
            ILaunchConfigurationWorkingCopy wc = conf.getWorkingCopy();
            String hostname = server.getHost();
            
            //check if launch configuration is configured to not attach any agent nor switch perspective
            boolean notAttachNswitch = wc.getAttribute(IProfileServerConstants.DO_NOT_ATTACH_AND_SWITCH_PERP,false);
            if (notAttachNswitch) return;
            
            //213367
            int port = wc.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PORT, CommonUIPlugin.getDefault().getPreferenceStore().getInt(CommonUIConstants.LOCALHOST_PORT));       	    
            Class profileOnServerAdapter = EclipseClass.findClass("org.eclipse.tptp.platform.profile.server.core",
                "org.eclipse.tptp.platform.profile.server.core.internal.ProfileOnServerAdapter");
            EclipseClass.invokeStatic(profileOnServerAdapter, "attachToAgents",
                new Class[]{ILaunchConfigurationWorkingCopy.class, String.class, int.class, Object.class},
                new Object[]{wc, hostname, new Integer(port), server});
        }
        catch (LinkageException e) {
            MessageDialog.openInformation(ServerPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(), "Error", "TPTP Profile on sever feature is not found");
        }
        catch (Exception e) {
            ServerPlugin.logError(e);
        }
    }
}

final class ServerListener implements IServerListener {
    public void serverChanged(final ServerEvent event) {

        /*
         * If the event is an event indicating that
         *      1. server changes just occurred (1st condition)
         *      2. the server has been started (2nd condition)
         *      3. the server was in profiling mode (3rd condition)
         * create the worker that will trigger the agent to be attached to.
         */
        if ((event.getKind() == (ServerEvent.STATE_CHANGE | ServerEvent.SERVER_CHANGE))
                && event.getState() == IServer.STATE_STARTED
                && event.getServer().getMode().equals(ILaunchManager.PROFILE_MODE)) {
            Display.getDefault().asyncExec(new Runnable() {
                public void run() {
                    ApplicationServerListener.attachToAgents(event.getServer());
                }
            });
        }
    }
}

final class ServerResourceAdapter implements IServerLifecycleListener {
    private final IServerListener serverListener;

    ServerResourceAdapter(IServerListener serverListener) {
        this.serverListener = serverListener;
    }

    public void serverAdded(IServer server) {
        server.addServerListener(serverListener,ServerEvent.STATE_CHANGE | ServerEvent.SERVER_CHANGE);
    }

    public void serverChanged(IServer server) {
        // Do nothing. Being here just to make Java happy.
    }

    public void serverRemoved(IServer server) {
        server.removeServerListener(serverListener);
    }
}