package org.eclipse.hyades.logging.events.cbe;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;
import org.eclipse.hyades.logging.core.LoggingCoreUtilities;

import com.ibm.icu.util.ULocale;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventException.java,v 1.6 2008/12/15 15:34:44 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Base for all exceptions in CBE package. 
 * 
 * The <code>EventException</code> class is a subclass
 * of <code>java.lang.Exception</code> that provides support for
 * exception chaining semantics for pre-JDK v1.4 run-times. 
 * <p>
 * It should be noted that this class provides only the exception
 * chaining semantics available in JDK v1.4 which is a subset of 
 * new exception functionality introduced in JDK v1.4. Most of 
 * the new functionality requires support from the JVM that is not 
 * available on earlier run-times.
 * <p>
 * Exceptions that subclass this exception only need to add the
 * various constructors and inherit everything else from this class.
 * <p>
 * See the API documentation for <code>java.lang.Throwable</code> in 
 * JDK v1.4 for a full description of the exception chaining functionality.
 * <p>
 * <code>EventException</code> also provides support for localized 
 * (language-specific) messages.
 * <p>
 * A <code>EventException</code> contains the following
 * components:
 * <ul>
 * <li>
 * A "key" which selects the text to be displayed from the message
 * ResourceBundle. (Separating the text from the classes which use it allows
 * the text to be translated into different national languages.)
 * </li>
 * <li>
 * A message ResourceBundle, which contains a set of messages that might be
 * displayed. Entries in the message ResourceBundle are stored as "key=value"
 * pairs.
 * </li>
 * <li>
 * A set of <code>Objects</code> which can be inserted into the
 * message text. These parameters are required only if the message
 * expects them. These objects are processed according to the rules
 * of the {@link java.text.MessageFormat} class.
 * </li>
 * <li>
 * A nested <code>java.lang.Throwable</code> whose message text will be
 * displayed along with that of this <code>EventException</code>.
 * </li>
 * </ul>
 * <p>
 *
 *
 * @author Denilson Nastacio
 * @author Paul E. Slauenwhite
 * @version 1.0.1
 * @since 1.0.1 
 * @see java.lang.Exception
 */
public class EventException extends Exception {

    /**
     * Stream-Unique IDentifier (SUID) of this class.
     * <p>
     * NOTE:  This value MUST not be modified unless an incompatible 
     *        change is introduced.  
     * <p>
     * See the 'Java Object Serialization Specification' for more details.
     * <p>
     * Compatible changes include:
     * <ul>
     * <li>Adding, modifying access or modifying <code>static</code>/<code>transient</code> to non-<code>static</code>/non-<code>transient</code> of fields.</li>
     * <li>Adding or removing classes.</li>
     * <li>Adding or removing <code>writeObject</code>/<code>readObject</code> methods.</li>
     * <li>Adding <code>java.io.Serializable</code> inheritance.</li>
     * </ul>
     * <p>
     * Incompatible changes include:
     * <ul>
     * <li>Deleting, modifying non-<code>static</code>/non-<code>transient</code> to <code>static</code>/<code>transient</code> or modifying the declared type of fields.</li>
     * <li>Modifying the hierarchy or interchanging/removing <code>java.io.Serializable</code>/<code>java.io.Externalizable</code> inheritance of the class.</li>
     * <li>Modifying <code>writeObject</code>/<code>readObject</code> methods.</li>
     * <li>Adding <code>writeReplace</code>/<code>readResolve</code> methods.</li>
     * </ul>
     */
    protected static final long serialVersionUID = -8100478137706108072L;

    /** 
     * Base name of the ResourceBundle in which the text is found. 
     */
    protected String resourceBundleName = null;

    /** 
     * The key of the message associated with this text. 
     */
    protected String key = null;

    /** 
     * Parameters to be included in the message. 
     */
    protected Object parameters[] = null;

    /**
     * The throwable that caused this throwable to get thrown, or
     * null if this throwable was not caused by another throwable,
     * or if the causative throwable is unknown.
     */
    protected Throwable cause = null;

    //~ Constructors .................................................

    /**
     * Construct a new <code>EventException</code> with
     * a null detail message and an uninitialized cause field. The
     * cause may subsequently be initialized via calling the {@link
     * #initCause} method.
     */
    public EventException() {
        super();
    }

    /**
     * Construct a new <code>EventException</code> with
     * the specified detail message and an uninitialized cause
     * field. The cause may subsequently be initialized via calling
     * the {@link #initCause} method.
     *
     * @param message a detail message. Null is tolerated.
     */
    public EventException(String message) {
        super(message);
    }

    /**
     * Creates a <code>EventException</code> with a detail message, message
     * translation information (resource bundle, message key and message
     * substitutions).
     *
     * @param key The message key.
     * @param resourceBundleName The name of the message ResourceBundle to use.
     */
    public EventException(String key, String resourceBundleName) {
        this(key, resourceBundleName, null);
    }

    /**
     * Construct a new <code>EventException</code> with
     * a null detail message and a cause field set to the specified
     * Throwable. Subsequent calls to the {@link #initCause} method
     * on this instance will result in an exception. The value of
     * the cause field may be retrieved at any time via the {@link
     * #getCause()} method.
     *
     * @param cause the Throwable that was caught and is considered
     *        the root cause of this exception. Null is tolerated.
     */
    public EventException(Throwable cause) {
       
        this(cause.getMessage());
        
        this.cause = cause;
    }

    /**
     * Construct a new <code>EventException</code> with
     * a null detail message and a cause field set to the specified
     * Throwable. Subsequent calls to the {@link #initCause} method
     * on this instance will result in an exception. The value of
     * the cause field may be retrieved at any time via the {@link
     * #getCause()} method.
     *
     * @param message the detail message. Null is tolerated.
     * @param cause the Throwable that was caught and is considered
     *        the root cause of this exception. Null is tolerated.
     */
    public EventException(String message, Throwable cause) {
        
        this(message);
        
        this.cause = cause;
    }

    /**
     * Creates a <code>EventException</code> with a
     * detail message, message translation information (resource
     * bundle, message key and message substitutions).
     *
     * @param key The message key.
     * @param resourceBundleName The name of the message ResourceBundle to use.
     * @param parameters An array of elements to be inserted into the
     *        message.
     */
    public EventException(String key, String resourceBundleName, Object parameters[]) {
        
        this.key = key;
        this.resourceBundleName = resourceBundleName;
        this.parameters = parameters;
    }

    /**
     * Creates a <code>EventException</code> with a
     * detail message, message translation information (resource
     * bundle, message key and message substitutions) and a nested
     * exception.
     *
     * @param key The message key.
     * @param resourceBundleName The name of the message ResourceBundle to use.
     * @param parameters An array of elements to be inserted into the
     *        message.
     * @param cause The nested exception, or <code>null</code>, if a
     *        nested exception is not appropriate.
     */
    public EventException(String key, String resourceBundleName, Object parameters[], Throwable cause) {
       
        this.key = key;
        this.resourceBundleName = resourceBundleName;
        this.parameters = parameters;
        this.cause = cause;
    }

    /**
     * Return the Throwable that is considered the root cause of this
     * <code>EventException</code>. Null is returned
     * if the root cause is nonexistent or unknown. The root cause
     * is the throwable that caused this
     * <code>EventException</code> to get thrown.
     * 
     * <p>
     * The Throwable that is returned is either the Throwable
     * supplied via one of the appropriate constructors, or that set
     * via the {@link #initCause(Throwable)} method. While it is
     * typically unnecessary to override this method, a subclass can
     * override it to return a cause set by some other means, such
     * as a legacy exception chaining infrastructure.
     * </p>
     *
     * @return the Throwable that is the cause of this
     *         <code>EventException</code>, or null if
     *         the cause is nonexistent or unknown.
     */
    public Throwable getCause() {
        return cause;
    }

    /**
     * Gets the message key.  The result is the key of the message
     * associated with this text.
     * 
     * <P></p>
     *
     * @return String   The message key .
     */
    public String getKey() {
        return key;
    }

    /**
     * Gets the text of the exception message, translated into the
     * current locale.
     *
     * @return The localized exception message.
     */
    public String getLocalizedMessage() {
        return getLocalizedMessage(com.ibm.icu.util.ULocale.forLocale(LoggingCoreUtilities.getClientLocale()));
    }

    /**
     * Gets the text of the exception message, translated into the
     * specified locale.
     *
     * @param locale The locale to use for translation of the
     *        exception message, or <code>null</code> to use the
     *        default locale.
     *
     * @return The localized exception message.
     * @deprecated As of TPTP V4.2.0, use {@link EventException#getMessage()} (English) and {@link EventException#getLocalizedMessage()} (default locale) for exception message localization.
     */
    public String getLocalizedMessage(ULocale locale) {
        
        StringBuffer buffer = new StringBuffer();

        if (key == null) {
            
            //NOTE: Calling super.getLocalizedMessage() will cause a recursive loop.
            buffer.append(super.getMessage());
        }
        else {
            
            try {
                
                ResourceBundle bundle = ResourceBundle.getBundle(resourceBundleName, locale.toLocale());
                String resource = bundle.getString(key);

                if ((parameters != null) && (parameters.length > 0)) {
                    
                	try {
                		buffer.append(InternationalizationUtilities.format(resource, parameters,locale));
					} 
                	catch (IllegalArgumentException i) {
						buffer.append(resource);
					}
                }
                else{
                    buffer.append(resource);
                }                    
            }
            catch (MissingResourceException m) {
                
                buffer.append(m.getLocalizedMessage());

                buffer.append(" : ");

                buffer.append(key);

                if ((parameters != null) && (parameters.length > 0)) {
                    
                    buffer.append(" {");

                    for (int counter = 0; counter < parameters.length; counter++) {
                        
                        if(parameters[counter] != null){
                            buffer.append(String.valueOf(parameters[counter]));
                        }
                        else{
                            buffer.append("null");
                        }
                        
                        if((counter + 1) < parameters.length){
                            buffer.append(", ");
                        }
                    }

                    buffer.append("}");
                }
            }
        }

        if (cause != null) {
            
            String causeMessage = null;

            if (cause instanceof EventException) {
                causeMessage = ((EventException) cause).getLocalizedMessage(locale);
            }
            else {
                causeMessage = cause.getLocalizedMessage();
            }
            
            if((causeMessage != null) && (causeMessage.trim().length() > 0)){
                buffer.append(" : ");
                buffer.append(causeMessage);
            }
        }

        return (buffer.toString());
    }

    /**
     * Gets the text of the exception message.  The result is the
     * same as that returned by
     * <code>getLocalizedMessage(Locale.ENGLISH)</code>.
     *
     * @return The exception message.
     */
    public String getMessage() {
        return getLocalizedMessage(ULocale.ENGLISH);
    }

    /**
     * Initialize the cause field for this
     * <code>EventException</code> to the specified
     * value. The cause is the Throwable that caused this
     * <code>EventException</code> to get thrown.
     * 
     * <p>
     * This method can be called at most once. It is generally called
     * from within a constructor that takes a Throwable, or
     * immediately after constructing this object with a constructor
     * that does not accept a Throwable. Thus, if a constructor that
     * takes Throwable as a parameter is used to construct this
     * object, it cannot be called at all.
     * </p>
     *
     * @param cause the Throwable which caused this
     *        <code>EventException</code> to be
     *        thrown. Null is tolerated.
     *
     * @return a reference to this <code>Throwable</code> instance.
     *
     * @exception IllegalStateException if this
     *            <code>EventException</code> was
     *            created with a constructor that specified a cause,
     *            or this method has already been called on this
     *            object.
     * @exception IllegalArgumentException if the specified cause is
     *            this <code>EventException</code>. An
     *            exception cannot be its own cause.
     */
    public synchronized Throwable initCause(Throwable cause) throws IllegalStateException, IllegalArgumentException {

        if (this.cause != null) {
            throw new IllegalStateException("Cannot overwrite cause");
        }

        if (this.cause == this) {
            throw new IllegalArgumentException("Self-causation not permitted");
        }

        this.cause = cause;

        return this;
    }

    /**
     * Print this <code>EventException</code> and its
     * backtrace to System.err.
     */
    public void printStackTrace() {
        printStackTrace(System.err);
    }

    /**
     * Print this <code>EventException</code> and its
     * backtrace to the specified print stream.
     *
     * @param stream the PrintStream to use for output
     */
    public void printStackTrace(PrintStream stream) {
        
        //First print out the stack trace for this exception frame:
        super.printStackTrace(stream);

        //Then print the nested exception, if it exists:
        if (cause != null) {
            
            stream.println("---- Begin backtrace for nested exception");
            
            cause.printStackTrace(stream);
        }
    }

    /**
     * Print this <code>EventException</code> and its
     * backtrace to the specified print writer.
     *
     * @param writer the PrintWriter to use for output
     */
    public void printStackTrace(PrintWriter writer) {
        
        //First print out the stack trace for this exception frame:
        super.printStackTrace(writer);

        //Then print the nested exception, if it exists:
        if (cause != null) {
            
            writer.println("---- Begin backtrace for nested exception");
            
            cause.printStackTrace(writer);
        }
    }
}