package org.eclipse.hyades.logging.events.cbe.internal.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.logging.core.LoggingCoreResourceBundle;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.util.EventListener;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventHandler.java,v 1.3 2008/02/12 22:35:57 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Event handler used with a <a href="http://www.saxproject.org/">Simple API 
 * for XML (SAX)</a> parser for de-serializing Common Base Event XML documents.
 * <p>
 * Callers register an {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} 
 * implementation with this event handler to process  
 * {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent}s and/or 
 * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine}s when 
 * de-serializing a stream of Common Base Event and/or Association Engine 
 * XML fragments in a Common Base Event XML document.
 * <p>
 * Once a Common Base Event XML and/or Association Engine fragment has been 
 * de-serialized by this event handler, a 
 * {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent} or 
 * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine} is constructed 
 * and broadcast to the registered 
 * {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} implementation 
 * for processing.
 * <p>
 * When de-serializing a stream of Common Base Event and Association Engine XML 
 * fragments in a Common Base Event XML document, callers can process 
 * de-serialized {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent}s and 
 * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine}s on a separate
 * thread before the entire stream of Common Base Event and Association Engine 
 * XML fragments in a Common Base Event XML document has been processed, resulting 
 * in a high level of parallelism and improved efficiency.
 * <p>
 * Noteworthy, callers are responsible for 
 * {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} instantiation, 
 * registration and clean-up.
 * <p>
 * 
 * 
 * @author  Denilson Nastacio
 * @author  Paul E. Slauenwhite
 * @version 1.0.1
 * @since   1.0.1
 * @see     org.xml.sax.helpers.DefaultHandler
 * @see     org.eclipse.hyades.logging.events.cbe.util.EventFormatter
 * @see     org.eclipse.hyades.logging.events.cbe.util.EventListener
 */
public class EventHandler extends DefaultHandler {

    /**
     * Buffer for any element XML fragments (e.g. CommonBaseEvent.otherData or
     * OtherSituation.anyData).
     */
    protected StringBuffer anyElementXMLFragment = null;

    /**
     * Buffer for Parsed Character DATA (PCDATA) (e.g.
     * ExtendedDataElement.values) since some event handlers do not return all
     * contiguous PCDATA in a single chunk.
     */
    protected StringBuffer charactersBuffer = null;

    /**
     * Buffer for name space mapping attributes.
     */
    protected StringBuffer nameSpaceAttribute = null;

    /**
     * The <code>EventListener</code> for this event handler.
     */
    protected EventListener eventListener = null;
    
    /**
     * Current document column position.
     */
    protected int column = 0;

    /**
     * Current document line position.
     */
    protected int line = 0;

    /**
     * Stack of event objects currently parsed.
     */
    protected List stack = null;

    /**
     * Current nesting depth of any element XML fragments (e.g.
     * CommonBaseEvent.otherData or OtherSituation.anyData).
     */
    protected int anyElementNestingDepth = 0;

    /**
     * Map of element names and respective structural feature.
     * <p>
     * The key is a <code>String</code> object representing the element name
     * and the value is a <code>EClass</code> object required to create the
     * Java instance through the <code>EventFactory</code> object.
     */
    protected final static HashMap ELEMENT_MAP = new HashMap();

    /**
     * Map of XML element names and its EMF structural features.
     * <p>
     * The key is a <code>String</code> object representing the element name
     * and the value is another <code>HashMap</code> representing the mapping
     * between the feature name and the feature itself.
     * <p>
     * For this map, the key is a <code>String</code> object representing the
     * feature name and the value is a <code>EStructuralFeature</code> object
     * required to create the Java instance through the
     * <code>EventFactory</code> object.
     */
    protected final static HashMap FEATURE_MAP = new HashMap();

    /**
     * Situation type needs special handling during parsing.
     */
    protected final static String SITUATION_TYPE_CLASS = EventPackage.eINSTANCE.getSituationType().getName();

    /**
     * CommonBaseEvent needs special handling during parsing.
     */
    protected final static String COMMON_BASE_EVENT_CLASS = EventPackage.eINSTANCE.getCommonBaseEvent().getName();

    /**
     * OtherSituation needs special handling during parsing.
     */
    protected final static String OTHER_SITUATION_CLASS = EventPackage.eINSTANCE.getOtherSituation().getName();

    /**
     * AssociationEngine needs special handling during parsing.
     */
    protected final static String ASSOCIATION_ENGINE_CLASS = EventPackage.eINSTANCE.getAssociationEngine().getName();

    /**
	 * Current thread lock for synchronization.
	 * <p>
	 */
	protected static final Object LOCK = new Object();
	
    static{
        
        Iterator iterator = EventPackage.eINSTANCE.eContents().iterator();

        while (iterator.hasNext()) {

            EClass eClass = (EClass) iterator.next();
            
            ELEMENT_MAP.put(eClass.getName(), eClass);

            List features = null;
            
            //NOTE:  Synchronization required for backwards compatibility with EMF v1.1 due to defect #49868:
            synchronized (LOCK) {
                features = eClass.getEAllStructuralFeatures();
            }

            for (int counter = 0; counter < features.size(); counter++) {

                EStructuralFeature structuralFeature = (EStructuralFeature) features.get(counter);

                //Remove the 'any' structural feature for special processing:
                if(!structuralFeature.getName().equals("any")){
                    
                    HashMap featureSubMap = (HashMap) FEATURE_MAP.get(eClass.getName());
    
                    if (featureSubMap == null) {
                        featureSubMap = new HashMap();
                        FEATURE_MAP.put(eClass.getName(), featureSubMap);
                    }
    
                    featureSubMap.put(structuralFeature.getName(), structuralFeature);
                }
            }
        }
    }

    /**
     * No argument constructor.
     * <p>
     */
    public EventHandler() {

        anyElementXMLFragment = new StringBuffer();

        charactersBuffer = new StringBuffer();
        
        stack = new ArrayList();
    }

    /**
     * Initializes the event handler to a newly constructed state.
     * <p>
     */
    public void init() {

        reset();

    	eventListener = null;
    }

    /**
     * Resets the event handler for parsing the next element.
     * <p>
     */
    protected void reset() {

        column = 0;

        line = 0;

        anyElementNestingDepth = 0;

        nameSpaceAttribute = null;

        if(stack.size() > 0){
        	stack.clear();
        }

        int stringBufferLength = anyElementXMLFragment.length();
        
        if(stringBufferLength > 0){
        	anyElementXMLFragment.setLength(0);
        }

        stringBufferLength = charactersBuffer.length();
        
        if(stringBufferLength > 0){
        	charactersBuffer.setLength(0);
        }     
    }

    /**
     * Registers the non-<code>null</code> parameter {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} 
     * implementation with this event handler.
     * <p>
     * The registered {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} 
     * implementation is notified to process 
     * {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent}s and/or 
     * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine}s when de-serializing 
     * a stream of Common Base Event and/or Association Engine XML fragments in a Common 
     * Base Event XML document.
     * <p>
     * 
     * @param eventListener A non-<code>null</code> {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} to be registered with this event handler.
     */
    public void addEventListener(EventListener eventListener) {
    	this.eventListener = eventListener;
    }
    
    /**
     * Retrieves the {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} 
     * implementation registered with this event handler.
     * <p>
     * The registered {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} 
     * implementation is notified to process 
     * {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent}s and/or 
     * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine}s when de-serializing 
     * a stream of Common Base Event and/or Association Engine XML fragments in a Common 
     * Base Event XML document.
     * <p>
     * 
     * @return The {@link org.eclipse.hyades.logging.events.cbe.util.EventListener} registered with this event handler.
     */
    public EventListener getEventListener() {
    	return eventListener;
    }

    /**
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator locator) {

        super.setDocumentLocator(locator);

        column = locator.getColumnNumber();
        line = locator.getLineNumber();
    }

    /**
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char characters[], int start, int length) throws SAXException {

        super.characters(characters, start, length);

        charactersBuffer.append(new String(characters, start, length));
    }

    /**
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String,
     *      java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri) throws SAXException {

        super.startPrefixMapping(prefix, uri);

        //NOTE: Xerces fires name space mapping events for all elements.
        //Do not capture name space mapping events for the 'xmlns' and 'xml' name space:
        if ((!prefix.equals("xmlns")) && (!prefix.equals("xml"))) {

            nameSpaceAttribute = new StringBuffer("xmlns");

            if ((prefix != null) && (prefix.trim().length() > 0)) {
                nameSpaceAttribute.append(':');
                nameSpaceAttribute.append(prefix);
            }

            nameSpaceAttribute.append("=\"");
            nameSpaceAttribute.append(uri);
            nameSpaceAttribute.append('\"');
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String,
     *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qualifiedName, Attributes attributes) throws SAXException {
        
    	super.startElement(uri, localName, qualifiedName, attributes);

        String elementName = getLocalElementName(localName, qualifiedName);

        if ((!elementName.equals("CommonBaseEvents")) && (!elementName.equals("TemplateEvent")) || anyElementNestingDepth > 0) {

            //The EMF class and object for this XML element:
            EObject emfObj = null;
            EClass emfObjClass = null;

            //If the stack is empty, this element should only be either a
            // <CommonBaseEvent> or <AssociationEngine> element:
            if (stack.isEmpty()) {

                if ((!elementName.equals(COMMON_BASE_EVENT_CLASS)) && (!elementName.equals(ASSOCIATION_ENGINE_CLASS))) { 
                    throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_",COMMON_BASE_EVENT_CLASS + " or " + ASSOCIATION_ENGINE_CLASS, elementName, String.valueOf(line), String.valueOf(column))); 
                }

                emfObjClass = (EClass) ELEMENT_MAP.get(elementName);
                emfObj = EventFactory.eINSTANCE.create(emfObjClass);

                //Set the Common Base Event's version property to <code>null</code> for Common Base Event XML elements without a version attribute (e.g. no version as compared to an empty version):
                if (elementName.equals(COMMON_BASE_EVENT_CLASS)) {
                    emfObj.eSet(EventPackage.eINSTANCE.getCommonBaseEvent_Version(),null);
                }
                
                stack.add(emfObj);
            } 
            else {

                EObject container = (EObject) stack.get(stack.size() - 1);
                String containerName = container.eClass().getName();

                HashMap featureSubMap = (HashMap) FEATURE_MAP.get(containerName);
                EStructuralFeature objStructuralFeature = (EStructuralFeature) featureSubMap.get(elementName);

                //This element is assumed to be an any element if no structural
                // feature was found:
                if (objStructuralFeature == null || anyElementNestingDepth > 0) {

                    if ((containerName.equals(COMMON_BASE_EVENT_CLASS)) || (containerName.equals(OTHER_SITUATION_CLASS))) {

                        if (anyElementNestingDepth > 0) {

                            String anyElementCharacters = charactersBuffer.toString();

                            //Do not persist white space (e.g. formatting
                            // characters) between elements:
                            if (anyElementCharacters.trim().length() > 0) {
                                anyElementXMLFragment.append(XmlUtility.normalize(anyElementCharacters));
                            }
                        }

                        anyElementNestingDepth++;

                        anyElementXMLFragment.append('<');
                        anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qualifiedName));

                        //Add the name space mapping attribute:
                        if(nameSpaceAttribute != null){
                            
                            anyElementXMLFragment.append(' ');
                            anyElementXMLFragment.append(nameSpaceAttribute.toString());
                        }
                        
                        for (int counter = 0; counter < attributes.getLength(); counter++) {

                            anyElementXMLFragment.append(' ');
                            anyElementXMLFragment.append(getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)));
                            anyElementXMLFragment.append("=\"");
                            anyElementXMLFragment.append(XmlUtility.normalize(attributes.getValue(counter)));
                            anyElementXMLFragment.append("\"");
                        }

                        anyElementXMLFragment.append('>');
                    } else {
                        throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_UNEXPECTED_XML_ELEMENT_EXC_", elementName, String.valueOf(line), String.valueOf(column)));
                    }
                } else {

                    //Special case the situation type:
                    String className = objStructuralFeature.getEType().getName();

                    if (className.equals(SITUATION_TYPE_CLASS)) {
                        
                        className = ((Situation) (container)).getCategoryName();
                        
                        //The situation's category is required to identify the class of situation type:
                        if((className == null) || (className.trim().length() == 0)){
                            throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", "categoryName", "<empty>", "0", "0"));
                        }
                    }

                    //Create instance:
                    if (className.equals("EString")) {

                        if (objStructuralFeature.isMany()) {
                            stack.add(((List) (container.eGet(objStructuralFeature))));
                        } else {
                            //This structural feature contains PCDATA to be set
                            // in the characters() API:
                            stack.add(objStructuralFeature);
                        }
                    } else {

                        emfObjClass = (EClass) ELEMENT_MAP.get(className);
                        emfObj = EventFactory.eINSTANCE.create(emfObjClass);

                        //Setting the object inside the container object:
                        if (objStructuralFeature.isMany()) {
                            ((List) (container.eGet(objStructuralFeature))).add(emfObj);
                        } else {
                            container.eSet(objStructuralFeature, emfObj);
                        }

                        stack.add(emfObj);
                    }
                }
            }

            //Fill instance attributes:
            if (emfObj != null && emfObjClass != null) {

                HashMap featureSubMap = (HashMap) FEATURE_MAP.get(emfObjClass.getName());
                EAttribute feature = null;
                String attributeTypeName = null;

                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    //Find structural feature:
                    feature = (EAttribute) featureSubMap.get(getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter)));

                    //The attribute could not be in the featureSubMap because
                    //it is an xmlns declaration or some other xml directive
                    // (e.g. <SituationType ... xsi:type="..."/>):
                    if (feature != null) {

                        attributeTypeName = feature.getEAttributeType().getName();

                        if (attributeTypeName.equals("ELong")) {
                        
                            try {
                                emfObj.eSet(feature, Long.valueOf(attributes.getValue(counter)));
                            } 
                            catch (NumberFormatException n) {
                                
                                //If the attribute value is not white space (e.g. empty), throw an exception:
                                if(attributes.getValue(counter).trim().length() > 0){
                                    throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)), attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                                }
                            }
                        } 
                        else if (attributeTypeName.equals("EShort")) {

                            try {
                                emfObj.eSet(feature, Short.valueOf(attributes.getValue(counter)));
                            } 
                            catch (NumberFormatException n) {
                                
                                //If the attribute value is not white space (e.g. empty), throw an exception:
                                if(attributes.getValue(counter).trim().length() > 0){
                                    throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)), attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                                }
                            }
                        } 
                        else {
                            emfObj.eSet(feature, attributes.getValue(counter));
                        }
                    }
                }
            }

            //Empty the buffer of characters (e.g. white space):
            charactersBuffer.setLength(0);

            //Release the name space mapping attribute since #startPrefixMapping(String, String) is <i>always</i> called before #startElement(String, String, String, Attributes):  
            nameSpaceAttribute = null;
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qualifiedName) throws SAXException {

        super.endElement(uri, localName, qualifiedName);

        String elementName = getLocalElementName(localName, qualifiedName);

        if ((!elementName.equals("CommonBaseEvents")) && (!elementName.equals("TemplateEvent")) || anyElementNestingDepth > 0) {

            if (stack.isEmpty()) {
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_UNEXPECTED_XML_ELEMENT_EXC_", elementName, String.valueOf(line), String.valueOf(column)));
            } 
            else {
                
                if (anyElementNestingDepth > 0) {

                    String anyElementCharacters = charactersBuffer.toString();
                    
                    //Do not persist white space (e.g. formatting characters)
                    // between elements:
                    if (anyElementCharacters.trim().length() > 0) {
                        anyElementXMLFragment.append(XmlUtility.normalize(anyElementCharacters));
                    }

                    anyElementXMLFragment.append("</");
                    anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qualifiedName));
                    anyElementXMLFragment.append('>');

                    if (--anyElementNestingDepth == 0) {

                        Object object = stack.get(stack.size() - 1);

                        if (object instanceof CommonBaseEvent) {
                            ((CommonBaseEvent) (object)).addAny(anyElementXMLFragment.toString());
                        } 
                        else if (object instanceof OtherSituation) {

                            OtherSituation otherSituation = ((OtherSituation) (object));
                            String currentAnyElement = otherSituation.getAny();

                            if ((currentAnyElement != null) && (currentAnyElement.trim().length() > 0)) {
                                otherSituation.setAny(currentAnyElement.concat(anyElementXMLFragment.toString()));
                            } 
                            else {
                                otherSituation.setAny(anyElementXMLFragment.toString());
                            }
                        }

                        anyElementXMLFragment.setLength(0);
                    }
                } 
                else {

                    Object object = stack.remove(stack.size() - 1);

                    if (object instanceof CommonBaseEvent) {

                        String anyElementCharacters = charactersBuffer.toString();
                        
                        //Do not persist white space (e.g. formatting characters)
                        // between elements:
                        if (anyElementCharacters.trim().length() > 0) {
                            ((CommonBaseEvent) (object)).addAny(anyElementCharacters.trim());
                        }
                        
                        if(eventListener != null){
                        	eventListener.processCommonBaseEvent(((CommonBaseEvent)(object)));
                        }

                        reset();

                    }
					else if (object instanceof OtherSituation) {

					    String anyElementCharacters = charactersBuffer.toString();
                        
                        //Do not persist white space (e.g. formatting characters)
                        // between elements:
                        if (anyElementCharacters.trim().length() > 0) {

                            OtherSituation otherSituation = ((OtherSituation) (object));
                            String currentAnyElement = otherSituation.getAny();

                            if ((currentAnyElement != null) && (currentAnyElement.trim().length() > 0)) {
                                otherSituation.setAny(currentAnyElement.concat(anyElementCharacters));
                            } 
                            else {
                                otherSituation.setAny(anyElementCharacters.trim());
                            }
				        }					
					}

					//NOTE: Must check for <AssociationEngine .../> elements
                    // since the object may be an <AssociationEngineInfo .../>
                    // element:
                    else if ((elementName.equals(ASSOCIATION_ENGINE_CLASS)) && (object instanceof AssociationEngine)) {

                        if(eventListener != null){
                        	eventListener.processAssociationEngine(((AssociationEngine)(object)));
                        }

                        reset();
                    }
                    //Persist PCDATA:
                    else{

                        if (stack.size() > 1) {

                            if (object instanceof List) {
                                ((List) (object)).add(charactersBuffer.toString());
                            }

                            if (object instanceof EStructuralFeature) {
                                ((EObject) (stack.get(stack.size() - 1))).eSet(((EStructuralFeature) (object)), charactersBuffer.toString());
                            }
                        }
                    }
                }
            }

            //Empty the buffer of characters (e.g. PCDATA):
            charactersBuffer.setLength(0);
        }
    }

    /**
     * @see org.xml.sax.helpers.DefaultHandler#warning(org.xml.sax.SAXParseException)
     */
    public void warning(SAXParseException saxParseException) throws SAXException {
        throw saxParseException;
    }

    /**
     * @see org.xml.sax.helpers.DefaultHandler#error(org.xml.sax.SAXParseException)
     */
    public void error(SAXParseException saxParseException) throws SAXException {
        throw saxParseException;
    }

    /**
     * @see org.xml.sax.helpers.DefaultHandler#fatalError(org.xml.sax.SAXParseException)
     */
    public void fatalError(SAXParseException saxParseException) throws SAXException {
        throw saxParseException;
    }
    
    /**
     * Resolves the local name of an element based on either the 
     * parameter local name or qualified name.
     * <p>
     * If the parameter local name is not <code>null</code> or
     * not empty, the local name is returned.  Otherwise, the
     * local name is resolved from the qualified name of the 
     * element.
     * <p>
     * 
     * @param localName The local name of an element.
     * @param qualifiedName The qualified name of an element
     * @return The local name of an element.
     */
    protected String getLocalElementName(String localName, String qualifiedName) {

        if ((localName == null) || (localName.trim().length() == 0)) {

            if (qualifiedName != null) {

                int lastColonIndex = qualifiedName.lastIndexOf(':');

                if (lastColonIndex != -1) {
                    return (qualifiedName.substring(lastColonIndex + 1).trim());
                } 
                else {
                    return (qualifiedName.trim());
                }
            }
            else {
                return "";
            }
        }

        return (localName.trim());
    }


    /**
     * Resolves the qualified name of an element based on either the 
     * parameter local name and URI or qualified name.
     * <p>
     * If the parameter qualified name is not <code>null</code> or
     * not empty, the qualified name is returned.  Otherwise, the
     * qualified name is resolved from the local name and URI of the 
     * element.
     * <p>
     * 
     * @param uri The URI of an element.
     * @param localName The local name of an element.
     * @param qualifiedName The qualified name of an element
     * @return The qualified name of an element.
     */
    protected String getQualifiedElementName(String uri, String localName, String qualifiedName) {

        if ((qualifiedName == null) || (qualifiedName.trim().length() == 0)) {

            if (localName != null) {

                if ((uri != null) && (uri.trim().length() > 0)) {
                    return (uri.concat(":").concat(localName));
                } 
                else {
                    return (localName.trim());
                }
            } 
            else {
                return "";
            }
        }

        return (qualifiedName.trim());
    }
}
