package org.eclipse.hyades.logging.events.cbe.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.StringReader;
import java.net.URL;
import java.security.AccessController;
import java.security.PrivilegedExceptionAction;
import java.util.List;

import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;
import org.eclipse.hyades.logging.core.SerializationException;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.hyades.logging.events.cbe.AssociatedEvent;
import org.eclipse.hyades.logging.events.cbe.AssociationEngine;
import org.eclipse.hyades.logging.events.cbe.AvailableSituation;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ConfigureSituation;
import org.eclipse.hyades.logging.events.cbe.ConnectSituation;
import org.eclipse.hyades.logging.events.cbe.ContextDataElement;
import org.eclipse.hyades.logging.events.cbe.CreateSituation;
import org.eclipse.hyades.logging.events.cbe.DependencySituation;
import org.eclipse.hyades.logging.events.cbe.DestroySituation;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.FeatureSituation;
import org.eclipse.hyades.logging.events.cbe.FormattingException;
import org.eclipse.hyades.logging.events.cbe.MsgCatalogToken;
import org.eclipse.hyades.logging.events.cbe.MsgDataElement;
import org.eclipse.hyades.logging.events.cbe.OtherSituation;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.RequestSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.SituationType;
import org.eclipse.hyades.logging.events.cbe.StartSituation;
import org.eclipse.hyades.logging.events.cbe.StopSituation;
import org.eclipse.hyades.logging.events.cbe.internal.util.EventHandler;
import org.eclipse.hyades.logging.events.cbe.internal.util.EventListenerImpl;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventFormatter.java,v 1.4 2008/02/12 22:35:57 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Utility class for converting, serializing and de-serializing
 * {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent}s and 
 * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine}s.
 * <p>
 * Conversion, serialization and de-serialization of 
 * {@link org.eclipse.hyades.logging.events.cbe.CommonBaseEvent}s and 
 * {@link org.eclipse.hyades.logging.events.cbe.AssociationEngine}s is based on
 * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
 * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
 * <p>
 * <b>Notes: </b>
 * <p>
 * <ul>
 * <li>For converting between Document Object Model (DOM) and XML documents, see the 
 * {@link org.eclipse.hyades.logging.core.XmlUtility#serialize(org.w3c.dom.Document)} and
 * {@link org.eclipse.hyades.logging.core.XmlUtility#deserialize(java.lang.String)} methods 
 * in {@link org.eclipse.hyades.logging.core.XmlUtility}.</li>
 * <li>Some JAXP implementations (e.g. Sun's Java V1.4.x with Crimson) do not support schema 
 * validation. As such, validation in these run-time environments is not supported.</li>
 * </ul>
 * <p>
 * 
 * @author  Richard K. Duggan
 * @author  Paul E. Slauenwhite
 * @author  John K. Gerken
 * @version 1.0.1
 * @since   1.0
 * @see     org.eclipse.hyades.logging.events.cbe.CommonBaseEvent
 * @see     org.eclipse.hyades.logging.events.cbe.AssociationEngine
 * @see     org.eclipse.hyades.logging.events.cbe.FormattingException
 * @see     org.eclipse.hyades.logging.events.cbe.util.EventListener
 * @see     org.eclipse.hyades.logging.core.XmlUtility
 */
public final class EventFormatter {

	/**
	 * The cached instance of the event handler used with a 
	 * <a href="http://www.saxproject.org/">Simple API for XML (SAX)</a> 
	 * parser for de-serializing Common Base Event XML documents.
	 * <p>
	 */
	protected static EventHandler eventHandler = null;

	/**
	 * The cached instance of the 
	 * <a href="http://www.saxproject.org/">Simple API for XML (SAX)</a> 
	 * parser for de-serializing Common Base Event XML documents.
	 * <p>
	 */
	protected static XMLReader xmlParser = null;

	/**
	 * Flag for checking if the cached instance of the 
	 * <a href="http://www.saxproject.org/">Simple API for XML (SAX)</a>
	 * supports validation.
	 * <p>
	 */
	protected static boolean isValidationSupported = true;

	/**
	 * Name of the Common Base Event V1.0.1 schema (XSD) used for validation.
	 * <p>
	 */
	protected final static String COMMON_BASE_EVENT_SCHEMA_NAME = "commonbaseevent1_0_1.xsd";

	/**
	 * Location (URL) of the Common Base Event V1.0.1 schema (XSD) used for validation.
	 * <p>
	 */
	protected final static String COMMON_BASE_EVENT_SCHEMA_LOCATION_URL = resolveCommonBaseEventSchemaLocation();

	/**
	 * Current thread lock for synchronization.
	 * <p>
	 */
	protected static final Object LOCK = new Object();

	/**
	 * Serializes an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} to an Association Engine 
	 * XML fragment contained in a Common Base Event XML document.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * by default. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param associationEngine The {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} to be serialized to an Association Engine XML fragment contained in a Common Base Event XML document.
	 * @return A Common Base Event XML document containing the serialized Association Engine XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLDocString(AssociationEngine associationEngine) {
		return (toCanonicalXMLDocString(associationEngine, true));
	}

	/**
	 * Serializes an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s to Association Engine 
	 * XML fragments contained in a Common Base Event XML document.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * by default. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param associationEngines The array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s to be serialized to Association Engine XML fragments contained in a Common Base Event XML document.
	 * @return A Common Base Event XML document containing the serialized Association Engine XML fragments.
	 * @since 1.0.1
	 */
	public static String toCanonicalXMLDocString(AssociationEngine[] associationEngines) {
		return (toCanonicalXMLDocString(associationEngines, true));
	}

	/**
	 * Serializes an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} to an Association Engine 
	 * XML fragment contained in a Common Base Event XML document.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * based on the <code>format</code> parameter. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param associationEngine The {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} to be serialized to an Association Engine XML fragment.
	 * @param format If the resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation).
	 * @return A Common Base Event XML document containing the serialized Association Engine XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLDocString(AssociationEngine associationEngine, boolean format) {
		return (toCanonicalXMLDocString(new AssociationEngine[] { associationEngine }, format));
	}

	/**
	 * Serializes an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s to Association Engine 
	 * XML fragments contained in a Common Base Event XML document.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * based on the <code>format</code> parameter. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param associationEngines The array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s to be serialized to Association Engine XML fragments contained in a Common Base Event XML document.
	 * @param format If the resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation).
	 * @return A Common Base Event XML document containing the serialized Association Engine XML fragments.
	 * @since 1.0.1
	 */
	public static String toCanonicalXMLDocString(AssociationEngine[] associationEngines, boolean format) {

		StringBuffer buffer = new StringBuffer(256);

		buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");

		if (format) {
			buffer.append(Constants.LINE_SEPARATOR);
		}

		buffer
				.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");

		if (format) {
			buffer.append(Constants.LINE_SEPARATOR);
		}

		if (associationEngines != null) {

			for (int index = 0; index < associationEngines.length; index++) {

				if (format) {
					buffer.append("\t");
				}

				buffer.append(toCanonicalXMLString(associationEngines[index]));

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
				}
			}
		}

		buffer.append("</CommonBaseEvents>");

		return (buffer.toString());
	}

	/**
	 * Serializes an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} to an Association Engine 
	 * XML fragment.
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param associationEngine The {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} to be serialized to an Association Engine XML fragment.
	 * @return A serialized Association Engine XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLString(AssociationEngine associationEngine) {

		StringBuffer buffer = new StringBuffer(256);

		buffer.append("<AssociationEngine");

		String attributeValue = associationEngine.getId();

		// For performance reasons, do not normalize attribute value since of
		// type xsd:ID which is guaranteed to not contain mark-up characters:
		if (attributeValue != null) {
			buffer.append(" id=\"");
			buffer.append(attributeValue);
			buffer.append("\"");
		}

		attributeValue = associationEngine.getName();

		// For performance reasons, do not normalize attribute value since of
		// type xsd:Name which is guaranteed to not contain mark-up characters:
		if (attributeValue != null) {
			buffer.append(" name=\"");
			buffer.append(attributeValue);
			buffer.append("\"");
		}

		attributeValue = associationEngine.getType();

		// For performance reasons, do not normalize attribute value since of
		// type xsd:Name which is guaranteed to not contain mark-up characters:
		if (attributeValue != null) {
			buffer.append(" type=\"");
			buffer.append(attributeValue);
			buffer.append("\"");
		}

		buffer.append("/>");

		return (buffer.toString());
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s. 
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(String xmlDocumentString) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(xmlDocumentString, false));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s. 
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(String xmlDocumentString, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(xmlDocumentString, false, eventListener);
	}
	
	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s. 
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(String xmlDocumentString, boolean validate) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(new InputSource(new StringReader(xmlDocumentString)), validate));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(String xmlDocumentString, boolean validate, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(new InputSource(new StringReader(xmlDocumentString)), validate, eventListener);
	}
	
	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputSource inputSource) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(inputSource, false));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(InputSource inputSource, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(inputSource, false, eventListener);
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputSource inputSource, boolean validate) throws FormattingException {

		EventListenerImpl eventListener = new EventListenerImpl();
		
		associationEnginesFromCanonicalXMLDoc(inputSource, validate, eventListener);
		
		return (eventListener.getAssociationEngines());
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(InputSource inputSource, boolean validate, EventListener eventListener) throws FormattingException {
		parse(inputSource, validate, eventListener);
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputStream inputStream) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(inputStream, false));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(InputStream inputStream, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(inputStream, false, eventListener);
	}
	
	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(InputStream inputStream, boolean validate) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(new InputSource(inputStream), validate));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(InputStream inputStream, boolean validate, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(new InputSource(inputStream), validate, eventListener);
	}
	
	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(Document document) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(document, false));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(Document document, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(document, false, eventListener);
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(Document document, boolean validate) throws FormattingException {

		try {
			return (associationEnginesFromCanonicalXMLDoc(XmlUtility.serialize(document), validate));
		} 
		catch (SerializationException s) {
			throw new FormattingException(s);
		}
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(Document document, boolean validate, EventListener eventListener) throws FormattingException {

		try {
			associationEnginesFromCanonicalXMLDoc(XmlUtility.serialize(document), validate, eventListener);
		} 
		catch (SerializationException s) {
			throw new FormattingException(s);
		}
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(File file) throws FormattingException {
		return (associationEnginesFromCanonicalXMLDoc(file, false));
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(File file, EventListener eventListener) throws FormattingException {
		associationEnginesFromCanonicalXMLDoc(file, false, eventListener);
	}
	
	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static AssociationEngine[] associationEnginesFromCanonicalXMLDoc(File file, boolean validate) throws FormattingException {

		try {
			return (associationEnginesFromCanonicalXMLDoc(new InputSource(new FileInputStream(file)), validate));
		} catch (FileNotFoundException f) {
			throw new FormattingException(f);
		}
	}

	/**
	 * De-serializes Association Engine XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void associationEnginesFromCanonicalXMLDoc(File file, boolean validate, EventListener eventListener) throws FormattingException {

		try {
			associationEnginesFromCanonicalXMLDoc(new InputSource(new FileInputStream(file)), validate, eventListener);
		} 
		catch (FileNotFoundException f) {
			throw new FormattingException(f);
		}
	}
	
	/**
	 * De-serializes an Association Engine XML fragment to an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}. 
	 * <p>
	 * The Association Engine XML fragment is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlFragmentString
	 *            The string with the Association Engine XML fragment to be de-serialized to an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}.
	 * @return An {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0
	 */
	public static AssociationEngine associationEngineFromCanonicalXML(String xmlFragmentString) throws FormattingException {
		return (associationEngineFromCanonicalXML(xmlFragmentString, false));
	}

	/**
	 * De-serializes an Association Engine XML fragment to an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}. 
	 * <p>
	 * The Association Engine XML fragment is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlFragmentString
	 *            The string with the Association Engine XML fragment to be de-serialized to an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}.
	 * @param validate If the Association Engine XML fragment is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0
	 */
	public static AssociationEngine associationEngineFromCanonicalXML(String xmlFragmentString, boolean validate) throws FormattingException {

		StringBuffer buffer = new StringBuffer(1024);

		buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		buffer
				.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
		buffer.append(xmlFragmentString);
		buffer.append("</CommonBaseEvents>");

		AssociationEngine[] engines = associationEnginesFromCanonicalXMLDoc(buffer.toString(), validate);

		if (engines != null) {
			return (engines[0]);
		}

		return null;
	}

	/**
	 * Serializes a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to a Common Base Event 
	 * XML fragment contained in a Common Base Event XML document.
	 * <p>
	 * The Common Base Event XML document is constructed by first serializing each of the {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s
	 * contained in each {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent} 
	 * ({@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngineInfo()}) for the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to an Association Engine XML fragment.  Next, the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} is serialized to a Common Base Event XML fragment.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * by default. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param commonBaseEvent The {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to be serialized to a Common Base Event XML fragment contained in a Common Base Event XML document.
	 * @return A Common Base Event XML document containing the serialized Common Base Event XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLDocString(CommonBaseEvent commonBaseEvent) {
		return (toCanonicalXMLDocString(commonBaseEvent, true));
	}

	/**
	 * Serializes an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s to Common Base Event 
	 * XML fragments contained in a Common Base Event XML document.
	 * <p>
	 * The Common Base Event XML document is constructed by first serializing each of the {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s
	 * contained in each {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent} 
	 * ({@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngineInfo()}) for each of the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s to an Association Engine XML fragment.  Next, each 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} is serialized to a Common Base Event XML fragment.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * by default. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param commonBaseEvents The array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s to be serialized to Common Base Event XML fragments contained in a Common Base Event XML document.
	 * @return A Common Base Event XML document containing the serialized Common Base Event XML fragments.
	 * @since 1.0.1
	 */
	public static String toCanonicalXMLDocString(CommonBaseEvent[] commonBaseEvents) {
		return (toCanonicalXMLDocString(commonBaseEvents, true));
	}

	/**
	 * Serializes a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to a Common Base Event 
	 * XML fragment contained in a Common Base Event XML document.
	 * <p>
	 * The Common Base Event XML document is constructed by first serializing each of the {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s
	 * contained in each {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent} 
	 * ({@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngineInfo()}) for the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to an Association Engine XML fragment.  Next, the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} is serialized to a Common Base Event XML fragment.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * based on the <code>format</code> parameter. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param commonBaseEvent The {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to be serialized to a Common Base Event XML fragment.
	 * @param format If the resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation).
	 * @return A Common Base Event XML document containing the serialized Common Base Event XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLDocString(CommonBaseEvent commonBaseEvent, boolean format) {
		return (toCanonicalXMLDocString(new CommonBaseEvent[] { commonBaseEvent }, format));
	}

	/**
	 * Serializes an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s to Common Base Event 
	 * XML fragments contained in a Common Base Event XML document.
	 * <p>
	 * The Common Base Event XML document is constructed by first serializing each of the {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s
	 * contained in each {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent} 
	 * ({@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngineInfo()}) for each of the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s to an Association Engine XML fragment.  Next, each 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} is serialized to a Common Base Event XML fragment.
	 * <p>
	 * The resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation) 
	 * based on the <code>format</code> parameter. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param commonBaseEvents The array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s to be serialized to Common Base Event XML fragments contained in a Common Base Event XML document.
	 * @param format If the resultant Common Base Event XML document is formatted (e.g. platform-dependent new lines and indentation).
	 * @return A Common Base Event XML document containing the serialized Common Base Event XML fragments.
	 * @since 1.0.1
	 */
	public static String toCanonicalXMLDocString(CommonBaseEvent[] commonBaseEvents, boolean format) {

		//Create a buffer containing the Common Base Event XML document:
		StringBuffer xmlDocument = new StringBuffer(1024);

		xmlDocument.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");

		if (format) {
			xmlDocument.append(Constants.LINE_SEPARATOR);
			xmlDocument.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
			xmlDocument.append(Constants.LINE_SEPARATOR);
		}
		else{
			xmlDocument.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
		}

		StringBuffer commonBaseEventsBuffer = new StringBuffer(1024);
		
		if (commonBaseEvents != null) {

			for (int index = 0; index < commonBaseEvents.length; index++) {

				List associatedEvents = commonBaseEvents[index].getAssociatedEvents();

				//If associatedEvents exist, append the serialized association engine to the top of the Common Base Event XML document:
				if (associatedEvents.size() > 0) {

					AssociationEngine associationEngine = null;

					for (int counter = 0; counter < associatedEvents.size(); counter++) {

						associationEngine = ((AssociatedEvent) (associatedEvents.get(counter))).getAssociationEngineInfo();

						if (associationEngine != null) {

							if (format) {
								xmlDocument.append("\t");
								xmlDocument.append(toCanonicalXMLString(associationEngine));
								xmlDocument.append(Constants.LINE_SEPARATOR);
							}
							else{
								xmlDocument.append(toCanonicalXMLString(associationEngine));								
							}
						}
					}
				}

				if (format) {

					// Add the Common Base Event element XML fragment to the Common Base Event buffer and append
					// an extra tab after every platform dependent new line character to indent all child XML fragments:
					String[] tokens = InternationalizationUtilities.tokenize(toCanonicalXMLString(commonBaseEvents[index], format), Constants.LINE_SEPARATOR);

					for (int counter = 0; counter < tokens.length; counter++) {
						
						commonBaseEventsBuffer.append("\t");
						commonBaseEventsBuffer.append(tokens[counter]);
						commonBaseEventsBuffer.append(Constants.LINE_SEPARATOR);
					}
				} else {

					// Add the Common Base Event element XML fragment to the Common Base Event buffer:
					commonBaseEventsBuffer.append(toCanonicalXMLString(commonBaseEvents[index], format));
				}
			}
		}

		//Append the Common Base Event buffer to the Common Base Event XML document:
		xmlDocument.append(commonBaseEventsBuffer);
		
		// Close and return the top level element:
		xmlDocument.append("</CommonBaseEvents>");

		return xmlDocument.toString();
	}

	/**
	 * Serializes a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to a Common Base Event 
	 * XML fragment.
	 * <p>
	 * The resultant Common Base Event XML fragment is formatted (e.g. platform-dependent new lines and indentation) 
	 * by default.
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param commonBaseEvent The {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to be serialized to a Common Base Event XML fragment.
	 * @return A serialized Common Base Event XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLString(CommonBaseEvent commonBaseEvent) {
		return (toCanonicalXMLString(commonBaseEvent, true));
	}

	/**
	 * Serializes a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to a Common Base Event 
	 * XML fragment.
	 * <p>
	 * The resultant Common Base Event XML fragment is formatted (e.g. platform-dependent new lines and indentation) 
	 * based on the <code>format</code> parameter. 
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param commonBaseEvent The {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} to be serialized to a Common Base Event XML fragment.
	 * @param format If the resultant Common Base Event XML fragment is formatted (e.g. platform-dependent new lines and indentation).
	 * @return A serialized Common Base Event XML fragment.
	 * @since 1.0
	 */
	public static String toCanonicalXMLString(CommonBaseEvent commonBaseEvent, boolean format) {

		// Start with a 1K buffer to load the XML String:
		StringBuffer buffer = new StringBuffer(1024);

		buffer.append("<CommonBaseEvent");

		// For performance reasons, do not normalize attribute value since of
		// type xsd:dateTime which is guaranteed to not contain mark-up
		// characters:
		if (commonBaseEvent.isSetCreationTime()) {
			buffer.append(" creationTime=\"");
			buffer.append(commonBaseEvent.getCreationTime());
			buffer.append("\"");
		}

		String attributeValue = commonBaseEvent.getExtensionName();

		// For performance reasons, do not normalize attribute value since of
		// type xsd:Name which is guaranteed to not contain mark-up characters:
		if (attributeValue != null) {
			buffer.append(" extensionName=\"");
			buffer.append(attributeValue);
			buffer.append("\"");
		}

		attributeValue = commonBaseEvent.getGlobalInstanceId();

		// For performance reasons, do not normalize attribute value since of
		// type xsd:ID which is guaranteed to not contain mark-up characters:
		if (attributeValue != null) {
			buffer.append(" globalInstanceId=\"");
			buffer.append(attributeValue);
			buffer.append("\"");
		}

		attributeValue = commonBaseEvent.getLocalInstanceId();

		if (attributeValue != null) {
			buffer.append(" localInstanceId=\"");
			buffer.append(XmlUtility.normalize(attributeValue));
			buffer.append("\"");
		}

		attributeValue = commonBaseEvent.getMsg();

		if (attributeValue != null) {
			buffer.append(" msg=\"");
			buffer.append(XmlUtility.normalize(attributeValue));
			buffer.append("\"");
		}

		// For performance reasons, do not normalize attribute value since of
		// type xsd:long which is guaranteed to not contain mark-up characters:
		if (commonBaseEvent.isSetElapsedTime()) {
			buffer.append(" elapsedTime=\"");
			buffer.append(commonBaseEvent.getElapsedTime());
			buffer.append("\"");
		}

		// For performance reasons, do not normalize attribute value since of
		// type xsd:short which is guaranteed to not contain mark-up characters:
		if (commonBaseEvent.isSetPriority()) {
			buffer.append(" priority=\"");
			buffer.append(commonBaseEvent.getPriority());
			buffer.append("\"");
		}

		// For performance reasons, do not normalize attribute value since of
		// type xsd:short which is guaranteed to not contain mark-up characters:
		if (commonBaseEvent.isSetRepeatCount()) {
			buffer.append(" repeatCount=\"");
			buffer.append(commonBaseEvent.getRepeatCount());
			buffer.append("\"");
		}

		// For performance reasons, do not normalize attribute value since of
		// type xsd:long which is guaranteed to not contain mark-up characters:
		if (commonBaseEvent.isSetSequenceNumber()) {
			buffer.append(" sequenceNumber=\"");
			buffer.append(commonBaseEvent.getSequenceNumber());
			buffer.append("\"");
		}

		// For performance reasons, do not normalize attribute value since of
		// type xsd:short which is guaranteed to not contain mark-up characters:
		if (commonBaseEvent.isSetSeverity()) {
			buffer.append(" severity=\"");
			buffer.append(commonBaseEvent.getSeverity());
			buffer.append("\"");
		}

		attributeValue = commonBaseEvent.getVersion();

		if (attributeValue != null) {
			buffer.append(" version=\"");
			buffer.append(XmlUtility.normalize(attributeValue));
			buffer.append("\"");
		}

		buffer.append(">");

		List contextDataElements = commonBaseEvent.getContextDataElements();
		ContextDataElement contextDataElement = null;

		for (int i = 0; i < contextDataElements.size(); i++) {

			contextDataElement = ((ContextDataElement) (contextDataElements.get(i)));

			if (contextDataElement != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t");
				}

				buffer.append("<contextDataElements");

				attributeValue = contextDataElement.getName();

				// For performance reasons, do not normalize attribute value
				// since of type xsd:Name which is guaranteed to not contain
				// mark-up characters:
				if (attributeValue != null) {
					buffer.append(" name=\"");
					buffer.append(attributeValue);
					buffer.append("\"");
				}

				attributeValue = contextDataElement.getType();

				// For performance reasons, do not normalize attribute value
				// since of type xsd:Name which is guaranteed to not contain
				// mark-up characters:
				if (attributeValue != null) {
					buffer.append(" type=\"");
					buffer.append(attributeValue);
					buffer.append("\"");
				}

				buffer.append(">");

				attributeValue = contextDataElement.getContextId();

				if (attributeValue != null) {

					if (format) {
						buffer.append(Constants.LINE_SEPARATOR);
						buffer.append("\t\t");
					}

					// For performance reasons, do not normalize element value
					// since of type xsd:NMTOKEN which is guaranteed to not
					// contain mark-up characters:
					buffer.append("<contextId>");
					buffer.append(attributeValue);
					buffer.append("</contextId>");
				} else {

					attributeValue = contextDataElement.getContextValue();

					if (attributeValue != null) {

						if (format) {
							buffer.append(Constants.LINE_SEPARATOR);
							buffer.append("\t\t");
						}

						buffer.append("<contextValue>");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("</contextValue>");
					}
				}

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t");
				}

				buffer.append("</contextDataElements>");
			}
		}

		List extendedDataElements = commonBaseEvent.getExtendedDataElements();

		for (int i = 0; i < extendedDataElements.size(); i++) {
			buffer.append(getExtendedDataElementXML(((ExtendedDataElement) (extendedDataElements.get(i))), "extendedDataElements", format, (format ? 1 : 0)));
		}

		List associatedEvents = commonBaseEvent.getAssociatedEvents();
		AssociatedEvent associatedEvent = null;

		for (int i = 0; i < associatedEvents.size(); i++) {

			associatedEvent = ((AssociatedEvent) (associatedEvents.get(i)));

			if (associatedEvent != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t");
				}

				buffer.append("<associatedEvents");

				String resolvedEvents = associatedEvent.getResolvedEvents();

				// For performance reasons, do not normalize element value since
				// of type xsd:NMTOKENS which is guaranteed to not contain
				// mark-up characters:
				if (resolvedEvents != null) {

					buffer.append(" resolvedEvents=\"");
					buffer.append(resolvedEvents);
					buffer.append("\"");
				}

				buffer.append(">");

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				AssociationEngine associationEngineInfo = associatedEvent.getAssociationEngineInfo();

				if (associationEngineInfo != null) {

					buffer.append("<associationEngineInfo");

					// Add the 'Info' to '<AssociationEngine'.
					buffer.append(toCanonicalXMLString(associationEngineInfo).substring(18)); // 18
					// is
					// the
					// length
					// of
					// '<AssociationEngine'.
				} else {

					attributeValue = associatedEvent.getAssociationEngine();

					// For performance reasons, do not normalize element value
					// since of type xsd:NMTOKEN which is guaranteed to not
					// contain mark-up characters:
					if (attributeValue != null) {

						buffer.append("<associationEngine>");
						buffer.append(attributeValue);
						buffer.append("</associationEngine>");
					}
				}

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t");
				}

				buffer.append("</associatedEvents>");
			}
		}

		ComponentIdentification componentId = commonBaseEvent.getReporterComponentId();

		if (componentId != null) {

			if (format) {
				buffer.append(Constants.LINE_SEPARATOR);
				buffer.append("\t");
			}

			buffer.append("<reporterComponentId");

			attributeValue = componentId.getApplication();

			if (attributeValue != null) {
				buffer.append(" application=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getComponent();

			if (attributeValue != null) {
				buffer.append(" component=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getComponentIdType();

			if (attributeValue != null) {
				buffer.append(" componentIdType=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getExecutionEnvironment();

			if (attributeValue != null) {
				buffer.append(" executionEnvironment=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getInstanceId();

			if (attributeValue != null) {
				buffer.append(" instanceId=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getLocation();

			if (attributeValue != null) {
				buffer.append(" location=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getLocationType();

			// For performance reasons, do not normalize attribute value since
			// of type xsd:Name which is guaranteed to not contain mark-up
			// characters:
			if (attributeValue != null) {
				buffer.append(" locationType=\"");
				buffer.append(attributeValue);
				buffer.append("\"");
			}

			attributeValue = componentId.getProcessId();

			if (attributeValue != null) {
				buffer.append(" processId=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getSubComponent();

			if (attributeValue != null) {
				buffer.append(" subComponent=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getThreadId();

			if (attributeValue != null) {
				buffer.append(" threadId=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getComponentType();

			if (attributeValue != null) {
				buffer.append(" componentType=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			buffer.append("/>");
		}

		componentId = commonBaseEvent.getSourceComponentId();

		if (componentId != null) {

			if (format) {
				buffer.append(Constants.LINE_SEPARATOR);
				buffer.append("\t");
			}

			buffer.append("<sourceComponentId");

			attributeValue = componentId.getApplication();

			if (attributeValue != null) {
				buffer.append(" application=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getComponent();

			if (attributeValue != null) {
				buffer.append(" component=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getComponentIdType();

			if (attributeValue != null) {
				buffer.append(" componentIdType=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getExecutionEnvironment();

			if (attributeValue != null) {
				buffer.append(" executionEnvironment=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getInstanceId();

			if (attributeValue != null) {
				buffer.append(" instanceId=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getLocation();

			if (attributeValue != null) {
				buffer.append(" location=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getLocationType();

			// For performance reasons, do not normalize attribute value since
			// of type xsd:Name which is guaranteed to not contain mark-up
			// characters:
			if (attributeValue != null) {
				buffer.append(" locationType=\"");
				buffer.append(attributeValue);
				buffer.append("\"");
			}

			attributeValue = componentId.getProcessId();

			if (attributeValue != null) {
				buffer.append(" processId=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getSubComponent();

			if (attributeValue != null) {
				buffer.append(" subComponent=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getThreadId();

			if (attributeValue != null) {
				buffer.append(" threadId=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			attributeValue = componentId.getComponentType();

			if (attributeValue != null) {
				buffer.append(" componentType=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			buffer.append("/>");
		}

		MsgDataElement msgDataElement = commonBaseEvent.getMsgDataElement();

		if (msgDataElement != null) {

			if (format) {
				buffer.append(Constants.LINE_SEPARATOR);
				buffer.append("\t");
			}

			buffer.append("<msgDataElement");

			attributeValue = msgDataElement.getMsgLocale();

			// For performance reasons, do not normalize attribute value since
			// of type xsd:language which is guaranteed to not contain mark-up
			// characters:
			if (attributeValue != null) {
				buffer.append(" msgLocale=\"");
				buffer.append(attributeValue);
				buffer.append("\"");
			}

			buffer.append(">");

			List tokens = msgDataElement.getMsgCatalogTokens();
			MsgCatalogToken msgCatalogToken = null;

			for (int i = 0; i < tokens.size(); i++) {

				msgCatalogToken = ((MsgCatalogToken) (tokens.get(i)));

				if ((msgCatalogToken != null) && (msgCatalogToken.getValue() != null)) {

					if (format) {
						buffer.append(Constants.LINE_SEPARATOR);
						buffer.append("\t\t");
					}

					buffer.append("<msgCatalogTokens value=\"");
					buffer.append(XmlUtility.normalize(msgCatalogToken.getValue()));
					buffer.append("\"/>");
				}
			}

			attributeValue = msgDataElement.getMsgId();

			if (attributeValue != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				buffer.append("<msgId>");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("</msgId>");
			}

			attributeValue = msgDataElement.getMsgIdType();

			if (attributeValue != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				// For performance reasons, do not normalize element value since
				// of type xsd:Name which is guaranteed to not contain mark-up
				// characters:
				buffer.append("<msgIdType>");
				buffer.append(attributeValue);
				buffer.append("</msgIdType>");
			}

			attributeValue = msgDataElement.getMsgCatalogId();

			if (attributeValue != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				buffer.append("<msgCatalogId>");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("</msgCatalogId>");
			}

			attributeValue = msgDataElement.getMsgCatalogType();

			if (attributeValue != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				// For performance reasons, do not normalize element value since
				// of type xsd:Name which is guaranteed to not contain mark-up
				// characters:
				buffer.append("<msgCatalogType>");
				buffer.append(attributeValue);
				buffer.append("</msgCatalogType>");
			}

			attributeValue = msgDataElement.getMsgCatalog();

			if (attributeValue != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				buffer.append("<msgCatalog>");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("</msgCatalog>");
			}

			if (format) {
				buffer.append(Constants.LINE_SEPARATOR);
				buffer.append("\t");
			}

			buffer.append("</msgDataElement>");
		}

		Situation situation = commonBaseEvent.getSituation();

		if (situation != null) {

			if (format) {
				buffer.append(Constants.LINE_SEPARATOR);
				buffer.append("\t");
			}

			buffer.append("<situation");

			attributeValue = situation.getCategoryName();

			if (attributeValue != null) {
				buffer.append(" categoryName=\"");
				buffer.append(XmlUtility.normalize(attributeValue));
				buffer.append("\"");
			}

			buffer.append(">");

			SituationType situationType = situation.getSituationType();

			if (situationType != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t\t");
				}

				buffer.append("<situationType xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"");

				attributeValue = situationType.getReasoningScope();

				if (situationType instanceof StartSituation) {

					buffer.append(" xsi:type=\"StartSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((StartSituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((StartSituation) (situationType)).getSituationQualifier();

					if (attributeValue != null) {

						buffer.append(" situationQualifier=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				} else if (situationType instanceof StopSituation) {

					buffer.append(" xsi:type=\"StopSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((StopSituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((StopSituation) (situationType)).getSituationQualifier();

					if (attributeValue != null) {

						buffer.append(" situationQualifier=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				} else if (situationType instanceof ConnectSituation) {

					buffer.append(" xsi:type=\"ConnectSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((ConnectSituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((ConnectSituation) (situationType)).getSituationDisposition();

					if (attributeValue != null) {

						buffer.append(" situationDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				} else if (situationType instanceof ReportSituation) {

					buffer.append(" xsi:type=\"ReportSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((ReportSituation) (situationType)).getReportCategory();

					if (attributeValue != null) {

						buffer.append(" reportCategory=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				}

				else if (situationType instanceof FeatureSituation) {

					buffer.append(" xsi:type=\"FeatureSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((FeatureSituation) (situationType)).getFeatureDisposition();

					if (attributeValue != null) {

						buffer.append(" featureDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				} else if (situationType instanceof ConfigureSituation) {

					buffer.append(" xsi:type=\"ConfigureSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((ConfigureSituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				} else if (situationType instanceof DependencySituation) {

					buffer.append(" xsi:type=\"DependencySituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((DependencySituation) (situationType)).getDependencyDisposition();

					if (attributeValue != null) {

						buffer.append(" dependencyDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				} else if (situationType instanceof CreateSituation) {

					buffer.append(" xsi:type=\"CreateSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((CreateSituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				}

				else if (situationType instanceof DestroySituation) {

					buffer.append(" xsi:type=\"DestroySituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((DestroySituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				}

				else if (situationType instanceof AvailableSituation) {

					buffer.append(" xsi:type=\"AvailableSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((AvailableSituation) (situationType)).getOperationDisposition();

					if (attributeValue != null) {

						buffer.append(" operationDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((AvailableSituation) (situationType)).getProcessingDisposition();

					if (attributeValue != null) {

						buffer.append(" processingDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((AvailableSituation) (situationType)).getAvailabilityDisposition();

					if (attributeValue != null) {

						buffer.append(" availabilityDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				}

				else if (situationType instanceof RequestSituation) {

					buffer.append(" xsi:type=\"RequestSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((RequestSituation) (situationType)).getSuccessDisposition();

					if (attributeValue != null) {

						buffer.append(" successDisposition=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					attributeValue = ((RequestSituation) (situationType)).getSituationQualifier();

					if (attributeValue != null) {

						buffer.append(" situationQualifier=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append("/>");
				}

				else if (situationType instanceof OtherSituation) {

					buffer.append(" xsi:type=\"OtherSituation\"");

					if (attributeValue != null) {
						buffer.append(" reasoningScope=\"");
						buffer.append(XmlUtility.normalize(attributeValue));
						buffer.append("\"");
					}

					buffer.append(">");

					String anyData = ((OtherSituation) (situationType)).getAny();

					if (anyData != null) {

						if (format) {
							buffer.append(Constants.LINE_SEPARATOR);
							buffer.append("\t\t\t");
						}

						// Normalizing since this string might not always be
		                // a valid XML fragment: Bugzilla 207806
		                if(XmlUtility.isValidXML(anyData))
		                {
		                    buffer.append(anyData);
		                }
		                else
		                {
		                    buffer.append(XmlUtility.normalize(anyData));
		                }
					}

					if (format) {
						buffer.append(Constants.LINE_SEPARATOR);
						buffer.append("\t\t");
					}

					buffer.append("</situationType>");
				} else {
					buffer.append("/>");
				}
			}

			if (format) {
				buffer.append(Constants.LINE_SEPARATOR);
				buffer.append("\t");
			}

			buffer.append("</situation>");
		}

		List anyElements = commonBaseEvent.getAny();
		String anyElement = null;

		for (int counter = 0; counter < anyElements.size(); counter++) {

			anyElement = ((String) (anyElements.get(counter)));

			if (anyElement != null) {

				if (format) {
					buffer.append(Constants.LINE_SEPARATOR);
					buffer.append("\t");
				}

				// Normalizing since this string might not always be
				// a valid XML fragment: Bugzilla 207806
				if(XmlUtility.isValidXML(anyElement))
				{
				    buffer.append(anyElement);
				}
				else
				{
				    buffer.append(XmlUtility.normalize(anyElement));
				}				
			}
		}

		if (format) {
			buffer.append(Constants.LINE_SEPARATOR);
		}

		buffer.append("</CommonBaseEvent>");

		return (buffer.toString());
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s. 
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(String xmlDocumentString) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(xmlDocumentString, false));
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s. 
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(String xmlDocumentString, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(xmlDocumentString, false, eventListener);
	}
	
	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(String xmlDocumentString, boolean validate) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(new InputSource(new StringReader(xmlDocumentString)), validate));
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param xmlDocumentString
	 *            The string with the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(String xmlDocumentString, boolean validate, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(new InputSource(new StringReader(xmlDocumentString)), validate, eventListener);
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputSource inputSource) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(inputSource, false));
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(InputSource inputSource, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(inputSource, false, eventListener);
	}
	
	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}.
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputSource inputSource, boolean validate) throws FormattingException {

		EventListenerImpl eventListener = new EventListenerImpl();
		
		eventsFromCanonicalXMLDoc(inputSource, validate, eventListener);
		
		return (eventListener.getCommonBaseEvents());
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link org.xml.sax.InputSource}. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(InputSource inputSource, boolean validate, EventListener eventListener) throws FormattingException {
		parse(inputSource, validate, eventListener);
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputStream inputStream) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(inputStream, false));
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(InputStream inputStream, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(inputStream, false, eventListener);
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}.
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(InputStream inputStream, boolean validate) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(new InputSource(inputStream), validate));
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from an {@link java.io.InputStream}. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputStream
	 *            The {@link java.io.InputStream} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(InputStream inputStream, boolean validate, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(new InputSource(inputStream), validate, eventListener);
	}
	
	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(Document document) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(document, false));
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(Document document, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(document, false, eventListener);
	}
	
	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}.
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(Document document, boolean validate) throws FormattingException {

		try {
			return (eventsFromCanonicalXMLDoc(XmlUtility.serialize(document), validate));
		} catch (SerializationException s) {
			throw new FormattingException(s);
		}
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link org.w3c.dom.Document}. 
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param document
	 *            The {@link org.w3c.dom.Document} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(Document document, boolean validate, EventListener eventListener) throws FormattingException {

		try {
			eventsFromCanonicalXMLDoc(XmlUtility.serialize(document), validate, eventListener);
		} catch (SerializationException s) {
			throw new FormattingException(s);
		}
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(File file) throws FormattingException {
		return (eventsFromCanonicalXMLDoc(file, false));
	}
	
	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD)
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(File file, EventListener eventListener) throws FormattingException {
		eventsFromCanonicalXMLDoc(file, false, eventListener);
	}
	
	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to an array 
	 * of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to an array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return An array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 */
	public static CommonBaseEvent[] eventsFromCanonicalXMLDoc(File file, boolean validate) throws FormattingException {

		try {
			return (eventsFromCanonicalXMLDoc(new InputSource(new FileInputStream(file)), validate));
		} catch (FileNotFoundException f) {
			throw new FormattingException(f);
		}
	}

	/**
	 * De-serializes Common Base Event XML fragments contained in a Common Base Event XML document to 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.  The Common Base Event XML document
	 * is resolved from a {@link java.io.File}.
	 * <p>
	 * The Common Base Event XML document is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param file
	 *            The {@link java.io.File} to resolve the Common Base Event XML fragments contained in a Common Base Event XML document to be de-serialized to {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0.1
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener
	 * @provisional As of TPTP v4.2.0, this method is marked as 'provisional'.
	 */
	public static void eventsFromCanonicalXMLDoc(File file, boolean validate, EventListener eventListener) throws FormattingException {

		try {
			eventsFromCanonicalXMLDoc(new InputSource(new FileInputStream(file)), validate, eventListener);
		} catch (FileNotFoundException f) {
			throw new FormattingException(f);
		}
	}
		
	/**
	 * De-serializes a Common Base Event XML fragment to a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}. 
	 * <p>
	 * The Common Base Event XML fragment is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) 
	 * by default.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlFragmentString
	 *            The string with the Common Base Event XML fragment to be de-serialized to a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}.
	 * @return A {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0
	 */
	public static CommonBaseEvent eventFromCanonicalXML(String xmlFragmentString) throws FormattingException {
		return (eventFromCanonicalXML(xmlFragmentString, false));
	}

	/**
	 * De-serializes a Common Base Event XML fragment to a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}. 
	 * <p>
	 * The Common Base Event XML fragment is <b>not</b> validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param xmlFragmentString
	 *            The string with the Common Base Event XML fragment to be de-serialized to a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}.
	 * @param validate If the Common Base Event XML fragment is validated using the Common Base Event V1.0.1 schema (XSD).         
	 * @return A {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}, otherwise <code>null</code>.
	 * @throws FormattingException A de-serialization error has occurred.
	 * @since 1.0
	 */
	public static CommonBaseEvent eventFromCanonicalXML(String xmlFragmentString, boolean validate) throws FormattingException {

		StringBuffer buffer = new StringBuffer(1024);

		buffer.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
		buffer
				.append("<CommonBaseEvents xmlns=\"http://www.ibm.com/AC/commonbaseevent1_0_1\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://www.ibm.com/AC/commonbaseevent1_0_1 commonbaseevent1_0_1.xsd\">");
		buffer.append(xmlFragmentString);
		buffer.append("</CommonBaseEvents>");

		CommonBaseEvent[] events = eventsFromCanonicalXMLDoc(buffer.toString(), validate);

		if (events != null) {
			return (events[0]);
		}

		return null;
	}

	/**
	 * Serializes an {@link org.eclipse.tptp.logging.events.cbe.ExtendedDataElement} to an Extended Data Element 
	 * XML fragment.
	 * <p>
	 * The name of the root tag or element of the Extended Data Element XML fragment is based on the <code>tagName</code> 
	 * parameter.  If <code>null</code>, "extendedDataElements" is used.
	 * <p>
	 * The resultant Extended Data Element XML fragment is formatted (e.g. platform-dependent new lines and indentation) 
	 * based on the <code>format</code> parameter.  If formatting is enabled, the number of spaces each tag or element 
	 * of the serialized Extended Data Element XML fragment is indented is based on the <code>indent</code> parameter.
	 * <p>
     * Serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * 
	 * @param extendedDataElement The {@link org.eclipse.tptp.logging.events.cbe.ExtendedDataElement} to be serialized to an Extended Data Element XML fragment.
	 * @param tagName The name of the root tag or element of the serialized Extended Data Element XML fragment, by default "extendedDataElements".
	 * @param format If the resultant Extended Data Element XML document is formatted (e.g. platform-dependent new lines and indentation).
	 * @param indent The number of spaces each tag or element of the serialized Extended Data Element XML fragment is indented, used is the <code>format</code> parameter is <code>true</code>. 
	 * @return A serialized Extended Data Element XML fragment.
	 */
	protected static String getExtendedDataElementXML(ExtendedDataElement extendedDataElement, String tagName, boolean format, int indent) {

		if (extendedDataElement != null) {

			StringBuffer buffer = new StringBuffer(512);

			if (format) {

				buffer.append(Constants.LINE_SEPARATOR);

				for (int counter = 0; counter < indent; counter++) {
					buffer.append("\t");
				}
			}

			buffer.append("<");

			if (tagName != null) {
				buffer.append(tagName);
			} else {
				buffer.append("extendedDataElements");
			}

			String attributeValue = extendedDataElement.getName();

			// For performance reasons, do not normalize attribute value since
			// of type xsd:Name which is guaranteed to not contain mark-up
			// characters:
			if (attributeValue != null) {
				buffer.append(" name=\"");
				buffer.append(attributeValue);
				buffer.append("\"");
			}

			attributeValue = extendedDataElement.getType();

			// For performance reasons, do not normalize attribute value since
			// of type xsd:Name which is guaranteed to not contain mark-up
			// characters:
			if (attributeValue != null) {
				buffer.append(" type=\"");
				buffer.append(attributeValue);
				buffer.append("\"");
			}

			buffer.append(">");

			if (extendedDataElement.getTypeAsInt() == ExtendedDataElement.TYPE_HEX_BINARY_VALUE) {

				String hexValues = extendedDataElement.getHexValue();

				if (hexValues != null) {

					if (format) {

						buffer.append(Constants.LINE_SEPARATOR);

						for (int counter = 0; counter < (indent + 1); counter++) {
							buffer.append("\t");
						}
					}

					// For performance reasons, do not normalize element value
					// since of type xsd:hexBinary which is guaranteed to not
					// contain mark-up characters:
					buffer.append("<hexValue>");
					buffer.append(hexValues);
					buffer.append("</hexValue>");
				}
			} else {

				List values = extendedDataElement.getValues();

				String valuesString = null;

				for (int counter = 0; counter < values.size(); counter++) {

					valuesString = ((String) (values.get(counter)));

					if (valuesString != null) {

						if (format) {

							buffer.append(Constants.LINE_SEPARATOR);

							for (int index = 0; index < (indent + 1); index++) {
								buffer.append("\t");
							}
						}

						buffer.append("<values>");
						buffer.append(XmlUtility.normalize(valuesString));
						buffer.append("</values>");
					}
				}
			}

			List childDataElements = extendedDataElement.getChildren();

			for (int counter = 0; counter < childDataElements.size(); counter++) {
				buffer.append(getExtendedDataElementXML(((ExtendedDataElement) childDataElements.get(counter)), "children", format, (format ? (indent + 1) : 0)));
			}

			if (format) {

				buffer.append(Constants.LINE_SEPARATOR);

				for (int counter = 0; counter < indent; counter++) {
					buffer.append("\t");
				}
			}

			buffer.append("</");

			if (tagName != null) {
				buffer.append(tagName);
			} else {
				buffer.append("extendedDataElements");
			}

			buffer.append(">");

			return (buffer.toString());
		} else {
			return "";
		}
	}

	/**
	 * Resolves the location (URL) of the Common Base Event V1.0.1 schema (XSD) used for validation.
	 * <p>
	 * 
	 * @return External form of the URL representing the location of the Common Base Event V1.0.1 schema (XSD).
	 */
	protected static String resolveCommonBaseEventSchemaLocation() {

		URL url = null;

		// Attempt to find the actual location of the
		// schema file on the classpath:
		try {

			// Consult the current class' class loader to find the
			// actual local location of the schema file
			// using privileged security:
			url = ((URL) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

				public Object run() throws Exception {
					return (org.eclipse.hyades.logging.events.cbe.util.EventFormatter.class.getClassLoader().getResource(COMMON_BASE_EVENT_SCHEMA_NAME));
				}
			})));
		} catch (Throwable t) {
			// Ignore exception since the current class' class loader is
			// null (e.g. current class was loaded by the bootstrap
			// class loader) or insufficient security privileges for
			// accessing the current class' class loader.
		}

		if (url == null) {

			try {

				// If the current class' class loader cannot find the actual
				// local location of the schema file,
				// consult the system's class loader to find the actual
				// local location of the schema file
				// using privileged security:
				url = ((URL) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

					public Object run() throws Exception {
						return (ClassLoader.getSystemClassLoader().getResource(COMMON_BASE_EVENT_SCHEMA_NAME));
					}
				})));
			} catch (Throwable t) {
				// Ignore exception since insufficient security privileges
				// for
				// accessing the system's class loader.
			}

			if (url == null) {

				try {

					// If the system's class loader cannot find the actual
					// local location of the schema file,
					// consult the current thread's class loader to find the
					// actual local location of the schema
					// file using privileged security:
					url = ((URL) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

						public Object run() throws Exception {
							return (Thread.currentThread().getContextClassLoader().getResource(COMMON_BASE_EVENT_SCHEMA_NAME));
						}
					})));
				} catch (Throwable t) {
					// Ignore exception since insufficient security
					// privileges for
					// accessing the current thread's class loader.
				}
			}
		}

		if (url != null) {
			return (url.toExternalForm());
		}

		return null;
	}

	/**
	 * De-serializes or parses Common Base Event and Association Engine XML fragments contained in a 
	 * Common Base Event XML document.  The Common Base Event XML document is resolved from an 
	 * {@link org.xml.sax.InputSource}.
	 * <p>
	 * The Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD) based
	 * on the <code>validate</code> parameter.
	 * <p>
     * De-serialization is based on
     * the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/CommonBaseEvent_SituationData_V1.0.1.pdf">Common Base Event Situation Data Specification v1.0.1</a>
     * and the <a href="http://www.eclipse.org/tptp/platform/documents/resources/cbe101spec/commonbaseevent1_0_1.xsd">Common Base Event XML schema</a>.
     * <p>
	 * Callers receive notification to process the de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * through the parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener}.  As such, callers can process 
	 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s on a separate thread before this method returns, 
	 * resulting in a high level of parallelism and improved efficiency.  Noteworthy, callers are responsible for listener instantiation, 
	 * registration and clean-up.
	 * <p>
	 * 
	 * @param inputSource
	 *            The {@link org.xml.sax.InputSource} to resolve the Common Base Event and Association Engine XML fragments contained in a Common Base Event XML document to be de-serialized or parsed.
	 * @param validate If the Common Base Event XML document is validated using the Common Base Event V1.0.1 schema (XSD).         
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be receive notification to process de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s.
	 * @throws FormattingException A de-serialization error has occurred.
	 */
	protected static void parse(InputSource inputSource, boolean validate, EventListener eventListener) throws FormattingException {

		synchronized (LOCK) {

			try {
	
				// Initialize the XML parser:
				if (xmlParser == null) {
	
					// Attempt to resolve the XML parser:
					try {
						xmlParser = XMLReaderFactory.createXMLReader();
					} 
					catch (SAXException s) {
	
						// NOTE: Sun's Java V1.4.x includes the Crimson XML parser implementation
						// which does not have a default XML parser implementation.
	
						// Attempt to resolve the XML parser based on the fully qualified class name of Crimson's
						// XML parser implementation as shipped in Sun's Java V1.4.x:
						xmlParser = XMLReaderFactory.createXMLReader("org.apache.crimson.parser.XMLReaderImpl");
					}
	
					// Instantiate the event handler:
					eventHandler = new EventHandler();
	
					// Set the event handler on the XML parser for both content and error handling:
					xmlParser.setContentHandler(eventHandler);
					xmlParser.setErrorHandler(eventHandler);
	
					// Attempt to configure the XML parser to validate the XML document based on the Common Base Event
					// schema:
					try {
						xmlParser.setProperty("http://apache.org/xml/properties/schema/external-schemaLocation", "http://www.ibm.com/AC/commonbaseevent1_0_1 "
								.concat(COMMON_BASE_EVENT_SCHEMA_LOCATION_URL));
					} catch (SAXNotRecognizedException s) {
	
						// NOTE: Sun's Java V1.4.x includes the Crimson XML parser implementation which does not support
						// schema validation.
	
						// If schema validation is supported, configure a non-validating XML parser:
						xmlParser.setFeature("http://xml.org/sax/features/validation", false);
						xmlParser.setFeature("http://xml.org/sax/features/namespaces", false);
	
						isValidationSupported = false;
					}
				}
	
				// Initialize the event handler:
				else {
					eventHandler.init();
				}
	
				//Add the event listener to the event handler:
				eventHandler.addEventListener(eventListener);
				
				// Configure a non-/validating XML parser only if schema validation is supported:
				if (isValidationSupported) {
	
					// Configure a non-/validating XML parser:
					xmlParser.setFeature("http://xml.org/sax/features/validation", validate);
					xmlParser.setFeature("http://xml.org/sax/features/namespaces", validate);
	
					try {
						xmlParser.setFeature("http://apache.org/xml/features/validation/schema", validate);
					} catch (SAXNotRecognizedException s) {
	
						// NOTE: Sun's Java V1.4.x includes the Crimson XML parser
						// implementation which does not support schema validation.
	
						// If schema validation is supported, configure a non-validating XML parser:
						xmlParser.setFeature("http://xml.org/sax/features/validation", false);
						xmlParser.setFeature("http://xml.org/sax/features/namespaces", false);
	
						isValidationSupported = false;
					}
				}
	
				//Parse the XML document:
				xmlParser.parse(inputSource);			
			} 
			catch (Exception e) {
				throw new FormattingException(e);
			}
		}
	}
}