package org.eclipse.hyades.logging.events.cbe.util;

import java.awt.EventQueue;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.internal.logging.core.Constants;
import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;
import org.eclipse.hyades.logging.core.LoggingCoreResourceBundle;
import org.eclipse.hyades.logging.core.LoggingCoreUtilities;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.CompletionException;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.ContentHandler;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.EventPackage;
import org.eclipse.hyades.logging.events.cbe.ExtendedDataElement;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.TemplateContentHandler;
import org.eclipse.hyades.logging.events.cbe.impl.TemplateContentHandlerImpl;

import com.ibm.icu.text.SimpleDateFormat;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventHelpers.java,v 1.7 2008/12/15 15:34:44 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Miscellaneous routines to support functions such as validation, mapping
 * and comparison.
 * 
 * @author CindyJ
 * @author Denilson Nastacio
 * @author Jason Cornpropst
 * @author Scott Brown
 * @author Paul E. Slauenwhite
 * @author Adriana Viman
 * @version March 30, 2007
 * @since 1.0.1
 */
public class EventHelpers {

    /**
     * Static flag for quickly determining if a JRE 1.4.x and above
     * run-time environment.
     * <p>
     * By default, a JRE 1.4.x and above run-time environment is assumed.
     */
    protected static boolean isJava14xRunTime = true;

    /** 
     * CBE list elements where order is important during comparison. 
     */
    protected static final List CBE_ORDERED_LISTS = Arrays.asList(new Object[] { EventPackage.eINSTANCE.getExtendedDataElement_Values().getName(), EventPackage.eINSTANCE.getMsgDataElement_MsgCatalogTokens().getName()});

    /**
     * CBE element (e.g. ExtendedDataElement#hexValue) where the text case is not important during comparison.
     */
    protected static final String CBE_CASE_INSENSITIVE_PROPERTY = EventPackage.eINSTANCE.getExtendedDataElement_HexValue().getName();

    /**
	 * Current thread lock for synchronization.
	 * <p>
	 */
	protected static final Object LOCK = new Object();

    /**
     * Generic serialization routine for EMF object.
     * 
     * @param o
     *            EMF object to be serialized
     * @param in
     *            Java IO input stream contained the serialized strem
     * 
     * @throws ClassNotFoundException
     *             if the class of an object inside the stream cannot be found
     *             in the classpath
     * @throws IOException
     *             if the input stream cannot be read for whatever reason
     * @deprecated As of TPTP 4.0.0, for performance reasons, object de-serialization is specific and localized to the serialized class.            
     */
    public static void serializableRead(EObject o, java.io.ObjectInputStream in) throws ClassNotFoundException, IOException {

        List features = null;

        //NOTE:  Synchronization required for backwards compatibility with EMF v1.1 due to defect #49868:
        synchronized (LOCK) {
            features = o.eClass().getEAllStructuralFeatures();;
        }
        
        for (int counter = 0; counter < features.size(); counter++) {
            o.eSet(((EStructuralFeature) (features.get(counter))), in.readObject());
        } 
    }

    /**
     * Generic serialization routine for EMF object.
     * 
     * @param o
     *            EMF object to be serialized
     * @param out
     *            Java IO stream to where the serialized object will be written.
     * 
     * @throws IOException
     *             if the object cannot be written to the output stream for
     *             whatever reason.
     * @deprecated As of TPTP 4.0.0, for performance reasons, object serialization is specific and localized to the serialized class.     
     */
    public static void serializableWrite(EObject o, java.io.ObjectOutputStream out) throws IOException {

        List features = null;
        
        //NOTE:  Synchronization required for backwards compatibility with EMF v1.1 due to defect #49868:
        synchronized (LOCK) {
            features = o.eClass().getEAllStructuralFeatures();
        }
        
        for (int counter = 0; counter < features.size(); counter++) {

            Object sourceValue = o.eGet(((EStructuralFeature) (features.get(counter))));
            
            //NOTE: IIOP does not process ELists so convert to an ArrayList:
            if (sourceValue instanceof EList) {
                out.writeObject(new ArrayList(((List) (sourceValue))));
            } 
            else {
                out.writeObject(sourceValue);
            }
        } 
    }

    /**
     * Deep-comparison for two EMF objects.
     * <P>
     * If the the two EMF objects are of the same type, it traverses their
     * features recursively until a difference is found.
     * </P>
     * <P>
     * When comparing lists, this method ignores order.
     * </P>
     * <P>
     * This is a helper for implementing the <code>equals</code> method on EMF
     * objects, where <code>obj1</code> is the object where
     * <code>equals</code> was called and <code>obj2</code> is the parameter
     * for that call.
     * <P>
     * Therefore <code>obj1</code> can never be <code>null</code> and when
     * <code>obj2</code> is <code>null</code>, the result is
     * <code>false</code>.
     * </P>
     * 
     * @param obj1
     *            first element in the comparison
     * @param obj2
     *            second element in the comparison
     * 
     * @return <code>true</code> if the two objects are identical instancesor
     *         <code>false</code> otherwise.
     */
    public static boolean compareEObject(Object obj1, Object obj2) {

        // Not throwing IllegalArgumentException when obj1 is null because the
        // precondition is that this object should not be null/
        if (obj2 == null) { return false; }

        // At this point we can ensure that both pointers are not null.
        // Are they both EObjects?
        if (((obj1 instanceof EObject) == false) || ((obj2 instanceof EObject) == false)) { return false; }
        if (obj1 == obj2) { return true; }

        if (obj1.getClass().equals(obj2.getClass()) == false) { 
            return false; 
        }
        
        EObject source = (EObject) obj1;
        EObject target = (EObject) obj2;
        boolean same = true;
        
        List features = null;
        
        //NOTE:  Synchronization required for backwards compatibility with EMF v1.1 due to defect #49868:
        synchronized (LOCK) {
            features = source.eClass().getEAllStructuralFeatures();
        }
        
        for (int i = 0; i < features.size(); i++) {
        
            EStructuralFeature feature = (EStructuralFeature) features.get(i);

            Object sourceValue = source.eGet(feature);
            Object targetValue = target.eGet(feature);
            boolean sourceIsSet = source.eIsSet(feature);
            boolean targetIsSet = target.eIsSet(feature);

            // Are the source and target value both null or both not null
            if (((sourceValue != null) && (targetValue == null)) || ((sourceValue == null) && (targetValue != null)) || (sourceIsSet != targetIsSet)) {
                same = false;
                break;
            }

            // At this point, either target and source are null (equal)
            // or not null, in which case equality must be checked.
            if (sourceValue != null) {
                if (sourceValue instanceof List) {
                    if (CBE_ORDERED_LISTS.contains(feature.getName())) {
                        same = EventHelpers.compareLists((List) sourceValue, (List) targetValue, true);
                        // consider order
                    } else {
                        same = EventHelpers.compareLists((List) sourceValue, (List) targetValue, false);
                        // consider order
                    }
                } else {
                    if (feature.getName().equals(CBE_CASE_INSENSITIVE_PROPERTY)) {
                        same = ((String) sourceValue).equalsIgnoreCase((String) targetValue);
                    } else {
                        same = sourceValue.equals(targetValue);
                    }
                }
                if (same == false) {
                    break;
                }
            } // if source value not null
        } // for features

        return same;
    }
  
    /**
     * Converts an XML Schema (XSD) 
     * <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a> time stamp 
     * to a long time stamp.
     * <p>
     * The XSD <code>dateTime</code> time stamp represents a finite-length sequence
     * of characters following the following format pattern:
     * <p>
     * <code>
     * yyyy-MM-dd'T'HH:mm:ss[.SSSS]['Z'|Z]
     * </code>
     * <p>
     * For more information on the meaning of the individual symbols in the XSD 
     * <code>dateTime</code> time stamp format pattern, see the class comment header 
     * for {@link SimpleDateFormat}.
     * <p>
     * For example, March 14, 2006 11:32:01.001 AM UTC/GMT would be represented as:
     * <p>
     * <code>
     * 2006-03-14T11:32:01.001Z
     * </code>
     * <p>
     * See the XSD <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a>
     * primitive datatype documentation for more information.
     * <p>
     * The long time stamp represents the difference, measured in milliseconds, between 
     * the current time (Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT)) 
     * and the standard base time.  The standard base time is known as "the epoch", namely 
     * January 1, 1970 12:00:00 AM Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT).  
     * <p>
     * <b>Note:</b> Due to inconsistencies between XSD V1.0 and later versions of the XSD 
     * specification on representing zero or negative XSD <code>dateTime</code> time stamps, 
     * such as year 0 and year 1 Before Common Era (BCE or BC), this  method will throw an 
     * {@link IllegalArgumentException} for parameter XSD <code>dateTime</code> time stamps 
     * before <code>0001-01-01T00:00:00.000Z</code> or <code>-62135769600000</code>.  See the 
     * XSD dateTime <a href="http://www.w3.org/TR/xmlschema-2/#year-zero">note</a> for more 
     * information.
     * <p>
     * 
     * @param xsdDateTime The XSD <code>dateTime</code> time stamp to be converted to a long (milliseconds) time stamp.
     * @return The long (milliseconds) time stamp representation of the parameter XSD <code>dateTime</code> time stamp.
     * @throws IllegalArgumentException If the parameter XSD <code>dateTime</code> time stamp is not a valid XSD <code>dateTime</code> time stamp.
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreUtilities#convertXsdDateTimeToMilliseconds(String)}.
     */    
    public static long dateToLong(String xsdDateTime) {
    	return (LoggingCoreUtilities.convertXsdDateTimeToMilliseconds(xsdDateTime));
    }
 
    /**
     * Converts along time stamp to an XML Schema (XSD) 
     * <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a> time stamp.
     * <p>
     * The long time stamp represents the difference, measured in milliseconds, between 
     * the current time (Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT)) 
     * and the standard base time.  The standard base time is known as "the epoch", namely 
     * January 1, 1970 12:00:00 AM Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT).  
     * <p>
     * The XSD <code>dateTime</code> time stamp represents a finite-length sequence
     * of characters following the following format pattern:
     * <p>
     * <code>
     * yyyy-MM-dd'T'HH:mm:ss[.SSSS]['Z'|Z]
     * </code>
     * <p>
     * For more information on the meaning of the individual symbols in the XSD 
     * <code>dateTime</code> time stamp format pattern, see the class comment header 
     * for {@link SimpleDateFormat}.
     * <p>
     * For example, March 14, 2006 11:32:01.001 AM UTC/GMT would be represented as:
     * <p>
     * <code>
     * 2006-03-14T11:32:01.001Z
     * </code>
     * <p>
     * See the XSD <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a>
     * primitive datatype documentation for more information.
     * <p>
     * <b>Note:</b> Due to inconsistencies between XSD V1.0 and later versions of the XSD 
     * specification on representing zero or negative XSD <code>dateTime</code> time stamps, 
     * such as year 0 and year 1 Before Common Era (BCE or BC), this method will return a 
     * positive XSD <code>dateTime</code> time stamp for parameter long (milliseconds) time 
     * stamps before <code>-62135769600000</code> or <code>0001-01-01T00:00:00.000Z</code>.  
     * See the XSD dateTime <a href="http://www.w3.org/TR/xmlschema-2/#year-zero">note</a> for 
     * more information.
     * <p>
     * 
     * @param milliseconds The long (milliseconds) time stamp to be converted to an XSD <code>dateTime</code> time stamp.
     * @return The XSD <code>dateTime</code> time stamp representation of the parameter long (milliseconds) time stamp .
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreUtilities#convertMillisecondsToXsdDateTime(long)}.
     */
    public static String longToDate(long milliseconds) {
    	return (LoggingCoreUtilities.convertMillisecondsToXsdDateTime(milliseconds));
    }
    
    /**
     * Converts a long representing a Coordinated Universal Time (UTC) date in 
     * milliseconds to a formatted string using the default date format pattern.  
     * <p>
     * The default date format is:
     * <p>
     * MMMM d, yyyy h:mm:ss.SSS a z
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleDateFormat</code>.
     * <p>
     * The time zone of the returned string is Coordinated Universal Time (UTC), represented
     * as Greenwich Mean Time (GMT).
     * <p>
     * 
     * @param milliseconds
     *            A long representing a Coordinated Universal Time (UTC) time stamp in milliseconds.
     * @return The date as a formatted string using the default date format pattern (e.g. MMMM d, yyyy h:mm:ss.SSS a z).
     * @throws IllegalArgumentException
     *             If the long representing a Coordinated Universal Time (UTC) time stamp in milliseconds is negative.
     * @see java.util.SimpleDateFormat
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreUtilities#getFormattedDateString(long)}.
     */
    public static String getFormattedDateString(long milliseconds) throws IllegalArgumentException{
    	return (LoggingCoreUtilities.getFormattedDateString(milliseconds));
    }
    
    /**
     * Converts a long representing a Coordinated Universal Time (UTC) date in 
     * milliseconds to a formatted string using the parameter date format pattern.  
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, date formatting is done by the 
	 * {@link com.ibm.icu.text.SimpleDateFormat} class using the {@link com.ibm.icu.util.ULocale} (parameter {@link Locale}
	 * converted to a {@link com.ibm.icu.util.ULocale} - see {@link com.ibm.icu.util.ULocale#forLocale(Locale)}), 
	 * {@link com.ibm.icu.util.GregorianCalendar} and {@link com.ibm.icu.util.TimeZone} (parameter time zone ID
	 * converted to a {@link com.ibm.icu.util.TimeZone} - see {@link com.ibm.icu.util.TimeZone#getTimeZone(String)}) classes, 
	 * otherwise date formatting is done by the {@link java.text.SimpleDateFormat} class using the {@link java.util.Locale}, 
	 * {@link java.util.GregorianCalendar} and {@link java.util.TimeZone} classes.  That is, if the 
	 * <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, the parameter date format pattern is based on the syntax and symbols 
	 * as specified in the class comment header of the {@link com.ibm.icu.text.SimpleDateFormat} class, otherwise the 
	 * parameter date format pattern is based on the syntax and symbols as specified in the class comment header of the 
	 * {@link java.util.SimpleDateFormat} class.
     * <p>
     * The time zone of the returned string is Coordinated Universal Time (UTC), represented
     * as Greenwich Mean Time (GMT).
     * <p>
     * 
     * @param milliseconds
     *            A long representing a Coordinated Universal Time (UTC) time stamp in milliseconds.
     * @param dateFormatPattern
     * 			  The date format pattern.
     * @return The date as a formatted string using the parameter date format pattern.
     * @throws IllegalArgumentException If the date format pattern invalid.
	 * @see java.text.SimpleDateFormat
	 * @see java.util.Locale
	 * @see java.util.GregorianCalendar
	 * @see java.util.TimeZone
	 * @see com.ibm.icu.text.SimpleDateFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see com.ibm.icu.util.GregorianCalendar
	 * @see com.ibm.icu.util.TimeZone
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreUtilities#getFormattedDateString(long, String))}.
     */
    public static String getFormattedDateString(long milliseconds, String dateFormatPattern) throws IllegalArgumentException{
    	return (LoggingCoreUtilities.getFormattedDateString(milliseconds, dateFormatPattern));
    }
    
    /**
     * Converts a string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date 
     * to a formatted string using the default date format pattern.  
     * <p>
     * For more information on the XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date, 
     * see <a href="http://www.w3.org/TR/NOTE-datetime">http://www.w3.org/TR/NOTE-datetime</a>.
     * <p>
     * The default date format is:
     * <p>
     * MMMM d, yyyy h:mm:ss.SSS a z
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleDateFormat</code>.
     * <p>
     * If the parameter string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date does not
     * contain any time zone information, the time zone of the returned string is Coordinated Universal Time (UTC), 
     * represented as Greenwich Mean Time (GMT).  Otherwise, the time zone of the returned string is represented as a
     * signed offset from Greenwich Mean Time (GMT).  For example, 'GMT-05:00' for Eastern Standard Time.
     * <p>
     * 
     * @param xsdDateTime
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @return The date as a formatted string using the default date format pattern (e.g. MMMM d, yyyy h:mm:ss.SSS a z).
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid.
     * @see java.util.SimpleDateFormat
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreUtilities#getFormattedDateString(String))}.
     */
    public static String getFormattedDateString(String xsdDateTime) throws IllegalArgumentException{
    	return (LoggingCoreUtilities.getFormattedDateString(xsdDateTime));
    }
        
    /**
     * Converts a string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date 
     * to a formatted string using the parameter date format pattern.  
     * <p>
     * For more information on the XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date, 
     * see <a href="http://www.w3.org/TR/NOTE-datetime">http://www.w3.org/TR/NOTE-datetime</a>.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, date formatting is done by the 
	 * {@link com.ibm.icu.text.SimpleDateFormat} class using the {@link com.ibm.icu.util.ULocale} (parameter {@link Locale}
	 * converted to a {@link com.ibm.icu.util.ULocale} - see {@link com.ibm.icu.util.ULocale#forLocale(Locale)}), 
	 * {@link com.ibm.icu.util.GregorianCalendar} and {@link com.ibm.icu.util.TimeZone} (parameter time zone ID
	 * converted to a {@link com.ibm.icu.util.TimeZone} - see {@link com.ibm.icu.util.TimeZone#getTimeZone(String)}) classes, 
	 * otherwise date formatting is done by the {@link java.text.SimpleDateFormat} class using the {@link java.util.Locale}, 
	 * {@link java.util.GregorianCalendar} and {@link java.util.TimeZone} classes.  That is, if the 
	 * <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, the parameter date format pattern is based on the syntax and symbols 
	 * as specified in the class comment header of the {@link com.ibm.icu.text.SimpleDateFormat} class, otherwise the 
	 * parameter date format pattern is based on the syntax and symbols as specified in the class comment header of the 
	 * {@link java.util.SimpleDateFormat} class.
     * <p>
     * If the parameter string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date does not
     * contain any time zone information, the time zone of the returned string is Coordinated Universal Time (UTC), 
     * represented as Greenwich Mean Time (GMT).  Otherwise, the time zone of the returned string is represented as a
     * signed offset from Greenwich Mean Time (GMT).  For example, 'GMT-05:00' for Eastern Standard Time.
     * <p>
     * 
     * @param xsdDateTime
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @param dateFormatPattern
     * 			  The date format pattern.
     * @return The date as a formatted string using the parameter date format pattern.
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid or 
     * 			   the date format pattern is invalid.
	 * @see java.text.SimpleDateFormat
	 * @see java.util.Locale
	 * @see java.util.GregorianCalendar
	 * @see java.util.TimeZone
	 * @see com.ibm.icu.text.SimpleDateFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see com.ibm.icu.util.GregorianCalendar
	 * @see com.ibm.icu.util.TimeZone
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreUtilities#getFormattedDateString(String, String)))}.
     */
    public static String getFormattedDateString(String xsdDateTime, String dateFormatPattern) throws IllegalArgumentException{
    	return (LoggingCoreUtilities.getFormattedDateString(xsdDateTime, dateFormatPattern));
    }
    
    /*
     * Private methods
     */

    /**
     * Compare two <code>List</code> objects, potentially ignoring order.
     * <p>
     * 
     * @param listOne The first list in the comparison.
     * @param listTwo The second list in the comparison.
     * @param considerOrder If order should be considered.
     * @return True if both parameter lists are equal, otherwise false.
     */
    protected static boolean compareLists(List listOne, List listTwo, boolean considerOrder) {

        int listTwoSize = listTwo.size();

        if (listOne.size() != listTwoSize) { 
            return false; 
        }

        Iterator listOneIterator = listOne.iterator();
        Object listOneElement = null;

        if(considerOrder){

            Iterator listTwoIterator = listTwo.iterator();
            
            while(listOneIterator.hasNext()) {
                
                listOneElement = listOneIterator.next();
                
                if(listOneElement == null ? listTwoIterator.next() != null : !listOneElement.equals(listTwoIterator.next())){
                    return false;
                }
            }
        }
        else{
            
            boolean[] listTwoMatches = new boolean[listTwoSize];
    
            while(listOneIterator.hasNext()) {
     
                listOneElement = listOneIterator.next();
                int counter = 0;
    
                while(counter < listTwoSize) {
                
                    if ((!listTwoMatches[counter]) && (listOneElement == null ? listTwo.get(counter) == null : listOneElement.equals(listTwo.get(counter)))) {
    
                        listTwoMatches[counter] = true;
                        
                        break;
                    }
                    
                    counter++;
                } 
    
                if (counter == listTwoSize) {
                    return false;
                }
            } 
        }
        
        return true;
    }

    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * from the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreResourceBundle#getString(String)}.
     */
    public static String getString(String key) {
    	return (LoggingCoreResourceBundle.getString(key));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameter from the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argument The first message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreResourceBundle#getString(String, Object))}.
     */
    public static String getString(String key, Object argument) {
    	return (LoggingCoreResourceBundle.getString(key,argument));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters from the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreResourceBundle#getString(String, Object, Object))}.
     */
    public static String getString(String key, Object argumentA, Object argumentB) {
    	return (LoggingCoreResourceBundle.getString(key,argumentA,argumentB));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters from the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @param argumentC The third message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreResourceBundle#getString(String, Object, Object, Object))}.
     */
    public static String getString(String key, Object argumentA, Object argumentB, Object argumentC) {
    	return (LoggingCoreResourceBundle.getString(key,argumentA,argumentB,argumentC));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters from the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @param argumentC The third message parameter for formatting in the localized and formatted message.
     * @param argumentD The fourth message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreResourceBundle#getString(String, Object, Object, Object, Object))}.
     */
    public static String getString(String key, Object argumentA, Object argumentB, Object argumentC, Object argumentD) {
    	return (LoggingCoreResourceBundle.getString(key,argumentA,argumentB,argumentC,argumentD));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters from the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
     * @deprecated As of TPTP V4.2.0, use {@link LoggingCoreResourceBundle#getString(String, Object[]))}.
     */
    public static String getString(String key, Object[] arguments) {
    	return (LoggingCoreResourceBundle.getString(key, arguments));
    }

    /**
     * Maps data of the parameter <code>java.lang.Throwable</code> object to an
     * Extended Data Element.
     * <p>
     * This API provides a structured representation of the parameter <code>java.lang.Throwable</code>
     * object as an Extended Data Element with the following advantages:
     * <p>
     * <ul>
     * <li>Efficient specialized parsibility and de-/serialization.</li>
     * <li>Identifiable Extended Data Element <code>name</code>s (e.g. "Throwable" and "Cause")</li>
     * <li>Captures all nested causal <code>java.lang.Throwable</code>s.</li>
     * <li>Adheres to the 1024-character restriction imposed on the Extended Data Element <code>values</code>.</li>
     * <li></li>
     * </ul>
     * <p>
     * NOTE: This API invoked in JRE 1.3.x and below run-time environments parses the 
     * stack trace produced by the <code>java.lang.Throwable</code>'s <code>printStackTrace()</code>
     * API (e.g. no causal <code>java.lang.Throwable</code>(s)).  Alternatively, this API invoked in JRE 1.4.x 
     * and above run-time environments utilizes the <code>java.lang.Throwable</code>'s causal 
     * <code>java.lang.Throwable</code> and <code>StackTraceElement</code>(s) properties.  
	 * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element:
     * <p>
     * <ul>
     * <li>name = "Throwable"</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
     * <code>java.lang.Throwable</code>'s localized message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt; <code>java.lang.Throwable</code>'s
     * stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>children[0] = &lt; <code>java.lang.Throwable</code>'s cause&gt;</li>
     * </ul>
     * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element if the 
     * parameter <code>java.lang.Throwable</code> object is <code>null</code>:
     * <p>
     * <ul>
     * <li>name = &lt;name&gt;</li>
     * <li>type = "stringArray"</li>
     * <li>values[0] = <code>null</code></li>
     * </ul>
     * <p>
     * Causal <code>java.lang.Throwable</code>s are recursively converted to Extended
     * Data Elements using the following mapping:
     * <p>
     * <ul>
     * <li>name = "Cause"</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt;cause's class name&gt;[: &lt;cause's localized
     * message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>[children[0] = &lt;cause's cause&gt;]</li>
     * </ul>
     * <p>
     * ...
     * <p>
     * NOTE: When the <code>java.lang.Throwable</code>'s class name and localized message
     * are greater than 1024 characters, the resultant class name and localized message string 
     * is segmented into a the first 1024-character elements of the <code>values</code> property.
     * As such, the <code>java.lang.Throwable</code>'s stack trace elements are transposed by the
     * number of 1024-character elements in the <code>values</code> property.
     * <p>
     * 
     * @param throwable
     *            The <code>java.lang.Throwable</code> object to be converted to an
     *            Extended Data Element.
     * @return The Extended Data Element representation of the parameter
     *         <code>java.lang.Throwable</code> object.
     */
    public static ExtendedDataElement convertToExtendedDataElement(Throwable throwable) {
        return (convertToExtendedDataElement(throwable,"Throwable"));
    }
    
    /**
     * Maps data of the parameter <code>java.lang.Throwable</code> object to an
     * Extended Data Element with the parameter <code>name</code>.
     * <p>
     * This API provides a structured representation of the parameter <code>java.lang.Throwable</code>
     * object as an Extended Data Element with the following advantages:
     * <p>
     * <ul>
     * <li>Efficient specialized parsibility and de-/serialization.</li>
     * <li>Identifiable Extended Data Element <code>name</code>s (e.g. &lt;name&gt; and "Cause")</li>
     * <li>Captures all nested causal <code>java.lang.Throwable</code>s.</li>
     * <li>Adheres to the 1024-character restriction imposed on the Extended Data Element <code>values</code>.</li>
     * <li></li>
     * </ul>
     * <p>
     * NOTE: This API invoked in JRE 1.3.x and below run-time environments parses the 
     * stack trace produced by the <code>java.lang.Throwable</code>'s <code>printStackTrace()</code>
     * API (e.g. no causal <code>java.lang.Throwable</code>(s)).  Alternatively, this API invoked in 
     * JRE 1.4.x and above run-time environments utilizes the <code>java.lang.Throwable</code>'s causal 
     * <code>java.lang.Throwable</code> and <code>StackTraceElement</code>(s) properties.  
	 * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element:
     * <p>
     * <ul>
     * <li>name = &lt;name&gt;</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt; <code>java.lang.Throwable</code>'s class name&gt;[: &lt;
     * <code>java.lang.Throwable</code>'s localized message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt; <code>java.lang.Throwable</code>'s
     * stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>children[0] = &lt; <code>java.lang.Throwable</code>'s cause&gt;</li>
     * </ul>
     * <p>
     * This API uses the following mapping to convert the parameter
     * <code>java.lang.Throwable</code> object to an Extended Data Element if the 
     * parameter <code>java.lang.Throwable</code> object is <code>null</code>:
     * <p>
     * <ul>
     * <li>name = &lt;name&gt;</li>
     * <li>type = "stringArray"</li>
     * <li>values[0] = <code>null</code></li>
     * </ul>
     * <p>
     * Causal <code>java.lang.Throwable</code>s are recursively converted to Extended
     * Data Elements using the following mapping:
     * <p>
     * <ul>
     * <li>name = "Cause"</li>
     * <li>type = "stringArray"</li>
     * <li>values[0...n] = &lt;cause's class name&gt;[: &lt;cause's localized
     * message&gt;]</li>
     * <li>values[(n + 1)...m] = &lt;cause's stackTraceElement[0...(m - (n + 1))]&gt;</li>
     * <li>[children[0] = &lt;cause's cause&gt;]</li>
     * </ul>
     * <p>
     * ...
     * <p>
     * NOTE: When the <code>java.lang.Throwable</code>'s class name and localized message
     * are greater than 1024 characters, the resultant class name and localized message string 
     * is segmented into a the first 1024-character elements of the <code>values</code> property.
     * As such, the <code>java.lang.Throwable</code>'s stack trace elements are transposed by the
     * number of 1024-character elements in the <code>values</code> property.
     * <p>
     * 
     * @param throwable
     *            The <code>java.lang.Throwable</code> object to be converted to an
     *            Extended Data Element.
     * @param name
     *            The <code>name</code> property of the Extended Data Element.
     * @return The Extended Data Element representation of the parameter
     *         <code>java.lang.Throwable</code> object.
     */
    public static ExtendedDataElement convertToExtendedDataElement(Throwable throwable, String name) {

        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
        extendedDataElement.setName(name);
        extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_STRING_ARRAY_VALUE);

        if (throwable == null) {
            extendedDataElement.setValues(new String[] { null});
        }
        else{

            //Attempt to use the <code>java.lang.Throwable</code> APIs available in a JRE 1.4.x and above run-time environment:
            if (isJava14xRunTime) {

                try {

                    Object[] stackTraceElements = ((Object[])(throwable.getClass().getMethod("getStackTrace", null).invoke(throwable, null)));
                    
                    if (stackTraceElements.length > 0) {

                        //Ensure that the <code>java.lang.Throwable</code>'s class name and localized message
                        //are segmented if greater than 1024 characters:
                        String[] throwableStringArray = getExtendedDataElementValuesArray(throwable.toString());
                        
                        String[] values = new String[throwableStringArray.length + stackTraceElements.length];

                        System.arraycopy(throwableStringArray,0,values,0,throwableStringArray.length);
                        
                        //ASSUMPTION: Stack trace elements are typically less than 1024 characters.
                        //For performance reasons, do not segment stack trace elements.
                        for (int counter = 0; counter < stackTraceElements.length; counter++) {
                            values[counter + throwableStringArray.length] = stackTraceElements[counter].toString();
                        }

                        extendedDataElement.setValues(values);
                    } 
                    else {
                        extendedDataElement.setValues(getExtendedDataElementValuesArray(throwable.toString()));
                    }

                    Throwable cause  = ((Throwable)(throwable.getClass().getMethod("getCause", null).invoke(throwable, null)));

                    if (cause != null) {
                        extendedDataElement.addChild(convertToExtendedDataElement(cause, "Cause"));
                    }
                    
                    return extendedDataElement;
                } 
                catch (Throwable t) {
                    isJava14xRunTime = false;
                }
            }
            
            //Default to standard <code>java.lang.Throwable</code> APIs available in a JRE 1.3.x and below run-time environment:
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();                
            PrintWriter writer = new PrintWriter(outputStream, true);

            throwable.printStackTrace(writer);
                        
            if(!writer.checkError()){
                
                //NOTE:  The <code>java.lang.Throwable</code>'s stack trace is <i>guaranteed</i> to not be <code>null</code> or empty (e.g. class name) if successfully serialized.
                String[] stackTraceElements = InternationalizationUtilities.tokenize(outputStream.toString().trim(),Constants.LINE_SEPARATOR);
                
                if(stackTraceElements.length > 1){

                    //Ensure that the <code>java.lang.Throwable</code>'s class name and localized message
                    //are segmented if greater than 1024 characters:
                    String[] throwableStringArray = getExtendedDataElementValuesArray(stackTraceElements[0].trim());
                    
                    String[] values = new String[throwableStringArray.length + stackTraceElements.length - 1];

                    System.arraycopy(throwableStringArray,0,values,0,throwableStringArray.length);

                    //ASSUMPTION: Stack trace elements are typically less than 1024 characters.
                    //For performance reasons, do not segment stack trace elements.
                    for (int counter = 1; counter < stackTraceElements.length; counter++) {
                    
                        if(stackTraceElements[counter].trim().startsWith("at")){
                        	
                        	//2 is the length of "at":
                        	values[counter + throwableStringArray.length - 1] = stackTraceElements[counter].trim().substring(2).trim();                             
                        }
						else{
						    values[counter + throwableStringArray.length - 1] = stackTraceElements[counter].trim();
						}
                    }

                    extendedDataElement.setValues(values);
                }
                else{
                    extendedDataElement.setValues(getExtendedDataElementValuesArray(stackTraceElements[0].trim()));
                }
			}
            else{
                extendedDataElement.setValues(getExtendedDataElementValuesArray(throwable.toString()));
            }
            
            writer.close();            
        }

        return extendedDataElement;
    }

    /**
     * Convenience API to transpose a <code>values</code> string to a
     * <code>values</code> array, thereby ensuring the parameter
     * <code>values</code> string does not exceed the 1024 character limit, as
     * stated in the Common Base Event v1.0.1 specification.
     * <p>
     * If the parameter <code>values</code> property is larger than 1024
     * characters, the string is segmented into a String array of 1024-character
     * elements. However, if the parameter <code>values</code> property is
     * 1024 or less characters or <code>null</code>, the string is set
     * directly on the first element a String array.
     * <p>
     * 
     * @param values
     *            The values string to be transposed to a values array.
     */
    public static String[] getExtendedDataElementValuesArray(String values) {

        if (values == null) {
            return (new String[] { null});
        } else {

            int valuesLength = values.length();

            if (valuesLength > 1024) {

                String[] valuesArray = new String[((int) (Math.ceil(valuesLength / 1024.0)))];

                for (int counter = 0; counter < valuesArray.length; counter++) {
                    valuesArray[counter] = values.substring((counter * 1024), Math.min(((counter + 1) * 1024), valuesLength));
                }

                return valuesArray;
            } else {
                return (new String[] { values});
            }
        }
    }
    
    /**
     * Converts the parameter object to a <code>CommonBaseEvent</code> 
     * based on the Common Base Event v1.0.1 schema.
     * <p>
     * This method uses the following rules:
     * <p>
     * <ol>
     * <li>The <code>creationTime</code> Common Base Event property is set to the current time.</li>
     * <li>The <code>globalInstanceId</code> Common Base Event property is set to a unique GUID.</li>
     * <li>The <code>situation</code> Common Base Event property is set to an internal, log and report situation.</li>
     * <li>The <code>sourceComponentId</code> Common Base Event property is set to a default component ID for the calling logging facility.</li>
     * <li>All simple instance properties of the object are the <code>value</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * <li>All complex instance properties of the object are child <code>ExtendedDataElement</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * </ol>
     * <p>
     * Simple type of objects include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * If the parameter object is an instance of a <code>CommonBaseEvent</code>, it is cast to a 
     * <code>CommonBaseEvent</code> and returned.
     * <p>
     * 
     * @param object
     *            The object to be converted to a Common Base Event.
     * @param maxReferences
     *            The maximum number of referenced complex objects traversed.
     * @return The generated Common Base Event from the parameter object.
     */
    public static CommonBaseEvent convertObjectToCommonBaseEvent(Object object, int maxReferences) {
       
        //Determine if the complex object is a Common Base Event:
        if (object instanceof CommonBaseEvent) {    
            return ((CommonBaseEvent)(object));
        }
        
        CommonBaseEvent commonBaseEvent = EventFactory.eINSTANCE.createCommonBaseEvent();
        convertObjectToCommonBaseEvent(commonBaseEvent,object,maxReferences);
        
        return commonBaseEvent;
    }
    
    /**
     * Converts the parameter object to a <code>CommonBaseEvent</code> 
     * based on the Common Base Event v1.0.1 schema and sets the Common
     * Base Event properties on the parameter Common Base Event.
     * <p>
     * This method uses the following rules:
     * <p>
     * <ol>
     * <li>The <code>creationTime</code> Common Base Event property is set to the current time, if not previously set.</li>
     * <li>The <code>globalInstanceId</code> Common Base Event property is set to a unique GUID, if not previously set.</li>
     * <li>The <code>situation</code> Common Base Event property is set to an internal, log and report situation, if not previously set.</li>
     * <li>The <code>sourceComponentId</code> Common Base Event property is set to a default component ID for the calling logging facility, if not previously set.</li>
     * <li>All simple instance properties of the object are the <code>value</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * <li>All complex instance properties of the object are child <code>ExtendedDataElement</code>(s) 
     * of a <code>ExtendedDataElement</code> with the package and class name of the object 
     * as the &apos;name&apos; property of the <code>ExtendedDataElement</code>.</li>
     * </ol>
     * <p>
     * Simple type of objects include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * If the parameter object is an instance of a <code>CommonBaseEvent</code>, its properties are
     * set on the parameter <code>CommonBaseEvent</code> using the precedence rules defined 
     * <code>org.eclipse.hyades.logging.events.cbe.impl.TemplateContentHandlerImpl</code>.
     * <p>
     * NOTE:  The parameter <code>CommonBaseEvent</code>'s existing <code>ContentHandler</code>,
     * if any, is preserved for future use.
     * <p>
     * 
     * @param commonBaseEvent
     *            The Common Base Event generated from converting the parameter object.
     * @param object
     *            The object to be converted to a Common Base Event.
     * @param maxReferences
     *            The maximum number of referenced complex objects traversed.
     */
    public static void convertObjectToCommonBaseEvent(CommonBaseEvent commonBaseEvent, Object object, int maxReferences) {
       
        //Determine if the complex object is a Common Base Event:
        if (object instanceof CommonBaseEvent) {            
            
            TemplateContentHandler templateContentHandler = new TemplateContentHandlerImpl();
            templateContentHandler.setTemplateEvent(((CommonBaseEvent) (object)));
            
            ContentHandler currentContentHandler = commonBaseEvent.getContentHandler();
            
            commonBaseEvent.setContentHandler(templateContentHandler);
            
            try {
                commonBaseEvent.complete();                
            } 
            catch (CompletionException c) {
                //Ignore since content completion is based on 'best-effort'.
            }        
            
            commonBaseEvent.setContentHandler(currentContentHandler);            
        } 
        else {

            //Step 1: Set the default properties on the Common Base Event:

            //Set the global instance ID:
            if(commonBaseEvent.getGlobalInstanceId() == null){
                commonBaseEvent.setGlobalInstanceId(EventFactory.eINSTANCE.createGlobalInstanceId());
            }

            //Set the creation time to the current time:
            if(!commonBaseEvent.isSetCreationTime()){
                commonBaseEvent.setCreationTimeAsLong(System.currentTimeMillis());
            }

            //Set the situation to a default situation:
            if(commonBaseEvent.getSituation() == null){
                
                Situation reportSituation = EventFactory.eINSTANCE.createSituation();
                reportSituation.setReportSituation("INTERNAL", "LOG");
    
                commonBaseEvent.setSituation(reportSituation);
            }

            //Set the source component ID to a default component ID:
            if(commonBaseEvent.getSourceComponentId() == null){
                commonBaseEvent.setSourceComponentId(null, "Logging", "Logger", "Logging_Application", "Application", Constants.LOCAL_HOST_IP_ADDRESS, ComponentIdentification.LOCATION_TYPE_IPV4);
            }

            //Step 2: Generate the Extended Data Element(s):
            commonBaseEvent.addExtendedDataElement(convertObjectToExtendedDataElement(object, null, maxReferences, 0));
        }
    }

    /**
     * Converts the parameter object to a <code>ExtendedDataElement</code>
     * using the following rules:
     * <p>
     * <ol>
     * <li>The package and class name of the object is the &apos;name&apos; property of the
     * <code>ExtendedDataElement</code>.</li>
     * <li>All simple instance properties of the object are the <code>value</code>(s) 
     * of the <code>ExtendedDataElement</code>.</li>
     * <li>All complex instance properties of the object are child <code>ExtendedDataElement</code>(s) 
     * of the <code>ExtendedDataElement</code>.</li>
     * </ol>
     * <p>
     * Simple type of objects include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * 
     * @param object
     *            The complex object to be converted to an Extended Data Element.
     * @param instanceName
     *            The instance name of the complex object to be converted to an Extended Data Element.
     * @param maxReferences
     *            The maximum number of referenced complex objects traversed.
     * @param currentReferenceCount
     *            The current number of referenced complex objects traversed.
     * @return The generated Extended Data Element of the parameter complex
     *         object.
     */
    public static ExtendedDataElement convertObjectToExtendedDataElement(Object object, String instanceName, int maxReferences, int currentReferenceCount) {

        ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();

        if (object != null) {

            Class objectsClass = object.getClass();

            //Case: Simple type:
            if (isSimpleType(object)) {
                
                String objectToString = object.toString().trim();
                int valuesLength = objectToString.length();

                if (valuesLength > 1024) {

                    ArrayList valuesParts = new ArrayList();

                    for (int counter = 0; counter < valuesLength; counter += 1024) {
                        valuesParts.add(objectToString.substring(counter, Math.min((counter + 1024), valuesLength)));
                    }

                    extendedDataElement.setValuesAsStringArray(((String[]) (valuesParts.toArray(new String[valuesParts.size()]))));
                } 
                else {
                    extendedDataElement.setValuesAsString(objectToString);
                }
            }

            //Case: Array:
            else if (objectsClass.isArray()) {

                //Case: boolean array:
                if (object instanceof boolean[]) {

                    boolean[] array = ((boolean[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Boolean(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: char array:
                else if (object instanceof char[]) {

                    char[] array = ((char[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Character(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: byte array:
                else if (object instanceof byte[]) {

                    byte[] array = ((byte[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Byte(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: short array:
                else if (object instanceof short[]) {

                    short[] array = ((short[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Short(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: int array:
                else if (object instanceof int[]) {

                    int[] array = ((int[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Integer(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: long array:
                else if (object instanceof long[]) {

                    long[] array = ((long[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Long(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: float array:
                else if (object instanceof float[]) {

                    float[] array = ((float[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Float(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: double array:
                else if (object instanceof double[]) {

                    double[] array = ((double[]) object);

                    for (int counter = 0; counter < array.length; counter++) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(new Double(array[counter]), null, maxReferences, currentReferenceCount));
                    }
                }

                //Case: Default (e.g. object arrays):
                else {

                    if(currentReferenceCount == maxReferences){            
                        extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                    }
                    else{

                        Object[] array = ((Object[]) (object));
    
                        for (int counter = 0; counter < array.length; counter++) {
                            extendedDataElement.addChild(convertObjectToExtendedDataElement(array[counter], null, maxReferences, (currentReferenceCount + 1)));
                        }
                    }
                }
            }

            //Case: Collection:
            else if (object instanceof Collection) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                }
                else{
    
                    Iterator iterator = ((Collection) (object)).iterator();
    
                    while (iterator.hasNext()) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(iterator.next(), null, maxReferences, (currentReferenceCount + 1)));
                    }
                }
            }

            //Case: Enumeration:
            else if (object instanceof Enumeration) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));                      
                }
                else{

                    Enumeration enumeration = ((Enumeration) (object));
    
                    while (enumeration.hasMoreElements()) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(enumeration.nextElement(), null, maxReferences, (currentReferenceCount + 1)));
                    }
                }
            }

            //Case: Iterator:
            else if (object instanceof Iterator) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                }
                else{

                    Iterator iterator = ((Iterator) (object));

                    while (iterator.hasNext()) {
                        extendedDataElement.addChild(convertObjectToExtendedDataElement(iterator.next(), null, maxReferences, (currentReferenceCount + 1)));
                    }
                }
            }

            //Case: Thread:
            else if (object instanceof Thread) {

                Thread thread = ((Thread) (object));

                extendedDataElement.addChild("Name", thread.getName());
                extendedDataElement.addChild("Priority", String.valueOf(thread.getPriority()));
                extendedDataElement.addChild("Group", thread.getThreadGroup().getName());
            }

            //Case: Throwable:
            //NOTE: Use the default Throwable --> Common Base Event mapping (see EventHelpers.convertToExtendedDataElement())
            //      to leverage existing tool.  As such, ignore the the maximum number of referenced causal Throwables that
            //      that are traversed.
            else if (object instanceof Throwable) {
                extendedDataElement = EventHelpers.convertToExtendedDataElement(((Throwable) (object)));
            }

            //Case: Map:
            else if (object instanceof Map) {

                if(currentReferenceCount == maxReferences){            
                    extendedDataElement.addChild(createEndOfNestingExtendedDataElement(instanceName));          
                }
                else{

                    Map map = ((Map) (object));
                    Iterator keys = map.keySet().iterator();
                    ExtendedDataElement entryExtendedDataElement = null;
                    ExtendedDataElement keyExtendedDataElement = null;
                    ExtendedDataElement valueExtendedDataElement = null;
                    int counter = 1;
    
                    while (keys.hasNext()) {
    
                        Object key = keys.next();
    
                        entryExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                        entryExtendedDataElement.setName("Entry_".concat(String.valueOf(counter++)));
                        entryExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
    
                        keyExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                        keyExtendedDataElement.setName("Key");
                        keyExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
                        keyExtendedDataElement.addChild(convertObjectToExtendedDataElement(key, null, maxReferences, (currentReferenceCount + 1)));
    
                        entryExtendedDataElement.addChild(keyExtendedDataElement);
    
                        valueExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                        valueExtendedDataElement.setName("Value");
                        valueExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
                        valueExtendedDataElement.addChild(convertObjectToExtendedDataElement(map.get(key), null, maxReferences, (currentReferenceCount + 1)));
    
                        entryExtendedDataElement.addChild(valueExtendedDataElement);
    
                        extendedDataElement.addChild(entryExtendedDataElement);
                    }
                }
            }
            
            //Case: EventQueue:
            else if (object instanceof EventQueue) {
                //Ignore.
            }

            //Case: Class:
            else if (object instanceof Class) {

                Class theClass = ((Class) (object));

                extendedDataElement.addChild("Name", getClassName(theClass));

                if (theClass.isPrimitive()) {
                    extendedDataElement.addChild("Type", "primitive");
                } 
                else if (theClass.isArray()) {
                    extendedDataElement.addChild("Type", "array");
                } 
                else if (theClass.isInterface()) {
                    extendedDataElement.addChild("Type", "interface");
                } 
                else {
                    extendedDataElement.addChild("Type", "class");
                }

                Package thePackage = theClass.getPackage();

                if (thePackage != null) {
                    extendedDataElement.addChild("Package", thePackage.getName());
                }

                extendedDataElement.addChild("Modifers", Modifier.toString(theClass.getModifiers()));

                extendedDataElement.addChild("Superclass", getClassName(theClass.getSuperclass()));
            }

            //Case: Default:
            else {

                Method[] objectsMethods = objectsClass.getMethods();
                String methodName = null;
                Object returnObject = null;

                for (int counter = 0; counter < objectsMethods.length; counter++) {

                    methodName = objectsMethods[counter].getName().trim();

                    if ((methodName.length() > 3) && (methodName.startsWith("get")) && (objectsMethods[counter].getParameterTypes().length == 0)) {

                        try {
                            returnObject = objectsMethods[counter].invoke(object, null);
                        } 
                        catch (Throwable t) {
                            returnObject = null;
                        }

                        if((returnObject != null) && (!isSimpleType(returnObject)) && (currentReferenceCount == maxReferences)){            
                            extendedDataElement.addChild(createEndOfNestingExtendedDataElement(methodName.substring(3)));           
                        }
                        else{
                            extendedDataElement.addChild(convertObjectToExtendedDataElement(returnObject, methodName.substring(3), maxReferences, (currentReferenceCount + 1)));
                        }
                    }
                }
            }
        }

        extendedDataElement.setName(getClassName(object));

        if (extendedDataElement.getType() == null) {
            extendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
        }

        if ((instanceName != null) && (instanceName.trim().length() > 0)) {
            extendedDataElement.addChild("Instance_Name", instanceName);
        }

        return extendedDataElement;
    }

    protected static ExtendedDataElement createEndOfNestingExtendedDataElement(String instanceName){
        
        ExtendedDataElement endExtendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
        endExtendedDataElement.setName("End_Of_Nesting");
        endExtendedDataElement.setTypeAsInt(ExtendedDataElement.TYPE_NO_VALUE_VALUE);
        
        if ((instanceName != null) && (instanceName.trim().length() > 0)) {
            endExtendedDataElement.addChild("Instance_Name", instanceName);
        }
        
        return endExtendedDataElement;
    }
    
    /**
     * Returns the package and class name of parameter object.
     * <p>
     * 
     * @param object
     *            The object for which its package and class name is derived.
     * @return String The package and class name of parameter object.
     */
    protected static String getClassName(Object object) {

        if (object == null) { 
            return "null"; 
        }

        Class theClass = object.getClass();
        String name = theClass.getName().trim();
        
        //Check if the class is an array:
        if (theClass.isArray()) {

            int index = (name.lastIndexOf('[') + 1);

            String encodingString = name.substring(index).trim();
            char encoding = InternationalizationUtilities.charAt(encodingString,0);
            String type = "";

            if ((encoding == 'L') && (encodingString.endsWith(";"))) {
                type = encodingString.substring(1, (encodingString.length() - 1));
            } 
            else if (encoding == 'B') {
                type = "byte";
            } 
            else if (encoding == 'C') {
                type = "char";
            } 
            else if (encoding == 'D') {
                type = "double";
            } 
            else if (encoding == 'F') {
                type = "float";
            } 
            else if (encoding == 'I') {
                type = "int";
            } 
            else if (encoding == 'J') {
                type = "long";
            } 
            else if (encoding == 'S') {
                type = "short";
            } 
            else if (encoding == 'Z') {
                type = "boolean";
            }
            else{
                type = encodingString.substring(1, encodingString.length());
            }

            StringBuffer nameBuffer = new StringBuffer();
            nameBuffer.append(type);
            
            int depth = name.substring(0, index).trim().length();

            for (int counter = 0; counter < depth; counter++) {
                nameBuffer.append("[]");
            }
            
            return (nameBuffer.toString());
        }

        return name;
    }
    
    /**
     * Determines if the parameter non-null object is a simple type.
     * <p>
     * Simple types include:
     * <p>
     * <ul>
     * <li>Java primitives (e.g. Boolean, Character, Byte, Short, Integer, Long, Float and Double)</li>
     * <li>StringBuffer</li>
     * <li>String</li>
     * <li>BigDecimal</li>
     * <li>BigInteger</li>
     * </ul>
     * <p>
     * 
     * @param theClass The non-null object for which its type is derived.
     * @return True if the parameter non-null object is a simple type, otherwise false.
     */
    protected static boolean isSimpleType(Object object) {
        return ((object.getClass().isPrimitive()) || (object instanceof Boolean) || (object instanceof Character) || (object instanceof Byte) || (object instanceof Short) || (object instanceof Integer) || (object instanceof Long) || (object instanceof Float) || (object instanceof Double) || (object instanceof BigDecimal) || (object instanceof BigInteger) || (object instanceof String) || (object instanceof StringBuffer));
    }
}