package org.eclipse.tptp.logging.events.cbe;

import java.io.Serializable;

/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociatedEvent.java,v 1.1 2006/09/06 17:53:59 cjin Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Interface representing an <code>AssociatedEvent</code>.
 * <p>
 * The following description is from the Common Base Event v1.0.1 specification
 * entitled "Canonical Situation Data Format: The Common Base Event V1.0.1":
 * <p>
 * The AssociatedEvent type allows for associated events to be grouped. It
 * allows for identifying associated events their association engines. The
 * associationEngine <em>may</em> be the reference to the application that
 * created the association.
 * <p>
 * 
 * 
 * @author Paul E. Slauenwhite
 * @version February 5, 2006
 * @since 1.0.1
 */
public interface AssociatedEvent extends Cloneable, Serializable {

    /**
     * Adds a Common Base Event reference to the array of resolved events.
     * 
     * Array of Common Base Event globalInstanceIds corresponding to the
     * event(s) that are associated with this event. This is a REQUIRED
     * property.
     * 
     * @param globalInstanceId
     *            The new Common Base Event globalInstanceId.
     * @see CommonBaseEvent
     * @since 1.0
     */
    void addResolvedEvent(String globalInstanceId) throws IllegalArgumentException;

    /**
     * Removes all Common Base Event references in the array of resolved events.
     * 
     * Array of Common Base Event globalInstanceIds corresponding to the
     * event(s) that are associated with this event. This is a REQUIRED
     * property.
     * 
     * @see CommonBaseEvent
     * @since 1.0
     */
    void clearResolvedEvents();

    /**
     * Removes a named Common Base Event reference in the array of resolved
     * events.
     * 
     * Array of Common Base Event globalInstanceIds corresponding to the
     * event(s) that are associated with this event. This is a REQUIRED
     * property.
     * 
     * @param globalInstanceId
     *            The removed Common Base Event globalInstanceId.
     * @see CommonBaseEvent
     * @since 1.0
     */
    void removeResolvedEvent(String globalInstanceId);

    /**
     * <P>
     * Returns the value of the '<em><b>Association Engine</b></em>'
     * attribute.
     * </P>
     *  
     * 
     * @return the value of the '<em>Association Engine</em>' attribute.
     * @see #setAssociationEngine(String)
     */
    String getAssociationEngine();

    /**
     * <P>
     * Sets the value of the '
     * {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngine <em>Association Engine</em>}'
     * attribute.
     * <P>
     * 
     * <P>
     * Reference to the AssocationEngine that created this AssociatedEvent. The
     * associationEngine and associationEnginInfo properties are mutually
     * exclusive. One of these properties <em>must</em> be defined. If this
     * attribute is set, the '<em><b>Association Engine Info</b></em>'
     * attribute should not be set.
     * </P>
     * 
     * @throws IllegalStateException
     *             if the '<em><b>Association Engine Info</b></em>'
     *             attribute is currently set. 
     * @param value
     *            the new value of the '<em>Association Engine</em>'
     *            attribute.
     * @see #getAssociationEngine()
     */
    void setAssociationEngine(String value);

    /**
     * <P>
     * Returns the value of the '<em><b>Resolved Events</b></em>'
     * attribute.
     * </P>
     * 
     * <p>
     * The returned string follows the IDREFS type as stated in the CBE and
     * defined in the XML Schema specification, which is a space-separated list
     * of IDREF strings.
     * </P>
     * 
     * 
     * @return the value of the '<em>Resolved Events</em>' attribute.
     * @see #setResolvedEvents(String)
     */
    String getResolvedEvents();

    /**
     * <P>
     * Returns the value of the '<em><b>Resolved Events</b></em>'
     * attribute.
     * </P>
     * 
     * <p>
     * The returned string follows the IDREFS type as stated in the CBE and
     * defined in the XML Schema specification. In this case it will be an array
     * of IDREF strings. If there are no resolved events then an empty array
     * will be returned.
     * </p>
     * 
     * 
     * @return the value of the '<em>Resolved Events</em>' attribute.
     * @see #setResolvedEvents(String)
     * @see #setResolvedEventsAsArray(String[])
     * @see #getResolvedEvents()
     */
    String[] getResolvedEventsAsArray();

    /**
     * <P>
     * Sets the value of the '
     * {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getResolvedEvents <em>Resolved Events</em>}'
     * attribute.
     * </P>
     * 
     * <P>
     * This is an array of globalInstanceIds corresponding to the events that
     * are associated with this event.
     * </P>
     * <P>
     * This is a <em>required</em> property, an array with at least one
     * element.
     * </P>
     * <P>
     * The parameter should follow the IDREFS type as stated in the CBE and
     * defined in the XML Schema specification, which is a space-separated list
     * of IDREF strings.
     * </P>
     * 
     * 
     * 
     * @param value
     *            the new value of the '<em>Resolved Events</em>' attribute.
     * @see #getResolvedEvents()
     * @see #validate()
     */
    void setResolvedEvents(String value);

    /**
     * <P>
     * Sets the value of the '
     * {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getResolvedEvents <em>Resolved Events</em>}'
     * attribute.
     * </P>
     * 
     * <P>
     * The parameter should follow the IDREFS type as stated in the CBE and
     * defined in the XML Schema specification. In this case it will be an array
     * of IDREF strings.
     * <P>
     * This method does not validate the parameter, which is left to separate
     * calls to the <code>validate</code> method.
     * </P>
     * 
     * 
     * 
     * @param value
     *            the new value of the '<em>Resolved Events</em>' attribute.
     * @see #setResolvedEvents(String)
     * @see #getResolvedEvents()
     * @see #getResolvedEventsAsArray()
     * @see #validate()
     */
    void setResolvedEventsAsArray(String[] value) throws IllegalArgumentException;

    /**
     * <P>
     * Returns the value of the '<em><b>Association Engine Info</b></em>'
     * containment reference.
     * </P>
     *  
     * 
     * @return the value of the '<em>Association Engine Info</em>'
     *         containment reference.
     * @see #setAssociationEngineInfo(AssociationEngine)
     */
    AssociationEngine getAssociationEngineInfo();

    /**
     * <P>
     * Sets the value of the '
     * {@link org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngineInfo <em>Association Engine Info</em>}'
     * containment reference.
     * </P>
     * 
     * <P>
     * Identifies the application that establishes association among related or
     * associated events. In addition, it provides properties to describe the
     * type of the association.
     * </P>
     * <P>
     * The associationEngine and associationEnginInfo properties are mutually
     * exclusive. One of these properties <em>must</em> be defined.
     * </P>
     * 
     * @throws IllegalStateException
     *             if the '<em><b>Association Engine</b></em>' attribute is
     *             currently set. 
     * @param value
     *            the new value of the '<em>Association Engine Info</em>'
     *            containment reference.
     * @see #getAssociationEngineInfo()
     */
    void setAssociationEngineInfo(AssociationEngine value);

    /**
     * <P>
     * Helper method for creating and setting the association engine information
     * for this object.
     * </P>
     * <P>
     * It creates an <code>AssociationEngine</code> object and sets it in the
     * respective field of this object.
     * </P>
     * 
     * @param associationEngineName
     *            name for the association engine.
     * @param associationEngineType
     *            type for the association engine.
     * @param associationEngineId
     *            identifier for the association engine.
     */
    void setAssociationEngineInfo(String associationEngineName, String associationEngineType, String associationEngineId);

    /**
     * <P>
     * Checks whether the contents of this object conforms to the CBE
     * specification.
     * </P>
     * 
     * @throws ValidationException
     *             if the object does not follow the CBE specification.
     */
    void validate() throws ValidationException;

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * All components are initialized to either zero or null.
     * 
     * @since 1.0
     */
    void init();
} // AssociatedEvent
