package org.eclipse.tptp.logging.events.cbe;

/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventFactoryHome.java,v 1.1 2006/09/06 17:54:00 cjin Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Interface representing a <code>EventFactoryHome</code>.
 * <p>
 * Event factory home for retrieving event factories.
 * <p>
 * <code>EventFactoryHome</code> instances provide resolution of named
 * <code>EventFactory</code> instances, configuration updates based on a
 * specific type of configuration template for the <code>EventFactory</code>
 * instances and releasing cached named <code>EventFactory</code> instances.
 * <p>
 * <code>EventFactoryHome</code> instances also provide an anonymous
 * <code>EventFactory</code> for retrieving an unnamed
 * <code>EventFactory</code> for component independent event creation. The
 * anonymous <code>EventFactory</code> may be retrieved and potentially
 * updated based on a specific type of configuration template for the
 * <code>EventFactory</code> instances, if one exists for unnamed
 * <code>EventFactory</code> instances. Note, the anonymous
 * <code>EventFactory</code> instance cannot be released once created.
 * <p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version February 9, 2006
 * @since   1.0.1
 * @see     org.eclipse.tptp.logging.events.cbe.EventFactory
 */
public interface EventFactoryHome {

    /**
     * Retrieves an instance of the anonymous <code>EventFactory</code>.
     * <p>
     * An instance of the anonymous <code>EventFactory</code> is created if
     * the anonymous <code>EventFactory</code> instance currently does not
     * exist.
     * <p>
     * Once the anonymous <code>EventFactory</code> is created, the instance
     * is cached for future calls to retrieve the same anonymous
     * <code>EventFactory</code>.
     * <p>
     * The anonymous <code>EventFactory</code> is used to for component
     * independent event creation and is unnamed. The anonymous
     * <code>EventFactory</code> may be assigned a <code>ContentHandler</code>
     * for configuration updates based on a specific type of configuration
     * template for the <code>EventFactory</code> instances, if one exists for
     * unnamed <code>EventFactory</code> instances.
     * <p>
     * 
     * @return The retrieved or newly created anonymous
     *         <code>EventFactory</code>.
     */
    public EventFactory getAnonymousEventFactory();

    /**
     * Retrieves an instance of the named <code>EventFactory</code>.
     * <p>
     * An instance of a named <code>EventFactory</code> is created if no named
     * instance current exists or a named instance is released.
     * <p>
     * Once a named <code>EventFactory</code> is created, the instance is
     * cached for future calls to retrieve the same named
     * <code>EventFactory</code>.
     * <p>
     * The name of the <code>EventFactory</code> uniquely identifies an
     * instance of an <code>EventFactory</code>. All subsequent calls will
     * return the same instance of the named <code>EventFactory</code>.
     * <p>
     * <code>EventFactory</code> names are hierarchal represented using the
     * standard Java dot-delimited name-space naming conventions.
     * <p>
     * Passing a <code>null</code> <code>EventFactory</code> name parameter
     * in will be equivalent to calling the {@link #getAnonymousEventFactory()}
     * API.
     * <p>
     * 
     * @param factoryName
     *            The name of the <code>EventFactory</code>.
     * @return The retrieved or newly created <code>EventFactory</code>.
     */
    public EventFactory getEventFactory(String factoryName);

    /**
     * Updates the configuration on the instance of the anonymous
     * <code>EventFactory</code>.
     * <p>
     * If the <code>EventFactoryHome</code> instance does not support
     * <code>ContentHandler</code> s for unnamed <code>EventFactory</code>
     * instances, no update occurs.
     * <p>
     * The anonymous <code>EventFactory</code> is used to for component
     * independent event creation and is unnamed. The anonymous
     * <code>EventFactory</code> may be assigned a <code>ContentHandler</code>
     * for configuration updates based on a specific type of configuration
     * template for the <code>EventFactory</code> instances, if one exists for
     * unnamed <code>EventFactory</code> instances.
     * <p>
     */
    public void updateAnonymousEventFactory();

    /**
     * Updates the configuration on the instance of the named
     * <code>EventFactory</code>.
     * <p>
     * If no named <code>EventFactory</code> instance exists, no update
     * occurs.
     * <p>
     * The name of the <code>EventFactory</code> uniquely identifies an
     * instance of an <code>EventFactory</code>. All calls will update the
     * configuration on the same named instance of the <code>EventFactory</code>.
     * <p>
     * <code>EventFactory</code> names are hierarchal represented using the
     * standard Java dot-delimited name-space naming conventions.
     * <p>
     * Passing a <code>null</code> <code>EventFactory</code> name parameter
     * in will be equivalent to calling the
     * {@link #updateAnonymousEventFactory()}API.
     * <p>
     * Passing an empty string (e.g. "") with or without white space or a single
     * asterisk (e.g. <code>*</code>) as the name parameter updates the
     * configuration on all cached <code>EventFactory</code> instances.
     * <p>
     * 
     * @param factoryName
     *            The name of the <code>EventFactory</code>.
     */
    public void updateEventFactory(String factoryName);

    /**
     * Releases the reference to the instance of the named
     * <code>EventFactory</code>.
     * <p>
     * If no named <code>EventFactory</code> instance exists, no release
     * occurs.
     * <p>
     * The name of the <code>EventFactory</code> uniquely identifies an
     * instance of an <code>EventFactory</code>.
     * <p>
     * <code>EventFactory</code> names are hierarchal represented using the
     * standard Java dot-delimited name-space naming conventions.
     * <p>
     * Passing an empty string (e.g. "") with or without white space or a single
     * asterisk (e.g. <code>*</code>) as the name parameter releases all
     * cached <code>EventFactory</code> instances.
     * <p>
     * 
     * @param factoryName
     *            The name of the <code>EventFactory</code>.
     */
    public void releaseEventFactory(String factoryName);
}