package org.eclipse.tptp.logging.events.cbe.impl;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.tptp.logging.events.cbe.ContentHandler;
import org.eclipse.tptp.logging.events.cbe.EventFactory;
import org.eclipse.tptp.logging.events.cbe.EventFactoryHome;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AbstractEventFactoryHome.java,v 1.3 2008/01/24 02:29:31 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * An abstract event factory home implementation for working with event
 * factories.
 * <p>
 * This abstract class provides resolution of named <code>EventFactory</code>
 * instances, persistence of retrieved <code>EventFactory</code> instances,
 * configuration updates based on a specific type of configuration template for
 * the <code>EventFactory</code> instances and releasing cached named
 * <code>EventFactory</code> instances.
 * <p>
 * This abstract class also provides an anonymous <code>EventFactory</code>
 * for retrieving an unnamed <code>EventFactory</code> for component
 * independent event creation. The anonymous <code>EventFactory</code> may be
 * retrieved and potentially updated based on a specific type of configuration
 * template for the <code>EventFactory</code> instances, if one exists for
 * unnamed <code>EventFactory</code> instances. Note, the anonymous
 * <code>EventFactory</code> instance cannot be released once created.
 * <p>
 * Concrete implementations of this abstract class implement the
 * <code>createContentHandler(String factoryName)</code> and
 * <code>resolveContentHandler()</code> APIs to provide a
 * <code>ContentHandler</code> implementation tightly coupled to the type of
 * concrete implementation of this abstract class. The
 * <code>resolveContentHandler()</code> API permits an alternative
 * <code>ContentHandler</code> to be associated with an instance of a concrete
 * implementation of this abstract class. This is achieved by extending the
 * concrete implementation of this abstract class and extend the
 * <code>resolveContentHandler()</code> API. Concrete implementations that do
 * not require a <code>ContentHandler</code> nor configuration template,
 * simply return <code>null</code> for each API.
 * <p>
 * For example, a concrete implementation of this abstract class may provide the
 * an implementation of the
 * <code>createContentHandler(String factoryName)</code> API that creates a
 * specialized <code>ContentHandler</code> pre-configured with an event
 * template resolved from a local event property file. The concrete
 * implementation of this abstract class would also provide an implementation of
 * the <code>resolveContentHandler()</code> API that retrieves a new instance
 * of the specialized <code>ContentHandler</code> used in the implementation
 * of the <code>createContentHandler(String factoryName)</code> API.
 * <p>
 * It is assumed that implementations of the
 * <code>createContentHandler(String factoryName)</code> API would reference
 * the implementation of the <code>resolveContentHandler()</code> API in the
 * same concrete implementation of this abstract class when creating the
 * returned <code>ContentHandler</code>.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version March 10, 2005
 * @since 1.0.1
 * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome
 */
public abstract class AbstractEventFactoryHome implements EventFactoryHome {

    protected Hashtable eventFactories = new Hashtable();

    protected EventFactory anonymousEventFactory = null;

    /**
     * Creates an instance of the <code>ContentHandler</code> implementation
     * which is tightly coupled to the type of the concrete implementation of
     * this abstract class.
     * <p>
     * The returned instance of the <code>ContentHandler</code> implementation
     * is pre-configured with a event properties resolved from a source tightly
     * coupled to the type of the concrete implementation of this abstract
     * class.
     * <p>
     * It is assumed that implementations of this method would reference the
     * implementation of the <code>resolveContentHandler()</code> method in
     * the same concrete implementation of this abstract class when creating the
     * returned <code>ContentHandler</code>.
     * <p>
     * 
     * @param factoryName
     *            The name of the event factory associated with the returned
     *            <code>ContentHandler</code> implementation.
     * @return An instance of the <code>ContentHandler</code> pre-configured
     *         with a event properties.
     */
    public abstract ContentHandler createContentHandler(String factoryName);

    /**
     * Resolves an instance of the <code>ContentHandler</code> implementation
     * which is tightly coupled to the type of the concrete implementation of
     * this abstract class.
     * <p>
     * It is assumed that implementations of the
     * <code>createContentHandler(String factoryName)</code> method would
     * reference the implementation of this method in the same concrete
     * implementation of this abstract class when creating the returned
     * <code>ContentHandler</code>.
     * <p>
     * 
     * @return An instance of the <code>ContentHandler</code> tightly coupled
     *         with this type of this event factory home implementation.
     */
    public abstract ContentHandler resolveContentHandler();

    /**
     * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome#getAnonymousEventFactory()
     */
    public EventFactory getAnonymousEventFactory() {

        if (anonymousEventFactory == null) {

            anonymousEventFactory = new EventFactoryImpl();

            updateAnonymousEventFactory();
        }

        return anonymousEventFactory;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome#getEventFactory(java.lang.String)
     */
    public EventFactory getEventFactory(String factoryName) {

        if (factoryName == null) { return (getAnonymousEventFactory()); }

        if (eventFactories.containsKey(factoryName)) { return (((EventFactory) (eventFactories.get(factoryName)))); }

        EventFactory eventFactory = new EventFactoryImpl();

        eventFactory.setContentHandler(createContentHandler(factoryName));

        eventFactories.put(factoryName, eventFactory);

        return eventFactory;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome#updateAnonymousEventFactory()
     */
    public void updateAnonymousEventFactory() {

        if (anonymousEventFactory != null) {

            try {
                anonymousEventFactory.setContentHandler(createContentHandler(null));
            } catch (NullPointerException n) {
                // Ignore since the event factory home implementation does not
                // support an anonymous event factory.
            }
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome#updateEventFactory(java.lang.String)
     */
    public void updateEventFactory(String factoryName) {

        if (factoryName == null) {
            updateAnonymousEventFactory();
        } else if ((factoryName.trim().length() == 0) || (factoryName.trim().equals("*"))) {

            Enumeration eventFactoryNames = eventFactories.keys();

            while (eventFactoryNames.hasMoreElements()) {
                updateEventFactory(((String) (eventFactoryNames.nextElement())));
            }
        } else if (eventFactories.containsKey(factoryName)) {
            ((EventFactory) (eventFactories.get(factoryName))).setContentHandler(createContentHandler(factoryName));
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome#releaseEventFactory(java.lang.String)
     */
    public void releaseEventFactory(String factoryName) {

        if (factoryName != null) {

            if ((factoryName.trim().length() == 0) || (factoryName.trim().equals("*"))) {
                eventFactories.clear();
            } else if (eventFactories.containsKey(factoryName)) {
                eventFactories.remove(factoryName);
            }
        }
    }
}