package org.eclipse.tptp.logging.events.cbe.impl;

import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;
import org.eclipse.hyades.logging.core.LoggingCoreResourceBundle;
import org.eclipse.tptp.logging.events.cbe.AssociatedEvent;
import org.eclipse.tptp.logging.events.cbe.AssociationEngine;
import org.eclipse.tptp.logging.events.cbe.EventFactory;
import org.eclipse.tptp.logging.events.cbe.MissingPropertyException;
import org.eclipse.tptp.logging.events.cbe.ValidationException;

/**********************************************************************
 * Copyright (c) 2005, 2006, 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociatedEventImpl.java,v 1.3 2008/01/24 02:29:31 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Class representing an implementation of an <code>AssociatedEvent</code>.
 * <p>
 * The following description is from the Common Base Event v1.0.1 specification
 * entitled "Canonical Situation Data Format: The Common Base Event V1.0.1":
 * <p>
 * The AssociatedEvent type allows for associated events to be grouped. It
 * allows for identifying associated events their association engines. The
 * associationEngine <em>may</em> be the reference to the application that
 * created the association.
 * <p>
 * 
 * @author cindyjin
 * @author Paul E. Slauenwhite
 * @version March 30, 2007
 * @since 1.0.1
 */
public class AssociatedEventImpl implements AssociatedEvent {

    /** 
     * Serialization version ID.
     */
    protected static final long serialVersionUID = -4327879421885580788L;

    /**
     * Association Engine instance variable.
     */
    protected String associationEngine = null;

    /**
     * Resolved Events instance variable.
     */
    protected String resolvedEvents = null;

    /**
     * Association Engine Info instance variable. 
     */
    protected AssociationEngine associationEngineInfo = null;

    /**
     * Protected constructor.
     */
    protected AssociatedEventImpl() {
        super();
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#setAssociationEngine(java.lang.String)
     */
    public void setAssociationEngine(String associationEngine) {
        this.associationEngine = associationEngine;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngine()
     */
    public String getAssociationEngine() {
        return associationEngine;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#setAssociationEngineInfo(org.eclipse.tptp.logging.events.cbe.AssociationEngine)
     */
    public void setAssociationEngineInfo(AssociationEngine associationEngineInfo) {
        this.associationEngineInfo = associationEngineInfo;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#setAssociationEngineInfo(java.lang.String, java.lang.String, java.lang.String)
     */
    public void setAssociationEngineInfo(String associationEngineName, String associationEngineType, String associationEngineId) {
        
        AssociationEngine associationEngine = EventFactory.eINSTANCE.createAssociationEngine();
        
        associationEngine.setName(associationEngineName);
        associationEngine.setType(associationEngineType);
        associationEngine.setId(associationEngineId);

        setAssociationEngineInfo(associationEngine);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getAssociationEngineInfo()
     */
    public AssociationEngine getAssociationEngineInfo() {
        return associationEngineInfo;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#setResolvedEvents(java.lang.String)
     */
    public void setResolvedEvents(String resolvedEvents) {
        this.resolvedEvents = resolvedEvents;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getResolvedEvents()
     */
    public String getResolvedEvents() {
        return resolvedEvents;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#setResolvedEventsAsArray(java.lang.String[])
     */
    public void setResolvedEventsAsArray(String[] resolvedEventsArray) throws IllegalArgumentException{

        //Need to check that each element of the string array contains no blanks:
        if (resolvedEventsArray != null) {

            StringBuffer buffer = new StringBuffer();

            for (int counter = 0; counter < resolvedEventsArray.length; counter++) {
            
                if (resolvedEventsArray[counter].indexOf(" ") != -1) { 
                    throw new IllegalArgumentException("Blanks are not permitted in the array element [" + resolvedEventsArray[counter] + "]"); 
                }

                buffer.append(" ");
                buffer.append(resolvedEventsArray[counter].trim());
            }

            setResolvedEvents(buffer.toString().trim());
        } 
        else{
            throw new IllegalArgumentException("The input array cannot be null.");
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#getResolvedEventsAsArray()
     */
    public String[] getResolvedEventsAsArray() {
    	 
    	String resolvedEvents = getResolvedEvents();
    	
    	if(resolvedEvents != null){
    		return (InternationalizationUtilities.tokenize(resolvedEvents));
    	}
    	else{
    		return (new String[0]);
    	}    	
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.AssociatedEvent#addResolvedEvent(java.lang.String)
     */
    public void addResolvedEvent(String globalInstanceId) throws IllegalArgumentException{

        if (globalInstanceId != null) {

            if (resolvedEvents == null) {
                setResolvedEvents(globalInstanceId);
            }
            // Do not persist duplicates:
            else if (resolvedEvents.indexOf(globalInstanceId) == -1) {
                setResolvedEvents(resolvedEvents.trim().concat(" ").concat(globalInstanceId.trim()));
            }
        } 
        else {
            throw new IllegalArgumentException("The input string cannot be null.");
        }
    }

    /**
     * Removes all Common Base Event references in the array of resolved events.
     * 
     * Array of Common Base Event globalInstanceIds corresponding to the
     * event(s) that are associated with this event. This is a REQUIRED
     * property.
     * 
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent
     * @since 1.0
     */
    public void clearResolvedEvents() {
        resolvedEvents = null;
    }

    /**
     * Removes a named Common Base Event reference in the array of resolved
     * events.
     * 
     * Array of Common Base Event globalInstanceIds corresponding to the
     * event(s) that are associated with this event. This is a REQUIRED
     * property.
     * 
     * @param globalInstanceId
     *            The removed Common Base Event globalInstanceId.
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent
     * @since 1.0
     */
    public void removeResolvedEvent(String globalInstanceId) {

        if (globalInstanceId != null) {

            if (resolvedEvents != null) {

                int index;

                // Remove any duplicates:
                while ((index = resolvedEvents.indexOf(globalInstanceId)) != -1) {
                    setResolvedEvents(resolvedEvents.substring(0, index).trim().concat(" ").concat(resolvedEvents.substring(index + globalInstanceId.length()).trim()));
                }
            }
        } else {
            throw new IllegalArgumentException("The input string cannot be null.");
        }
    }


    /**
     * Overrides the <code>java.lang.Object</code>'s equals() API to 
     * determine if the parameter <code>object</code> is equal to this object.
     * <p>
     * 
     * @param object The <code>java.lang.Object</code> to be compared to this object.
     * @return True, if this object is the same as the parameter <code>object</code>, otherwise false.
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {

        synchronized (this) {
            
            //Check if the parameter object is the same object reference as this object (e.g. equal):
            if (this == object){
                return true;
            }
            else {

                //Check if the parameter object is null and is the a different type as this object(e.g. not equal):
                if ((object != null) && (getClass() == object.getClass())) {

                    //Compare the value(s) of all object properties:
                    Object thisObjectsProperty = getAssociationEngine();
                    Object parameterObjectsProperty = ((AssociatedEventImpl) (object)).getAssociationEngine();

                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                        thisObjectsProperty = getAssociationEngineInfo();
                        parameterObjectsProperty = ((AssociatedEventImpl)(object)).getAssociationEngineInfo();

                        if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                            thisObjectsProperty = getResolvedEvents();
                            parameterObjectsProperty = ((AssociatedEventImpl) (object)).getResolvedEvents();

                            if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {
                                return true;
                            }
                        }
                    }
                }
            }

            return false;
        }
    }

    /**
     *  
     * 
     */
    public String toString() {

        StringBuffer result = new StringBuffer(super.toString());
        result.append(" (associationEngine: ");
        result.append(associationEngine);
        result.append(", resolvedEvents: ");
        result.append(resolvedEvents);
        result.append(", associationEngineInfo: ");
        result.append(associationEngineInfo);
        result.append(')');
        return result.toString();
    }

    /**
     * Checks whether the contents of the instance conform to the Common Base Event 
     * specification as described in "Canonical Situation Data Format: The Common Base Event 
     * V1.0.1".
     * <p>
     * 
     * @throws ValidationException If the event instance does not conform to the Common Base Event specification.
     */
    public void validate() throws ValidationException {

        synchronized (this) {

            //Step 1:  Check for required properties:
            if(resolvedEvents == null){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.AssociatedEvent.ResolvedEvents")); 
            }
            
            //Step 2:  Check for mutually exclusive properties:
            if(((associationEngine == null) && (associationEngineInfo == null)) || ((associationEngine != null) && (associationEngineInfo != null))){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MUTUALLY_EXCLUSIVE_ATTRIBUTES_EXC_", "CommonBaseEvent.AssociatedEvent.AssociationEngine", "CommonBaseEvent.AssociatedEvent.AssociationEngineInfo"));                 
            }

            //Step 3:  Check content of each property:
            String[] resolvedEventsArray = getResolvedEventsAsArray();
            
            for (int counter = 0; counter < resolvedEventsArray.length; counter++) {
                    
                if((resolvedEventsArray[counter].length() < 32) || (resolvedEventsArray[counter].length() > 64)){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_LENGTH_EXC_","CommonBaseEvent.AssociatedEvent.ResolvedEvents[".concat(String.valueOf(counter + 1)).concat("]"), resolvedEventsArray[counter]));
                }
                else if(!Character.isLetter(InternationalizationUtilities.charAt(resolvedEventsArray[counter],0))){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_EXC_","CommonBaseEvent.AssociatedEvent.ResolvedEvents[".concat(String.valueOf(counter + 1)).concat("]"), resolvedEventsArray[counter]));
                }
            }
            
            if(associationEngineInfo != null){
                associationEngineInfo.validate();
            }
            else {
                
                if((associationEngine.length() < 32) || (associationEngine.length() > 64)){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_LENGTH_EXC_","CommonBaseEvent.AssociatedEvent.AssociationEngine", associationEngine));
                }
                else if(!Character.isLetter(InternationalizationUtilities.charAt(associationEngine,0))){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_EXC_","CommonBaseEvent.AssociatedEvent.AssociationEngine", associationEngine));
                }
            }
        }
    }

    /**
     * Return a cloned (copy by value) object of this object. 
     * <p>
     * This is a deep copy version, in which all the objects 
     * within this object will be copied by value.
     * <p>
     * 
     * @return Object The cloned (copy by value) object of this object.
     * @throws CloneNotSupportedException If a instance variable of this object does not implement the <code>Cloneable</code> interface.   
     */
    public Object clone() throws CloneNotSupportedException {

        synchronized (this) {

            AssociatedEventImpl copy = ((AssociatedEventImpl) (super.clone()));

            if(this.associationEngine != null){
                copy.setAssociationEngine(new String(this.associationEngine));                                
            }

            if(this.associationEngineInfo != null){

                try {
                    copy.setAssociationEngineInfo(((AssociationEngine) ((AssociationEngineImpl) (this.associationEngineInfo)).clone()));                
                } 
                catch (ClassCastException c) {
                    //Ignore since cloning is supported.
                }
            }
            
            if(this.resolvedEvents != null){
                copy.setResolvedEvents(new String(this.resolvedEvents));                                
            }

            return copy;
        }
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * All components are initialized to either zero or null.
     * 
     * @since 1.0
     */
    public void init() {

        setAssociationEngine(null);
        setAssociationEngineInfo(null);
        setResolvedEvents(null);
    }
}