package org.eclipse.tptp.logging.events.cbe.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;
import org.eclipse.hyades.logging.core.LoggingCoreResourceBundle;
import org.eclipse.hyades.logging.core.LoggingCoreUtilities;
import org.eclipse.tptp.logging.events.cbe.AssociatedEvent;
import org.eclipse.tptp.logging.events.cbe.CommonBaseEvent;
import org.eclipse.tptp.logging.events.cbe.CompletionException;
import org.eclipse.tptp.logging.events.cbe.ComponentIdentification;
import org.eclipse.tptp.logging.events.cbe.ContentHandler;
import org.eclipse.tptp.logging.events.cbe.ContextDataElement;
import org.eclipse.tptp.logging.events.cbe.EventFactory;
import org.eclipse.tptp.logging.events.cbe.ExtendedDataElement;
import org.eclipse.tptp.logging.events.cbe.MissingPropertyException;
import org.eclipse.tptp.logging.events.cbe.MsgCatalogToken;
import org.eclipse.tptp.logging.events.cbe.MsgDataElement;
import org.eclipse.tptp.logging.events.cbe.Situation;
import org.eclipse.tptp.logging.events.cbe.ValidationException;
import org.eclipse.tptp.logging.events.cbe.util.EventFormatter;
import org.eclipse.tptp.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005, 2006, 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonBaseEventImpl.java,v 1.3 2008/01/24 02:29:31 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Class representing an implementation of an <code>CommonBaseEvent</code>.
 * <p>
 * The following description is from the Common Base Event v1.0.1 specification
 * entitled "Canonical Situation Data Format: The Common Base Event V1.0.1":
 * <p>
 * The purpose of the Common Base Event is to facilitate the effective
 * intercommunication among disparate enterprise components that support
 * logging, management, problem determination, autonomic computing and
 * e-business functions in an enterprise.
 * <p>
 * Common Base Event definition ensures completeness of the data by providing
 * properties to publish the following information:
 * <p>
 * <ol>
 * <li>The identification of the component that is reporting the situation
 * </li>
 * <li>The identification of the component that is affected by the situation
 * (which may be the same as the component that is reporting the situation)
 * </li>
 * <li>The situation itself</li>
 * </ol>
 * <p>
 * 
 * @author cindyjin
 * @author Paul E. Slauenwhite
 * @version March 30, 2006
 * @since 1.0.1
 */
public class CommonBaseEventImpl implements CommonBaseEvent {

    /** Automatically generated javadoc for: serialVersionUID */
    protected static final long serialVersionUID = -7210409289362234835L;

    /**
     * Mediation handler associated to this factory.
     */
    protected transient ContentHandler contentHandler = null;

    /**
     * Flag to denote if the associated content handler has been previously been
     * invoked.
     */
    protected transient boolean isComplete = false;

    /**
     * The cached value of the '
     * {@link #getCreationTime() <em>Creation Time</em>}' attribute. <!--
     * begin-user-doc --> 
     * 
     * @see #getCreationTime()
     */
    protected String creationTime = null;

    /**
     * This is true if the Creation Time attribute has been set. <!--
     * begin-user-doc --> 
     * 
     * @deprecated
     */
    protected boolean creationTimeESet = false;

    /**
     * The cached value of the '
     * {@link #getGlobalInstanceId() <em>Global Instance Id</em>}' attribute.
     *  
     * 
     * @see #getGlobalInstanceId()
     */
    protected String globalInstanceId = null;

    /**
     * The cached value of the '{@link #getElapsedTime() <em>Elapsed Time</em>}'
     * attribute.  
     * 
     * @see #getElapsedTime()
     */
    protected long elapsedTime = 0L;

    /**
     * This is true if the Elapsed Time attribute has been set. <!--
     * begin-user-doc --> 
     * 
     */
    protected boolean elapsedTimeESet = false;

    /**
     * The cached value of the '
     * {@link #getExtensionName() <em>Extension Name</em>}' attribute. <!--
     * begin-user-doc --> 
     * 
     * @see #getExtensionName()
     */
    protected String extensionName = null;

    /**
     * The cached value of the '
     * {@link #getLocalInstanceId() <em>Local Instance Id</em>}' attribute.
     *  
     * 
     * @see #getLocalInstanceId()
     */
    protected String localInstanceId = null;

    /**
     * The cached value of the '{@link #getMsg() <em>Msg</em>}' attribute.
     *  
     * 
     * @see #getMsg()
     */
    protected String msg = null;

    /**
     * The cached value of the '{@link #getPriority() <em>Priority</em>}'
     * attribute.  
     * 
     * @see #getPriority()
     */
    protected short priority = 0;

    /**
     * This is true if the Priority attribute has been set. <!-- begin-user-doc
     * --> 
     * 
     */
    protected boolean priorityESet = false;

    /**
     * The cached value of the '{@link #getRepeatCount() <em>Repeat Count</em>}'
     * attribute.  
     * 
     * @see #getRepeatCount()
     */
    protected short repeatCount = 0;

    /**
     * This is true if the Repeat Count attribute has been set. <!--
     * begin-user-doc --> 
     * 
     */
    protected boolean repeatCountESet = false;

    /**
     * The cached value of the '
     * {@link #getSequenceNumber() <em>Sequence Number</em>}' attribute. <!--
     * begin-user-doc --> 
     * 
     * @see #getSequenceNumber()
     */
    protected long sequenceNumber = 0L;

    /**
     * This is true if the Sequence Number attribute has been set. <!--
     * begin-user-doc --> 
     * 
     */
    protected boolean sequenceNumberESet = false;

    /**
     * The cached value of the '{@link #getSeverity() <em>Severity</em>}'
     * attribute.  
     * 
     * @see #getSeverity()
     */
    protected short severity = 0;

    /**
     * This is true if the Severity attribute has been set. <!-- begin-user-doc
     * --> 
     * 
     */
    protected boolean severityESet = false;

    /**
     * The cached value of the '{@link #getVersion() <em>Version</em>}'
     * attribute.  
     * 
     * @see #getVersion()
     */
    protected String version = "1.0.1";

    /**
     * The cached value of the '{@link #getAny() <em>Any</em>}' attribute
     * list.  
     * 
     * @see #getAny()
     */
    protected List any = null;

    /**
     * The cached value of the '{@link #getSituation() <em>Situation</em>}'
     * containment reference.  
     * 
     * @see #getSituation()
     */
    protected Situation situation = null;

    /**
     * The cached value of the '
     * {@link #getAssociatedEvents() <em>Associated Events</em>}' containment
     * reference list.  
     * 
     * @see #getAssociatedEvents()
     */
    protected List associatedEvents = null;

    /**
     * The cached value of the '
     * {@link #getContextDataElements() <em>Context Data Elements</em>}'
     * containment reference list.  
     * 
     * @see #getContextDataElements()
     */
    protected List contextDataElements = null;

    /**
     * The cached value of the '
     * {@link #getExtendedDataElements() <em>Extended Data Elements</em>}'
     * containment reference list.  
     * 
     * @see #getExtendedDataElements()
     */
    protected List extendedDataElements = null;

    /**
     * The cached value of the '
     * {@link #getMsgDataElement() <em>Msg Data Element</em>}' containment
     * reference.  
     * 
     * @see #getMsgDataElement()
     */
    protected MsgDataElement msgDataElement = null;

    /**
     * The cached value of the '
     * {@link #getReporterComponentId() <em>Reporter Component Id</em>}'
     * containment reference.  
     * 
     * @see #getReporterComponentId()
     */
    protected ComponentIdentification reporterComponentId = null;

    /**
     * The cached value of the '
     * {@link #getSourceComponentId() <em>Source Component Id</em>}'
     * containment reference.  
     * 
     * @see #getSourceComponentId()
     */
    protected ComponentIdentification sourceComponentId = null;

    /**
     *  
     * 
     */
    protected CommonBaseEventImpl() {
        super();
    }

    /**
     *  
     * 
     */
    public List getAny() {
        
        if (any == null) {
            any = new ArrayList();
        }
        return any;
    }

    /**
     *  
     * 
     */
    public List getAssociatedEvents() {
        if (associatedEvents == null) {
            associatedEvents = new ArrayList();
        }
        return associatedEvents;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setContentHandler(org.eclipse.tptp.logging.events.cbe.ContentHandler)
     */
    public void setContentHandler(ContentHandler contentHandler) {
        this.contentHandler = contentHandler;
        isComplete = false;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#getContentHandler()
     */
    public ContentHandler getContentHandler() {
        return contentHandler;
    }

    /**
     *  
     * 
     */
    public List getContextDataElements() {
        if (contextDataElements == null) {
            contextDataElements = new ArrayList();
        }
        return contextDataElements;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#getContextDataElements(String)
     */
    public List getContextDataElements(String name) {

        ListIterator it = getContextDataElements().listIterator();
        ArrayList res = new ArrayList();
        ContextDataElement cde = null;

        while (it.hasNext()) {

            cde = ((ContextDataElement) (it.next()));

            if ((cde != null) && (cde.getName().equals(name))) {
                res.add(cde);
            }
        }

        return res;
    }

    /**
     *  
     * 
     */
    public void setCreationTime(String newCreationTime) {
        creationTime = newCreationTime;
        creationTimeESet = true;
    }

    /**
     *  
     * 
     */
    public String getCreationTime() {
        return creationTime;
    }

    /**
     * Sets the XSD <code>dateTime</code> '<em><b>Creation Time</b></em>' attribute 
     * as a long (milliseconds) time stamp.
     * <p>
     * See {@link EventHelpers#dateToLong(String)} for more information 
     * on the long (milliseconds) and XSD <code>dateTime</code> time stamp
     * formats and the conversion from a long time stamp to an XSD 
     * <code>dateTime</code> time stamp.
     * <p>
     * 
     * @param inDateTime The long (milliseconds) time stamp to be converted to the XSD <code>dateTime</code> '<em><b>Creation Time</b></em>' attribute.
     * @see org.eclipse.hyades.logging.events.cbe.CommonBaseEvent#setCreationTimeAsLong(long)
     * @see EventHelpers#dateToLong(String)
     */
    public void setCreationTimeAsLong(long inDateTime) {
        setCreationTime(LoggingCoreUtilities.convertMillisecondsToXsdDateTime(inDateTime));
    }

    /**
     * Resolves the XSD <code>dateTime</code> '<em><b>Creation Time</b></em>' attribute 
     * as a long (milliseconds) time stamp.
     * <p>
     * See {@link EventHelpers#dateToLong(String)} for more information 
     * on the XSD <code>dateTime</code> and long (milliseconds) time stamp
     * formats and the conversion from an XSD <code>dateTime</code> time 
     * stamp to a long time stamp.
     * <p>
     * 
     * @return The long (milliseconds) time stamp representation of the XSD <code>dateTime</code> '<em><b>Creation Time</b></em>' attribute.
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#getCreationTimeAsLong()
     * @see EventHelpers#dateToLong(String)
     */
    public long getCreationTimeAsLong() {

        long result = 0;

        if (isSetCreationTime()) {

            try {
                result = LoggingCoreUtilities.convertXsdDateTimeToMilliseconds(getCreationTime());
            } catch (IllegalArgumentException e) {
                throw new IllegalStateException(getCreationTime() + " is not a valid xsd:dateTime string");
            }
        }

        return result;
    }

    /**
     *  
     * 
     */
    public void setElapsedTime(long newElapsedTime) {
        elapsedTime = newElapsedTime;
        elapsedTimeESet = true;
    }
    
    /**
     *  
     * 
     */
    public long getElapsedTime() {
        return elapsedTime;
    }

    /**
     *  
     * 
     */
    public List getExtendedDataElements() {
        if (extendedDataElements == null) {
            extendedDataElements = new ArrayList();
        }
        return extendedDataElements;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#getExtendedDataElements(String)
     */
    public List getExtendedDataElements(String name) {

        ListIterator it = getExtendedDataElements().listIterator();
        ArrayList res = new ArrayList();
        ExtendedDataElement ede = null;

        while (it.hasNext()) {

            ede = ((ExtendedDataElement) (it.next()));

            if ((ede != null) && (ede.getName().equals(name))) {
                res.add(ede);
            }
        }

        return res;
    }

    /**
     *  
     * 
     */
    public void setExtensionName(String newExtensionName) {
        extensionName = newExtensionName;
    }

    /**
     *  
     * 
     */
    public String getExtensionName() {
        return extensionName;
    }

    /**
     *  
     * 
     */
    public void setGlobalInstanceId(String newGlobalInstanceId) {
        globalInstanceId = newGlobalInstanceId;
    }

    /**
     *  
     * 
     */
    public String getGlobalInstanceId() {
        return globalInstanceId;
    }

    /**
     *  
     * 
     */
    public void setLocalInstanceId(String newLocalInstanceId) {
        localInstanceId = newLocalInstanceId;
    }

    /**
     *  
     * 
     */
    public String getLocalInstanceId() {
        return localInstanceId;
    }

    /**
     *  
     * 
     */
    public void setMsg(String newMsg) {
        msg = newMsg;
    }

    /**
     *  
     * 
     */
    public String getMsg() {
        return msg;
    }

    /**
     *  
     * 
     */
    public void setMsgDataElement(MsgDataElement newMsgDataElement) {
        msgDataElement = newMsgDataElement;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setMsgDataElement(String,
     *      String, MsgCatalogToken[], String, String, String, String)
     */
    public MsgDataElement setMsgDataElement(String msgCatalog, String msgCatalogId, MsgCatalogToken[] tokens, String msgCatalogType, String msgId, String msgIdType, String msgCatalogLocale) {
        MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog(msgCatalog);
        mde.setMsgCatalogId(msgCatalogId);
        mde.setMsgCatalogTokens(tokens);
        mde.setMsgCatalogType(msgCatalogType);
        mde.setMsgId(msgId);
        mde.setMsgIdType(msgIdType);
        mde.setMsgLocale(msgCatalogLocale);
        setMsgDataElement(mde);

        return mde;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setMsgDataElement(String,
     *      String, String[], String, String, String, String)
     */
    public MsgDataElement setMsgDataElement(String msgCatalog, String msgCatalogId, String[] tokens, String msgCatalogType, String msgId, String msgIdType, String msgCatalogLocale) {
        MsgDataElement mde = EventFactory.eINSTANCE.createMsgDataElement();
        mde.setMsgCatalog(msgCatalog);
        mde.setMsgCatalogId(msgCatalogId);
        mde.setMsgCatalogTokensAsStrings(tokens);
        mde.setMsgCatalogType(msgCatalogType);
        mde.setMsgId(msgId);
        mde.setMsgIdType(msgIdType);
        mde.setMsgLocale(msgCatalogLocale);
        setMsgDataElement(mde);

        return mde;
    }

    /**
     *  
     * 
     */
    public MsgDataElement getMsgDataElement() {
        return msgDataElement;
    }

    /**
     *  
     * 
     */
    public void setPriority(short newPriority) {
        priority = newPriority;
        priorityESet = true;
    }

    /**
     *  
     * 
     */
    public short getPriority() {
        return priority;
    }

    /**
     *  
     * 
     */
    public void setRepeatCount(short newRepeatCount) {
        repeatCount = newRepeatCount;
        repeatCountESet = true;
    }

    /**
     *  
     * 
     */
    public short getRepeatCount() {
        return repeatCount;
    }

    /**
     *  
     * 
     */
    public void setReporterComponentId(ComponentIdentification newReporterComponentId) {
        reporterComponentId = newReporterComponentId;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setReporterComponentId(String,
     *      String, String, String, String, String, String, String, String,
     *      String,String)
     */
    public void setReporterComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String executionEnvironment, String instanceId, String location, String locationType, String processId, String threadId) {
        ComponentIdentification cid = EventFactory.eINSTANCE.createComponentIdentification();

        cid.setApplication(application);
        cid.setComponent(component);
        cid.setComponentType(componentType);
        cid.setComponentIdType(componentIdType);
        cid.setExecutionEnvironment(executionEnvironment);
        cid.setInstanceId(instanceId);
        cid.setLocation(location);
        cid.setLocationType(locationType);
        cid.setProcessId(processId);
        cid.setApplication(application);
        cid.setSubComponent(subComponent);
        cid.setThreadId(threadId);

        setReporterComponentId(cid);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setReporterComponentId(String,
     *      String, String, String, String, String, String)
     */
    public void setReporterComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String location, String locationType) {
        setReporterComponentId(application, component, subComponent, componentType, componentIdType, null, // environment
                null, // instanceId
                location, locationType, null, // process
                null); // threadid
    }

    /**
     *  
     * 
     */
    public ComponentIdentification getReporterComponentId() {
        return reporterComponentId;
    }

    /**
     *  
     * 
     */
    public void setSequenceNumber(long newSequenceNumber) {
        sequenceNumber = newSequenceNumber;
        sequenceNumberESet = true;
    }

    /**
     *  
     * 
     */
    public long getSequenceNumber() {
        return sequenceNumber;
    }

    /**
     *  
     * 
     * @deprecated
     */
    public boolean isSetCreationTime() {
        return creationTimeESet;
    }

    /**
     *  
     * 
     */
    public boolean isSetElapsedTime() {
        return elapsedTimeESet;
    }

    /**
     *  
     * 
     */
    public boolean isSetPriority() {
        return priorityESet;
    }

    /**
     *  
     * 
     */
    public boolean isSetRepeatCount() {
        return repeatCountESet;
    }

    /**
     *  
     * 
     */
    public boolean isSetSequenceNumber() {
        return sequenceNumberESet;
    }

    /**
     *  
     * 
     */
    public boolean isSetSeverity() {
        return severityESet;
    }

    /**
     *  
     * 
     */
    public void setSeverity(short newSeverity) {
        severity = newSeverity;
        severityESet = true;
    }
    
    /**
     *  
     * 
     */
    public short getSeverity() {
        return severity;
    }

    /**
     *  
     * 
     */
    public void setSituation(Situation newSituation) {
        situation = newSituation;
    }

    /**
     *  
     * 
     */
    public Situation getSituation() {
        return situation;
    }

    /**
     *  
     * 
     */
    public void setSourceComponentId(ComponentIdentification newSourceComponentId) {
        sourceComponentId = newSourceComponentId;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setSourceComponentId(String,
     *      String, String, String, String, String, String, String, String,
     *      String,String)
     */
    public void setSourceComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String executionEnvironment, String instanceId, String location, String locationType, String processId, String threadId) {
        ComponentIdentification cid = EventFactory.eINSTANCE.createComponentIdentification();

        cid.setApplication(application);
        cid.setComponent(component);
        cid.setSubComponent(subComponent);
        cid.setComponentType(componentType);
        cid.setComponentIdType(componentIdType);
        cid.setExecutionEnvironment(executionEnvironment);
        cid.setInstanceId(instanceId);
        cid.setLocation(location);
        cid.setLocationType(locationType);
        cid.setProcessId(processId);
        cid.setThreadId(threadId);

        setSourceComponentId(cid);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#setSourceComponentId(String,
     *      String, String, String, String, String,String)
     */
    public void setSourceComponentId(String application, String component, String subComponent, String componentType, String componentIdType, String location, String locationType) {
        setSourceComponentId(application, component, subComponent, componentType, componentIdType, null, // environment
                null, // instanceId
                location, locationType, null, // process
                null); // threadid
    }

    /**
     *  
     * 
     */
    public ComponentIdentification getSourceComponentId() {
        return sourceComponentId;
    }

    /**
     *  
     * 
     */
    public void setVersion(String newVersion) {
        version = newVersion;
    }

    /**
     *  
     * 
     */
    public String getVersion() {
        return version;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addAny(String)
     */
    public void addAny(String any) {
        getAny().add(any);
    }

    /**
     * Removes all any strings that this event is referencing
     */
    public void clearAny() {
        getAny().clear();
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addAssociatedEvent(String,
     *      String, String, String[])
     */
    public AssociatedEvent addAssociatedEvent(String associationEngineName, String associationEngineType, String associationEngineId, String[] resolvedEvents) {
        AssociatedEvent ase = EventFactory.eINSTANCE.createAssociatedEvent();
        ase.setAssociationEngineInfo(associationEngineName, associationEngineType, associationEngineId);

        ase.setResolvedEventsAsArray(resolvedEvents);

        getAssociatedEvents().add(ase);

        return ase;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addContextDataElementWithValue(String,
     *      String, String)
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addContextDataElementWithId(String,
     *      String, String)
     */
    protected ContextDataElement addContextDataElement(String contextType, String contextName, String contextValue, String contextId) {
        ContextDataElement cde = EventFactory.eINSTANCE.createContextDataElement();

        if (contextId != null) {
            cde.setContextId(contextId);
        }

        if (contextValue != null) {
            cde.setContextValue(contextValue);
        }

        cde.setName(contextName);
        cde.setType(contextType);
        getContextDataElements().add(cde);

        return cde;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addContextDataElementWithValue(String,
     *      String, String)
     */
    public ContextDataElement addContextDataElementWithValue(String contextType, String contextName, String contextValue) {
        return addContextDataElement(contextType, contextName, contextValue, null);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addContextDataElementWithId(String,
     *      String,String)
     */
    public ContextDataElement addContextDataElementWithId(String contextType, String contextName, String contextId) {
        return addContextDataElement(contextType, contextName, null, contextId);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElement(String,
     *      int, String)
     */
    public ExtendedDataElement addExtendedDataElement(String name, int type, String value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setTypeAsInt(type);
        if (ExtendedDataElement.TYPE_HEX_BINARY_VALUE == type) {
            ede.setHexValue(value);
        } else {
            ede.getValues().add(value);
        }

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElement(String,
     *      int, String[])
     */
    public ExtendedDataElement addExtendedDataElement(String name, int type, String[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setTypeAsInt(type);
        ede.getValues().addAll(Arrays.asList(values));

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElement(String,
     *      String)
     */
    public ExtendedDataElement addExtendedDataElement(String name, String value) {
        return addExtendedDataElement(name, ExtendedDataElement.TYPE_STRING_VALUE, value);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElement(String,
     *      String[])
     */
    public ExtendedDataElement addExtendedDataElement(String name, String[] values) {
        return addExtendedDataElement(name, ExtendedDataElement.TYPE_STRING_ARRAY_VALUE, values);
    }

    /**
     * Adds a new <code>ExtendedDataElement</code> to this object.
     * 
     * @param extendedDataElement
     *            the new extended data element.
     * @return a reference to the extended data element added to this object.
     * @see #getExtendedDataElements()
     */
    public ExtendedDataElement addExtendedDataElement(ExtendedDataElement extendedDataElement) {

        getExtendedDataElements().add(extendedDataElement);

        return extendedDataElement;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElement(String)
     */
    public ExtendedDataElement addExtendedDataElement(String name) {
        ExtendedDataElement ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(name);
        getExtendedDataElements().add(ede);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithByteArrayValue(String,
     *      byte[])
     */
    public ExtendedDataElement addExtendedDataElementWithByteArrayValue(String name, byte[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsByteArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithByteValue(String,
     *      byte)
     */
    public ExtendedDataElement addExtendedDataElementWithByteValue(String name, byte value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsByte(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithNoValue(String)
     */
    public ExtendedDataElement addExtendedDataElementWithNoValue(String name) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setType(ExtendedDataElement.TYPE_NO_VALUE);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithBooleanArrayValue(java.lang.String,
     *      boolean[])
     */
    public ExtendedDataElement addExtendedDataElementWithBooleanArrayValue(String name, boolean[] values) {

        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsBooleanArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithBooleanValue(java.lang.String,
     *      boolean)
     */
    public ExtendedDataElement addExtendedDataElementWithBooleanValue(String name, boolean value) {

        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsBoolean(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithDateArrayValue(String,
     *      String[])
     */
    public ExtendedDataElement addExtendedDataElementWithDateArrayValue(String name, String[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsDateArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithDateAsLongValue(String,
     *      long)
     */
    public ExtendedDataElement addExtendedDataElementWithDateAsLongValue(String name, long value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsDateAsLong(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithDateValue(String,
     *      String)
     */
    public ExtendedDataElement addExtendedDataElementWithDateValue(String name, String value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsDate(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithDatesAsLongValue(String,
     *      long[])
     */
    public ExtendedDataElement addExtendedDataElementWithDatesAsLongValue(String name, long[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsDateAsLongArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithDoubleArrayValue(String,
     *      double[])
     */
    public ExtendedDataElement addExtendedDataElementWithDoubleArrayValue(String name, double[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsDoubleArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithDoubleValue(String,
     *      double)
     */
    public ExtendedDataElement addExtendedDataElementWithDoubleValue(String name, double value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsDouble(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithFloatArrayValue(String,
     *      float[])
     */
    public ExtendedDataElement addExtendedDataElementWithFloatArrayValue(String name, float[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsFloatArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithFloatValue(String,
     *      float)
     */
    public ExtendedDataElement addExtendedDataElementWithFloatValue(String name, float value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsFloat(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithIntArrayValue(String,
     *      int[])
     */
    public ExtendedDataElement addExtendedDataElementWithIntArrayValue(String name, int[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsIntArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithIntValue(String,
     *      int)
     */
    public ExtendedDataElement addExtendedDataElementWithIntValue(String name, int value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsInt(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithLongArrayValue(String,
     *      long[])
     */
    public ExtendedDataElement addExtendedDataElementWithLongArrayValue(String name, long[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsLongArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithLongValue(String,
     *      long)
     */
    public ExtendedDataElement addExtendedDataElementWithLongValue(String name, long value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsLong(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithShortArrayValue(String,
     *      short[])
     */
    public ExtendedDataElement addExtendedDataElementWithShortArrayValue(String name, short[] values) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsShortArray(values);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithShortValue(String,
     *      short)
     */
    public ExtendedDataElement addExtendedDataElementWithShortValue(String name, short value) {
        ExtendedDataElement ede = addExtendedDataElement(name);
        ede.setValuesAsShort(value);

        return ede;
    }

    /*
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithHexValue(java.lang.String,
     *      byte[])
     */
    public ExtendedDataElement addExtendedDataElementWithHexValue(String name, byte[] hexValue) {
        ExtendedDataElement result = EventFactory.eINSTANCE.createExtendedDataElement();
        result.setType(ExtendedDataElement.TYPE_HEX_BINARY);
        result.setName(name);
        result.setHexValueAsByteArray(hexValue);
        getExtendedDataElements().add(result);
        return result;
    }

    /*
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#addExtendedDataElementWithHexValue(java.lang.String,
     *      java.lang.String)
     */
    public ExtendedDataElement addExtendedDataElementWithHexValue(String name, String hexValue) {
        ExtendedDataElement result = EventFactory.eINSTANCE.createExtendedDataElement();
        result.setType(ExtendedDataElement.TYPE_HEX_BINARY);
        result.setName(name);
        result.setHexValue(hexValue);
        getExtendedDataElements().add(result);
        return result;
    }

    /**
     * Removes all ExtenedDataElements that this event is referencing
     */
    public void clearExtendedDataElements() {
        getExtendedDataElements().clear();
    }

    /**
     * Remove a named extended data element that this event is referencing
     * 
     * @param name
     *            the name of the element to be removed
     */
    public void removeExtendedDataElements(String name) {

        if (getExtendedDataElements() != null) {

            for (int counter = 0; counter < getExtendedDataElements().size(); counter++) {

                if (((ExtendedDataElement) (extendedDataElements.get(counter))).getName().equals(name)) {
                    extendedDataElements.remove(counter--);
                }
            }
        }
    }

    /**
     * Adds a new <code>ContextDataElement</code> to this object.
     * 
     * @param contextDataElement
     *            the new context data element.
     * @return a reference to the context data element added to this object.
     * @see #getContextDataElements()
     */
    public ContextDataElement addContextDataElement(ContextDataElement contextDataElement) {

        getContextDataElements().add(contextDataElement);

        return contextDataElement;
    }

    /**
     * Removes all ExtenedDataElements that this event is referencing
     */
    public void clearContextDataElements() {
        getContextDataElements().clear();
    }

    /**
     * Remove a named context data element that this event is referencing
     * 
     * @param name
     *            the name of the element to be removed
     */
    public void removeContextDataElements(String name) {

        if (getContextDataElements() != null) {

            for (int counter = 0; counter < getContextDataElements().size(); counter++) {

                if (((ContextDataElement) (getContextDataElements().get(counter))).getName().equals(name)) {
                    getContextDataElements().remove(counter--);
                }
            }
        }
    }

    /**
     * Adds a new <code>AssociatedEvent</code> to this object.
     * 
     * @param associatedEvent
     *            the new context data element.
     * @return a reference to the context data element added to this object.
     * @see #getAssociatedEvents()
     */
    public AssociatedEvent addAssociatedEvent(AssociatedEvent associatedEvent) {

        getAssociatedEvents().add(associatedEvent);

        return associatedEvent;
    }

    /**
     * Removes all ExtenedDataElements that this event is referencing
     */
    public void clearAssociatedEvents() {
        getAssociatedEvents().clear();
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#complete
     */
    public void complete() throws CompletionException {

        if (contentHandler != null) {

            synchronized (this) {
                contentHandler.completeEvent(this);
            }

            isComplete = true;
        }
    }

    /*
     * @see org.eclipse.tptp.logging.events.cbe.CommonBaseEvent#isComplete()
     */
    public boolean isComplete() {
        return isComplete;
    }

    /**
     * Overrides the <code>java.lang.Object</code>'s equals() API to 
     * determine if the parameter <code>object</code> is equal to this object.
     * <p>
     * 
     * @param object The <code>java.lang.Object</code> to be compared to this object.
     * @return True, if this object is the same as the parameter <code>object</code>, otherwise false.
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {

        synchronized (this) {
                
            //Check if the parameter object is the same object reference as this object (e.g. equal):
            if (this == object){
                return true;
            }
            else {

                //Check if the parameter object is null and is the a different type as this object(e.g. not equal):
                if ((object != null) && (getClass() == object.getClass())) {

                    //Compare the value(s) of all object properties:
                    Object thisObjectsProperty = getGlobalInstanceId();
                    Object parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getGlobalInstanceId();

                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                        thisObjectsProperty = getCreationTime();
                        parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getCreationTime();

                        if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {
                        
                            thisObjectsProperty = getLocalInstanceId();
                            parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getLocalInstanceId();

                            if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                                thisObjectsProperty = getVersion();
                                parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getVersion();

                                if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                                    thisObjectsProperty = getExtensionName();
                                    parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getExtensionName();

                                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {
                                        
                                        thisObjectsProperty = getSituation();
                                        parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getSituation();

                                        if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {
                                            
                                            thisObjectsProperty = getMsg();
                                            parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getMsg();

                                            if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {
                                                
                                                thisObjectsProperty = getMsgDataElement();
                                                parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getMsgDataElement();

                                                if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                                                    thisObjectsProperty = getReporterComponentId();
                                                    parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getReporterComponentId();

                                                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                                                        thisObjectsProperty = getSourceComponentId();
                                                        parameterObjectsProperty = ((CommonBaseEventImpl) (object)).getSourceComponentId();

                                                        if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                                                            List thisObjectsListProperty = getContextDataElements();
                                                            List parameterObjectsListProperty = ((CommonBaseEventImpl) (object)).getContextDataElements();

                                                            if (((thisObjectsListProperty != null) && (EventHelpers.compareUnorderedLists(thisObjectsListProperty, parameterObjectsListProperty))) || (((thisObjectsListProperty == null) || (thisObjectsListProperty.size() == 0)) && ((parameterObjectsListProperty == null) || (parameterObjectsListProperty.size() == 0)))) {

                                                                thisObjectsListProperty = getExtendedDataElements();
                                                                parameterObjectsListProperty = ((CommonBaseEventImpl) (object)).getExtendedDataElements();

                                                                if (((thisObjectsListProperty != null) && (EventHelpers.compareUnorderedLists(thisObjectsListProperty, parameterObjectsListProperty))) || (((thisObjectsListProperty == null) || (thisObjectsListProperty.size() == 0)) && ((parameterObjectsListProperty == null) || (parameterObjectsListProperty.size() == 0)))) {

                                                                    thisObjectsListProperty = getAssociatedEvents();
                                                                    parameterObjectsListProperty = ((CommonBaseEventImpl) (object)).getAssociatedEvents();

                                                                    if (((thisObjectsListProperty != null) && (EventHelpers.compareUnorderedLists(thisObjectsListProperty, parameterObjectsListProperty))) || (((thisObjectsListProperty == null) || (thisObjectsListProperty.size() == 0)) && ((parameterObjectsListProperty == null) || (parameterObjectsListProperty.size() == 0)))) {

                                                                        thisObjectsListProperty = getAny();
                                                                        parameterObjectsListProperty = ((CommonBaseEventImpl) (object)).getAny();

                                                                        if (((thisObjectsListProperty != null) && (EventHelpers.compareUnorderedLists(thisObjectsListProperty, parameterObjectsListProperty))) || (((thisObjectsListProperty == null) || (thisObjectsListProperty.size() == 0)) && ((parameterObjectsListProperty == null) || (parameterObjectsListProperty.size() == 0)))) { return (((isSetElapsedTime() == ((CommonBaseEventImpl) (object)).isSetElapsedTime()) && (getElapsedTime() == ((CommonBaseEventImpl) (object)).getElapsedTime())) && ((isSetPriority() == ((CommonBaseEventImpl) (object)).isSetPriority()) && (getPriority() == ((CommonBaseEventImpl) (object)).getPriority())) && ((isSetRepeatCount() == ((CommonBaseEventImpl) (object)).isSetRepeatCount()) && (getRepeatCount() == ((CommonBaseEventImpl) (object)).getRepeatCount())) && ((isSetSequenceNumber() == ((CommonBaseEventImpl) (object)).isSetSequenceNumber()) && (getSequenceNumber() == ((CommonBaseEventImpl) (object)).getSequenceNumber())) && ((isSetSeverity() == ((CommonBaseEventImpl) (object))
                                                                                .isSetSeverity()) && (getSeverity() == ((CommonBaseEventImpl) (object)).getSeverity()))); }
                                                                    }
                                                                }
                                                            }
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return false;
        }
    }

    /**
     *  
     * 
     */
    public String toString() {

        StringBuffer result = new StringBuffer(super.toString());
        result.append(" (creationTime: ");
        if (creationTimeESet)
            result.append(creationTime);
        else
            result.append("<unset>");
        result.append(", globalInstanceId: ");
        result.append(globalInstanceId);
        result.append(", elapsedTime: ");
        if (elapsedTimeESet)
            result.append(elapsedTime);
        else
            result.append("<unset>");
        result.append(", extensionName: ");
        result.append(extensionName);
        result.append(", localInstanceId: ");
        result.append(localInstanceId);
        result.append(", msg: ");
        result.append(msg);
        result.append(", priority: ");
        if (priorityESet)
            result.append(priority);
        else
            result.append("<unset>");
        result.append(", repeatCount: ");
        if (repeatCountESet)
            result.append(repeatCount);
        else
            result.append("<unset>");
        result.append(", sequenceNumber: ");
        if (sequenceNumberESet)
            result.append(sequenceNumber);
        else
            result.append("<unset>");
        result.append(", severity: ");
        if (severityESet)
            result.append(severity);
        else
            result.append("<unset>");
        result.append(", version: ");
        result.append(version);
        result.append(", any: ");
        result.append(any);
        result.append(", situation: ");
        result.append(situation);
        result.append(", associatedEvents: ");
        result.append(associatedEvents);
        result.append(", contextDataElements: ");
        result.append(contextDataElements);
        result.append(", extendedDataElements: ");
        result.append(extendedDataElements);
        result.append(", msgDataElement: ");
        result.append(msgDataElement);
        result.append(", reporterComponentId: ");
        result.append(reporterComponentId);
        result.append(", sourceComponentId: ");
        result.append(sourceComponentId);
        result.append(')');
        return result.toString();
    }

    /**
     *  
     * 
     * @deprecated
     */
    public void unsetCreationTime() {
        creationTime = null;
        creationTimeESet = false;
    }

    /**
     *  
     * 
     */
    public void unsetElapsedTime() {
        elapsedTime = 0L;
        elapsedTimeESet = false;
    }

    /**
     *  
     * 
     */
    public void unsetPriority() {
        priority = 0;
        priorityESet = false;
    }

    /**
     *  
     * 
     */
    public void unsetRepeatCount() {
        repeatCount = 0;
        repeatCountESet = false;
    }

    /**
     *  
     * 
     */
    public void unsetSequenceNumber() {
        sequenceNumber = 0L;
        sequenceNumberESet = false;
    }

    /**
     *  
     * 
     */
    public void unsetSeverity() {
        severity = 0;
        severityESet = false;
    }

    /**
     * Checks whether the contents of the instance conform to the Common Base Event 
     * specification as described in "Canonical Situation Data Format: The Common Base Event 
     * V1.0.1".
     * <p>
     * 
     * @throws ValidationException If the event instance does not conform to the Common Base Event specification.
     */
    public void validate() throws ValidationException {

        synchronized (this) {

            //Step 1:  Check for required properties:
            if(sourceComponentId == null){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.SourceComponentId")); 
            }

            if(situation == null){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.Situation")); 
            }

            if(creationTime == null){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.CreationTime")); 
            }

            //Step 2:  Check for mutually inclusive properties:
            if ((repeatCount > 0) && (!isSetElapsedTime())) { 
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.ElapsedTime")); 
            }

            //Step 3:  Check content of each property:
            if((version != null) && (!version.equals("1.0.1"))){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_VERSION_EXC_", "CommonBaseEvent.Version", version));
            }
            
            if((localInstanceId != null) && (localInstanceId.length() > 128)){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LENGTH_EXC_", "CommonBaseEvent.LocalInstanceId", localInstanceId, new Integer(localInstanceId.length()), new Integer(128)));
            }

            if((extensionName != null) && (extensionName.length() > 64)){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LENGTH_EXC_", "CommonBaseEvent.ExtensionName", extensionName, new Integer(extensionName.length()), new Integer(64)));
            }

            if((msg != null) && (msg.length() > 1024)){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LENGTH_EXC_", "CommonBaseEvent.Msg", msg, new Integer(msg.length()), new Integer(1024)));
            }

            if (globalInstanceId != null){

                if((globalInstanceId.length() < 32) || (globalInstanceId.length() > 64)){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_LENGTH_EXC_","CommonBaseEvent.GlobalInstanceId", globalInstanceId));
                }
                else if(!Character.isLetter(InternationalizationUtilities.charAt(globalInstanceId,0))){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_GLOBAL_INSTANCE_ID_EXC_","CommonBaseEvent.GlobalInstanceId", globalInstanceId));
                }
            }

            try {
                
                if(LoggingCoreUtilities.convertXsdDateTimeToMilliseconds(creationTime) == 0){
                    throw new IllegalArgumentException();
                }
            } 
            catch (IllegalArgumentException ex) {
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_CREATION_TIME_EXC_", "CommonBaseEvent.CreationTime", creationTime));            
            }

            if((severityESet) && ((severity < 0) || (severity > 70))){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_VALUE_OUT_OF_RANGE_EXC_", new Short(severity), "CommonBaseEvent.Severity", new Short((short)0), new Short((short)70))); 
            }

            if((priorityESet) && ((priority < 0) || (priority > 100))){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_VALUE_OUT_OF_RANGE_EXC_", new Short(priority), "CommonBaseEvent.Priority", new Short((short)0), new Short((short)100))); 
            }

            if((repeatCountESet) && (repeatCount < 0)){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_VALUE_OUT_OF_RANGE_EXC_", new Short(repeatCount), "CommonBaseEvent.RepeatCount", new Short((short)0), new Short(Short.MAX_VALUE))); 
            }

            if((elapsedTimeESet) && (elapsedTime < 0)){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_VALUE_OUT_OF_RANGE_EXC_", new Long(elapsedTime), "CommonBaseEvent.ElapsedTime", new Long((long)0), new Long(Long.MAX_VALUE))); 
            }

            if((sequenceNumberESet) && (sequenceNumber < 0)){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_VALUE_OUT_OF_RANGE_EXC_", new Long(sequenceNumber), "CommonBaseEvent.SequenceNumber", new Long((long)0), new Long(Long.MAX_VALUE))); 
            }

            if(reporterComponentId != null){
                reporterComponentId.validate();
            }
            
            if(msgDataElement != null){
                msgDataElement.validate();
            }
            
            sourceComponentId.validate();
            
            situation.validate();
            
            if (contextDataElements != null) {

                ListIterator iterator = contextDataElements.listIterator();
                ContextDataElement contextDataElement = null;

                while (iterator.hasNext()) {

                    contextDataElement = ((ContextDataElement) (iterator.next()));

                    try {
                        contextDataElement.validate();
                    } 
                    catch (ValidationException v) {
                        throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LIST_ELEMENT_ATTRIBUTE_EXC_", "CommonBaseEvent.ContextDataElement".concat("[").concat(String.valueOf(iterator.nextIndex())).concat("]")), v);
                    }
                }
            }

            if (associatedEvents != null) {

                ListIterator iterator = associatedEvents.listIterator();
                AssociatedEvent associatedEvent = null;

                while (iterator.hasNext()) {

                    associatedEvent = ((AssociatedEvent) (iterator.next()));

                    try {
                        associatedEvent.validate();
                    } 
                    catch (ValidationException v) {
                        throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LIST_ELEMENT_ATTRIBUTE_EXC_", "CommonBaseEvent.AssociatedEvent".concat("[").concat(String.valueOf(iterator.nextIndex())).concat("]")), v);
                    }
                }
            }

            if (extendedDataElements != null) {

                ListIterator iterator = getExtendedDataElements().listIterator();
                ExtendedDataElement extendedDataElement = null;

                while (iterator.hasNext()) {

                    extendedDataElement = ((ExtendedDataElement) (iterator.next()));

                    try {
                        extendedDataElement.validate();
                    } 
                    catch (ValidationException v) {
                        throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LIST_ELEMENT_ATTRIBUTE_EXC_", "CommonBaseEvent.ExtendedDataElement".concat("[").concat(String.valueOf(iterator.nextIndex())).concat("]")), v);
                    }
                }
            }
        }
    }

    /**
     * @see org.eclipse.hyades.logging.core.IExternalizableToXml#externalizeCanonicalXmlString()
     */
    public String externalizeCanonicalXmlString() {
        return EventFormatter.toCanonicalXMLString(this);
    }

    /**
     * @see org.eclipse.hyades.logging.core.IExternalizableToXml#externalizeCanonicalXmlDocString()
     */
    public String externalizeCanonicalXmlDocString() {
        return EventFormatter.toCanonicalXMLDocString(this);
    }

    /**
     * @see org.eclipse.hyades.logging.core.IInternalizableFromXml#internalizeCanonicalXmlString(java.lang.String)
     */
    public void internalizeCanonicalXmlString(String value) {

        try {

            CommonBaseEvent event = EventFormatter.eventFromCanonicalXML(value);

            if (event != null) {

                TemplateContentHandlerImpl contentHandler = new TemplateContentHandlerImpl();
                contentHandler.setTemplateEvent(event);

                setContentHandler(contentHandler);

                complete();

                setContentHandler(null);
            }
        } catch (Exception e) {
            // Ignore for backwards compatibility.
        }
    }

    /**
     * @see org.eclipse.hyades.logging.core.IInternalizableFromXml#internalizeCanonicalXmlDocString(java.lang.String)
     */
    public void internalizeCanonicalXmlDocString(String value) {

        try {

            CommonBaseEvent[] events = EventFormatter.eventsFromCanonicalXMLDoc(value);

            if ((events != null) && (events[0] != null)) {

                TemplateContentHandlerImpl contentHandler = new TemplateContentHandlerImpl();
                contentHandler.setTemplateEvent(events[0]);

                setContentHandler(contentHandler);

                complete();

                setContentHandler(null);
            }
        } catch (Exception e) {
            // Ignore for backwards compatibility.
        }
    }

    /**
     * Return a cloned (copy by value) object of this object. 
     * <p>
     * This is a deep copy version, in which all the objects 
     * within this object will be copied by value.
     * <p>
     * 
     * @return Object The cloned (copy by value) object of this object.
     * @throws CloneNotSupportedException If a instance variable of this object does not implement the <code>Cloneable</code> interface.   
     */
    public Object clone() throws CloneNotSupportedException {

        synchronized (this) {

            CommonBaseEventImpl copy = ((CommonBaseEventImpl) (super.clone()));

            if(this.creationTime != null){
                copy.setCreationTime(new String(this.creationTime));                                
            }

            if(this.extensionName != null){
                copy.setExtensionName(new String(this.extensionName));                                
            }

            if(this.globalInstanceId != null){
                copy.setGlobalInstanceId(new String(this.globalInstanceId));                                
            }

            if(this.localInstanceId != null){
                copy.setLocalInstanceId(new String(this.localInstanceId));                                
            }

            if(this.msg != null){
                copy.setMsg(new String(this.msg));                                
            }

            if(this.version != null){
                copy.setVersion(new String(this.version));                                
            }

            if (this.contextDataElements != null) {
                
                copy.contextDataElements = new ArrayList();
                
                for (int counter = 0; counter < this.contextDataElements.size(); counter++) {
                    
                    try{
                        copy.addContextDataElement(((ContextDataElement)(((ContextDataElementImpl)(this.contextDataElements.get(counter))).clone())));
                    } 
                    catch (ClassCastException c) {
                        //Ignore since cloning is supported.
                    }
                }
            }

            if (this.extendedDataElements != null) {

                copy.extendedDataElements = new ArrayList();
                
                for (int counter = 0; counter < this.extendedDataElements.size(); counter++) {
                    
                    try{
                        copy.addExtendedDataElement(((ExtendedDataElement)(((ExtendedDataElementImpl)(this.extendedDataElements.get(counter))).clone())));
                    } 
                    catch (ClassCastException c) {
                        //Ignore since cloning is supported.
                    }
                }
            }

            if (this.any != null) {
                
                copy.any = new ArrayList();

                for (int counter = 0; counter < this.any.size(); counter++) {
                    
                    try{
                        copy.addAny(new String((String)(this.any.get(counter))));
                    } 
                    catch (ClassCastException c) {
                        //Ignore since cloning is supported.
                    }
                }
            }

            if (this.associatedEvents != null) {

                copy.associatedEvents = new ArrayList();

                for (int counter = 0; counter < this.associatedEvents.size(); counter++) {
                    
                    try{
                        copy.addAssociatedEvent(((AssociatedEvent)(((AssociatedEventImpl)(this.associatedEvents.get(counter))).clone())));
                    } 
                    catch (ClassCastException c) {
                        //Ignore since cloning is supported.
                    }
                }
            }

            if(this.msgDataElement != null){

                try {
                    copy.setMsgDataElement(((MsgDataElement) ((MsgDataElementImpl) (this.msgDataElement)).clone()));                
                } 
                catch (ClassCastException c) {
                    //Ignore since cloning is supported.
                }
            }

            if(this.situation != null){

                try {
                    copy.setSituation(((Situation) ((SituationImpl) (this.situation)).clone()));                
                } 
                catch (ClassCastException c) {
                    //Ignore since cloning is supported.
                }
            }

            if(this.sourceComponentId != null){

                try {
                    copy.setSourceComponentId(((ComponentIdentification) ((ComponentIdentificationImpl) (this.sourceComponentId)).clone()));                
                } 
                catch (ClassCastException c) {
                    //Ignore since cloning is supported.
                }
            }

            if(this.reporterComponentId != null){

                try {
                    copy.setReporterComponentId(((ComponentIdentification) ((ComponentIdentificationImpl) (this.reporterComponentId)).clone()));                
                } 
                catch (ClassCastException c) {
                    //Ignore since cloning is supported.
                }
            }

            if(this.msgDataElement != null){

                try {
                    copy.setMsgDataElement(((MsgDataElement) ((MsgDataElementImpl) (this.msgDataElement)).clone()));                
                } 
                catch (ClassCastException c) {
                    //Ignore since cloning is supported.
                }
            }

            return copy;
        }
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * <p>
     * 
     * @since 1.0
     */
    public void init() {

        unsetElapsedTime();
        unsetPriority();
        unsetRepeatCount();
        unsetSequenceNumber();
        unsetSeverity();

        setCreationTime(null);
        setExtensionName(null);
        setLocalInstanceId(null);
        setMsg(null);
        setVersion(null);
        setGlobalInstanceId(null);

        clearExtendedDataElements();
        clearContextDataElements();
        clearAssociatedEvents();
        clearAny();

        setSituation(null);
        setMsgDataElement(null);
        setReporterComponentId(null);
        setSourceComponentId(null);

        isComplete = false;
    }
}
