package org.eclipse.tptp.logging.events.cbe.impl;

import java.security.AccessController;
import java.security.PrivilegedExceptionAction;

import org.eclipse.tptp.logging.events.cbe.EventFactoryHome;

/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventFactoryContext.java,v 1.5 2008/01/24 02:29:31 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Generic look-up service for locating <code>EventFactoryHome</code>
 * instances.
 * <p>
 * This class is implemented as a Singleton design pattern.
 * <p>
 * <code>EventFactoryHome</code> instances may be directly retrieved by
 * passing the <code>EventFactoryHome</code>'s fully qualified package and
 * class name as a parameter to the
 * <code>getEventFactoryHome(String eventFactoryHomeType)</code> API.
 * Alternatively, the <code>EventFactoryContext</code> may be configured to
 * retrieve the context-specific <code>EventFactoryHome</code> instance by
 * calling the <code>getEventFactoryHome()</code> API. The
 * <code>EventFactoryContext</code> may be configured using the setter to set
 * its <code>EventFactoryHome</code> type.
 * <p>
 * The <code>EventFactoryContext</code>'s<code>EventFactoryHome</code>
 * type property contains an <code>EventFactoryHome</code>'s fully qualified
 * package and class name.
 * <p>
 * The desired <code>EventFactoryHome</code> must be on the classpath and
 * accessible by this class. Otherwise, the desired
 * <code>EventFactoryHome</code> should be on the classpath and accessible by
 * the user specified class loader. Users may supply a class loader for
 * scenarios where the <code>EventFactoryHome</code> is not on the classpath
 * and accessible by this class. Users may pass a reference to their class
 * loader to allow loading <code>EventFactoryHome</code> implementation
 * outside of the scope of this class' class loader. The
 * <code>EventFactoryHome</code> is attempted to be instantiated using the
 * following class loader hierarchy:
 * <p>
 * <ol>
 * <li>The user specified class loader, if provided.</li>
 * <li>The <code>EventFactoryContext</code> class' class loader.</li>
 * <li>The system's class loader.</li>
 * <li>The current thread's context class loader.</li>
 * </ol>
 * <p>
 * This class also provides a singleton reference to
 * {@link org.eclipse.tptp.logging.events.cbe.impl.SimpleEventFactoryHomeImpl <em>simple</em>}
 * <code>EventFactoryHome</code> as a convenient mechanism to retrieve an
 * <code>EventFactory</code> for template independent event creation. As such,
 * the <code>EventFactory</code> retrieved from this
 * <code>EventFactoryHome</code> has no associated <code>ContentHandler</code>
 * and are used to create events that do not have an associated
 * <code>ContentHandler</code> nor a configuration template.
 * <p>
 * The benefit of this look-up service is that application code does not have to
 * import or invoke typed <code>EventFactoryHome</code> s directly. This
 * look-up service is optional and is not required to create an
 * <code>EventFactoryHome</code>.
 * <p>
 * 
 * @author Paul E. Slauenwhite
 * @version October 27, 2007
 * @since 1.0.1
 * @see org.eclipse.tptp.logging.events.cbe.EventFactoryHome
 */
public class EventFactoryContext {

    protected String eventFactoryHomeType = null;

    protected static EventFactoryContext instance = null;

    protected EventFactoryHome simpleEventFactoryHome = new SimpleEventFactoryHomeImpl();

    /**
     * Privatized constructor since this class is modeled after the Singleton
     * design pattern.
     */
    protected EventFactoryContext() {
    }

    /**
     * Retrieves the context-specific <code>EventFactoryHome</code> instance.
     * <p>
     * The desired <code>EventFactoryHome</code> must be on the classpath and
     * accessible by this class. The <code>EventFactoryHome</code> is
     * attempted to be instantiated using the following class loader hierarchy:
     * <p>
     * <ol>
     * <li>The <code>EventFactoryContext</code> class' class loader.</li>
     * <li>The system's class loader.</li>
     * <li>The current thread's context class loader.</li>
     * </ol>
     * <p>
     * The <code>EventFactoryContext</code> may be configured to retrieve a
     * context-specific <code>EventFactoryHome</code> using the setter to set
     * its <code>EventFactoryHome</code> type.
     * <p>
     * The <code>EventFactoryContext</code>'s<code>EventFactoryHome</code>
     * type property contains an <code>EventFactoryHome</code>'s fully
     * qualified package and class name.
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @return The context-specific <code>EventFactoryHome</code> instance.
     */
    public static EventFactoryContext getInstance() {

        if (instance == null) {
            instance = new EventFactoryContext();
        }

        return instance;
    }

    /**
     * Retrieves the an <code>EventFactoryHome</code> instance based on the
     * parameter <code>EventFactoryHome</code>'s fully qualified package and
     * class name.
     * <p>
     * The desired <code>EventFactoryHome</code> must be on the classpath and
     * accessible by this class. The <code>EventFactoryHome</code> is
     * attempted to be instantiated using the following class loader hierarchy:
     * <p>
     * <ol>
     * <li>The <code>EventFactoryContext</code> class' class loader.</li>
     * <li>The system's class loader.</li>
     * <li>The current thread's context class loader.</li>
     * </ol>
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @param eventFactoryHomeType
     *            An <code>EventFactoryHome</code>'s fully qualified package
     *            and class name
     * @return The <code>EventFactoryHome</code> instance based on the
     *         parameter <code>EventFactoryHome</code>'s fully qualified
     *         package and class name, otherwise <code>null</code>.
     */
    public EventFactoryHome getEventFactoryHome(String eventFactoryHomeType) {
        return (getEventFactoryHome(eventFactoryHomeType, null));
    }

    /**
     * Retrieves the an <code>EventFactoryHome</code> instance based on the
     * parameter <code>EventFactoryHome</code>'s fully qualified package and
     * class name, as loaded by the parameter class loader.
     * <p>
     * This API is used for scenarios where the <code>EventFactoryHome</code>
     * is not on the classpath and accessible by this class. Users may pass a
     * reference to their class loader to allow loading
     * <code>EventFactoryHome</code> implementation outside of the scope of
     * this class' class loader.
     * <p>
     * The desired <code>EventFactoryHome</code> should be on the classpath
     * and accessible by the parameter class loader. Otherwise, the desired
     * <code>EventFactoryHome</code> must be on the classpath and accessible
     * by this class. The <code>EventFactoryHome</code> is attempted to be
     * instantiated using the following class loader hierarchy:
     * <p>
     * <ol>
     * <li>The parameter class loader.</li>
     * <li>The current class' (e.g. <code>EventFactoryContext</code>) class
     * loader.</li>
     * <li>The system's class loader.</li>
     * <li>The current thread's context class loader.</li>
     * </ol>
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @param eventFactoryHomeType
     *            An <code>EventFactoryHome</code>'s fully qualified package
     *            and class name
     * @param classLoader
     *            A class loader used for loading the
     *            <code>EventFactoryHome</code> instance.
     * @return The <code>EventFactoryHome</code> instance based on the
     *         parameter <code>EventFactoryHome</code>'s fully qualified
     *         package and class name, otherwise <code>null</code>.
     */
    public EventFactoryHome getEventFactoryHome(final String eventFactoryHomeType, final ClassLoader classLoader) {

        if ((eventFactoryHomeType != null) && (eventFactoryHomeType.trim().length() > 0)) {

            Class eventFactoryHomeClass = null;

            //Attempt to load the EventFactoryHome class using the parameter class loader:
            if (classLoader != null) {

                try {
                    
                    //Attempt to load the EventFactoryHome class using the parameter class loader and privileged security:
                    eventFactoryHomeClass = ((Class) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                        public Object run() throws Exception {
                            return (classLoader.loadClass(eventFactoryHomeType));
                        }
                    })));
                } 
                catch (Throwable t) {
                	//Ignore since insufficient security privileges for loading the EventFactoryHome class using the parameter class loader and privileged security.
                }
            }

            //If the EventFactoryHome class could not be loaded using the parameter class loader, attempt the class load using the current class' (e.g. EventFactoryContext) class loader:
            if (eventFactoryHomeClass == null) {

                try {

                    //Attempt to load the EventFactoryHome class using the current class' (e.g. EventFactoryContext) class loader and privileged security:
                    eventFactoryHomeClass = ((Class) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                        public Object run() throws Exception {
                            return (org.eclipse.tptp.logging.events.cbe.impl.EventFactoryContext.class.getClassLoader().loadClass(eventFactoryHomeType));
                        }
                    })));
                } 
                catch (Throwable t) {
                	//Ignore since insufficient security privileges for loading the EventFactoryHome class using the current class' (e.g. EventFactoryContext) class loader and privileged security.
                }
                
                //If the EventFactoryHome class could not be loaded using the current class' (e.g. EventFactoryContext) class loader, attempt the class load using the system's class loader:
                if(eventFactoryHomeClass == null){
                    
                	try {

                        //Attempt to load the EventFactoryHome class using the system's class loader and privileged security:
                        eventFactoryHomeClass = ((Class) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                            public Object run() throws Exception {
                                return (ClassLoader.getSystemClassLoader().loadClass(eventFactoryHomeType));
                            }
                        })));
                    } 
                    catch (Throwable t) {
                    	//Ignore since insufficient security privileges for loading the EventFactoryHome class using the system's class loader and privileged security.
                    }
                
                    //If the EventFactoryHome class could not be loaded using the system's class loader, attempt the class load using the current thread's context class loader:
                    if(eventFactoryHomeClass == null){
                        
                    	try {

                            //Attempt to load the EventFactoryHome class using the system's class loader:
                            eventFactoryHomeClass = ((Class) (AccessController.doPrivileged(new PrivilegedExceptionAction() {

                                public Object run() throws Exception {
                                    return (Thread.currentThread().getContextClassLoader().loadClass(eventFactoryHomeType));
                                }
                            })));
                        } 
                        catch (Throwable t) {
                        	//Ignore since insufficient security privileges for loading the EventFactoryHome class using the system's class loader.
                        }
                    }
                }
            }

            if (eventFactoryHomeClass != null) {

                try {

                    // Attempt to create an instance of the class using the
                    // no-argument constructor:
                    return ((EventFactoryHome) (eventFactoryHomeClass.newInstance()));
                } catch (Throwable t) {

                    // Catch all instantiation exceptions.
                    try {

                        // If the no-argument constructor of the class is
                        // privatized, attempt to retrieve the reference
                        // to the singleton instance of the class:
                        return ((EventFactoryHome) (eventFactoryHomeClass.getMethod("getInstance", null).invoke(null, null)));
                    } catch (Throwable tt) {
                        // Ignore since the APIs post condition states a null
                        // return value if the EventFactoryHome instance cannot
                        // be
                        // resolved and/or instantiated.
                    }
                }
            }
        }

        return null;
    }

    /**
     * Retrieves the an <code>EventFactoryHome</code> instance based on the
     * <code>EventFactoryContext</code>'s<code>EventFactoryHome</code>
     * type property as an <code>EventFactoryHome</code>'s fully qualified
     * package and class name.
     * <p>
     * The desired <code>EventFactoryHome</code> must be on the classpath and
     * accessible by this class. The <code>EventFactoryHome</code> is
     * attempted to be instantiated using the following class loader hierarchy:
     * <p>
     * <ol>
     * <li>The <code>EventFactoryContext</code> class' class loader.</li>
     * <li>The system's class loader.</li>
     * <li>The current thread's context class loader.</li>
     * </ol>
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @return The <code>EventFactoryHome</code> instance based on the
     *         <code>EventFactoryContext</code>'s
     *         <code>EventFactoryHome</code> type property as an
     *         <code>EventFactoryHome</code>'s fully qualified package and
     *         class name, otherwise <code>null</code>.
     */
    public EventFactoryHome getEventFactoryHome() {
        return (getEventFactoryHome(eventFactoryHomeType));
    }

    /**
     * Retrieves the an <code>EventFactoryHome</code> instance based on the
     * <code>EventFactoryContext</code>'s<code>EventFactoryHome</code>
     * type property as an <code>EventFactoryHome</code>'s fully qualified
     * package and class name, as loaded by the parameter class loader.
     * <p>
     * This API is used for scenarios where the <code>EventFactoryHome</code>
     * is not on the classpath and accessible by this class. Users may pass a
     * reference to their class loader to allow loading
     * <code>EventFactoryHome</code> implemenation outside of the scope of
     * this class' class loader.
     * <p>
     * The desired <code>EventFactoryHome</code> should be on the classpath
     * and accessible by the parameter class loader. Otherwise, the desired
     * <code>EventFactoryHome</code> must be on the classpath and accessible
     * by this class. The <code>EventFactoryHome</code> is attempted to be
     * instantiated using the following class loader hierarchy:
     * <p>
     * <ol>
     * <li>The parameter class loader.</li>
     * <li>The <code>EventFactoryContext</code> class' class loader.</li>
     * <li>The system's class loader.</li>
     * <li>The current thread's context class loader.</li>
     * </ol>
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @param classLoader
     *            A class loader used for loading the
     *            <code>EventFactoryHome</code> instance.
     * @return The <code>EventFactoryHome</code> instance based on the
     *         <code>EventFactoryContext</code>'s
     *         <code>EventFactoryHome</code> type property as an
     *         <code>EventFactoryHome</code>'s fully qualified package and
     *         class name, otherwise <code>null</code>.
     */
    public EventFactoryHome getEventFactoryHome(ClassLoader classLoader) {
        return (getEventFactoryHome(eventFactoryHomeType, classLoader));
    }

    /**
     * Retrieves a singleton reference to
     * {@link org.eclipse.tptp.logging.events.cbe.impl.SimpleEventFactoryHomeImpl <em>simple</em>}
     * <code>EventFactoryHome</code> as a convenient mechanism to retrieve an
     * <code>EventFactory</code> for template independent event creation.
     * <p>
     * As such, the <code>EventFactory</code> retrieved from this
     * <code>EventFactoryHome</code> has no associated
     * <code>ContentHandler</code> and are used to create events that do not
     * have an associated <code>ContentHandler</code> nor a configuration
     * template.
     * <p>
     * 
     * @return A singleton reference to
     *         {@link org.eclipse.tptp.logging.events.cbe.impl.SimpleEventFactoryHomeImpl <em>simple</em>}
     *         <code>EventFactoryHome</code>, otherwise <code>null</code>.
     * @see org.eclipse.tptp.logging.events.cbe.impl.SimpleEventFactoryHomeImpl
     */
    public EventFactoryHome getSimpleEventFactoryHome() {
        return (simpleEventFactoryHome);
    }

    /**
     * Sets the <code>EventFactoryContext</code>'s
     * <code>EventFactoryHome</code> type property.
     * <p>
     * The <code>EventFactoryContext</code> may be configured to retrieve a
     * context-specific <code>EventFactoryHome</code> using this method to set
     * its <code>EventFactoryHome</code> type.
     * <p>
     * The <code>EventFactoryContext</code>'s<code>EventFactoryHome</code>
     * type property contains an <code>EventFactoryHome</code>'s fully
     * qualified package and class name.
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @param eventFactoryHomeType
     *            The <code>EventFactoryContext</code>'s
     *            <code>EventFactoryHome</code> type property as an
     *            <code>EventFactoryHome</code>'s fully qualified package and
     *            class name.
     */
    public void setEventFactoryHomeType(String eventFactoryHomeType) {
        this.eventFactoryHomeType = eventFactoryHomeType;
    }

    /**
     * Gets the <code>EventFactoryContext</code>'s
     * <code>EventFactoryHome</code> type property.
     * <p>
     * The <code>EventFactoryContext</code> may be configured to retrieve a
     * context-specific <code>EventFactoryHome</code> using this method to set
     * its <code>EventFactoryHome</code> type.
     * <p>
     * The <code>EventFactoryContext</code>'s<code>EventFactoryHome</code>
     * type property contains an <code>EventFactoryHome</code>'s fully
     * qualified package and class name.
     * <p>
     * The benefit of this look-up service is that application code does not
     * have to import or invoke typed <code>EventFactoryHome</code> s
     * directly.
     * <p>
     * 
     * @return The <code>EventFactoryContext</code>'s
     *         <code>EventFactoryHome</code> type property as an
     *         <code>EventFactoryHome</code>'s fully qualified package and
     *         class name.
     */
    public String getEventFactoryHomeType() {
        return eventFactoryHomeType;
    }
}