package org.eclipse.tptp.logging.events.cbe.impl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.ListIterator;

import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;
import org.eclipse.hyades.logging.core.LoggingCoreResourceBundle;
import org.eclipse.hyades.logging.core.LoggingCoreUtilities;
import org.eclipse.tptp.logging.events.cbe.EventFactory;
import org.eclipse.tptp.logging.events.cbe.ExtendedDataElement;
import org.eclipse.tptp.logging.events.cbe.MissingPropertyException;
import org.eclipse.tptp.logging.events.cbe.ValidationException;
import org.eclipse.tptp.logging.events.cbe.util.EventHelpers;

/**********************************************************************
 * Copyright (c) 2005, 2006, 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExtendedDataElementImpl.java,v 1.3 2008/01/24 02:29:31 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Class representing an implementation of an <code>ExtendedDataElement</code>.
 * <p>
 * The following description is from the Common Base Event v1.0.1 specification
 * entitled "Canonical Situation Data Format: The Common Base Event V1.0.1":
 * <p>
 * The ExtendedDataElement allows for application-supplied name-type-value
 * collections to be specified for extensibility purposes. This is the mechanism
 * by which other attributes not specified in the CommonBaseEvent data model can
 * be added. Collections specified here are assumed to be product specific data.
 * <p>
 * The named properties can be filtered, searched or referenced by the
 * correlation rules. The name is user defined, however, the nonexclusive
 * reserved keywords are as follows:
 * <ul>
 * <li>RawData - This keyword is indicative of "as is" data that is meaningful
 * only to the producer of the data (typically proprietary data). It may be in
 * any form, including binary. It is intended to allow the data to be retrieved
 * verbatim and to support tools that understand the format of the context.
 * </li>
 * <li>RootHeader - This keyword is intended to identify the root
 * ExtendedDataElement for a hierarchy of ExtendedDataElements that are defined
 * by dataRefs.</li>
 * <li>LoggingLevel - This keyword is intended to specify the level of logging
 * represeted as an integer. Higher numbers <em>must</em> include all the
 * granularity of lower numbers.</li>
 * </ul>
 * <p>
 * 
 * @author Cindy Jin
 * @author Paul E. Slauenwhite
 * @version March 30, 2007
 * @since 1.0.1
 */
public class ExtendedDataElementImpl implements ExtendedDataElement {

    /** Automatically generated javadoc for: serialVersionUID */
    private static final long serialVersionUID = -6200886839335499445L;

    /**
     * Support for the getters and setters based on hexadecimal
     * values.
     *
     * @see #getHexValueAsByteArray()
     * @see #setHexValueAsByteArray(byte[])
     */
    protected static final char[] HEX_BASE_CHARS = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };

    /**
     * The cached value of the '{@link #getHexValue() <em>Hex Value</em>}'
     * attribute.  
     * 
     * @see #getHexValue()
     */
    protected String hexValue = null;

    /**
     * The cached value of the '{@link #getName() <em>Name</em>}' attribute.
     *  
     * 
     * @see #getName()
     */
    protected String name = null;

    /**
     * The cached value of the '{@link #getType() <em>Type</em>}' attribute.
     *  
     * 
     * @see #getType()
     */
    protected String type = null;

    /**
     * The cached value of the '{@link #getValues() <em>Values</em>}'
     * attribute list.  
     * 
     * @see #getValues()
     */
    protected List values = null;

    /**
     * The cached value of the '{@link #getChildren() <em>Children</em>}'
     * containment reference list.  
     * 
     * @see #getChildren()
     */
    protected List children = null;

    /**
     *  
     * 
     */
    protected ExtendedDataElementImpl() {
        super();
    }

    /**
     *  
     * 
     */
    public List getChildren() {
        if (children == null) {
            children = new ArrayList();
        }
        return children;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getChildren(String)
     */
    public List getChildren(String newName) {

        ListIterator it = getChildren().listIterator();
        ArrayList res = new ArrayList();
        ExtendedDataElement ede = null;

        while (it.hasNext()) {

            ede = ((ExtendedDataElement) (it.next()));

            if ((ede != null) && (ede.getName().equals(newName))) {
                res.add(ede);
            }
        }

        return res;
    }

    /**
     *  
     * 
     */
    public void setHexValue(String newHexValue) {
        hexValue = newHexValue;
    }

    /**
     *  
     * 
     */
    public String getHexValue() {
        return hexValue;
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.ExtendedDataElement#setHexValueAsByteArray(byte[])
     */
    public void setHexValueAsByteArray(byte[] value) {
        setHexValue(byteArrayToHexBinary(value));
        setType(ExtendedDataElement.TYPE_HEX_BINARY);
    }

    /**
     * Sets the hexValue of the extended data element as an array of chars.
     *
     * <P>
     * It also changes the '<em><b>Type</b></em>' attribute to {@link
     * #TYPE_HEX_BINARY}.
     * </p>
     *
     * @param value the new hexValue for the extended data element.
     */
    public void setHexValueAsCharArray(char[] value) {
        setHexValue(new String(value));
        setType(ExtendedDataElement.TYPE_HEX_BINARY);
    }

    /**
     * @see org.eclipse.hyades.logging.events.cbe.ExtendedDataElement#getHexValueAsByteArray()
     */
    public byte[] getHexValueAsByteArray() {
        byte[] result = null;

        if (getType().equals(TYPE_HEX_BINARY)) {
            String localHexValue = getHexValue();

            if (localHexValue != null) {
                try {
                    result = hexBinaryToByteArray(localHexValue);
                }
                catch (IllegalArgumentException ex) {
                    throw new IllegalStateException(localHexValue + " is not a valid xsd:hexBinary value"); /* NOI18N */
                }
            }

            // if hexValue not null
        }
        else {
            throw new IllegalStateException("type is not hexBinary"); //$NON-NLS-1$
        }

        // if type is right
        return result;
    }

    /**
     * <p>
     * Returns the hexValue of the extended data element cast as char array
     * values.
     * </p>
     *
     * @return the hexValue of the extended data element.
     *
     * @throws IllegalStateException if the '<em><b>Type</b></em>' attribute is
     *         not {@link #TYPE_HEX_BINARY}.
     */
    public char[] getHexValueAsCharArray() {

        char[] result = null;

        if (getType().equals(TYPE_HEX_BINARY)) {

            String localHexValue = getHexValue();

            if (localHexValue != null) {
                try {
                    result = localHexValue.toCharArray();
                }
                catch (IllegalArgumentException ex) {
                    throw new IllegalStateException(localHexValue + " is not a valid xsd:hexBinary value"); /* NOI18N */
                }
            }

            // if hexValue not null
        }
        else {
            throw new IllegalStateException("type is not hexBinary"); //$NON-NLS-1$
        }

        // if type is right
        return result;
    }

    /**
     *  
     * 
     */
    public void setName(String newName) {
        name = newName;
    }

    /**
     *  
     * 
     */
    public String getName() {
        return name;
    }

    /**
     *  
     * 
     */
    public void setType(String newType) {
        type = newType;
    }

    /**
     *  
     * 
     */
    public String getType() {
        return type;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setTypeAsInt(int)
     */
    public void setTypeAsInt(int value) {
        switch (value) {
        case TYPE_BYTE_VALUE:
            setType(TYPE_BYTE);

            break;

        case TYPE_SHORT_VALUE:
            setType(TYPE_SHORT);

            break;

        case TYPE_INT_VALUE:
            setType(TYPE_INT);

            break;

        case TYPE_LONG_VALUE:
            setType(TYPE_LONG);

            break;

        case TYPE_FLOAT_VALUE:
            setType(TYPE_FLOAT);

            break;

        case TYPE_DOUBLE_VALUE:
            setType(TYPE_DOUBLE);

            break;

        case TYPE_STRING_VALUE:
            setType(TYPE_STRING);

            break;

        case TYPE_DATE_TIME_VALUE:
            setType(TYPE_DATE_TIME);

            break;

        case TYPE_BOOLEAN_VALUE:
            setType(TYPE_BOOLEAN);

            break;

        case TYPE_BYTE_ARRAY_VALUE:
            setType(TYPE_BYTE_ARRAY);

            break;

        case TYPE_SHORT_ARRAY_VALUE:
            setType(TYPE_SHORT_ARRAY);

            break;

        case TYPE_INT_ARRAY_VALUE:
            setType(TYPE_INT_ARRAY);

            break;

        case TYPE_LONG_ARRAY_VALUE:
            setType(TYPE_LONG_ARRAY);

            break;

        case TYPE_FLOAT_ARRAY_VALUE:
            setType(TYPE_FLOAT_ARRAY);

            break;

        case TYPE_DOUBLE_ARRAY_VALUE:
            setType(TYPE_DOUBLE_ARRAY);

            break;

        case TYPE_STRING_ARRAY_VALUE:
            setType(TYPE_STRING_ARRAY);

            break;

        case TYPE_DATE_TIME_ARRAY_VALUE:
            setType(TYPE_DATE_TIME_ARRAY);

            break;

        case TYPE_BOOLEAN_ARRAY_VALUE:
            setType(TYPE_BOOLEAN_ARRAY);

            break;

        case TYPE_HEX_BINARY_VALUE:
            setType(TYPE_HEX_BINARY);

            break;

        default: // TYPE_NO_VALUE_VALUE or others
            setType(TYPE_NO_VALUE);

            break;
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getTypeAsInt()
     */
    public int getTypeAsInt() {
        String t = getType();

        if (t != null) {
            if (t.equals(TYPE_BYTE)) { return TYPE_BYTE_VALUE; }

            if (t.equals(TYPE_SHORT)) { return TYPE_SHORT_VALUE; }

            if (t.equals(TYPE_INT)) { return TYPE_INT_VALUE; }

            if (t.equals(TYPE_LONG)) { return TYPE_LONG_VALUE; }

            if (t.equals(TYPE_FLOAT)) { return TYPE_FLOAT_VALUE; }

            if (t.equals(TYPE_DOUBLE)) { return TYPE_DOUBLE_VALUE; }

            if (t.equals(TYPE_STRING)) { return TYPE_STRING_VALUE; }

            if (t.equals(TYPE_DATE_TIME)) { return TYPE_DATE_TIME_VALUE; }

            if (t.equals(TYPE_BOOLEAN)) { return TYPE_BOOLEAN_VALUE; }

            if (t.equals(TYPE_BYTE_ARRAY)) { return TYPE_BYTE_ARRAY_VALUE; }

            if (t.equals(TYPE_SHORT_ARRAY)) { return TYPE_SHORT_ARRAY_VALUE; }

            if (t.equals(TYPE_INT_ARRAY)) { return TYPE_INT_ARRAY_VALUE; }

            if (t.equals(TYPE_LONG_ARRAY)) { return TYPE_LONG_ARRAY_VALUE; }

            if (t.equals(TYPE_FLOAT_ARRAY)) { return TYPE_FLOAT_ARRAY_VALUE; }

            if (t.equals(TYPE_DOUBLE_ARRAY)) { return TYPE_DOUBLE_ARRAY_VALUE; }

            if (t.equals(TYPE_STRING_ARRAY)) { return TYPE_STRING_ARRAY_VALUE; }

            if (t.equals(TYPE_DATE_TIME_ARRAY)) { return TYPE_DATE_TIME_ARRAY_VALUE; }

            if (t.equals(TYPE_BOOLEAN_ARRAY)) { return TYPE_BOOLEAN_ARRAY_VALUE; }

            if (t.equals(TYPE_HEX_BINARY)) { return TYPE_HEX_BINARY_VALUE; }
        }

        return TYPE_NO_VALUE_VALUE;
    }

    /**
     *  
     * 
     */
    public List getValues() {
        if (values == null) {
            values = new ArrayList();
        }
        return values;
    }

    /**
     * The array of values for this extended data element as a string
     * representation of the type specified above, excluding hexBinary.
     * hexBinary values MUST be defined using the hexValue property. This is an
     * OPTIONAL property.
     * 
     * @param values
     *            the values to set
     */
    public void setValues(String[] values) {

        clearValues();
        List valuList = getValues();
        valuList.addAll(Arrays.asList(values));
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsBoolean(boolean)
     */
    public void setValuesAsBoolean(boolean value) {
        setValuesAsString(String.valueOf(value));
        setType(TYPE_BOOLEAN);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsBoolean()
     */
    public boolean getValuesAsBoolean() {
        if (getType().equals(TYPE_BOOLEAN) && (getValues().size() == 1)) {
            return (new Boolean((String) getValues().get(0))).booleanValue();
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsBooleanArray(boolean[])
     */
    public void setValuesAsBooleanArray(boolean[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(String.valueOf(value[i]));
        }

        setType(TYPE_BOOLEAN_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsBooleanArray()
     */
    public boolean[] getValuesAsBooleanArray() {
        if (getType().equals(TYPE_BOOLEAN_ARRAY)) {
            List newValues = getValues();
            boolean[] booleanArray = new boolean[newValues.size()];

            for (int i = 0; i < newValues.size(); i++) {
                booleanArray[i] = (new Boolean((String) newValues.get(i))).booleanValue();
            }

            return booleanArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /** 
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsByte(byte)
     */
    public void setValuesAsByte(byte value) {
        setValues(String.valueOf(value));
        setType(TYPE_BYTE);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsByte()
     */
    public byte getValuesAsByte() {
        if (getType().equals(TYPE_BYTE) && (getValues().size() == 1)) {
            return (new Byte((String) getValues().get(0))).byteValue();
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsByteArray(byte[])
     */
    public void setValuesAsByteArray(byte[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(String.valueOf(value[i]));
        }

        setType(TYPE_BYTE_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsByteArray()
     */
    public byte[] getValuesAsByteArray() {
        if (getType().equals(TYPE_BYTE_ARRAY)) {
            List newValues = getValues();
            byte[] byteArray = new byte[newValues.size()];

            for (int i = 0; i < newValues.size(); i++) {
                byteArray[i] = (new Byte((String) newValues.get(i))).byteValue();
            }

            return byteArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsDate(String)
     */
    public void setValuesAsDate(String value) {
        setValues(value);
        setType(TYPE_DATE_TIME);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsDate()
     */
    public String getValuesAsDate() {
        if (getType().equals(TYPE_DATE_TIME) && (getValues().size() != 0)) {
            return (String) getValues().get(0);
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsDateArray(String[])
     */
    public void setValuesAsDateArray(String[] value) {
        setValues(value);
        setType(TYPE_DATE_TIME_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsDateArray()
     */
    public String[] getValuesAsDateArray() {
        if (getType().equals(TYPE_DATE_TIME_ARRAY)) {
            ArrayList valueArray = new ArrayList(getValues());
            String[] retArray = new String[valueArray.size()];

            return (String[]) valueArray.toArray(retArray);
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * Sets the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute 
     * as a long (milliseconds) time stamp and sets the '<em><b>Type</b></em>' 
     * attribute to {@link #TYPE_DATE_TIME}.
     * <p>
     * See {@link EventHelpers#dateToLong(String)} for more information 
     * on the long (milliseconds) and XSD <code>dateTime</code> time stamp
     * formats and the conversion from a long time stamp to an XSD 
     * <code>dateTime</code> time stamp.
     * <p>
     * 
     * @param value The long (milliseconds) time stamp to be converted to the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute.
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsDateAsLong(long)
     * @see EventHelpers#dateToLong(String)
     */
    public void setValuesAsDateAsLong(long value) {
        setValuesAsDate(LoggingCoreUtilities.convertMillisecondsToXsdDateTime(value));
    }

    /**
     * Resolves the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute 
     * as a long (milliseconds) time stamp.
     * <p>
     * See {@link EventHelpers#dateToLong(String)} for more information 
     * on the XSD <code>dateTime</code> and long (milliseconds) time stamp
     * formats and the conversion from an XSD <code>dateTime</code> time 
     * stamp to a long time stamp.
     * <p>
     * 
     * @return The long (milliseconds) time stamp representation of the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute.
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsDateAsLong()
     * @see EventHelpers#dateToLong(String)
     * @throws IllegalStateException If the '<em><b>Type</b></em>' attribute is not {@link #TYPE_DATE_TIME}.
     */
    public long getValuesAsDateAsLong() {
        long result = 0;

        if (getType().equals(TYPE_DATE_TIME)) {
            if (getValues().size() != 0) {
                try {
                    result = LoggingCoreUtilities.convertXsdDateTimeToMilliseconds(getValuesAsDate());
                } catch (IllegalArgumentException e) {
                    throw new IllegalStateException(getValuesAsDate() + " is not a valid xsd:dateTime"); /* NOI18N */
                }
            } else {
                throw new IllegalStateException("There are " + getValues().size() + " elements in the values array"); /* NOI18N *//* NOI18N */
            }
        } else {
            throw new IllegalStateException("Invalid type " + getType()); /* NOI18N */
        }

        return result;
    }

    /**
     * Sets the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute 
     * as an array of long (milliseconds) time stamps and sets the '<em><b>Type</b></em>' 
     * attribute to {@link #TYPE_DATE_TIME_ARRAY}.
     * <p>
     * See {@link EventHelpers#dateToLong(String)} for more information 
     * on the long (milliseconds) and XSD <code>dateTime</code> time stamp
     * formats and the conversion from a long time stamp to an XSD 
     * <code>dateTime</code> time stamp.
     * <p>
     * 
     * @param newValues The array of long (milliseconds) time stamps to be converted to the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute.
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsDateAsLongArray(long[])
     * @see EventHelpers#dateToLong(String)
     */
    public void setValuesAsDateAsLongArray(long[] newValues) {
        String[] dateArray = new String[newValues.length];

        for (int i = 0; i < newValues.length; i++) {
            dateArray[i] = LoggingCoreUtilities.convertMillisecondsToXsdDateTime(newValues[i]);
        }

        setValuesAsDateArray(dateArray);
    }

    /**
     * Resolves the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute 
     * as an array of long (milliseconds) time stamps.
     * <p>
     * See {@link EventHelpers#dateToLong(String)} for more information 
     * on the XSD <code>dateTime</code> and long (milliseconds) time stamp
     * formats and the conversion from an XSD <code>dateTime</code> time 
     * stamp to a long time stamp.
     * <p>
     * 
     * @return The array of long (milliseconds) time stamps representation of the XSD <code>dateTime</code> '<em><b>values</b></em>' attribute.
     * @see org.eclipse.hyades.logging.events.cbe.ExtendedDataElement#getValuesAsDateAsLongArray()
     * @see EventHelpers#dateToLong(String)
     * @throws IllegalStateException If the '<em><b>Type</b></em>' attribute is not {@link #TYPE_DATE_TIME_ARRAY}.
     */
    public long[] getValuesAsDateAsLongArray() {
        String[] dateArray = getValuesAsDateArray();
        long[] longArray = new long[dateArray.length];

        for (int i = 0; i < dateArray.length; i++) {
            try {
                longArray[i] = LoggingCoreUtilities.convertXsdDateTimeToMilliseconds(dateArray[i]);
            } catch (IllegalArgumentException e) {
                throw new IllegalStateException(dateArray[i] + " is not a valid xsd:dateTime value"); /* NOI18N */
            }
        }

        return longArray;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsDouble(double)
     */
    public void setValuesAsDouble(double value) {
        setValues(String.valueOf(value));
        setType(TYPE_DOUBLE);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsDouble()
     */
    public double getValuesAsDouble() {
        if (getType().equals(TYPE_DOUBLE) && (getValues().size() == 1)) {
            return (new Double((String) getValues().get(0))).doubleValue();
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsDoubleArray(double[])
     */
    public void setValuesAsDoubleArray(double[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(String.valueOf(value[i]));
        }

        setType(TYPE_DOUBLE_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsDoubleArray()
     */
    public double[] getValuesAsDoubleArray() {
        if (getType().equals(TYPE_DOUBLE_ARRAY)) {
            List localValues = getValues();
            double[] doubleArray = new double[localValues.size()];

            for (int i = 0; i < localValues.size(); i++) {
                doubleArray[i] = (new Double((String) localValues.get(i))).doubleValue();
            }

            return doubleArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsFloat(float)
     */
    public void setValuesAsFloat(float value) {
        setValues(String.valueOf(value));
        setType(TYPE_FLOAT);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsFloat()
     */
    public float getValuesAsFloat() {
        if (getType().equals(TYPE_FLOAT) && (getValues().size() == 1)) {
            return (new Float((String) getValues().get(0))).floatValue();
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsFloatArray(float[])
     */
    public void setValuesAsFloatArray(float[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(String.valueOf(value[i]));
        }

        setType(TYPE_FLOAT_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsFloatArray()
     */
    public float[] getValuesAsFloatArray() {
        if (getType().equals(TYPE_FLOAT_ARRAY)) {
            List localValues = getValues();
            float[] floatArray = new float[localValues.size()];

            for (int i = 0; i < localValues.size(); i++) {
                floatArray[i] = (new Float((String) localValues.get(i))).floatValue();
            }

            return floatArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsInt(int)
     */
    public void setValuesAsInt(int value) {
        setValues(Integer.toString(value));
        setType(TYPE_INT);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsInt()
     */
    public int getValuesAsInt() {
        if (getType().equals(TYPE_INT) && (getValues().size() == 1)) {
            return Integer.parseInt((String) getValues().get(0));
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsIntArray(int[])
     */
    public void setValuesAsIntArray(int[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(Integer.toString(value[i]));
        }

        setType(TYPE_INT_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsIntArray()
     */
    public int[] getValuesAsIntArray() {
        if (getType().equals(TYPE_INT_ARRAY)) {
            List localValues = getValues();
            int[] intArray = new int[localValues.size()];

            for (int i = 0; i < localValues.size(); i++) {
                intArray[i] = (new Integer((String) localValues.get(i))).intValue();
            }

            return intArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsLong(long)
     */
    public void setValuesAsLong(long value) {
        setValues(String.valueOf(value));
        setType(TYPE_LONG);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsLong()
     */
    public long getValuesAsLong() {
        if (getType().equals(TYPE_LONG) && (getValues().size() == 1)) {
            return (new Long((String) getValues().get(0))).longValue();
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsLongArray(long[])
     */
    public void setValuesAsLongArray(long[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(String.valueOf(value[i]));
        }

        setType(TYPE_LONG_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsLongArray()
     */
    public long[] getValuesAsLongArray() {
        if (getType().equals(TYPE_LONG_ARRAY)) {
            List localValues = getValues();
            long[] longArray = new long[localValues.size()];

            for (int i = 0; i < localValues.size(); i++) {
                longArray[i] = (new Long((String) localValues.get(i))).longValue();
            }

            return longArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsShort(short)
     */
    public void setValuesAsShort(short value) {
        setValues(String.valueOf(value));
        setType(TYPE_SHORT);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsShort()
     */
    public short getValuesAsShort() {
        if (getType().equals(TYPE_SHORT) && (getValues().size() == 1)) {
            return (new Short((String) getValues().get(0))).shortValue();
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsShortArray(short[])
     */
    public void setValuesAsShortArray(short[] value) {

        clearValues();
        List valuList = getValues();

        for (int i = 0; i < value.length; i++) {
            valuList.add(String.valueOf(value[i]));
        }

        setType(TYPE_SHORT_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsShortArray()
     */
    public short[] getValuesAsShortArray() {
        if (getType().equals(TYPE_SHORT_ARRAY)) {
            List localValues = getValues();
            short[] shortArray = new short[localValues.size()];

            for (int i = 0; i < localValues.size(); i++) {
                shortArray[i] = (new Short((String) localValues.get(i))).shortValue();
            }

            return shortArray;
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsString(java.lang.String)
     */
    public void setValuesAsString(String value) {
        setValues(value);
        setType(TYPE_STRING);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsString()
     */
    public String getValuesAsString() {
        if (getType().equals(TYPE_STRING) && (getValues().size() == 1)) {
            return (String) getValues().get(0);
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#setValuesAsStringArray(String[])
     */
    public void setValuesAsStringArray(String[] value) {
        setValues(value);
        setType(TYPE_STRING_ARRAY);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#getValuesAsStringArray()
     */
    public String[] getValuesAsStringArray() {
        if (getType().equals(TYPE_STRING_ARRAY)) {
            ArrayList valueArray = new ArrayList(getValues());
            String[] retArray = new String[valueArray.size()];

            return (String[]) valueArray.toArray(retArray);
        } else {
            throw new IllegalStateException();
        }
    }

    /**
     * Adds a new child <code>ExtendedDataElement</code> to this object.
     * 
     * @param child
     *            the new child extended data element.
     * @return a reference to the child extended data element added to this
     *         object.
     * @see #getChildren()
     */
    public ExtendedDataElement addChild(ExtendedDataElement child) {

        getChildren().add(child);

        return child;
    }

    /**
     * Removes all child ExtenedDataElements that this event is referencing
     */
    public void clearChildren() {
        getChildren().clear();
    }

    /**
     * Remove a named child extended data element that this event is referencing
     * 
     * @param name
     *            the name of the child element to be removed
     */
    public void removeChildren(String name) {

        if (getChildren() != null) {

            for (int counter = 0; counter < getChildren().size(); counter++) {

                if (((ExtendedDataElement) (getChildren().get(counter))).getName().equals(name)) {
                    getChildren().remove(counter--);
                }
            }
        }
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String)
     */
    public ExtendedDataElement addChild(String newName) {
        ExtendedDataElement ede = EventFactory.eINSTANCE.createExtendedDataElement();
        ede.setName(newName);
        getChildren().add(ede);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String,
     *      int, String)
     */
    public ExtendedDataElement addChild(String newName, int newType, String value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setTypeAsInt(newType);
        if (TYPE_HEX_BINARY_VALUE == newType) {
            ede.setHexValue(value);
        } else {
            ede.getValues().add(value);
        }

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String,
     *      int, String[])
     */
    public ExtendedDataElement addChild(String newName, int newType, String[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setTypeAsInt(newType);
        ede.getValues().addAll(Arrays.asList(newValues));

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String,
     *      String)
     */
    public ExtendedDataElement addChild(String newName, String value) {
        return addChild(newName, TYPE_STRING_VALUE, value);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String,
     *      String[])
     */
    public ExtendedDataElement addChild(String newName, String[] newValues) {
        return addChild(newName, TYPE_STRING_ARRAY_VALUE, newValues);
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithByteArrayValue(String,
     *      byte[])
     */
    public ExtendedDataElement addChildWithBooleanArrayValue(String newName, boolean[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsBooleanArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithByteValue(String,
     *      byte)
     */
    public ExtendedDataElement addChildWithBooleanValue(String newName, boolean value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsBoolean(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithNoValue(String)
     */
    public ExtendedDataElement addChildWithNoValue(String newName) {
        
    	ExtendedDataElement ede = addChild(newName);
        setTypeAsInt(TYPE_NO_VALUE_VALUE);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithByteArrayValue(String,
     *      byte[])
     */
    public ExtendedDataElement addChildWithByteArrayValue(String newName, byte[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsByteArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithByteValue(String,
     *      byte)
     */
    public ExtendedDataElement addChildWithByteValue(String newName, byte value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsByte(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithDateArrayValue(String,
     *      String[])
     */
    public ExtendedDataElement addChildWithDateArrayValue(String newName, String[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsDateArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithDateAsLongValue(String,
     *      long)
     */
    public ExtendedDataElement addChildWithDateAsLongValue(String newName, long value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsDateAsLong(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithDateValue(String,
     *      String)
     */
    public ExtendedDataElement addChildWithDateValue(String newName, String value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsDate(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithDatesAsLongValue(String,
     *      long[])
     */
    public ExtendedDataElement addChildWithDatesAsLongValue(String newName, long[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsDateAsLongArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithDoubleArrayValue(String,
     *      double[])
     */
    public ExtendedDataElement addChildWithDoubleArrayValue(String newName, double[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsDoubleArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithDoubleValue(String,
     *      double)
     */
    public ExtendedDataElement addChildWithDoubleValue(String newName, double value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsDouble(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithFloatArrayValue(String,
     *      float[])
     */
    public ExtendedDataElement addChildWithFloatArrayValue(String newName, float[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsFloatArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithFloatValue(String,
     *      float)
     */
    public ExtendedDataElement addChildWithFloatValue(String newName, float value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsFloat(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String,
     *      String)
     */
    public ExtendedDataElement addChildWithHexValue(String newName, String value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setType(TYPE_HEX_BINARY);
        ede.setHexValue(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChild(String,
     *      String)
     */
    public ExtendedDataElement addChildWithHexValueAsByteArray(String newName, byte[] value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setType(TYPE_HEX_BINARY);
        ede.setHexValueAsByteArray(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithIntArrayValue(String,
     *      int[])
     */
    public ExtendedDataElement addChildWithIntArrayValue(String newName, int[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsIntArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithIntValue(String,
     *      int)
     */
    public ExtendedDataElement addChildWithIntValue(String newName, int value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsInt(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithLongArrayValue(String,
     *      long[])
     */
    public ExtendedDataElement addChildWithLongArrayValue(String newName, long[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsLongArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithLongValue(String,
     *      long)
     */
    public ExtendedDataElement addChildWithLongValue(String newName, long value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsLong(value);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithShortArrayValue(String,
     *      short[])
     */
    public ExtendedDataElement addChildWithShortArrayValue(String newName, short[] newValues) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsShortArray(newValues);

        return ede;
    }

    /**
     * @see org.eclipse.tptp.logging.events.cbe.ExtendedDataElement#addChildWithShortValue(String,
     *      short)
     */
    public ExtendedDataElement addChildWithShortValue(String newName, short value) {
        ExtendedDataElement ede = addChild(newName);
        ede.setValuesAsShort(value);

        return ede;
    }

    /**
     * Overrides the <code>java.lang.Object</code>'s equals() API to 
     * determine if the parameter <code>object</code> is equal to this object.
     * <p>
     * 
     * @param object The <code>java.lang.Object</code> to be compared to this object.
     * @return True, if this object is the same as the parameter <code>object</code>, otherwise false.
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(Object object) {

        synchronized (this) {
            
            //Check if the parameter object is the same object reference as this object (e.g. equal):
            if (this == object){
                return true;
            }
            else {

                //Check if the parameter object is null and is the a different type as this object(e.g. not equal):
                if ((object != null) && (getClass() == object.getClass())) {

                    //Compare the value(s) of all object properties:
                    String thisObjectsProperty = getHexValue();
                    String parameterObjectsProperty = ((ExtendedDataElementImpl) (object)).getHexValue();

                    if (((thisObjectsProperty != null) && (thisObjectsProperty.equalsIgnoreCase(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))) {

                        thisObjectsProperty = getName();
                        parameterObjectsProperty = ((ExtendedDataElementImpl) (object)).getName();

                        if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))){

                            thisObjectsProperty = getType();
                            parameterObjectsProperty = ((ExtendedDataElementImpl) (object)).getType();
    
                            if (((thisObjectsProperty != null) && (thisObjectsProperty.equals(parameterObjectsProperty))) || ((thisObjectsProperty == null) && (parameterObjectsProperty == null))){
    
                                List thisObjectsListProperty = getChildren();
                                List parameterObjectsListProperty = ((ExtendedDataElementImpl) (object)).getChildren();
    
                                if (((thisObjectsListProperty != null) && (EventHelpers.compareUnorderedLists(thisObjectsListProperty, parameterObjectsListProperty))) || (((thisObjectsListProperty == null) || (thisObjectsListProperty.size() == 0)) && ((parameterObjectsListProperty == null) || (parameterObjectsListProperty.size() == 0)))) {
    
                                    thisObjectsListProperty = getValues();
                                    parameterObjectsListProperty = ((ExtendedDataElementImpl) (object)).getValues();
    
                                    if (((thisObjectsListProperty != null) && (thisObjectsListProperty.equals(parameterObjectsListProperty))) || (((thisObjectsListProperty == null) || (thisObjectsListProperty.size() == 0)) && ((parameterObjectsListProperty == null) || (parameterObjectsListProperty.size() == 0)))) {
                                        return true;
                                    }
                                }
                            }
                        }
                    }
                }
            }

            return false;
        }
    }

    /**
     *  
     * 
     */
    public String toString() {

        StringBuffer result = new StringBuffer(super.toString());
        result.append(" (hexValue: ");
        result.append(hexValue);
        result.append(", name: ");
        result.append(name);
        result.append(", type: ");
        result.append(type);
        result.append(", values: ");
        result.append(values);
        result.append(", children: ");
        result.append(children);
        result.append(')');
        return result.toString();
    }

    public static boolean isValid(int type) {
        return (typeToString(type) != null);
    }

    public static boolean isValid(String type) {
        return (isValid(typeToInt(type)));
    }

    public static String typeToString(int type) {

        switch (type) {
        case TYPE_NO_VALUE_VALUE:
            return TYPE_NO_VALUE;
        case TYPE_BYTE_VALUE:
            return TYPE_BYTE;
        case TYPE_SHORT_VALUE:
            return TYPE_SHORT;
        case TYPE_INT_VALUE:
            return TYPE_INT;
        case TYPE_LONG_VALUE:
            return TYPE_LONG;
        case TYPE_FLOAT_VALUE:
            return TYPE_FLOAT;
        case TYPE_DOUBLE_VALUE:
            return TYPE_DOUBLE;
        case TYPE_STRING_VALUE:
            return TYPE_STRING;
        case TYPE_DATE_TIME_VALUE:
            return TYPE_DATE_TIME;
        case TYPE_BOOLEAN_VALUE:
            return TYPE_BOOLEAN;
        case TYPE_BYTE_ARRAY_VALUE:
            return TYPE_BYTE_ARRAY;
        case TYPE_SHORT_ARRAY_VALUE:
            return TYPE_SHORT_ARRAY;
        case TYPE_INT_ARRAY_VALUE:
            return TYPE_INT_ARRAY;
        case TYPE_LONG_ARRAY_VALUE:
            return TYPE_LONG_ARRAY;
        case TYPE_FLOAT_ARRAY_VALUE:
            return TYPE_FLOAT_ARRAY;
        case TYPE_DOUBLE_ARRAY_VALUE:
            return TYPE_DOUBLE_ARRAY;
        case TYPE_STRING_ARRAY_VALUE:
            return TYPE_STRING_ARRAY;
        case TYPE_DATE_TIME_ARRAY_VALUE:
            return TYPE_DATE_TIME_ARRAY;
        case TYPE_BOOLEAN_ARRAY_VALUE:
            return TYPE_BOOLEAN_ARRAY;
        case TYPE_HEX_BINARY_VALUE:
            return TYPE_HEX_BINARY;
        default:
            return null;
        }
    }

    public static int typeToInt(String type) {
        
        if (type.equals(TYPE_NO_VALUE)) {
            return TYPE_NO_VALUE_VALUE;
        }

        else if (type.equals(TYPE_BYTE)) {
            return TYPE_BYTE_VALUE;
        }

        else if (type.equals(TYPE_SHORT)) {
            return TYPE_SHORT_VALUE;
        }

        else if (type.equals(TYPE_INT)) {
            return TYPE_INT_VALUE;
        }

        else if (type.equals(TYPE_LONG)) {
            return TYPE_LONG_VALUE;
        }

        else if (type.equals(TYPE_FLOAT)) {
            return TYPE_FLOAT_VALUE;
        }

        else if (type.equals(TYPE_DOUBLE)) {
            return TYPE_DOUBLE_VALUE;
        }

        else if (type.equals(TYPE_STRING)) {
            return TYPE_STRING_VALUE;
        }

        else if (type.equals(TYPE_DATE_TIME)) {
            return TYPE_DATE_TIME_VALUE;
        }

        else if (type.equals(TYPE_BOOLEAN)) {
            return TYPE_BOOLEAN_VALUE;
        }

        else if (type.equals(TYPE_BYTE_ARRAY)) {
            return TYPE_BYTE_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_SHORT_ARRAY)) {
            return TYPE_SHORT_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_INT_ARRAY)) {
            return TYPE_INT_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_LONG_ARRAY)) {
            return TYPE_LONG_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_FLOAT_ARRAY)) {
            return TYPE_FLOAT_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_DOUBLE_ARRAY)) {
            return TYPE_DOUBLE_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_STRING_ARRAY)) {
            return TYPE_STRING_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_DATE_TIME_ARRAY)) {
            return TYPE_DATE_TIME_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_BOOLEAN_ARRAY)) {
            return TYPE_BOOLEAN_ARRAY_VALUE;
        }

        else if (type.equals(TYPE_HEX_BINARY)) { 
            return TYPE_HEX_BINARY_VALUE; 
        }

        return -1;
    }

    /**
     * Checks whether the contents of the instance conform to the Common Base Event 
     * specification as described in "Canonical Situation Data Format: The Common Base Event 
     * V1.0.1".
     * <p>
     * 
     * @throws ValidationException If the event instance does not conform to the Common Base Event specification.
     */
    public void validate() throws ValidationException {

        synchronized (this) {

            //Step 1:  Check for required properties:
            if(name == null){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.ExtendedDataElement.Name")); 
            }

            if(type == null){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.ExtendedDataElement.Type")); 
            }

            //Step 2:  Check for mutually exclusive properties:
            if((values != null) && (hexValue != null)){
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MUTUALLY_EXCLUSIVE_ATTRIBUTES_EXC_", "CommonBaseEvent.ExtendedDataElement.Values", "CommonBaseEvent.ExtendedDataElement.HexValue"));                 
            }

            //Step 3:  Check content of each property:
            if(name.length() > 64){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LENGTH_EXC_", "CommonBaseEvent.ExtendedDataElement.Name", name, new Integer(name.length()), new Integer(64)));
            }
            
            if(type.length() > 64){
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LENGTH_EXC_", "CommonBaseEvent.ExtendedDataElement.Type", type, new Integer(type.length()), new Integer(64)));
            }

            if (!isValid(type)) { 
                throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_EXTENDED_DATA_INVALID_TYPE_EXC_", name, type)); 
            }

            int typeAsInt = typeToInt(type);

            if (typeAsInt == TYPE_HEX_BINARY_VALUE) {

                if (hexValue == null){ 
                    throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.ExtendedDataElement.HexValue"));
                }
                else {
                   
                   if ((hexValue.length() % 2) != 0) { 
                       throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_EXTENDED_DATA_INVALID_HEX_BINARY_1_EXC_", name, type, hexValue)); 
                   }

                   for (int counter = 0; counter < hexValue.length(); counter++) {
                        
                       char character = Character.toUpperCase(InternationalizationUtilities.charAt(hexValue,counter));
                       int index = 0;

                       while(index < HEX_BASE_CHARS.length){
                           
                           if(character == HEX_BASE_CHARS[index]){
                               break;
                           }

                           index++;
                       }
                       
                       if(index == HEX_BASE_CHARS.length){ 
                           throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_EXTENDED_DATA_INVALID_HEX_BINARY_2_EXC_", name, type, hexValue, new Character(character))); 
                       }
                    }
                }
            } 
            else if (typeAsInt == TYPE_NO_VALUE_VALUE) {

                if (((values != null) && (values.size() > 0)) || (hexValue != null)) { 
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INCONSISTENT_ATTRIBUTES_EXC_", "CommonBaseEvent.ExtendedDataElement.Type", type, "CommonBaseEvent.ExtendedDataElement.Values, CommonBaseEvent.ExtendedDataElement.HexValue")); 
                }           
            }
            else if ((values == null) || (values.size() == 0)) {
                throw new MissingPropertyException(LoggingCoreResourceBundle.getString("LOG_MISSING_REQUIRED_ATTRIBUTE_EXC_", "CommonBaseEvent.ExtendedDataElement.Values")); 
            }
            else {

                if(((typeAsInt == TYPE_BOOLEAN_VALUE) || (typeAsInt == TYPE_BYTE_VALUE) || (typeAsInt == TYPE_DATE_TIME_VALUE) || (typeAsInt == TYPE_DOUBLE_VALUE) || (typeAsInt == TYPE_FLOAT_VALUE) || (typeAsInt == TYPE_INT_VALUE) || (typeAsInt == TYPE_LONG_VALUE) || (typeAsInt == TYPE_SHORT_VALUE) || (typeAsInt == TYPE_STRING_VALUE)) && (values.size() > 1)){
                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_VALUE_ARRAY_SIZE_MISMATCH_EXC_", new Integer(values.size()), name, type));
                }
                
                ListIterator iterator = values.listIterator();                
                String value = null;

                while (iterator.hasNext()) {

                    value = ((String) (iterator.next()));

                    if(value == null){
                        throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_EXTENDED_DATA_VALUE_NULL_EXC_", name, type)); 
                    }
                    else if(value.length() > 1024){
                        throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LIST_ELEMENT_LENGTH_EXC_", value, "CommonBaseEvent.ExtendedDataElement.Value".concat("[").concat(String.valueOf(iterator.nextIndex())).concat("]"), new Integer(1024)));                        
                    }
                    else{
                    
                        try {
    
                            if ((typeAsInt == TYPE_BYTE_VALUE) || (typeAsInt == TYPE_BYTE_ARRAY_VALUE)) {
                                Byte.parseByte(value);
                            } 
                            else if ((typeAsInt == TYPE_DOUBLE_VALUE) || (typeAsInt == TYPE_DOUBLE_ARRAY_VALUE)) {
                                Double.parseDouble(value);
                            } 
                            else if ((typeAsInt == TYPE_FLOAT_VALUE) || (typeAsInt == TYPE_FLOAT_ARRAY_VALUE)) {
                                Float.parseFloat(value);
                            } 
                            else if ((typeAsInt == TYPE_INT_VALUE) || (typeAsInt == TYPE_INT_ARRAY_VALUE)) {
                                Integer.parseInt(value);
                            } 
                            else if ((typeAsInt == TYPE_LONG_VALUE) || (typeAsInt == TYPE_LONG_ARRAY_VALUE)) {
                                Long.parseLong(value);
                            } 
                            else if ((typeAsInt == TYPE_SHORT_VALUE) || (typeAsInt == TYPE_SHORT_ARRAY_VALUE)) {
                                Short.parseShort(value);
                            }
                            else if ((typeAsInt == TYPE_DATE_TIME_VALUE) || (typeAsInt == TYPE_DATE_TIME_ARRAY_VALUE)) {
                                
                                try {
                                                            
                                    if(LoggingCoreUtilities.convertXsdDateTimeToMilliseconds(value) == 0){
                                        throw new IllegalArgumentException();
                                    }
                                } 
                                catch (IllegalArgumentException ex) {
                                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_EXTENDED_DATA_INVALID_DATE_EXC_", name, type, value)); 
                                }
                            } 
                            else if ((typeAsInt == TYPE_BOOLEAN_VALUE) || (typeAsInt == TYPE_BOOLEAN_ARRAY_VALUE)) {

                                if (!((value.equalsIgnoreCase("true")) || (value.equalsIgnoreCase("false")) || (value.equals("1")) || (value.equals("0")))) { 
                                    throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_EXTENDED_DATA_INVALID_BOOLEAN_EXC_", name, type, value)); 
                                }
                            }
                        } 
                        catch (NumberFormatException e) {
                            throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_TYPE_CONVERSION_EXC_", value, type, name));
                        }        
                    }
                }
            }

            if (children != null) {

                ListIterator iterator = children.listIterator();
                ExtendedDataElement extendedDataElement = null;

                while (iterator.hasNext()) {

                    extendedDataElement = ((ExtendedDataElement) (iterator.next()));

                    try {
                        extendedDataElement.validate();
                    } 
                    catch (ValidationException v) {
                        throw new ValidationException(LoggingCoreResourceBundle.getString("LOG_INVALID_LIST_ELEMENT_ATTRIBUTE_EXC_", "CommmonBaseEvent.ExtendedDataElement.Child".concat("[").concat(String.valueOf(iterator.nextIndex())).concat("]")), v);
                    }
                }
            }
        }        
    }

    /**
     * Removes all values that this event is referencing
     */
    public void clearValues() {
        getValues().clear();
    }

    /**
     * Return a cloned (copy by value) object of this object. 
     * <p>
     * This is a deep copy version, in which all the objects 
     * within this object will be copied by value.
     * <p>
     * 
     * @return Object The cloned (copy by value) object of this object.
     * @throws CloneNotSupportedException If a instance variable of this object does not implement the <code>Cloneable</code> interface.   
     */
    public Object clone() throws CloneNotSupportedException {

        synchronized (this) {

            ExtendedDataElementImpl copy = ((ExtendedDataElementImpl) (super.clone()));

            if (this.children != null) {
                
                copy.children = new ArrayList();

                for (int counter = 0; counter < this.children.size(); counter++) {
                    
                    try{
                        copy.addChild(((ExtendedDataElement)(((ExtendedDataElementImpl)(this.children.get(counter))).clone())));
                    } 
                    catch (ClassCastException c) {
                        //Ignore since cloning is supported.
                    }
                }
            }
            
            if(this.hexValue != null){
                copy.setHexValue(new String(this.hexValue));                                
            }

            if(this.name != null){
                copy.setName(new String(this.name));                                
            }

            if(this.type != null){
                copy.setType(new String(this.type));                                
            }

            if ((this.values != null) && (this.values.size() > 0)) {
                
                String[] valuesArray = new String[this.values.size()];
                
                for (int counter = 0; counter < valuesArray.length; counter++) {
                    
                    try{
                        valuesArray[counter] = new String((String)(this.values.get(counter)));
                    } 
                    catch (ClassCastException c) {
                        //Ignore since cloning is supported.
                    }
                }
                
                copy.setValues(valuesArray);
            }

            return copy;
        }            
    }

    /**
     * Resets the object's properties to their initial (e.g. null) state.
     * 
     * All components are initialized to either zero or null.
     * 
     * @since 1.0
     */
    public void init() {

        setHexValue(null);
        setName(null);
        setType(null);

        clearValues();
        clearChildren();
    }
    
    /**
     * Sets the value of the extended data element as String value.
     * 
     * @param value
     *            the new value for the extended data element
     */
    protected void setValues(String value) {

        clearValues();
        List valuList = getValues();
        valuList.add(value);
    }
    
    /**
     * @see org.eclipse.hyades.logging.events.cbe.ExtendedDataElement#setHexValueAsByteArray(byte[])
     *         
     * Converts a byte array to a hexBinary string
     *
     * @param inByteArray A byte array
     *
     * @return String Conversion of the byte array to a hexBinary string.
     */
    protected String byteArrayToHexBinary(byte[] inByteArray) {
        String result = null;

        if (inByteArray != null) {
            StringBuffer hexValueAsString = new StringBuffer(inByteArray.length * 2);
            for (int i = 0; i < inByteArray.length; i++) {
                hexValueAsString.append(HEX_BASE_CHARS[(inByteArray[i] & 0xF0) >> 4]);
                hexValueAsString.append(HEX_BASE_CHARS[inByteArray[i] & 0x0F]);
            }

            result = hexValueAsString.toString();
        }

        return result;
    }

    /**
     * Converts a <code>xsd:hexBinary</code> string to the corresponding
     * array of bytes. 
     * 
     * @param inHexData A String containing hexBinary data
     *
     * @return               conversion of the string to a byte array.  
     *                       If the input is <code>null</code>, a null byte 
     *                       array will be returned.
     * 
     * @throws IllegalStateException 
     *                       if the input is not an even length or if the
     *                       input does not contain hex characters
     */
    protected byte[] hexBinaryToByteArray(String inHexData) throws IllegalStateException {
        byte[] result = null;

        if (inHexData != null) {
            int count = inHexData.length();
            if ((count % 2) == 1) {
                throw new IllegalArgumentException("inHexData must have an even length"); /* NOI18N */
            }
            result = new byte[count / 2];
            // The idea is to iterate over the bytes and come up with
            // the hexadecimal pair of characters that represent it.
            byte byteValue = 0;
            for (int i = 0; i < count; i = i + 2) {
                int resultIndex = i / 2;
                for (int j = 0; j < 2; j++) {
                    char c =InternationalizationUtilities.charAt(inHexData,i + j);
                    switch (Character.toUpperCase(c)) {
                        case '0' :
                            byteValue = 0;
                            break;
                        case '1' :
                            byteValue = 1;
                            break;
                        case '2' :
                            byteValue = 2;
                            break;
                        case '3' :
                            byteValue = 3;
                            break;
                        case '4' :
                            byteValue = 4;
                            break;
                        case '5' :
                            byteValue = 5;
                            break;
                        case '6' :
                            byteValue = 6;
                            break;
                        case '7' :
                            byteValue = 7;
                            break;
                        case '8' :
                            byteValue = 8;
                            break;
                        case '9' :
                            byteValue = 9;
                            break;
                        case 'A' :
                            byteValue = 10;
                            break;
                        case 'B' :
                            byteValue = 11;
                            break;
                        case 'C' :
                            byteValue = 12;
                            break;
                        case 'D' :
                            byteValue = 13;
                            break;
                        case 'E' :
                            byteValue = 14;
                            break;
                        case 'F' :
                            byteValue = 15;
                            break;
                        default :
                            {
                                throw new IllegalStateException("string is not a valid hexBinary"); /* NOI18N */
                            }
                    } // switch hex character

                    if (j == 0) {
                        result[resultIndex] = (byte) (byteValue << 4);
                    }
                    else {
                        result[resultIndex] += byteValue;
                    }
                } // for 1st, 2nd byte
            } // for characters
        } // if inHexData not null

        return result;
    }
 } 