package org.eclipse.tptp.logging.events.cbe.internal.util;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.logging.core.LoggingCoreResourceBundle;
import org.eclipse.hyades.logging.core.XmlUtility;
import org.eclipse.tptp.logging.events.cbe.AssociatedEvent;
import org.eclipse.tptp.logging.events.cbe.AssociationEngine;
import org.eclipse.tptp.logging.events.cbe.AvailableSituation;
import org.eclipse.tptp.logging.events.cbe.CommonBaseEvent;
import org.eclipse.tptp.logging.events.cbe.ComponentIdentification;
import org.eclipse.tptp.logging.events.cbe.ConfigureSituation;
import org.eclipse.tptp.logging.events.cbe.ConnectSituation;
import org.eclipse.tptp.logging.events.cbe.ContextDataElement;
import org.eclipse.tptp.logging.events.cbe.CreateSituation;
import org.eclipse.tptp.logging.events.cbe.DependencySituation;
import org.eclipse.tptp.logging.events.cbe.DestroySituation;
import org.eclipse.tptp.logging.events.cbe.EventFactory;
import org.eclipse.tptp.logging.events.cbe.ExtendedDataElement;
import org.eclipse.tptp.logging.events.cbe.FeatureSituation;
import org.eclipse.tptp.logging.events.cbe.MsgCatalogToken;
import org.eclipse.tptp.logging.events.cbe.MsgDataElement;
import org.eclipse.tptp.logging.events.cbe.OtherSituation;
import org.eclipse.tptp.logging.events.cbe.ReportSituation;
import org.eclipse.tptp.logging.events.cbe.RequestSituation;
import org.eclipse.tptp.logging.events.cbe.Situation;
import org.eclipse.tptp.logging.events.cbe.StartSituation;
import org.eclipse.tptp.logging.events.cbe.StopSituation;
import org.eclipse.tptp.logging.events.cbe.util.EventListener;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventHandler.java,v 1.4 2008/02/28 02:33:17 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Event handler used with a <a href="http://www.saxproject.org/">Simple API 
 * for XML (SAX)</a> parser for de-serializing Common Base Event XML documents.
 * <p>
 * Callers register an {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
 * implementation with this event handler to process  
 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s and/or 
 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s when 
 * de-serializing a stream of Common Base Event and/or Association Engine 
 * XML fragments in a Common Base Event XML document.
 * <p>
 * Once a Common Base Event XML and/or Association Engine fragment has been 
 * de-serialized by this event handler, a 
 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} or 
 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} is constructed 
 * and broadcast to the registered 
 * {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} implementation 
 * for processing.
 * <p>
 * When de-serializing a stream of Common Base Event and Association Engine XML 
 * fragments in a Common Base Event XML document, callers can process 
 * de-serialized {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s and 
 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s on a separate
 * thread before the entire stream of Common Base Event and Association Engine 
 * XML fragments in a Common Base Event XML document has been processed, resulting 
 * in a high level of parallelism and improved efficiency.
 * <p>
 * Noteworthy, callers are responsible for 
 * {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} instantiation, 
 * registration and clean-up.
 * <p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version February 13, 2006
 * @since   February 6, 2006
 * @see     org.xml.sax.helpers.DefaultHandler
 * @see     org.eclipse.tptp.logging.events.cbe.util.EventFormatter
 * @see     org.eclipse.tptp.logging.events.cbe.util.EventListener
 */
public class EventHandler extends DefaultHandler {

    /**
     * Buffer for any element XML fragments (e.g. CommonBaseEvent.otherData or
     * OtherSituation.anyData).
     */
    protected StringBuffer anyElementXMLFragment = null;

    /**
     * Buffer for Parsed Character DATA (PCDATA) (e.g.
     * ExtendedDataElement.values) since some event handlers do not return all
     * contiguous PCDATA in a single chunk.
     */
    protected StringBuffer charactersBuffer = null;

    /**
     * Buffer for name space mapping attributes.
     */
    protected StringBuffer nameSpaceAttribute = null;

    /**
     * The <code>EventListener</code> for this event handler.
     */
    protected EventListener eventListener = null;

    /**
     * Stack of event elements currently parsed.
     */
    protected List parsedElements = null;
    
    /**
     * Indicator of the parent element in the current XML fragment.
     * <p>
     * This indicator is required for performance reasons to eliminate expensive 
     * <code>instanceof</code> calls to determine the top element on the parsed 
     * elements stack.
     * <p>
     * This indicator uses the following notation:
     * <p>
     * [Document]:                     0
     * CommonBaseEvents/TemplateEvent: 1
     * CommonBaseEvent:                2
     * contextDataElements:            3
     * extendedDataElements:           4
     * children:                       5
     * associatedEvents:               6
     * msgDataElement:                 7
     * situation:                      8
     * OtherSituation:                 9
     */
    protected short parentElement = 0;
    
    /**
     * Current document column position.
     */
    protected int column = 0;

    /**
     * Current document line position.
     */
    protected int line = 0;

    /**
     * Current nesting depth of any element XML fragments (e.g.
     * CommonBaseEvent.otherData or OtherSituation.anyData).
     */
    protected int anyElementNestingDepth = 0;

    /**
     * No argument constructor.
     * <p>
     */
    public EventHandler() {

        parsedElements = new ArrayList();

        anyElementXMLFragment = new StringBuffer();

        charactersBuffer = new StringBuffer();
    }

    /**
     * Initializes the event handler to a newly constructed state.
     * <p>
     */
    public void init() {

        reset();

    	eventListener = null;

        parentElement = 0;
    }

    /**
     * Resets the event handler for parsing the next element.
     * <p>
     */
    protected void reset() {

        column = 0;

        line = 0;

        anyElementNestingDepth = 0;

        nameSpaceAttribute = null;

        if(parsedElements.size() > 0){
        	parsedElements.clear();
        }

        int stringBufferLength = anyElementXMLFragment.length();
        
        if(stringBufferLength > 0){
        	anyElementXMLFragment.setLength(0);
        }

        stringBufferLength = charactersBuffer.length();
        
        if(stringBufferLength > 0){
        	charactersBuffer.setLength(0);
        }     
    }

    /**
     * Registers the non-<code>null</code> parameter {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
     * implementation with this event handler.
     * <p>
     * The registered {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
     * implementation is notified to process 
     * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s and/or 
     * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s when de-serializing 
     * a stream of Common Base Event and/or Association Engine XML fragments in a Common 
     * Base Event XML document.
     * <p>
     * 
     * @param eventListener A non-<code>null</code> {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} to be registered with this event handler.
     */
    public void addEventListener(EventListener eventListener) {
    	this.eventListener = eventListener;
    }
    
    /**
     * Retrieves the {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
     * implementation registered with this event handler.
     * <p>
     * The registered {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
     * implementation is notified to process 
     * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s and/or 
     * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s when de-serializing 
     * a stream of Common Base Event and/or Association Engine XML fragments in a Common 
     * Base Event XML document.
     * <p>
     * 
     * @return The {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} registered with this event handler.
     */
    public EventListener getEventListener() {
    	return eventListener;
    }

    /**
     * @see org.xml.sax.ContentHandler#setDocumentLocator(org.xml.sax.Locator)
     */
    public void setDocumentLocator(Locator locator) {

        super.setDocumentLocator(locator);

        column = locator.getColumnNumber();
        line = locator.getLineNumber();
    }

    /**
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char characters[], int start, int length) throws SAXException {

        super.characters(characters, start, length);

        charactersBuffer.append(new String(characters, start, length));
    }

    /**
     * @see org.xml.sax.ContentHandler#startPrefixMapping(java.lang.String,
     *      java.lang.String)
     */
    public void startPrefixMapping(String prefix, String uri) throws SAXException {

        super.startPrefixMapping(prefix, uri);

        //NOTE: Xerces fires name space mapping events for all elements.
        //Do not capture name space mapping events for the 'xmlns' and 'xml' name space:
        if ((!prefix.equals("xmlns")) && (!prefix.equals("xml"))) {

            nameSpaceAttribute = new StringBuffer("xmlns");

            if ((prefix != null) && (prefix.trim().length() > 0)) {
                nameSpaceAttribute.append(':');
                nameSpaceAttribute.append(prefix);
            }

            nameSpaceAttribute.append("=\"");
            nameSpaceAttribute.append(uri);
            nameSpaceAttribute.append('\"');
        }
    }

    /**
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String,
     *      java.lang.String, java.lang.String, org.xml.sax.Attributes)
     */
    public void startElement(String uri, String localName, String qualifiedName, Attributes attributes) throws SAXException {

        super.startElement(uri, localName, qualifiedName, attributes);

        String elementName = getLocalElementName(localName, qualifiedName);
        String attributeName = null;

        //[Document]: 0
        if(parentElement == 0){
            
            if ((elementName.equals("CommonBaseEvents")) || (elementName.equals("TemplateEvent"))) {            
                parentElement = 1;
            }
            else{
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "CommonBaseEvents or TemplateEvent", elementName, String.valueOf(line), String.valueOf(column)));                 
            }
        }
        
        //CommonBaseEvents/TemplateEvent: 1
        else if(parentElement == 1 && anyElementNestingDepth == 0){
            
            if(elementName.equals("CommonBaseEvent")){

                CommonBaseEvent commonBaseEvent = EventFactory.eINSTANCE.createCommonBaseEvent();
                
                //Set the Common Base Event's version property to <code>null</code> for Common Base Event XML elements without a version attribute (e.g. no version as compared to an empty version):
                commonBaseEvent.setVersion(null);

                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("creationTime")){
                        commonBaseEvent.setCreationTime(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("globalInstanceId")){
                        commonBaseEvent.setGlobalInstanceId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("version")){
                        commonBaseEvent.setVersion(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("severity")){
                        
                        try {
                            commonBaseEvent.setSeverity(Short.parseShort(attributes.getValue(counter)));                            
                        } 
                        catch (NumberFormatException n) {

                            //If the attribute value is not white space (e.g. empty), throw an exception:
                            if(attributes.getValue(counter).trim().length() > 0){
                                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", attributeName, attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                            }
                        }
                    }
                    else if(attributeName.equals("msg")){
                        commonBaseEvent.setMsg(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("sequenceNumber")){
                        
                        try {
                            commonBaseEvent.setSequenceNumber(Long.parseLong(attributes.getValue(counter)));                            
                        } 
                        catch (NumberFormatException n) {

                            //If the attribute value is not white space (e.g. empty), throw an exception:
                            if(attributes.getValue(counter).trim().length() > 0){
                                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", attributeName, attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                            }
                        }
                    }
                    else if(attributeName.equals("priority")){
                        
                        try {
                            commonBaseEvent.setPriority(Short.parseShort(attributes.getValue(counter)));                            
                        } 
                        catch (NumberFormatException n) {

                            //If the attribute value is not white space (e.g. empty), throw an exception:
                            if(attributes.getValue(counter).trim().length() > 0){
                                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", attributeName, attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                            }
                        }
                    }
                    else if(attributeName.equals("repeatCount")){
                        
                        try {
                            commonBaseEvent.setRepeatCount(Short.parseShort(attributes.getValue(counter)));                            
                        } 
                        catch (NumberFormatException n) {

                            //If the attribute value is not white space (e.g. empty), throw an exception:
                            if(attributes.getValue(counter).trim().length() > 0){
                                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", attributeName, attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                            }
                        }
                    }
                    else if(attributeName.equals("elapsedTime")){
                        
                        try {
                            commonBaseEvent.setElapsedTime(Long.parseLong(attributes.getValue(counter)));                            
                        } 
                        catch (NumberFormatException n) {

                            //If the attribute value is not white space (e.g. empty), throw an exception:
                            if(attributes.getValue(counter).trim().length() > 0){
                                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", attributeName, attributes.getValue(counter), String.valueOf(line), String.valueOf(column)));
                            }
                        }
                    }
                    else if(attributeName.equals("extensionName")){
                        commonBaseEvent.setExtensionName(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("localInstanceId")){
                        commonBaseEvent.setLocalInstanceId(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(commonBaseEvent);
                
                parentElement = 2;
            }
            else if(elementName.equals("AssociationEngine")){

                AssociationEngine associationEngine = EventFactory.eINSTANCE.createAssociationEngine();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("id")){
                        associationEngine.setId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("type")){
                        associationEngine.setType(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("name")){
                        associationEngine.setName(attributes.getValue(counter));
                    }
                }
                
                if(eventListener != null){
                	eventListener.processAssociationEngine(associationEngine);
                }

                reset();
            }
            else{
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "CommonBaseEvent or AssociationEngine", elementName, String.valueOf(line), String.valueOf(column)));                 
            }
        }

        //CommonBaseEvent: 2
        else if(parentElement == 2){

            if(elementName.equals("contextDataElements") && anyElementNestingDepth == 0){

                ContextDataElement contextDataElement = EventFactory.eINSTANCE.createContextDataElement();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("name")){
                        contextDataElement.setName(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("type")){
                        contextDataElement.setType(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(contextDataElement);
                
                parentElement = 3;
            }
            else if(elementName.equals("extendedDataElements") && anyElementNestingDepth == 0){
                
                ExtendedDataElement extendedDataElement = EventFactory.eINSTANCE.createExtendedDataElement();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("name")){
                        extendedDataElement.setName(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("type")){
                        extendedDataElement.setType(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(extendedDataElement);
                
                parentElement = 4;
            }
            else if(elementName.equals("associatedEvents") && anyElementNestingDepth == 0){
                
                AssociatedEvent associatedEvent = EventFactory.eINSTANCE.createAssociatedEvent();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("resolvedEvents")){
                        associatedEvent.setResolvedEvents(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(associatedEvent);
                
                parentElement = 6;
            }
            else if(elementName.equals("reporterComponentId") && anyElementNestingDepth == 0){
                                
                ComponentIdentification reporterComponentId = EventFactory.eINSTANCE.createComponentIdentification();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("component")){
                        reporterComponentId.setComponent(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("subComponent")){
                        reporterComponentId.setSubComponent(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("componentIdType")){
                        reporterComponentId.setComponentIdType(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("instanceId")){
                        reporterComponentId.setInstanceId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("application")){
                        reporterComponentId.setApplication(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("executionEnvironment")){
                        reporterComponentId.setExecutionEnvironment(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("location")){
                        reporterComponentId.setLocation(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("locationType")){
                        reporterComponentId.setLocationType(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("processId")){
                        reporterComponentId.setProcessId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("threadId")){
                        reporterComponentId.setThreadId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("componentType")){
                        reporterComponentId.setComponentType(attributes.getValue(counter));
                    }
                }
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).setReporterComponentId(reporterComponentId);
            }
            else if(elementName.equals("sourceComponentId") && anyElementNestingDepth == 0){
                
                ComponentIdentification sourceComponentId = EventFactory.eINSTANCE.createComponentIdentification();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("component")){
                        sourceComponentId.setComponent(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("subComponent")){
                        sourceComponentId.setSubComponent(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("componentIdType")){
                        sourceComponentId.setComponentIdType(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("instanceId")){
                        sourceComponentId.setInstanceId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("application")){
                        sourceComponentId.setApplication(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("executionEnvironment")){
                        sourceComponentId.setExecutionEnvironment(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("location")){
                        sourceComponentId.setLocation(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("locationType")){
                        sourceComponentId.setLocationType(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("processId")){
                        sourceComponentId.setProcessId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("threadId")){
                        sourceComponentId.setThreadId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("componentType")){
                        sourceComponentId.setComponentType(attributes.getValue(counter));
                    }
                }
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).setSourceComponentId(sourceComponentId);
            }
            else if(elementName.equals("msgDataElement") && anyElementNestingDepth == 0){
                
                MsgDataElement msgDataElement = EventFactory.eINSTANCE.createMsgDataElement();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("msgLocale")){
                        msgDataElement.setMsgLocale(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(msgDataElement);
                
                parentElement = 7;
            }
            else if(elementName.equals("situation") && anyElementNestingDepth == 0){
                
                Situation situation = EventFactory.eINSTANCE.createSituation();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("categoryName")){
                        situation.setCategoryName(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(situation);
                
                parentElement = 8;
            }
            else{
                
                //Do not persist white space (e.g. formatting characters) between elements:
                if ((anyElementNestingDepth > 0) && (charactersBuffer.toString().trim().length() > 0)) {
                    anyElementXMLFragment.append(XmlUtility.normalize(charactersBuffer.toString()));
                }
                    
                anyElementXMLFragment.append('<');
                anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qualifiedName));
                    
                //Add the name space mapping attribute:
                if (nameSpaceAttribute != null) {
                    
                    anyElementXMLFragment.append(' ');
                    anyElementXMLFragment.append(nameSpaceAttribute.toString());
                }
                    
                for (int counter = 0; counter < attributes.getLength(); counter++) {
                    
                    anyElementXMLFragment.append(' ');
                    anyElementXMLFragment.append(getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)));
                    anyElementXMLFragment.append("=\"");
                    anyElementXMLFragment.append(XmlUtility.normalize(attributes.getValue(counter)));
                    anyElementXMLFragment.append("\"");
                }
                    
                anyElementXMLFragment.append('>');            

                anyElementNestingDepth++;                
            }
        }

        //extendedDataElements: 4
        //children: 5
        else if((parentElement == 4) || (parentElement == 5) && anyElementNestingDepth == 0){
            
            //Validate the remaining child elements when processing the ExtendedDataElements end element:
            if(elementName.equals("children")){
                
                ExtendedDataElement child = EventFactory.eINSTANCE.createExtendedDataElement();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("name")){
                        child.setName(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("type")){
                        child.setType(attributes.getValue(counter));
                    }
                }
                
                parsedElements.add(child);
                
                parentElement = 5;
            }
        }

        //associatedEvents: 6
        else if(parentElement == 6 && anyElementNestingDepth == 0){
            
            //Validate the remaining child elements when processing the msgDataElement end element:
            if(elementName.equals("associationEngineInfo")){

                AssociationEngine associationEngineInfo = EventFactory.eINSTANCE.createAssociationEngine();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("id")){
                        associationEngineInfo.setId(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("type")){
                        associationEngineInfo.setType(attributes.getValue(counter));
                    }
                    else if(attributeName.equals("name")){
                        associationEngineInfo.setName(attributes.getValue(counter));
                    }
                }
                
                ((AssociatedEvent)(parsedElements.get(parsedElements.size() - 1))).setAssociationEngineInfo(associationEngineInfo);
            }
        }

        //msgDataElement: 7
        else if(parentElement == 7 && anyElementNestingDepth == 0){
            
            //Validate the remaining child elements when processing the msgDataElement end element:
            if(elementName.equals("msgCatalogTokens")){
                
                MsgCatalogToken msgCatalogToken = EventFactory.eINSTANCE.createMsgCatalogToken();
                
                //Set the properties from the attributes on the object of the element:
                for (int counter = 0; counter < attributes.getLength(); counter++) {

                    attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                    
                    if(attributeName.equals("value")){
                        msgCatalogToken.setValue(attributes.getValue(counter));
                    }
                }
                
                ((MsgDataElement)(parsedElements.get(parsedElements.size() - 1))).addMsgCatalogToken(msgCatalogToken);
            }
        }

        //situation: 8
        else if(parentElement == 8 && anyElementNestingDepth == 0){

            if(elementName.equals("situationType")){

                //Resolve the type of situation from the situation's category name:
                String situationType = ((Situation)(parsedElements.get(parsedElements.size() - 1))).getCategoryName();
                
                if((situationType == null) || ((situationType = situationType.trim()).length() == 0)){
                    throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", "categoryName", String.valueOf(situationType), String.valueOf(line), String.valueOf(column)));                                            
                }                
                else if(situationType.equals("StartSituation")){
                    
                    StartSituation startSituation = EventFactory.eINSTANCE.createStartSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            startSituation.setSuccessDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("situationQualifier")){
                            startSituation.setSituationQualifier(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            startSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(startSituation);
                }
                else if(situationType.equals("StopSituation")){
                    
                    StopSituation stopSituation = EventFactory.eINSTANCE.createStopSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            stopSituation.setSuccessDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("situationQualifier")){
                            stopSituation.setSituationQualifier(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            stopSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(stopSituation);
                }
                else if(situationType.equals("ConnectSituation")){
                    
                    ConnectSituation connectSituation = EventFactory.eINSTANCE.createConnectSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            connectSituation.setSuccessDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("situationDisposition")){
                            connectSituation.setSituationDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            connectSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(connectSituation);
                }
                else if(situationType.equals("RequestSituation")){
                    
                    RequestSituation requestSituation = EventFactory.eINSTANCE.createRequestSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            requestSituation.setSuccessDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("situationQualifier")){
                            requestSituation.setSituationQualifier(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            requestSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(requestSituation);
                }
                else if(situationType.equals("ConfigureSituation")){
                    
                    ConfigureSituation configureSituation = EventFactory.eINSTANCE.createConfigureSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            configureSituation.setSuccessDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            configureSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(configureSituation);
                }
                else if(situationType.equals("AvailableSituation")){
                    
                    AvailableSituation availableSituation = EventFactory.eINSTANCE.createAvailableSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("operationDisposition")){
                            availableSituation.setOperationDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("availabilityDisposition")){
                            availableSituation.setAvailabilityDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("processingDisposition")){
                            availableSituation.setProcessingDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            availableSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(availableSituation);
                }
                else if(situationType.equals("ReportSituation")){
                    
                    ReportSituation reportSituation = EventFactory.eINSTANCE.createReportSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("reportCategory")){
                            reportSituation.setReportCategory(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            reportSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(reportSituation);
                }
                else if(situationType.equals("CreateSituation")){
                    
                    CreateSituation createSituation = EventFactory.eINSTANCE.createCreateSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            createSituation.setSuccessDisposition(attributes.getValue(counter));
                        }                        
                        else if(attributeName.equals("reasoningScope")){
                            createSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(createSituation);
                }
                else if(situationType.equals("DestroySituation")){
                    
                    DestroySituation destroySituation = EventFactory.eINSTANCE.createDestroySituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("successDisposition")){
                            destroySituation.setSuccessDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            destroySituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(destroySituation);
                }
                else if(situationType.equals("FeatureSituation")){
                    
                    FeatureSituation featureSituation = EventFactory.eINSTANCE.createFeatureSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("featureDisposition")){
                            featureSituation.setFeatureDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            featureSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(featureSituation);
                }
                else if(situationType.equals("DependencySituation")){
                    
                    DependencySituation dependencySituation = EventFactory.eINSTANCE.createDependencySituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("dependencyDisposition")){
                            dependencySituation.setDependencyDisposition(attributes.getValue(counter));
                        }
                        else if(attributeName.equals("reasoningScope")){
                            dependencySituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(dependencySituation);
                }
                else if(situationType.equals("OtherSituation")){
                    
                    OtherSituation otherSituation = EventFactory.eINSTANCE.createOtherSituation();
                    
                    //Set the properties from the attributes on the object of the element:
                    for (int counter = 0; counter < attributes.getLength(); counter++) {

                        attributeName = getLocalElementName(attributes.getLocalName(counter), attributes.getQName(counter));
                        
                        if(attributeName.equals("reasoningScope")){
                            otherSituation.setReasoningScope(attributes.getValue(counter));
                        }
                    }
                    
                    parsedElements.add(otherSituation);
                    
                    parentElement = 9;
                }
                else{
                    throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INVALID_XML_ATTRIBUTE_EXC_", "categoryName", situationType, String.valueOf(line), String.valueOf(column)));                                            
                }
            }
            else{
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "SituationType", elementName, String.valueOf(line), String.valueOf(column)));                 
            }                    
        }

        //OtherSituation: 9
        else if(parentElement == 9){

            //Do not persist white space (e.g. formatting characters) between elements:
            if ((anyElementNestingDepth > 0) && (charactersBuffer.toString().trim().length() > 0)) {
                anyElementXMLFragment.append(XmlUtility.normalize(charactersBuffer.toString()));
            }
                
            anyElementXMLFragment.append('<');
            anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qualifiedName));
                
            //Add the name space mapping attribute:
            if (nameSpaceAttribute != null) {
                
                anyElementXMLFragment.append(' ');
                anyElementXMLFragment.append(nameSpaceAttribute.toString());
            }
                
            for (int counter = 0; counter < attributes.getLength(); counter++) {
                
                anyElementXMLFragment.append(' ');
                anyElementXMLFragment.append(getQualifiedElementName(attributes.getURI(counter), attributes.getLocalName(counter), attributes.getQName(counter)));
                anyElementXMLFragment.append("=\"");
                anyElementXMLFragment.append(XmlUtility.normalize(attributes.getValue(counter)));
                anyElementXMLFragment.append("\"");
            }
                
            anyElementXMLFragment.append('>');            

            anyElementNestingDepth++;                
        }
        
        //Validate the child elements of the following elements when processing their end element:
        //-ContextDataElements
        else if(parentElement != 3 && anyElementNestingDepth == 0){
            throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_UNEXPECTED_XML_ELEMENT_EXC_", elementName, String.valueOf(line), String.valueOf(column)));                             
        }
        
        //Empty the buffer of characters (e.g. white space):
        charactersBuffer.setLength(0);

        //Release the name space mapping attribute since startPrefixMapping(String, String) is <i>always</i> called before startElement(String, String, String, Attributes):
        nameSpaceAttribute = null;
    }

    /**
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String,
     *      java.lang.String, java.lang.String)
     */
    public void endElement(String uri, String localName, String qualifiedName) throws SAXException {

        super.endElement(uri, localName, qualifiedName);
        
        String elementName = getLocalElementName(localName, qualifiedName);

        //CommonBaseEvent: 2
        if(parentElement == 2){

            if(elementName.equals("CommonBaseEvent") && anyElementNestingDepth == 0){
                
                CommonBaseEvent commonBaseEvent = ((CommonBaseEvent)(parsedElements.remove(parsedElements.size() - 1)));

                //Do not persist white space (e.g. formatting characters) between elements:
                if (charactersBuffer.toString().trim().length() > 0) {
                    commonBaseEvent.addAny(charactersBuffer.toString().trim());
                }

                if(eventListener != null){
                	eventListener.processCommonBaseEvent(commonBaseEvent);
                }

                parentElement = 1;
                
                reset();                
            }
            else if (anyElementNestingDepth > 0) {

                //Do not persist white space (e.g. formatting characters) between elements:
                if (charactersBuffer.toString().trim().length() > 0) {
                    anyElementXMLFragment.append(XmlUtility.normalize(charactersBuffer.toString()));
                }

                anyElementXMLFragment.append("</");
                anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qualifiedName));
                anyElementXMLFragment.append('>');

                if(--anyElementNestingDepth == 0){                    

                    ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).addAny(anyElementXMLFragment.toString().trim());
                    
                    anyElementXMLFragment.setLength(0);
                }
            } 
        }

        //contextDataElements: 3
        else if(parentElement == 3 && anyElementNestingDepth == 0){

            if(elementName.equals("contextValue")){
                ((ContextDataElement)(parsedElements.get(parsedElements.size() - 1))).setContextValue(charactersBuffer.toString());
            }
            else if(elementName.equals("contextId")){
            ((ContextDataElement)(parsedElements.get(parsedElements.size() - 1))).setContextId(charactersBuffer.toString());
            }
            else if(elementName.equals("contextDataElements")){

                ContextDataElement contextDataElement = ((ContextDataElement)(parsedElements.remove(parsedElements.size() - 1)));
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).addContextDataElement(contextDataElement);

                parentElement = 2;
            }
            else {
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "ContextValue or ContextId", elementName, String.valueOf(line), String.valueOf(column)));                 
            }
        }
        
        //extendedDataElements: 4
        //children: 5
        else if((parentElement == 4) || (parentElement == 5) && anyElementNestingDepth == 0){
            
            if(elementName.equals("values")){
                ((ExtendedDataElement)(parsedElements.get(parsedElements.size() - 1))).getValues().add(charactersBuffer.toString());
            }
            else if(elementName.equals("hexValue")){
                ((ExtendedDataElement)(parsedElements.get(parsedElements.size() - 1))).setHexValue(charactersBuffer.toString());
            }
            else if(elementName.equals("extendedDataElements")){

                ExtendedDataElement extendedDataElement = ((ExtendedDataElement)(parsedElements.remove(parsedElements.size() - 1)));
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).addExtendedDataElement(extendedDataElement);

                parentElement = 2;
            }
            else if(elementName.equals("children")){

                ExtendedDataElement child = ((ExtendedDataElement)(parsedElements.remove(parsedElements.size() - 1)));

                ((ExtendedDataElement)(parsedElements.get(parsedElements.size() - 1))).addChild(child);

                parentElement = 4;
            }
            else {
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "Values, HexValue or Children", elementName, String.valueOf(line), String.valueOf(column)));                 
            }
        }

        //associatedEvents: 6
        else if(parentElement == 6 && anyElementNestingDepth == 0){
            
            if(elementName.equals("associationEngine")){
                ((AssociatedEvent)(parsedElements.get(parsedElements.size() - 1))).setAssociationEngine(charactersBuffer.toString());
            }            
            else if(elementName.equals("associatedEvents")){

                AssociatedEvent associatedEvent = ((AssociatedEvent)(parsedElements.remove(parsedElements.size() - 1)));
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).addAssociatedEvent(associatedEvent);

                parentElement = 2;
            }            
            else if(!elementName.equals("associationEngineInfo")){
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "AssociatedEvents", elementName, String.valueOf(line), String.valueOf(column)));                 
            }
        }

        //msgDataElement: 7
        else if(parentElement == 7 && anyElementNestingDepth == 0){
            
            if(elementName.equals("msgId")){
                ((MsgDataElement)(parsedElements.get(parsedElements.size() - 1))).setMsgId(charactersBuffer.toString());
            }
            else if(elementName.equals("msgIdType")){
                ((MsgDataElement)(parsedElements.get(parsedElements.size() - 1))).setMsgIdType(charactersBuffer.toString());
            }
            else if(elementName.equals("msgCatalogId")){
                ((MsgDataElement)(parsedElements.get(parsedElements.size() - 1))).setMsgCatalogId(charactersBuffer.toString());
            }
            else if(elementName.equals("msgCatalogType")){
                ((MsgDataElement)(parsedElements.get(parsedElements.size() - 1))).setMsgCatalogType(charactersBuffer.toString());
            }
            else if(elementName.equals("msgCatalog")){
                ((MsgDataElement)(parsedElements.get(parsedElements.size() - 1))).setMsgCatalog(charactersBuffer.toString());
            }
            else if(elementName.equals("msgDataElement")){

                MsgDataElement msgDataElement = ((MsgDataElement)(parsedElements.remove(parsedElements.size() - 1)));
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).setMsgDataElement(msgDataElement);

                parentElement = 2;
            }
            else if(!elementName.equals("msgCatalogTokens")){
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "MsgId, MsgIdType, MsgCatalog, MsgCatalogType or MsgCatalogId", elementName, String.valueOf(line), String.valueOf(column)));                 
            }
        }

        //situation: 8
        else if(parentElement == 8 && anyElementNestingDepth == 0){

            if(elementName.equals("situation")){

                Situation situation = ((Situation)(parsedElements.remove(parsedElements.size() - 1)));
                
                ((CommonBaseEvent)(parsedElements.get(parsedElements.size() - 1))).setSituation(situation);

                parentElement = 2;
            }            
            else if(!elementName.equals("situationType")){
                throw new SAXException(LoggingCoreResourceBundle.getString("LOG_EVENT_SAX_PARSER_INCORRECT_XML_ELEMENT_EXC_", "Situation", elementName, String.valueOf(line), String.valueOf(column)));                 
            }            
        }

        //OtherSituation: 9
        else if(parentElement == 9){

            if(elementName.equals("situationType") && anyElementNestingDepth == 0){
                
                OtherSituation otherSituation = ((OtherSituation)(parsedElements.remove(parsedElements.size() - 1)));

                //Do not persist white space (e.g. formatting characters) between elements:
                if (anyElementXMLFragment.toString().trim().length() > 0) {
                	
                    otherSituation.setAny(anyElementXMLFragment.toString().trim());
                
                    anyElementXMLFragment.setLength(0);
                }
    
                //Do not persist white space (e.g. formatting characters) between elements:
                if (charactersBuffer.toString().trim().length() > 0) {
                    otherSituation.setAny(charactersBuffer.toString().trim());
                }                    

                ((Situation)(parsedElements.get(parsedElements.size() - 1))).setSituationType(otherSituation);

                parentElement = 8;
            }
            else if (anyElementNestingDepth > 0) {

                //Do not persist white space (e.g. formatting characters) between elements:
                if (charactersBuffer.toString().trim().length() > 0) {
                    anyElementXMLFragment.append(XmlUtility.normalize(charactersBuffer.toString()));
                }

                anyElementXMLFragment.append("</");
                anyElementXMLFragment.append(getQualifiedElementName(uri, localName, qualifiedName));
                anyElementXMLFragment.append('>');

                anyElementNestingDepth--;
            }             
        }        
        
        // Empty the buffer of characters (e.g. PCDATA):
        charactersBuffer = new StringBuffer();
    }

    /**
     * @see org.xml.sax.helpers.DefaultHandler#warning(org.xml.sax.SAXParseException)
     */
    public void warning(SAXParseException saxParseException) throws SAXException {
        throw saxParseException;
    }

    /**
     * @see org.xml.sax.helpers.DefaultHandler#error(org.xml.sax.SAXParseException)
     */
    public void error(SAXParseException saxParseException) throws SAXException {
        throw saxParseException;
    }

    /**
     * @see org.xml.sax.helpers.DefaultHandler#fatalError(org.xml.sax.SAXParseException)
     */
    public void fatalError(SAXParseException saxParseException) throws SAXException {
        throw saxParseException;
    }
    
    /**
     * Resolves the local name of an element based on either the 
     * parameter local name or qualified name.
     * <p>
     * If the parameter local name is not <code>null</code> or
     * not empty, the local name is returned.  Otherwise, the
     * local name is resolved from the qualified name of the 
     * element.
     * <p>
     * 
     * @param localName The local name of an element.
     * @param qualifiedName The qualified name of an element
     * @return The local name of an element.
     */
    protected String getLocalElementName(String localName, String qualifiedName) {

        if ((localName == null) || (localName.trim().length() == 0)) {

            if (qualifiedName != null) {

                int lastColonIndex = qualifiedName.lastIndexOf(':');

                if (lastColonIndex != -1) {
                    return (qualifiedName.substring(lastColonIndex + 1).trim());
                } 
                else {
                    return (qualifiedName.trim());
                }
            }
            else {
                return "";
            }
        }

        return (localName.trim());
    }


    /**
     * Resolves the qualified name of an element based on either the 
     * parameter local name and URI or qualified name.
     * <p>
     * If the parameter qualified name is not <code>null</code> or
     * not empty, the qualified name is returned.  Otherwise, the
     * qualified name is resolved from the local name and URI of the 
     * element.
     * <p>
     * 
     * @param uri The URI of an element.
     * @param localName The local name of an element.
     * @param qualifiedName The qualified name of an element
     * @return The qualified name of an element.
     */
    protected String getQualifiedElementName(String uri, String localName, String qualifiedName) {

        if ((qualifiedName == null) || (qualifiedName.trim().length() == 0)) {

            if (localName != null) {

                if ((uri != null) && (uri.trim().length() > 0)) {
                    return (uri.concat(":").concat(localName));
                } 
                else {
                    return (localName.trim());
                }
            } 
            else {
                return "";
            }
        }

        return (qualifiedName.trim());
    }
}
