package org.eclipse.tptp.logging.events.cbe.internal.util;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.tptp.logging.events.cbe.AssociationEngine;
import org.eclipse.tptp.logging.events.cbe.CommonBaseEvent;
import org.eclipse.tptp.logging.events.cbe.util.EventListener;

/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventListenerImpl.java,v 1.3 2008/02/28 02:33:17 jkubasta Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Implementation of the 
 * {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
 * interface for caching 
 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s and 
 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s for future 
 * access when de-serializing a stream of Common Base Event and Association 
 * Engine XML fragments.
 * <p>
 * The {@link org.eclipse.tptp.logging.events.cbe.util.EventFormatter} provides 
 * methods for callers to register an implementation of this listener interface,
 * such as this implementation, when de-serializing a stream of Common Base Event 
 * and Association Engine XML fragments.  
 * <p>
 * Noteworthy, callers to the methods in the 
 * {@link org.eclipse.tptp.logging.events.cbe.util.EventFormatter} are 
 * responsible for listener instantiation, registration and clean-up.
 * <p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @version     February 13, 2006
 * @since       February 6, 2006
 * @see         org.eclipse.tptp.logging.events.cbe.util.EventFormatter
 * @see         org.eclipse.tptp.logging.events.cbe.util.EventListener
 */
public class EventListenerImpl implements EventListener{

    /**
     * List or cache of <code>CommonBaseEvent</code>s.
     */
    protected List commonBaseEvents = null;

    /**
     * List or cache of <code>AssociationEngine</code>s.
     */
    protected List associationEngines = null;

    /**
     * No argument constructor.
     * <p>
     */
    public EventListenerImpl() {

        associationEngines = new ArrayList();

        commonBaseEvents = new ArrayList();
    }
    
    /**
     * Initializes the <code>EventListenerImpl</code> to a newly 
     * constructed state.
     * <p>
     */
    public void init() {

        associationEngines.clear();

        commonBaseEvents.clear();
    }

	/**
	 * Receives notification to cache a 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} for future 
	 * access when de-serializing a Common Base Event XML fragment.
	 * <p>
	 * Once a Common Base Event XML fragment has been de-serialized by the 
	 * {@link org.eclipse.tptp.logging.events.cbe.util.EventFormatter}, 
	 * a {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} is 
	 * constructed and broadcast to this method of the registered  
	 * {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
	 * implementation for caching the 
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}.
	 * <p>
	 * 
	 * @param commonBaseEvent The {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent} 
	 * to be cached.
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener#processCommonBaseEvent(org.eclipse.tptp.logging.events.cbe.CommonBaseEvent)
	 */
	public void processCommonBaseEvent(CommonBaseEvent commonBaseEvent){
		
		if(commonBaseEvent != null){
			commonBaseEvents.add(commonBaseEvent);
		}
	}
	
	/**
	 * Receives notification to cache a 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} for future 
	 * access when de-serializing an Association Engine XML fragment.
	 * <p>
	 * Once an Association Engine XML fragment has been de-serialized by the 
	 * {@link org.eclipse.tptp.logging.events.cbe.util.EventFormatter}, 
	 * an {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} is 
	 * constructed and broadcast to this method of the registered  
	 * {@link org.eclipse.tptp.logging.events.cbe.util.EventListener} 
	 * implementation for caching the 
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}.
	 * <p>
	 * 
	 * @param associationEngine The {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine} 
	 * to be cached.
	 * @see org.eclipse.tptp.logging.events.cbe.util.EventListener#processAssociationEngine(org.eclipse.tptp.logging.events.cbe.AssociationEngine)
	 */
	public void processAssociationEngine(AssociationEngine associationEngine){
		
		if(associationEngine != null){
			associationEngines.add(associationEngine);
		}
	}

    /**
     * Returns the list or cache of one or more
	 * {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s 
	 * created from de-serializing Common Base Event XML fragments.
     * <p>
     * 
     * @return Array of {@link org.eclipse.tptp.logging.events.cbe.CommonBaseEvent}s, otherwise <code>null</code>.
     */
    public CommonBaseEvent[] getCommonBaseEvents() {

        if (!commonBaseEvents.isEmpty()) { 
            return (((CommonBaseEvent[]) (commonBaseEvents.toArray(new CommonBaseEvent[commonBaseEvents.size()])))); 
        }

        return null;
    }

    /**
     * Returns a list or cache of one or more
	 * {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s 
	 * created from de-serializing Association Engine XML fragments.
     * <p>
     * 
     * @return Array of {@link org.eclipse.tptp.logging.events.cbe.AssociationEngine}s, otherwise <code>null</code>.
     */
    public AssociationEngine[] getAssociationEngines() {

        if (!associationEngines.isEmpty()) { 
            return (((AssociationEngine[]) (associationEngines.toArray(new AssociationEngine[associationEngines.size()])))); 
        }

        return null;
    }
}
