package org.eclipse.hyades.internal.logging.core.internationalization;

import java.util.Date;
import java.util.StringTokenizer;

import com.ibm.icu.text.MessageFormat;
import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.util.GregorianCalendar;
import com.ibm.icu.util.TimeZone;
import com.ibm.icu.util.ULocale;
import com.ibm.icu.text.UTF16;


/**********************************************************************
 * Copyright (c) 2006,2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ICUInternationalizationUtilities.java,v 1.2 2007/04/09 04:35:53 cjin Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * <a href="http://icu.sourceforge.net/">International 
 * Components for Unicode (ICU)</a> internationalization utilities 
 * class containing operations for <a href="http://icu.sourceforge.net/">International 
 * Components for Unicode (ICU)</a>-enabled run-times.  That is, 
 * these internationalization utilities require the <a href="http://icu.sourceforge.net/">International 
 * Components for Unicode (ICU)</a> dependency at run-time.
 * <p>
 * 
 * @author      cindyjin
 * @author      Paul E. Slauenwhite
 * @version     March 30, 2007
 * @since       April 7, 2006
 */
final class ICUInternationalizationUtilities {
	
    /**
     * Resolves the formatted date string associated with the parameter long representing a 
     * Coordinated Universal Time (UTC) date in milliseconds using the parameter date format 
     * pattern based on the parameter {@link ULocale} and time zone ID.
     * <p>
     * It is assumed that the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time.  As such, date formatting is done by the 
	 * {@link SimpleDateFormat} class using the {@link ULocale}, 
	 * {@link GregorianCalendar} and {@link TimeZone} classes.  That is, the parameter date 
	 * format pattern is based on the syntax and symbols as specified in the class comment header of the 
	 * {@link SimpleDateFormat} class.  Also, the parameter time zone ID is based on the syntax 
	 * as specified in the class comment header of the {@link TimeZone} class.
     * <p>
     * 
     * @param milliseconds A long representing a Coordinated Universal Time (UTC) time stamp in milliseconds.
     * @param dateFormatPattern The date format pattern.
     * @param locale The {@link ULocale} for date formatting.
     * @param timeZoneID The time zone ID for date formatting.
     * @return The date as a formatted string using the parameter date format pattern, locale and time zone ID.
     * @throws IllegalArgumentException If the date format pattern is invalid.
	 * @see SimpleDateFormat
	 * @see ULocale
	 * @see GregorianCalendar
	 * @see TimeZone
	 * @see ICUInternationalizationUtilities#format(long, String, ULocale, String)
     */
    protected static String format(long milliseconds, String dateFormatPattern, ULocale locale, String timeZoneID) throws IllegalArgumentException {
    	
    	SimpleDateFormat simpleDateFormat = new SimpleDateFormat(dateFormatPattern, locale);
        simpleDateFormat.setCalendar(new GregorianCalendar(TimeZone.getTimeZone(timeZoneID)));
        
        String formattedDateString = simpleDateFormat.format(new Date(milliseconds));
        
        //Work-around: 
        if(timeZoneID.equals("GMT")){
        	return (formattedDateString.substring(0, formattedDateString.lastIndexOf("+00:00")));      
        }
        
        return formattedDateString;
    }
    
    /**
     * Resolves the formatted message associated with the localized parameter <code>message</code>
     * and message parameters based on the parameter {@link ULocale}.
     * <p>
	 * It is assumed that the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time.  As such, message formatting is done by the 
	 * {@link MessageFormat} class using the {@link ULocale} class.
     * <p>
     * 
     * @param message The localized message to be formatted.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @param locale The {@link ULocale} for message formatting.
     * @return The localized and formatted message.
	 * @throws IllegalArgumentException If the localized message is invalid and cannot be formatted.
	 * @see MessageFormat
	 * @see ULocale
     */
    protected static String format(String message, Object[] arguments, ULocale locale) throws IllegalArgumentException {
		return (new MessageFormat(message, locale).format(arguments));
    }

	/**
	 * Tokenizes the parameter {@link String} to an array of 
	 * tokens based on the separating token characters 
	 * in the <code>delimiters</code> parameter.  The separating 
	 * token characters are not returned.
	 * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for 
	 * Unicode (ICU)</a> JAR(s) are available on the class path at run-time, the 
	 * {@link StringTokenizer} class is used, otherwise the 
	 * {@link StringTokenizer} class is used.
	 * <p>
	 * 
	 * @param  string The {@link String} to be tokenized.
	 * @param  delimiters The separating token characters.
	 * @return Array containing the tokenized parameter {@link String}, otherwise <code>null</code>.
	 * @see    StringTokenizer    
	 * @see    StringTokenizer
	 */
	protected static String[] tokenize(String string, String delimiters){
		
		String[] tokens = null;
		
		if(string != null){
			
			StringTokenizer tokenizer = new StringTokenizer(string,delimiters);
	        tokens = new String[tokenizer.countTokens()];
	            	
	        for (int counter = 0; counter < tokens.length; counter++) {
	        	tokens[counter] = tokenizer.nextToken();
			}
		}

		return tokens;
	}
	/**
	 * Returns the char value at the specified index for a given String.
	 * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for 
	 * Unicode (ICU)</a> JAR(s) are available on the class path at run-time, the 
	 * {@link com.ibm.icu.text.UTF16} class is used, otherwise the 
	 * {@link String} class is used.
	 * <p>
	 * @param string The String to get the char value at the specified index from.
	 * @param index the index of the char value.
	 * @return the char value at the specified index of this string.
	 * @throws IndexOutOfBoundsException
	 * @see String
	 * @see com.ibm.icu.text.UTF16
	 */
	
	protected static char charAt(String string, int index)throws IndexOutOfBoundsException 
	{
		int ch = UTF16.charAt(string, index);
		return (char)ch;
		 
	}
}