package org.eclipse.hyades.internal.logging.core.internationalization;

import java.util.StringTokenizer;

import org.eclipse.hyades.logging.core.LoggingCoreUtilities;

import com.ibm.icu.util.ULocale;

/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: InternationalizationUtilities.java,v 1.6 2008/12/15 15:34:43 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Internationalization utilities facade class containing operations to eliminate 
 * the <a href="http://icu.sourceforge.net/">International Components for 
 * Unicode (ICU)</a> dependency in some run-times.
 * <p>
 * 
 * @author      cindyjin
 * @author      Paul E. Slauenwhite
 * @version     March 30, 2007
 * @since       April 6, 2006
 */
public final class InternationalizationUtilities {

	/**
     * Static flag for quickly determining if the
     * ICU classes are available on the classpath.
     * <p>
     * By default, the ICU classes are assumed to 
     * be available on the classpath.
     */
    protected static boolean isICUAvailable = true;
   
    /**
     * Resolves the formatted date string associated with the parameter long representing a 
     * Coordinated Universal Time (UTC) date in milliseconds using the parameter date format 
     * pattern based on the parameter {@link Locale} and time zone ID.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, date formatting is done by the 
	 * {@link com.ibm.icu.text.SimpleDateFormat} class using the {@link com.ibm.icu.util.ULocale} (parameter {@link Locale}
	 * converted to a {@link com.ibm.icu.util.ULocale} - see {@link com.ibm.icu.util.ULocale#forLocale(Locale)}), 
	 * {@link com.ibm.icu.util.GregorianCalendar} and {@link com.ibm.icu.util.TimeZone} (parameter time zone ID
	 * converted to a {@link com.ibm.icu.util.TimeZone} - see {@link com.ibm.icu.util.TimeZone#getTimeZone(String)}) classes, 
	 * otherwise date formatting is done by the {@link java.text.SimpleDateFormat} class using the {@link java.util.Locale}, 
	 * {@link java.util.GregorianCalendar} and {@link java.util.TimeZone} classes.  That is, if the 
	 * <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, the parameter date format pattern is based on the syntax and symbols 
	 * as specified in the class comment header of the {@link com.ibm.icu.text.SimpleDateFormat} class, otherwise the 
	 * parameter date format pattern is based on the syntax and symbols as specified in the class comment header of the 
	 * {@link java.util.SimpleDateFormat} class.  Also, if the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> 
	 * JAR(s) are available on the class path at run-time, the parameter time zone ID is based on the syntax 
	 * as specified in the class comment header of the {@link com.ibm.icu.util.TimeZone} class, otherwise the 
	 * parameter time zone ID is based on the syntax as specified in the class comment header of the 
	 * {@link java.util.TimeZone} class.
     * <p>
     * 
     * @param milliseconds A long representing a Coordinated Universal Time (UTC) time stamp in milliseconds.
     * @param dateFormatPattern The date format pattern.
     * @param locale The {@link Locale} for date formatting.
     * @param timeZoneID The time zone ID for date formatting.
     * @return The date as a formatted string using the parameter date format pattern, locale and time zone ID.
     * @throws IllegalArgumentException If the date format pattern is invalid.
	 * @see java.text.SimpleDateFormat
	 * @see java.util.Locale
	 * @see java.util.GregorianCalendar
	 * @see java.util.TimeZone
	 * @see com.ibm.icu.text.SimpleDateFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see com.ibm.icu.util.GregorianCalendar
	 * @see com.ibm.icu.util.TimeZone
	 * @see InternationalizationUtilities#format(long, String, com.ibm.icu.util.ULocale, String)
     */
    public static String format(long milliseconds, String dateFormatPattern, ULocale locale, String timeZoneID) throws IllegalArgumentException {

    	if (isICUAvailable) {

			try {
		        return (ICUInternationalizationUtilities.format(milliseconds, dateFormatPattern, locale, timeZoneID));
			} 
			catch (NoClassDefFoundError n) {
				isICUAvailable = false;
			}
		}

        return (JavaInternationalizationUtilities.format(milliseconds, dateFormatPattern, locale, timeZoneID));
    }
    
    
    /**
     * Resolves the formatted message associated with the localized parameter <code>message</code>
     * and message parameters based on the default locale.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * 
     * @param message The localized message to be formatted.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized and formatted message.
	 * @throws IllegalArgumentException If the localized message is invalid and cannot be formatted.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see InternationalizationUtilities#format(String, Object[], java.util.Locale)
	 * @see InternationalizationUtilities#format(String, Object[], com.ibm.icu.util.ULocale)
     */
    public static String format(String message, Object[] arguments) throws IllegalArgumentException {
		return (format(message, arguments, com.ibm.icu.util.ULocale.forLocale(LoggingCoreUtilities.getClientLocale())));
    }
    
    /**
     * Resolves the formatted message associated with the localized parameter <code>message</code>
     * and message parameters based on the parameter {@link Locale}.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class (parameter {@link Locale}
	 * converted to a {@link com.ibm.icu.util.ULocale} - see {@link com.ibm.icu.util.ULocale#forLocale(Locale)}), 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * 
     * @param message The localized message to be formatted.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @param locale The {@link Locale} for message formatting.
     * @return The localized and formatted message.
	 * @throws IllegalArgumentException If the localized message is invalid and cannot be formatted.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see InternationalizationUtilities#format(String, Object[], com.ibm.icu.util.ULocale)
     */
    public static String format(String message, Object[] arguments, ULocale locale) throws IllegalArgumentException {

    	if (isICUAvailable) {

			try {
				return (ICUInternationalizationUtilities.format(message, arguments, locale));
			} 
			catch (NoClassDefFoundError n) {
				isICUAvailable = false;
			}
		}

    	return (JavaInternationalizationUtilities.format(message, arguments, locale));
    }
    

    /**
	 * Tokenizes the parameter {@link String} to an array of tokens based on the following separating token characters:
	 * <p>
	 * <ul>
	 * <li>Space (e.g. ' ')</li>
	 * <li>Tab (e.g. '\t')</li>
	 * <li>Newline (e.g. '\n')</li>
	 * <li>Carriage-return (e.g. '\r')</li>
	 * <li>Form-feed (e.g. '\f')</li>
	 * <p>
	 * The separating token characters are not returned.
	 * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) are
	 * available on the class path at run-time, the {@link com.ibm.icu.util.StringTokenizer} class is used, otherwise
	 * the {@link StringTokenizer} class is used.
	 * <p>
	 * 
	 * @param string
	 *            The {@link String} to be tokenized.
	 * @return Array containing the tokenized parameter {@link String}, otherwise <code>null</code>.
	 * @see StringTokenizer
	 * @see com.ibm.icu.util.StringTokenizer
	 * @see InternationalizationUtilities#tokenize(String, String)
	 */
	public static String[] tokenize(String string){
		return (tokenize(string, " \t\n\r\f"));
	}	
	
	/**
	 * Tokenizes the parameter {@link String} to an array of 
	 * tokens based on the separating token characters 
	 * in the <code>delimiters</code> parameter.  The separating 
	 * token characters are not returned.
	 * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for 
	 * Unicode (ICU)</a> JAR(s) are available on the class path at run-time, the 
	 * {@link com.ibm.icu.util.StringTokenizer} class is used, otherwise the 
	 * {@link StringTokenizer} class is used.
	 * <p>
	 * 
	 * @param  string The {@link String} to be tokenized.
	 * @param  delimiters The separating token characters.
	 * @return Array containing the tokenized parameter {@link String}, otherwise <code>null</code>.
	 * @see    StringTokenizer    
	 * @see    com.ibm.icu.util.StringTokenizer
	 */
	public static String[] tokenize(String string, String delimiters){
		
		if (isICUAvailable) {

			try {
	        	return (ICUInternationalizationUtilities.tokenize(string, delimiters));
	        } 
	        catch (NoClassDefFoundError n) {
	        	isICUAvailable = false;
	        } 
	    }

        return (JavaInternationalizationUtilities.tokenize(string, delimiters));
	}
	/**
	 * Returns the char value at the specified index for a given String.
	 * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for 
	 * Unicode (ICU)</a> JAR(s) are available on the class path at run-time, the 
	 * {@link com.ibm.icu.text.UTF16} class is used, otherwise the 
	 * {@link String} class is used.
	 * <p>
	 * @param string The String to get the char value at the specified index from.
	 * @param index the index of the char value.
	 * @return the char value at the specified index of this string.
	 * @throws IndexOutOfBoundsException
	 * @see String
	 * @see com.ibm.icu.text.UTF16
	 */
	public static char charAt(String string, int index )throws IndexOutOfBoundsException 
	{
		if (isICUAvailable) {

			try {
	        	return (ICUInternationalizationUtilities.charAt(string, index));
	        } 
	        catch (NoClassDefFoundError n) {
	        	isICUAvailable = false;
	        } 
	    }
		return JavaInternationalizationUtilities.charAt(string, index);

	}
}