package org.eclipse.hyades.logging.core;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;

/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoggingCoreResourceBundle.java,v 1.7 2008/12/15 15:34:43 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Logging Core (org.eclipse.hyades.logging.core) resource bundle.
 * <p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @version     April 7, 2006
 * @since       April 7, 2006
 */
public class LoggingCoreResourceBundle {

    /**
     * Static reference to the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code>
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     */
    protected static ResourceBundle resourceBundle = null;

	/**
     * Static constant representing the name 
     * (<code>org.eclipse.hyades.logging.core.properties.message</code>) 
     * of the resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     */
    protected static final String RESOURCE_BUNDLE_NAME = "org.eclipse.hyades.logging.core.properties.message";

    /**
     * Resolves the localized message associated with the parameter <code>key</code>
     * based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * Noteworthy, the resultant message is not formatted (e.g. no message parameter substitution). 
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @return The localized message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
     */
    public static String getString(String key) {

        synchronized(RESOURCE_BUNDLE_NAME){
            
            //Load the resource bundle:
            if (resourceBundle == null || !resourceBundle.getLocale().equals(LoggingCoreUtilities.getClientLocale())) {                
                //Attempt to load the 'org.eclipse.hyades.logging.core.properties.message[_<language code>[_<country code>]].properties' resource bundle:
                try {
                    resourceBundle = ResourceBundle.getBundle(RESOURCE_BUNDLE_NAME,LoggingCoreUtilities.getClientLocale());
                }
                catch (MissingResourceException m) {
                    return key;
                }
            }       

            try{
                return (resourceBundle.getString(key.trim()).trim());    
            }
            catch(Exception e){
            }
        } 
        return key;
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameter based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argument The first message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see LoggingCoreResourceBundle#getString(String, Object[])
     */
    public static String getString(String key, Object argument) {
        return (getString(key, new Object[] { argument}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see LoggingCoreResourceBundle#getString(String, Object[])
     */
    public static String getString(String key, Object argumentA, Object argumentB) {
        return (getString(key, new Object[] { argumentA, argumentB}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @param argumentC The third message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see LoggingCoreResourceBundle#getString(String, Object[])
     */
    public static String getString(String key, Object argumentA, Object argumentB, Object argumentC) {
        return (getString(key, new Object[] { argumentA, argumentB, argumentC}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param argumentA The first message parameter for formatting in the localized and formatted message.
     * @param argumentB The second message parameter for formatting in the localized and formatted message.
     * @param argumentC The third message parameter for formatting in the localized and formatted message.
     * @param argumentD The fourth message parameter for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see LoggingCoreResourceBundle#getString(String, Object[])
     */
    public static String getString(String key, Object argumentA, Object argumentB, Object argumentC, Object argumentD) {
        return (getString(key, new Object[] { argumentA, argumentB, argumentC, argumentD}));
    }

    /**
     * Resolves the localized and formatted message associated with the parameter <code>key</code>
     * and message parameters based on the default locale from the 
     * <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, message formatting is done by the 
	 * {@link com.ibm.icu.text.MessageFormat} class using the {@link com.ibm.icu.util.ULocale} class, 
	 * otherwise message formatting is done by the {@link java.text.MessageFormat} class using the {@link Locale} 
	 * class.
     * <p>
     * If the <code>key</code> does not exist in the <code>plugin[_&lt;language code&gt;[_&lt;country code&gt;]].properties</code> 
     * resource bundle for the <code>org.eclipse.hyades.logging.*</code> classes, the <code>key</code> is
     * returned.
     * <p>
     * 
     * @param key The <code>key</code> of the message in the resource bundle.
     * @param arguments The array of message parameters for formatting in the localized and formatted message.
     * @return The localized and formatted message associated with the parameter <code>key</code> from the resource bundle, otherwise the <code>key</code>.
	 * @see java.text.MessageFormat
	 * @see java.util.Locale
	 * @see com.ibm.icu.text.MessageFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see LoggingCoreResourceBundle#getString(String)
     */
    public static String getString(String key, Object[] arguments) {

    	try {
	    	return (InternationalizationUtilities.format(getString(key), arguments));
		} 
		catch (Exception e) {
			return key;
		}
    }
}