package org.eclipse.hyades.logging.core;

import org.eclipse.hyades.internal.logging.core.internationalization.InternationalizationUtilities;

import com.ibm.icu.text.SimpleDateFormat;
import com.ibm.icu.util.Calendar;
import com.ibm.icu.util.GregorianCalendar;
import com.ibm.icu.util.SimpleTimeZone;
import com.ibm.icu.util.ULocale;

import java.util.Locale;

/**********************************************************************
 * Copyright (c) 2006, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoggingCoreUtilities.java,v 1.6 2008/12/15 15:34:43 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/**
 * Logging Core (org.eclipse.hyades.logging.core) utilities.
 * <p>
 * 
 * @author      cindyjin
 * @author      Paul E. Slauenwhite
 * @version     March 30, 2007
 * @since       April 7, 2006
 */
public class LoggingCoreUtilities {

	//Notes: 1) Set the locale of the date format to English since the 
    //          XSD:dateTime format is locale agnostic (week not represented).
    //       2) Set the time zone of the date format to UTC since the 
    //          XSD:dateTime format is time zone agnostic (UTC or Zulu (e.g. Z) 
    //          time zone).
    protected static SimpleDateFormat xsdDateTimeFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss", ULocale.ENGLISH);

    //Notes: 1) Set the locale of the cached calendar to English since the 
    //          XSD:dateTime format is locale agnostic (week not represented).
    //       2) Set the time zone of the cached calendar to UTC since the 
    //          XSD:dateTime format is time zone agnostic (UTC or Zulu (e.g. Z) 
    //          time zone).
    protected static Calendar cachedCalendar = new GregorianCalendar(new SimpleTimeZone(0, "UTC"), ULocale.ENGLISH);

    static {

    	cachedCalendar.setLenient(false);

        //Notes: 1) Set the locale of the date format's calendar to English since the 
        //          XSD:dateTime format is locale agnostic (week not represented).
        //       2) Set the time zone of the date format's calendar to UTC since the 
        //          XSD:dateTime format is time zone agnostic (UTC or Zulu (e.g. Z) 
        //          time zone).
    	Calendar xsdDateTimeFormatCalendar = new GregorianCalendar(new SimpleTimeZone(0, "UTC"), ULocale.ENGLISH);
    	xsdDateTimeFormatCalendar.setLenient(false);

        xsdDateTimeFormat.setLenient(false);
        xsdDateTimeFormat.setCalendar(xsdDateTimeFormatCalendar);
    }
    
    /**
     * Static constant representing the default date format
     * (<code>MMMM d, yyyy h:mm:ss.SSS a z</code>).
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleDateFormat</code>.
     * <p>
     * @see java.util.SimpleDateFormat
     */
	protected static final String DEFAULT_FORMATTED_DATE_FORMAT = "MMMM d, yyyy h:mm:ss.SSS a z";

	/**
     * Converts an XML Schema (XSD) 
     * <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a> time stamp 
     * to a long time stamp.
     * <p>
     * The XSD <code>dateTime</code> time stamp represents a finite-length sequence
     * of characters following the following format pattern:
     * <p>
     * <code>
     * yyyy-MM-dd'T'HH:mm:ss[.SSSS]['Z'|Z]
     * </code>
     * <p>
     * For more information on the meaning of the individual symbols in the XSD 
     * <code>dateTime</code> time stamp format pattern, see the class comment header 
     * for {@link SimpleDateFormat}.
     * <p>
     * For example, March 14, 2006 11:32:01.001 AM UTC/GMT would be represented as:
     * <p>
     * <code>
     * 2006-03-14T11:32:01.001Z
     * </code>
     * <p>
     * See the XSD <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a>
     * primitive datatype documentation for more information.
     * <p>
     * The long time stamp represents the difference, measured in milliseconds, between 
     * the current time (Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT)) 
     * and the standard base time.  The standard base time is known as "the epoch", namely 
     * January 1, 1970 12:00:00 AM Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT).  
     * <p>
     * <b>Note:</b> Due to inconsistencies between XSD V1.0 and later versions of the XSD 
     * specification on representing zero or negative XSD <code>dateTime</code> time stamps, 
     * such as year 0 and year 1 Before Common Era (BCE or BC), this  method will throw an 
     * {@link IllegalArgumentException} for parameter XSD <code>dateTime</code> time stamps 
     * before <code>0001-01-01T00:00:00.000Z</code> or <code>-62135769600000</code>.  See the 
     * XSD dateTime <a href="http://www.w3.org/TR/xmlschema-2/#year-zero">note</a> for more 
     * information.
     * <p>
     * 
     * @param xsdDateTime The XSD <code>dateTime</code> time stamp to be converted to a long (milliseconds) time stamp.
     * @return The long (milliseconds) time stamp representation of the parameter XSD <code>dateTime</code> time stamp.
     * @throws IllegalArgumentException If the parameter XSD <code>dateTime</code> time stamp is not a valid XSD <code>dateTime</code> time stamp.
     */    
    public static long convertXsdDateTimeToMilliseconds(String xsdDateTime) {
    	
    	try {
        
        	long millisecondTimeStamp = 0;

        	//Step 1: Convert the required characters (e.g. [-]yyyy-MM-ddTHH:mm:ss) of the XSD dateTime string to a millisecond time stamp:
        	//Note:  The year (e.g. [-]yyyy) is a four-or-more digit optionally negative-signed numeral .
    		synchronized (xsdDateTimeFormat) {
    			millisecondTimeStamp = xsdDateTimeFormat.parse(xsdDateTime).getTime();
            }
    		
    		//Step 2: Process the optional characters (e.g. [.SSSS]['Z'|Z]) of the XSD dateTime string, if any exist:
    		//Note: 9 is the length of "THH:mm:ss".
    		String optionalCharactersString = xsdDateTime.substring(xsdDateTime.indexOf('T') + 9).trim();
    		int optionalCharactersStringLength = optionalCharactersString.length();
    		
    		if(optionalCharactersStringLength > 0){

    			//Step 2A: Add/subtract the optional time zone offset from UTC (e.g. ['Z'|Z]), if one exists:
    			int timeZoneStartIndex = optionalCharactersStringLength;
    			
        		//Note: 1 is the length of "Z".
		    	if(InternationalizationUtilities.charAt(optionalCharactersString,(optionalCharactersStringLength - 1)) == 'Z'){
		    		timeZoneStartIndex--;		    			
		    	}

		    	//Note: 6 is the length of "[+|-]HH:mm".
		    	else if(optionalCharactersStringLength >= 6){
		    		
		    		switch (InternationalizationUtilities.charAt(optionalCharactersString,(optionalCharactersStringLength - 6))) {
						
		    			case '+':{
							
				    		millisecondTimeStamp -= ((Long.parseLong(optionalCharactersString.substring((optionalCharactersStringLength - 5), (optionalCharactersStringLength - 3))) * 60 * 60 * 1000) + (Long.parseLong(optionalCharactersString.substring(optionalCharactersStringLength - 2)) * 60 * 1000));
							
				    		timeZoneStartIndex = (optionalCharactersStringLength - 6);			    			
				    		
				    		break;
						}
						case '-':{
							
				    		millisecondTimeStamp += ((Long.parseLong(optionalCharactersString.substring((optionalCharactersStringLength - 5), (optionalCharactersStringLength - 3))) * 60 * 60 * 1000) + (Long.parseLong(optionalCharactersString.substring(optionalCharactersStringLength - 2)) * 60 * 1000));
							
				    		timeZoneStartIndex = (optionalCharactersStringLength - 6);			    			

				    		break;
						}
					}
		    	}
				    
	    		if(timeZoneStartIndex != 0){
	    			
			    	//Step 2B: Add the optional fractional seconds (e.g. [.SSSS]), if one exists: 	
					//Note: Despite "0.5", "0.50", and "0.500" being a numerically equal number of milliseconds.
	    			if((InternationalizationUtilities.charAt(optionalCharactersString,0) == '.') && (Character.isDigit(InternationalizationUtilities.charAt(optionalCharactersString,1)))){
	    		
	    		    	//Note: 1 is the length of ".".
						switch (timeZoneStartIndex - 1) {
									
							case 1:{						
								millisecondTimeStamp += (Long.parseLong(optionalCharactersString.substring(1, timeZoneStartIndex)) * 100);
								break;
							}
							case 2:{						
								millisecondTimeStamp += (Long.parseLong(optionalCharactersString.substring(1, timeZoneStartIndex)) * 10);
								break;
							}
							case 3:{						
								millisecondTimeStamp += Long.parseLong(optionalCharactersString.substring(1, timeZoneStartIndex));
								break;
							}
							default:{
								millisecondTimeStamp += Math.round(Long.parseLong(optionalCharactersString.substring(1, timeZoneStartIndex)) / Math.pow(10, (timeZoneStartIndex - 4)));
								break;
							}
						}
	    			}
	    			
			    	//Step 2C: Invalid optional characters (e.g. [.SSSS]['Z'|Z]) in the XSD dateTime string:
					else{
						throw new Exception();
					}
	    		}	    		
    		}    		
    		
		    return millisecondTimeStamp;
	    } 
		catch (Exception e) {	
	    	throw new IllegalArgumentException(LoggingCoreResourceBundle.getString("LOG_INVALID_XSD_DATE_TIME_VALUE_EXC_",xsdDateTime));
	    }
    }
 
    /**
     * Converts along time stamp to an XML Schema (XSD) 
     * <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a> time stamp.
     * <p>
     * The long time stamp represents the difference, measured in milliseconds, between 
     * the current time (Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT)) 
     * and the standard base time.  The standard base time is known as "the epoch", namely 
     * January 1, 1970 12:00:00 AM Coordinated Universal Time (UTC) or Greenwich Mean Time (GMT).  
     * <p>
     * The XSD <code>dateTime</code> time stamp represents a finite-length sequence
     * of characters following the following format pattern:
     * <p>
     * <code>
     * yyyy-MM-dd'T'HH:mm:ss[.SSSS]['Z'|Z]
     * </code>
     * <p>
     * For more information on the meaning of the individual symbols in the XSD 
     * <code>dateTime</code> time stamp format pattern, see the class comment header 
     * for {@link SimpleDateFormat}.
     * <p>
     * For example, March 14, 2006 11:32:01.001 AM UTC/GMT would be represented as:
     * <p>
     * <code>
     * 2006-03-14T11:32:01.001Z
     * </code>
     * <p>
     * See the XSD <a href="http://www.w3.org/TR/xmlschema-2/#dateTime">dateTime</a>
     * primitive datatype documentation for more information.
     * <p>
     * <b>Note:</b> Due to inconsistencies between XSD V1.0 and later versions of the XSD 
     * specification on representing zero or negative XSD <code>dateTime</code> time stamps, 
     * such as year 0 and year 1 Before Common Era (BCE or BC), this method will return a 
     * positive XSD <code>dateTime</code> time stamp for parameter long (milliseconds) time 
     * stamps before <code>-62135769600000</code> or <code>0001-01-01T00:00:00.000Z</code>.  
     * See the XSD dateTime <a href="http://www.w3.org/TR/xmlschema-2/#year-zero">note</a> for 
     * more information.
     * <p>
     * 
     * @param milliseconds The long (milliseconds) time stamp to be converted to an XSD <code>dateTime</code> time stamp.
     * @return The XSD <code>dateTime</code> time stamp representation of the parameter long (milliseconds) time stamp .
     */
    public static String convertMillisecondsToXsdDateTime(long milliseconds) {

    	synchronized (cachedCalendar) {

	        //Set the millisecond date on the calendar:
	        cachedCalendar.setTimeInMillis(milliseconds);
	        
	        //Create the string buffer and build the XSD:dateTime string:
	        //Notes:  1) All fields less than two numbers in size must be zero padded.
	        //Assumptions: 1) The XSD:dateTime string is at most 24 characters in length 
	        //                (YYYY-MM-dd'T'HH:mm:ss.SSS'Z').
	        StringBuffer stringBuffer = new StringBuffer(24);
	        
	        int year = cachedCalendar.get(Calendar.YEAR);
	        
	        if(year < 10){
	        	stringBuffer.append("000");        	
	        }
	        else if(year < 100){
	        	stringBuffer.append("00");        	
	        }
	        else if(year < 1000){
	        	stringBuffer.append("0");        	
	        }

        	stringBuffer.append(year);        	

	        stringBuffer.append('-');
	        	
	        //NOTE: Numeric months in Java start from 0.
	        int month = (cachedCalendar.get(Calendar.MONTH) + 1);
	        
	        if (month < 10) {
	            stringBuffer.append('0');
	        }
	        	
	        stringBuffer.append(month);
	        stringBuffer.append('-');
	        	
	        int day = cachedCalendar.get(Calendar.DAY_OF_MONTH);
	        
	        if (day < 10) {
	            stringBuffer.append('0');
	        }
	        	
	        stringBuffer.append(day);
	        stringBuffer.append('T');
	        	
	        int hour = cachedCalendar.get(Calendar.HOUR_OF_DAY);
	        
	        if (hour < 10) {
	            stringBuffer.append('0');
	        }
	        	
	        stringBuffer.append(hour);
	        stringBuffer.append(':');
	        	
	        int minute = cachedCalendar.get(Calendar.MINUTE);
	        
	        if (minute < 10) {
	            stringBuffer.append('0');
	        }
	        	
	        stringBuffer.append(minute);
	        stringBuffer.append(':');
	        
	        int second = cachedCalendar.get(Calendar.SECOND);
	        
	        if (second < 10) {
	            stringBuffer.append('0');
	        }
	        	
	        stringBuffer.append(String.valueOf(second));
	        	
	        stringBuffer.append('.');
	        	
	        int millisecondValue = cachedCalendar.get(Calendar.MILLISECOND);
	        
	        if (millisecondValue < 10) {
	            stringBuffer.append("00");
	        } 
	        else if (millisecondValue < 100) {
	            stringBuffer.append('0');
	        }
	        	
	        stringBuffer.append(millisecondValue);
	        	
	        //Notes:  1) All XSD:dateTimes are time zone agnostic (UTC or Zulu (e.g. Z) 
	        //           time zone).
	        stringBuffer.append('Z');
	        
	        return (stringBuffer.toString());
    	}
    }
    
	/**
     * Converts a long representing a Coordinated Universal Time (UTC) date in 
     * milliseconds to a formatted string using the default date format pattern.  
     * <p>
     * The default date format is:
     * <p>
     * MMMM d, yyyy h:mm:ss.SSS a z
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleDateFormat</code>.
     * <p>
     * The time zone of the returned string is Coordinated Universal Time (UTC), represented
     * as Greenwich Mean Time (GMT).
     * <p>
     * 
     * @param milliseconds
     *            A long representing a Coordinated Universal Time (UTC) time stamp in milliseconds.
     * @return The date as a formatted string using the default date format pattern (e.g. MMMM d, yyyy h:mm:ss.SSS a z).
     * @throws IllegalArgumentException
     *             If the long representing a Coordinated Universal Time (UTC) time stamp in milliseconds is negative.
     * @see java.util.SimpleDateFormat
     */
    public static String getFormattedDateString(long milliseconds) throws IllegalArgumentException{
        return (getFormattedDateString(milliseconds,DEFAULT_FORMATTED_DATE_FORMAT));   
    }
    
    /**
     * Converts a long representing a Coordinated Universal Time (UTC) date in 
     * milliseconds to a formatted string using the parameter date format pattern.  
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, date formatting is done by the 
	 * {@link com.ibm.icu.text.SimpleDateFormat} class using the {@link com.ibm.icu.util.ULocale} (parameter {@link ULocale}
	 * converted to a {@link com.ibm.icu.util.ULocale} - see {@link com.ibm.icu.util.ULocale#forLocale(Locale)}), 
	 * {@link com.ibm.icu.util.GregorianCalendar} and {@link com.ibm.icu.util.TimeZone} (parameter time zone ID
	 * converted to a {@link com.ibm.icu.util.TimeZone} - see {@link com.ibm.icu.util.TimeZone#getTimeZone(String)}) classes, 
	 * otherwise date formatting is done by the {@link java.text.SimpleDateFormat} class using the {@link java.util.Locale}, 
	 * {@link java.util.GregorianCalendar} and {@link java.util.TimeZone} classes.  That is, if the 
	 * <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, the parameter date format pattern is based on the syntax and symbols 
	 * as specified in the class comment header of the {@link com.ibm.icu.text.SimpleDateFormat} class, otherwise the 
	 * parameter date format pattern is based on the syntax and symbols as specified in the class comment header of the 
	 * {@link java.util.SimpleDateFormat} class.
     * <p>
     * The time zone of the returned string is Coordinated Universal Time (UTC), represented
     * as Greenwich Mean Time (GMT).
     * <p>
     * 
     * @param milliseconds
     *            A long representing a Coordinated Universal Time (UTC) time stamp in milliseconds.
     * @param dateFormatPattern
     * 			  The date format pattern.
     * @return The date as a formatted string using the parameter date format pattern.
     * @throws IllegalArgumentException If the date format pattern invalid.
	 * @see java.text.SimpleDateFormat
	 * @see java.util.Locale
	 * @see java.util.GregorianCalendar
	 * @see java.util.TimeZone
	 * @see com.ibm.icu.text.SimpleDateFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see com.ibm.icu.util.GregorianCalendar
	 * @see com.ibm.icu.util.TimeZone
     */
    public static String getFormattedDateString(long milliseconds, String dateFormatPattern) throws IllegalArgumentException{
    	        
        //Note: Set the locale of the date format to English since the 
        //Gregorian calendar format is used which is locale agnostic.
        return (InternationalizationUtilities.format(milliseconds, dateFormatPattern, ULocale.ENGLISH, "GMT"));
    }
    
    /**
     * Converts a string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date 
     * to a formatted string using the default date format pattern.  
     * <p>
     * For more information on the XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date, 
     * see <a href="http://www.w3.org/TR/NOTE-datetime">http://www.w3.org/TR/NOTE-datetime</a>.
     * <p>
     * The default date format is:
     * <p>
     * MMMM d, yyyy h:mm:ss.SSS a z
     * <p>
     * For more information on the meaning of the individual symbols in the
     * default date format pattern, see the class comment header for 
     * <code>java.util.SimpleDateFormat</code>.
     * <p>
     * If the parameter string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date does not
     * contain any time zone information, the time zone of the returned string is Coordinated Universal Time (UTC), 
     * represented as Greenwich Mean Time (GMT).  Otherwise, the time zone of the returned string is represented as a
     * signed offset from Greenwich Mean Time (GMT).  For example, 'GMT-05:00' for Eastern Standard Time.
     * <p>
     * 
     * @param xsdDateTime
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @return The date as a formatted string using the default date format pattern (e.g. MMMM d, yyyy h:mm:ss.SSS a z).
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid.
     * @see java.util.SimpleDateFormat
     */
    public static String getFormattedDateString(String xsdDateTime) throws IllegalArgumentException{
        return (getFormattedDateString(xsdDateTime,DEFAULT_FORMATTED_DATE_FORMAT));
    }
        
    /**
     * Converts a string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date 
     * to a formatted string using the parameter date format pattern.  
     * <p>
     * For more information on the XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date, 
     * see <a href="http://www.w3.org/TR/NOTE-datetime">http://www.w3.org/TR/NOTE-datetime</a>.
     * <p>
	 * When the <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, date formatting is done by the 
	 * {@link com.ibm.icu.text.SimpleDateFormat} class using the {@link com.ibm.icu.util.ULocale} (parameter {@link Locale}
	 * converted to a {@link com.ibm.icu.util.ULocale} - see {@link com.ibm.icu.util.ULocale#forLocale(Locale)}), 
	 * {@link com.ibm.icu.util.GregorianCalendar} and {@link com.ibm.icu.util.TimeZone} (parameter time zone ID
	 * converted to a {@link com.ibm.icu.util.TimeZone} - see {@link com.ibm.icu.util.TimeZone#getTimeZone(String)}) classes, 
	 * otherwise date formatting is done by the {@link java.text.SimpleDateFormat} class using the {@link java.util.Locale}, 
	 * {@link java.util.GregorianCalendar} and {@link java.util.TimeZone} classes.  That is, if the 
	 * <a href="http://icu.sourceforge.net/">International Components for Unicode (ICU)</a> JAR(s) 
	 * are available on the class path at run-time, the parameter date format pattern is based on the syntax and symbols 
	 * as specified in the class comment header of the {@link com.ibm.icu.text.SimpleDateFormat} class, otherwise the 
	 * parameter date format pattern is based on the syntax and symbols as specified in the class comment header of the 
	 * {@link java.util.SimpleDateFormat} class.
     * <p>
     * If the parameter string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date does not
     * contain any time zone information, the time zone of the returned string is Coordinated Universal Time (UTC), 
     * represented as Greenwich Mean Time (GMT).  Otherwise, the time zone of the returned string is represented as a
     * signed offset from Greenwich Mean Time (GMT).  For example, 'GMT-05:00' for Eastern Standard Time.
     * <p>
     * 
     * @param xsdDateTime
     *            A string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date.
     * @param dateFormatPattern
     * 			  The date format pattern.
     * @return The date as a formatted string using the parameter date format pattern.
     * @throws IllegalArgumentException
     *             If the string representing an XML Schema dateTime (e.g. yyyy-MM-ddTHH:mm:ss) date is invalid or 
     * 			   the date format pattern is invalid.
	 * @see java.text.SimpleDateFormat
	 * @see java.util.Locale
	 * @see java.util.GregorianCalendar
	 * @see java.util.TimeZone
	 * @see com.ibm.icu.text.SimpleDateFormat
	 * @see com.ibm.icu.util.ULocale
	 * @see com.ibm.icu.util.GregorianCalendar
	 * @see com.ibm.icu.util.TimeZone
     */
    public static String getFormattedDateString(String xsdDateTime, String dateFormatPattern) throws IllegalArgumentException{
   
        //Step 1: Convert to a long representing a UTC date in milliseconds:
        long utcMilliseconds = convertXsdDateTimeToMilliseconds(xsdDateTime);
                
        //Step 2: Resolve the time zone in the XSD:dateTime and return the formatted string:
        //Notes:  1) Set the locale of the date format to English since the 
        //           XSD:dateTime format is locale agnostic (week not represented).
        //        2) Since an IllegalArgumentException was not thrown from the call 
        //           to convertXsdDateTimeToMilliseconds(xsdDateTime), the XSD:dateTime 
        //           is assumed to be valid (e.g. at least 19 characters in length).
        switch (InternationalizationUtilities.charAt(xsdDateTime.trim(),xsdDateTime.trim().length() - 6)) {

        	case '+':
        	case '-':{
        		return (InternationalizationUtilities.format(utcMilliseconds, dateFormatPattern, ULocale.ENGLISH, "GMT".concat(xsdDateTime.substring(xsdDateTime.trim().length() - 6).trim())));
        	}
		}

		return (InternationalizationUtilities.format(utcMilliseconds, dateFormatPattern, ULocale.ENGLISH, "GMT"));        		
    }
    
    public final static String CLIENT_LOCALE_LANGUAGE_KEY = "client_locale_language";
    public final static String CLIENT_LOCALE_COUNTRY_KEY = "client_locale_country";
    public final static String CLIENT_LOCALE_VARIANT_KEY = "client_locale_variant";
    
    private static ULocale _clientLocale = null;
    
    /**
     * Returns either the Locale specified using System properties LoggingCoreUtilities.CLIENT_LOCALE_LANGUAGE_KEY, 
     * LoggingCoreUtilities.CLIENT_LOCALE_COUNTRY_KEY and LoggingCoreUtilities.CLIENT_LOCALE_VARIANT_KEY or the current Default Locale.
     * @return Locale
     */
    public static Locale getClientLocale()
    {
        if(_clientLocale == null)
        {
            String clientLanguage = System.getProperty(CLIENT_LOCALE_LANGUAGE_KEY);
            
            if(clientLanguage !=null)
            {
                String clientCountry = System.getProperty(CLIENT_LOCALE_COUNTRY_KEY);
                
                if(clientCountry!=null)
                {
                    String variant = System.getProperty(CLIENT_LOCALE_VARIANT_KEY);
                    
                    if(variant!=null)
                    {
                        _clientLocale = new ULocale(clientLanguage,clientCountry,variant);
                    }
                    else{
                        _clientLocale = new ULocale(clientLanguage,clientCountry);
                    }
                }
                else
                {
                    _clientLocale = new ULocale(clientLanguage);
                }
            }
        }        
        
        return (_clientLocale!=null?_clientLocale:ULocale.getDefault()).toLocale();
    }
}