/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CorrelatorStack.java,v 1.2 2005/02/25 22:17:10 hleung Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

import java.util.Comparator;

/**
 * This is a stack for managing the instances of {@link ICorrelatorData} for a specific <code>key</code>,
 * which implements <code>java.util.Comparator</code>.  Contextual data can be pushed onto and popped off
 * the stack, and an operation counter also exists to keep track of the number of local {@link #push(Object)}
 * operations.
 * 
 * @author Qiyan Li
 */
public class CorrelatorStack extends Stack {
    private Comparator key; // the identifier of the owner
    private long operationCounter; // the number of pushes on this stack

    /**
     * Creates an empty stack of <code>size</code> with an <code>increment</code> for holding instances of
     * {@link ICorrelatorData} associated with <code>key</code>.
     * 
     * @param key       any user-defined unique identifier for a this stack
     * @param size      the initial size of the stack
     * @param increment the size by which the stack grows if its current capacity is insufficient
     */
     public CorrelatorStack(Comparator key, int size, int increment) {
        super(size, increment);
        this.key = key;
        operationCounter = 0;
    }

    /**
     * Returns the key assocaited with this stack.
     * 
     * @return  the key associated with this stack.
     */
    public Comparator getKey() {
        return key;
    }

    /**
     * Returns the current value of the operation counter.
     * 
     * @return  the current value of the operation counter.
     */
    public synchronized long getOperationCounter() {
        return operationCounter;
    }

    /**
     * Increments the current value of the operation counter by 1, and returns its new value.
     * 
     * @return  the current value of the operation counter, plus 1
     */
    public synchronized long incrementOperationCounter() {
        return ++operationCounter;
    }
}