/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IApplication.java,v 1.2 2005/02/25 22:17:10 hleung Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

import java.util.Comparator;

/**
 * This interface defines a model via which correlation data is captured in a series of push and pop operations
 * of correlators on a collection of stacks.  Each stack is identified by a key, and it is the responsibility of
 * an application to define the exact semantics of the key.
 * 
 * @author Qiyan Li
 */
public interface IApplication
{
    /**
     * Returns the unique identifier of this application.
     *
     * @return the unique identifier of this application.
     */
    public int getId();

    /**
     * Returns the version of this application.
     * 
     * @return the version of this application.
     */
    int getVersion();

    /**
     * Initializes this application so that it is ready to be (re-)used.
     */
    public void initialize();

    /**
     * Creates an initialized copy of {@link ICorrelatorData ICorrelatorData}.
     * 
     * @return  a fresh copy of the application-specific correlator.
     */
    public ICorrelatorData createCorrelatorData();

    /**
     * Releases an application-specific correlator association for later re-use.  This method does not release
     * any correlators themselves automatically, and applications must therefore release these correlators manually.
     * 
     * @param association   a valid application-specific context
     */
    public void releaseAssociation(CorrelatorAssociation association);

    /**
     * Creates and handles the call context by forwarding the current active correlator data for the key to the
     * destination of a remote invocation.  This context occurs before a remote invocation request is sent,
     * and should be followed by a later return context handled by {@link #handleReturn(Comparator, ICorrelatorData)}.
     * The context handler should be {@link IContextHandler#handleSend(CorrelatorAssociation)}.  This API
     * is created in attempt to pass more information to the engine so that a richer set of contexts can be
     * handled.
     * 
     * @param key       the key to the context to be created
     * @param parent    the parent of the context representing the local context
     * @return          the context created.
     * @see             handleReturn(Comparator, ICorrelatorData, ICorrelatorData)
     */
    public CorrelatorAssociation handleCall(Comparator key, ICorrelatorData parent);

    /**
     * Is exactly the same as <code>handleCall(key, null)</code>.
     * 
     * @param key   the key to the context to be created
     * @return      the context created.
     * @see         handleCall(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation handleCall(Comparator key);

    /**
     * Creates and handles the return context by receiving the current active correlator data for the key from the
     * destination of a remote invocation.  This context occurs after a remote invocation reply is received,
     * and should be following an earlier call context handled by {@link #handleCall(Comparator, ICorrelator)}.
     * The context handler should be {@link IContextHandler#handleReturn(CorrelatorAssociation)}. This API
     * is created in attempt to pass more information to the engine so that a richer set of contexts can be
     * handled.
     * 
     * @param key       the key to the context to be created
     * @param parent    the updated correlator data for the parent from the destination
     * @param child     the correlator data for the child
     * @return          the context created.
     * @see             handleCall(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation handleReturn(Comparator key, ICorrelatorData parent, ICorrelatorData child);

    /**
     * Is exactly the same as <code>handleReturn(key, parent, null)</code>.
     * 
     * @param key       the key to the context to be created
     * @param parent    the updated correlator data for the parent from the destination
     * @return          the context created.
     * @see             handleReturn(Comparator, ICorrelatorData, ICorrelatorData)
     */
    public CorrelatorAssociation handleReturn(Comparator key, ICorrelatorData parent);

    /**
     * Creates and handles the push-child context by pushing the child correlator onto the correlator stack
     * for the key.  This context occurs before a local inocation takes place, and should be followed by a later
     * pop-child context handled by {@link #popChild(Comparator)}.  The context handler is
     * {@link IContextHandler#handleEntry(CorrelatorAssociation)}.
     * 
     * @param key   the key to the context stack
     * @param child the correlator data for the child
     * @return      the context created.
     * @see         popChild(Comparator)
     */
    public CorrelatorAssociation pushChild(Comparator key, ICorrelatorData child);

    /**
     * Creates and handles the pop-child context by popping the child correlator off the correlator stack
     * for the key.  This context occurs after a local inocation has finished, and should be following an earlier
     * push-child context handled by {@link #pushChild(Comparator, ICorrelatorData)}.  The context handler is
     * {@link IContextHandler#handleExit(CorrelatorAssociation)}.
     * 
     * @param key   the key to the context stack
     * @return      the context created.
     * @see         pushChild(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation popChild(Comparator key);

    /**
     * Creates and handles the push-parent context by pushing the parent correlator onto the correlator stack
     * for the key.  This context occurs after a remote request is received at the server but before the actual
     * local inocation takes place, and should be followed by a later pop-parent context
     * handled by {@link #popParent(Comparator)}.  The context handler is
     * {@link IContextHandler#handleReceive(CorrelatorAssociation)}.  This API is created in attempt to pass
     * more information to the engine so that a richer set of contexts can be handled.
     * 
     * @param key       the key to the context stack
     * @param parent    the correlator data for the parent
     * @param child     the correlator data for the child
     * @return          the context created.
     * @see             popParent(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation pushParent(Comparator key, ICorrelatorData parent, ICorrelatorData child);

    /**
     * Is exactly the same as <code>pushParent(key, parent, null)</code>.
     * 
     * @param key       the key to the context stack
     * @param parent    the correlator data for the parent
     * @return          the context created.
     * @see             pushParent(Comparator, ICorrelatorData, ICorrelatorData)
     */
    public CorrelatorAssociation pushParent(Comparator key, ICorrelatorData parent);

    /**
     * Creates and handles the pop-parent context by popping the parent correlator off the correlator stack
     * for the key.  This context occurs after the local invocation has finished but before the reply is sent
     * to the client, and should be following an earlier push-parent context
     * handled by {@link #pushParent(Comparator, ICorrelatorData)}.  The context handler is
     * {@link IContextHandler#handleReply(CorrelatorAssociation)}.  This API is created in attempt to pass
     * more information to the engine so that a richer set of contexts can be handled.
     * 
     * @param key       the key to the context stack
     * @param parent    the correlator data for the parent
     * @return          the context created.
     * @see             pushParent(Comparator, ICorrelatorData, ICorrelatorData)
     */
    public CorrelatorAssociation popParent(Comparator key, ICorrelatorData parent);

    /**
     * Is exactly the same as <code>popParent(key, null)</code>.
     * 
     * @param key       the key to the context stack
     * @return          the context created.
     * @see             popParent(Comparator, ICorrelatorData)
     */
    public CorrelatorAssociation popParent(Comparator key);

    /**
     * Returns the most recent correlator for this key in this application.  If there is no correlator data
     * for the key, a <code>null</code> pointer will be returned.
     * 
     * @param key   the key for which the most recent correlator data in this application is to be located
     * @return      the most recent correlator data for this key in this application, or <code>null</code>
     *              if no such data exists.
     */
    public ICorrelatorData getActiveCorrelatorByKey(Comparator key);
}
