/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ServiceRegistry.java,v 1.3 2007/03/08 07:10:54 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.correlation;

/**
 * This is the application registry with which each application must register using a unique identifier.
 * An application can be located in the registry, or deregistered from the registry using the same credential.
 * The registry itself is a singleton, i.e., all applications within the same environment should be registered
 * with the same registry.
 * 
 * @author  Qiyan Li
 * @author  Paul E. Slauenwhite
 * @version March 6, 2007
 * @since   February 25, 2005
 */
public class ServiceRegistry extends Stack {
    private static final int INITIAL_REGISTRY_SIZE = 5; // the initial size of the registry
    private static final int INCREMENT_REGISTRY_SIZE = 5; // the increment of the registry
    private static ServiceRegistry registry; // a single instance of the application regitry

    /**
     * Creates an application registry.  In order to avoid mutliple instances of the registry, this constructor is
     * made private.  Users should obtain a reference to the registry by invoking {@link #getRegistry()}.
     */
    private ServiceRegistry() {
        super(INITIAL_REGISTRY_SIZE, INCREMENT_REGISTRY_SIZE);
    }

    /**
     * Gets a singleton registry.  If the registry has not been instantiated, a new instance will be created.
     * 
     * @return a singleton application registry
     */
    public static synchronized ServiceRegistry getRegistry() {

        // An instance of the registry will be created on the first invocation.        
        if (registry == null) {
            registry = new ServiceRegistry();
        }
        return registry;
    }

    /**
     * Returns the array of all applications registered.
     * 
     * @return the array of all applications registered.
     */
    public synchronized IApplication[] getApplications() {

    	IApplication[] applications = new IApplication[stack.length];
    	
    	for (int counter = 0; counter < stack.length; counter++) {
			applications[counter] = ((IApplication)(stack[counter]));
		}
    	
        return applications;
    }

    /**
     * Locate an application using an identifier.
     * 
     * @param id    the identifier of the target application
     * @return a reference to the application, or <code>null</code> if no such application is found.
     */
    public synchronized IApplication getApplicationById(int id) {
        IApplication registered; // a registered application

        // Perform a linear search on the stack of applications.
        for (int i = 0; i <= top; i++) {
            registered = (IApplication) stack[i];
            if (registered.getId() == id) {
                return registered;
            }
        }
        return null;
    }

    /**
     * Registers an application with the registry using the identifier returned by
     * {@link IApplication#getId()}.  It is the responsibility of the applications to ensure that this ID
     * does not conflict with other IDs.  Consequently, an ID can be used to register at most once.
     * 
     * @param application   the application to be registered
     * @return whether the application is registered or not.
     * @see #deregisterApplication(IApplication)
     */
    public synchronized boolean registerApplication(IApplication application) {
        
        // Each application can be registered only once, i.e., its ID must be unique within the registry.
        // So, make sure this is indeed the case before the application is registered.
        if (getApplicationById(application.getId()) == null) {
            push(application);
            return true;
        } else {
            return false;
        }
    }

    /**
     * Deregisters an application from the registry using the identifier returned by {@link IApplication#getId()}.
     * 
     * @param application   the application to be deregistered
     * @return whether the application is deregistered or not.
     * @see #registerApplication(IApplication)
     */
    public synchronized boolean deregisterApplication(IApplication application) {
        IApplication registered; // a registered application

        // Locate the application to be deregistered, and replace its slot with the application on top of the stack.
        // This avoids unnecessary shuffling of the registered applications and unwanted searches.
        for (int i = 0; i <= top; i++) {
            registered = (IApplication) stack[i];
            if (registered == application) {
                stack[i] = stack[top--];
                return true;
            }
        }
        return false;
    }
}
