/**********************************************************************
 Copyright (c) 2005, 2009 IBM Corporation and others.
 All rights reserved.  This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 $Id: ConfigFile.java,v 1.17 2009/03/18 20:13:46 jcayne Exp $
 
 Contributors:
  IBM Rational - initial implementation
 **********************************************************************/
package org.eclipse.tptp.platform.agentcontroller.config;

import java.awt.Toolkit;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.StringTokenizer;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

public class ConfigFile extends DefaultHandler {
	protected static String TAG = ConfigUtility.getString("Config.AgentControllerConfig.Tag");

	private static String lineSeparator = System.getProperty("line.separator");

	protected DocumentBuilderFactory docBuildFactory = null;

	protected DocumentBuilder docBuild = null;

	protected Document doc = null;

	protected Element acConfig = null;

	protected Element holder = null;
	
	private boolean inSecurityEnabledElement = false;
	private boolean inKeystoreElement = false;
	private boolean inKeystorePasswordElement = false;
	
	private final static int size = 100;

	private Hashtable hash = new Hashtable(size);

	private final static Hashtable validateHashClass = new Hashtable(size);

	private final Hashtable validateHashInstance = new Hashtable(size);

	private String filename = null;

	public ConfigFile(String name) {
		docBuildFactory = DocumentBuilderFactory.newInstance();
		try {
			docBuild = docBuildFactory.newDocumentBuilder();
			createNewDocument();
		} catch (ParserConfigurationException e) {
			Logger.err("Cannot create document builder");
		}

		filename = name;
		File f = new File(filename);
		if (f.exists()) {
			loadExistingDocument(filename);
		}
	}

	public void saveToFile() {
		if (filename != null) {
			saveToFile(filename);
		}
	}

//	public void saveToFile(String fileName) {
//		PrintWriter writer = null;
//
//		try {
//			writer = new PrintWriter(new FileWriter(fileName));
//		} catch (Exception e) {
//			Logger.err(ConfigUtility.getString("Config.Writer.Error.CannotWriteFile") + " " + fileName);
//			Logger.err(e.getMessage());
//		}
//
//		if (writer != null) {
//			writer.print(toString());
//			writer.flush();
//			writer.close();
//		}
//	}
//
	public void saveToFile(String fileName) {
		FileOutputStream fout = null;
		byte[] u_bytes;

		try {
			fout = new FileOutputStream(fileName);
		} catch (Exception e) {
			String errMsg = ConfigUtility.getString("Config.Writer.Error.CannotWriteFile") + " " + fileName;
			
			System.out.println();			
			System.out.println(errMsg);
			
			Logger.err(errMsg);
			Logger.err(e.getMessage());
			
			// This is caught by SetConfig.generateConfig(....) ;
			throw(new ConfigFileException(errMsg));
			
		}

		if (fout != null) {
			try {
				u_bytes = toString().getBytes(PlatformObject.encoding);
			} catch (UnsupportedEncodingException e) {
				u_bytes = toString().getBytes();
			}

			try {
				fout.write(u_bytes);
				fout.flush();
				fout.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	public String toString() {
		StringBuffer strbuf = new StringBuffer();
		String encoding = PlatformObject.encoding; // Set a default value
		String version = "1.0"; // Set a default value

		//
		// XML header information
		//
		strbuf.append("<?xml");

		//
		// Version and encoding
		//
		strbuf.append(" version=\"" + version + "\"");
		strbuf.append(" encoding=\"" + encoding + "\"");

		//
		// Closing the header
		//
		strbuf.append("?>" + ConfigFile.lineSeparator);

		//
		// Print each child node
		//
		NodeList nodes = holder.getChildNodes();
		if (nodes != null) {
			for (int i = 0; i < nodes.getLength(); i++) {
				strbuf.append(ConfigUtility.print(nodes.item(i)));
			}
		}

		return strbuf.toString();
	}

	public void generateConfiguration() {
		generateHyadesConfiguration();
	}

	/**
	 * Generate the basic Hyades Data Collection Engine configuration
	 */
	public void generateHyadesConfiguration() {
	}

	public void init(Hashtable hash) {
		Enumeration keys = hash.keys();
		while (keys.hasMoreElements()) {
			String key = (String) keys.nextElement();
			String value = (String) hash.get(key);
			setValue(key, value);
		}
	}

	public void createNewDocument() {
		DOMImplementation dom = docBuild.getDOMImplementation();
		doc = dom.createDocument(null, TAG, null);
		holder = doc.createElement("ElementHolder");
	}

	public void loadExistingDocument(String file) {
		SAXParser saxp = null;
		XMLReader parser = null;

		//
		// Create the parser
		//
		try {
			saxp = SAXParserFactory.newInstance().newSAXParser();
			parser = saxp.getXMLReader();
			parser.setContentHandler(this);
			parser.setErrorHandler(this);
		} catch (Exception e) {
			Logger.err(ConfigUtility.getString("Config.Parser.Error.CreateParser") + " " + parser);
			Logger.err(e.getMessage());
		}

		if (parser != null) {
			try {
				parser.parse(new InputSource(new FileInputStream(new File(file))));
			} catch (Exception e) {
				Logger.err(ConfigUtility.getString("Config.Parser.Error.ParseFile") + " " + file);
				Logger.err(e.getMessage());
				if (e instanceof SAXException) {
					Exception e1 = ((SAXException) e).getException();
					if (e1 != null) {
						Logger.err(e1.getMessage());
					}
				}
			}
		}
	}

	public void startElement(String uri, String local, String raw, Attributes attrs) throws SAXException {
		//
		// Security, first via the new tech AC approach, then via the RAC approach
		//
		if (raw.equals(SecurityEnabled.TAG)) {
			inSecurityEnabledElement = true;
		}
		else if (raw.equals(Keystore.TAG)) {
			inKeystoreElement = true;
		}
		else if (raw.equals(KeystorePassword.TAG)) {
			inKeystorePasswordElement = true;
		}

		else if (raw.equals(Security.TAG)) {
			String keystore = null;
			String keystorePassword = null;
			setValue("SECURITY", "TRUE");

			for (int i = 0; i < attrs.getLength(); i++) {
				if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Security.Keystore.Tag"))) {
					keystore = attrs.getValue(i);
					setValue("SECURITY_KEYSTORE", keystore);
				} else if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Security.KeystorePassword.Tag"))) {
					keystorePassword = attrs.getValue(i);
					setValue("SECURITY_KEYSTOREPASSWORD", keystorePassword);
				} else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}
		}

		//
		// UserDefinition
		//
		else if (raw.equals(UserDefinition.TAG)) {
			if(getValue("SECURITY").toUpperCase().equals("TRUE")) {
				
				for (int i = 0; i < attrs.getLength(); i++) {
					if (attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.Name.Tag"))) {
						String name = attrs.getValue(i);
						if (getValue("USERS") == null) {
							setValue("USERS", name);
						} else {
							setValue("USERS", getValue("USERS") + "," + name);
						}
					}else if (attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.User.Type.Tag"))) {
						setValue(Constants.USERTYPE, attrs.getValue(i));
					}else if (attrs.getQName(i).equals(ConfigUtility.getString("Config.UserDefinition.User.List.Tag"))) {
						setValue(Constants.USERLIST, attrs.getValue(i));
					} else {
						Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
					}
				}
			}
		}

		//
		// Variable
		//
		else if (raw.equals(Variable.TAG)) {
			String name = null;
			String value = null;
			String position = null;

			for (int i = 0; i < attrs.getLength(); i++) {
				if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Name.Tag"))) {
					name = attrs.getValue(i);
				} else if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Value.Tag"))) {
					value = attrs.getValue(i);
				} else if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Variable.Position.Tag"))) {
					position = attrs.getValue(i);
				} else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}

			if ((name != null) && (value != null) && (position != null)) {
				//
				// Populate the hash table, do not add CLASSPATH or PATH
				//
				if (!name.equals("CLASSPATH") && !name.equals("PATH")) {
					setValue(name, value);
				} else {
					setValidateValue(name, value);
				}
			}
		}

		//
		// Allow
		//
		else if (raw.equals(Allow.TAG)) {
			String type = null, hostlist = null;

			for (int i = 0; i < attrs.getLength(); i++) {
				if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Allow.Host.Type.Tag"))) {
					type = attrs.getValue(i);
				}if (attrs.getQName(i).equals(ConfigUtility.getString("Config.Allow.Host.List.Tag"))) {
					hostlist = attrs.getValue(i);
				}else {
					Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedAttribute") + " " + attrs.getQName(i));
				}
			}
			
			if (type != null) {
				setValue(Constants.ALLOW_HOST_TYPE, type.toUpperCase());
				if (type.toLowerCase().equals(Constants.ALLOW_ALL) || type.toLowerCase().equals(Constants.ALLOW_LOCAL)) {
					removeKey(Constants.HOSTS);
				} else {
					setValue(Constants.HOSTS, hostlist);
				}
			}else{
				//something is wrong, use default
				setValue(Constants.ALLOW_HOST_TYPE, Constants.ALLOW_LOCAL);
				removeKey(Constants.HOSTS);
			}
		}

		//
		// Unrecognized element name
		//
		else {
			Logger.out(ConfigUtility.getString("Config.Parser.Warning.UnrecognizedElement") + " " + local);
		}
	}
	
	public void endElement(String uri, String local, String raw) {
		if (raw.equals(SecurityEnabled.TAG)) {
			inSecurityEnabledElement = false;
		}
		else if (raw.equals(Keystore.TAG)) {
			inKeystoreElement = false;
		}
		else if (raw.equals(KeystorePassword.TAG)) {
			inKeystorePasswordElement = false;
		}
	}
	
	public void characters(char chs[], int start, int length) {
		if (inSecurityEnabledElement) {
			String chData = (new String(chs, start, length)).trim();
			if (chData.toUpperCase().equals("TRUE")) {
				setValue("SECURITY", "TRUE");
			}
		}
		else if (inKeystoreElement) {
			String keystore = (new String(chs, start, length)).trim();
			setValue("SECURITY_KEYSTORE", keystore);
		}
		else if (inKeystorePasswordElement) {
			String keystorePassword = (new String(chs, start, length)).trim();
			setValue("SECURITY_KEYSTOREPASSWORD", keystorePassword);
		}
	}

	/**
	 * @return Returns the acConfig.
	 */
	public Element getAcConfig() {
		return acConfig;
	}

	private String getValidateValue(String tag) {
		String value = this.getValidateValueInstance(tag);
		if (value == null) {
			value = this.getValidateValueClass(tag);
		}
		return value;
	}

	private String getValidateValueClass(String tag) {
		return (String) ConfigFile.validateHashClass.get(tag);
	}

	private String getValidateValueInstance(String tag) {
		return (String) this.validateHashInstance.get(tag);
	}

	public String getValue(String tag) {
		return (String) hash.get(tag);
	}

	public void setValue(String tag, String value) {
		if ((tag != null) && (value != null)) {
			hash.put(tag, value);
			this.setValidateValue(tag, value);
		}
	}

	private void setValidateValue(String tag, String value) {
		if ((tag != null) && (value != null)) {
			ConfigFile.validateHashClass.put(tag, value);
			String existing = (String) this.validateHashInstance.get(tag);
			if (existing != null) {
				value = existing + File.pathSeparator + value;
			}
			this.validateHashInstance.put(tag, value);
		}
	}

	public void removeKey(String tag) {
		hash.remove(tag);
	}

	public boolean isExist(String tag) {
		return hash.containsKey(tag);
	}

	/**
	 * @return Returns the holder.
	 */
	public Element getHolder() {
		return holder;
	}

	/**
	 * @return Returns the doc.
	 */
	public Document getDoc() {
		return doc;
	}

	// Bug 87150
	public void setFileName(String filename) {
		this.filename = filename;
	}

	/**
	 * Detects values that are file paths, resolves them to a path with
	 * variables replaced with concrete values and then validate that the paths
	 * are indeed existing paths (or files as appropriate)
	 * 
	 * This is the place to put other validation checks if required as well,
	 * such as variable assignments that are required by missing, conflicting
	 * variable values, etc
	 * 
	 * @param showAll
	 *            indicates if all is rpeorted on just invalid entries
	 */
	public void validate(Boolean showAll) {
		Enumeration elements = this.validateHashInstance.elements();
		while (elements.hasMoreElements()) {
			String path = (String) elements.nextElement();
			this.validate(path, showAll);
		}
	}

	/**
	 * Validate that the given path is existent, recursively resolves variables
	 * surrounded by % characters and then tries to validate the resultant new
	 * path
	 * 
	 * This is the place to put other validation checks if required as well,
	 * such as variable assignments that are required by missing, conflicting
	 * variable values, etc
	 * 
	 * @param path
	 *            the path to validate as an existing file or directory
	 * @param showAll
	 *            indicates if all is reported on or just invalid entries
	 */
	private void validate(String path, Boolean showAll) {
		StringBuffer resolvedPath = new StringBuffer();
		StringTokenizer variables = new StringTokenizer(path, "%", true);
		if (variables.countTokens() > 1) {
			int delimeters = 0;
			String previousVariable = null;
			while (variables.hasMoreTokens()) {
				String variable = variables.nextToken();
				if (variable.equals("%")) {
					delimeters++;
					if (delimeters % 2 == 0) {
						String value = this.getValidateValue(previousVariable);
						if (value == null) {
							System.err.println("Variable is unresolvable (manually verify your configuration files): " + previousVariable);
						}
						resolvedPath.append(value);
					}
				} else {
					if (delimeters % 2 == 0) {
						resolvedPath.append(variable);
					}
				}
				previousVariable = variable;
			}
			this.validate(resolvedPath.toString(), showAll);
		} else {
			StringTokenizer paths = new StringTokenizer(path, File.pathSeparator);
			while (paths.hasMoreTokens()) {
				path = paths.nextToken();
				if (path != null && !path.equalsIgnoreCase("null") && !path.equalsIgnoreCase("ALL")) {
					File file = new File(path);
					if (file.isAbsolute()) {
						if (!file.exists()) {
							this.beep();
							System.out.println("\t--> A well-formed but non-existent file or directory is detected: '" + file + "'");
						} else {
							if (showAll.booleanValue()) {
								if (file.isFile()) {
									System.out.println("\tA existent file is validated: '" + file + "'");
								} else {
									System.out.println("\tA existent directory is validated: '" + file + "'");
								}
							}
						}
					} else {
						this.beep();
						System.out.println("\t--> Cannot ascertain the validity of (might not be intended as path): '" + file + "'");
					}
				}
			}
		}
	}

	/**
	 * Beeps the system bell if available
	 */
	private void beep() {
		Toolkit.getDefaultToolkit().beep();
	}

}

class ConfigFileException extends RuntimeException {

	public ConfigFileException() {
		super();
	}
	public ConfigFileException(String msg) {
		super(msg);
	}

	private static final long serialVersionUID = 1L;
	
}
