/**********************************************************************
 Copyright (c) 2008 IBM Corporation and others.
 All rights reserved.  This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 $Id: ConfigUpgrade.java,v 1.4 2008/04/16 13:39:19 jcayne Exp $
 
 Contributors:
	IBM Rational - initial implementation
 **********************************************************************/
package org.eclipse.tptp.platform.agentcontroller.config;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.eclipse.tptp.platform.agentcontroller.config.Allow;
import org.eclipse.tptp.platform.agentcontroller.config.ConfigFile;
import org.eclipse.tptp.platform.agentcontroller.config.ConfigUtility;
import org.eclipse.tptp.platform.agentcontroller.config.Constants;
import org.eclipse.tptp.platform.agentcontroller.config.Logger;
import org.eclipse.tptp.platform.agentcontroller.config.SecurityEnabled;
import org.eclipse.tptp.platform.agentcontroller.config.UserDefinition;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Sample usage: <br/>
 * ConfigUpgrade upgrade = ConfigUpgrade.getInstance(); <br/>
 * upgrade.parseDocument("AC HOME\\config\\serviceconfig.xml");
 * 
 * @author Joel Cayne
 * @version March 3, 2008
 * @since February 29, 2008
 * @provisional
 * 
 */
public class ConfigUpgrade {

	protected ConfigFile configFile;
	protected Document doc;
	protected Properties properties;
	
	private static ConfigUpgrade instance = null;
	
	/**
	 * The method called to acquire a reference of the singleton ConfigUpgrade.
	 * @return An instance of ConfigUpgrade.
	 */
	public static ConfigUpgrade getInstance() {
		if(instance == null) {
			instance = new ConfigUpgrade();
		}
		return instance;
	}

	private ConfigUpgrade() {
	} 

	/**
	 * Provides for upgrading an existing Agent Controller installation. Parses the serviceconfig.xml and obtains the user provided values.
	 * @param filename The path to the serviceconfig.xml to retrieve an existing configuration from.
	 * @return The properties hash with the user entered values.
	 */
	public Properties parseDocument(String filename) {
		try {
			doc = DocumentBuilderFactory.newInstance().newDocumentBuilder().parse(new File(filename));
			configFile = new ConfigFile(filename);
			properties = new Properties();
		} catch (SAXException e) {
			Logger.err(e.getMessage());
		} catch (IOException e) {
			Logger.err(e.getMessage());
		} catch (ParserConfigurationException e) {
			Logger.err(e.getMessage());
		}
		
		properties.put(Constants.JAVA_PATH, configFile.getValue(Constants.JAVA_PATH));
		addElement(SecurityEnabled.TAG);
		addElement(UserDefinition.TAG);
		addElement(Allow.TAG, ConfigUtility.getString("Config.Allow.Host.Tag"));
		return properties;
	}
	
	/**
	 * Searches for the tag in the serviceconfig.xml and adds it to the properties hash.
	 * @param tag The tag to search for.
	 */
	protected void addElement(String tag) {
		addElement(tag, null);
	}
	
	/**
	 * Searches for the tag containing the id attribute in serviceconfig.xml and adds it to the properties hash.
	 * @param tag The tag to search for.
	 * @param id The id attribute to search for in the tag.
	 */
	protected void addElement(String tag, String id) {
		NodeList nl = doc.getElementsByTagName(tag);
		if(nl != null) {
			/* Traverse the nodes with the tag name adding them to the properties hash */
			for(int i = 0; i < nl.getLength(); i++) {
				Node n = nl.item(i);
				String text = null;
				/* Obtain the attribute value and add it to the properties hash */
				if(id != null) {
					n = n.getAttributes().getNamedItem(id);
					if(n != null) {
						/* Obtain the Attribute value */
						text = n.getNodeValue();
						/* The tag exists in the properties hash */ 
						if (properties.containsKey(tag)) {
							text = (String) properties.get(tag) + "," + text;
						}
					}
				} else {
					/* Obtain a unique tag value and add it to the properties hash */
					/* The value is stored in the first child of the Element node */
					text = n.getFirstChild().getNodeValue();
				}
				if(text != null) {
					properties.put(tag, text);
				}
			}
		}	
	}

	/**
	 * Outputs the user entered values to a property file and provides a properties hash table containing the user entered values.
	 * @param filename The name of the property file to store the user entered values to.
	 */
	public void getPropertyFile(File filename) {
		try {
			if(filename != null) {
				properties.store(new FileOutputStream(filename), null);
			}
		} catch (IOException e) {
			Logger.err(e.getMessage());
		}
	}
}
