/**********************************************************************
 Copyright (c) 2005, 2008 IBM Corporation and others.
 All rights reserved. This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 
 Contributors:
     IBM Rational - initial implementation
     
 **********************************************************************/
package org.eclipse.tptp.platform.agentcontroller.config;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * SetIACConfig.java.
 * 
 * 
 * @author  Bing Xu
 * @version April 16, 2008
 * @since   April 16,2008
 */
public class IACConfigHelper {

	static final String FILE_SEPARATOR=System.getProperty("file.separator");
	static final String PLUGIN_DIR = "plugins";
	static final String CONFIG_JAR_NAME = "config.jar";
	
	/**
	 * 
	 * @param IACHome home of IAC, ie <IAC Plugins>/agent_controller
	 * @return File[]  array of sub directories under IACHome that contains a config.jar file.   
	 * @throws Exception
	 * 
	 * Assumption:  1. all TPTP plugins must be installed in the same directory with IAC
	 * 				2. IAC's SetConfig script is run from <IAC Home>/agent_controller/bin directory.
	 */
	public static File[] getIACSubPlugins(String IACHome) throws IOException {	
		
		HashMap hashMap = new HashMap(11);
		
		String path = IACHome.substring(0, IACHome.indexOf(SetConfig.IAC_PLUGIN_NAME)-1);
		File pluginsDir = new File(path);
		
		if(!pluginsDir.exists())
			return null;
		
	    // This filter only returns directories
	    FileFilter fileFilter = new FileFilter() {
	        public boolean accept(File file) {
	            return file.isDirectory();
	        }
	    };
	    File[] children = pluginsDir.listFiles(fileFilter);
		
	    if (children == null) {
	    	return null;
	    } else {
	        for (int i=0; i<children.length; i++) {
	        	String dirName = children[i].getName();
	        	
	        	if ((new File(children[i].getCanonicalFile()+FILE_SEPARATOR+SetConfig.CONFIG_JAR_NAME).exists())){
	        		String key = dirName.substring(0, dirName.indexOf("_"));
		        	//if there are multiple versions of same plugin, use the latest one
	        		if(hashMap.containsKey(key)){
		        		String oldDir = ((File) hashMap.get(key)).getName();
		        		if(oldDir.compareTo(dirName) < 0 ){
		        			hashMap.put(key, children[i]);
		        		}			
		        	}else{
		        		hashMap.put(key, children[i]);
		        		//System.out.println("adding "+  dirName + " to plugins list.");  	
		        	}	        	
	        	}      					
	        }
	    }
	    return (File[]) hashMap.values().toArray(new File[hashMap.values().size()]);

	} 
	
	/**
	 * 
	 * @param subdirectory is a sub-directory under consuming product's plugin installation directory. 
	 * @return the plugins installation directory used by consuming product or null.
	 * 
	 */
	public static File findParentPluginDir(String subdirectory){
		
		File directory = new File(subdirectory);
		if(!directory.exists())
			return null;
		
		while(directory != null){
			if(isValidParentPluginDir(directory))
				return directory;
			else{
				directory = directory.getParentFile();
			}			
		}
		return null;
	}
	
	/**
	 * 
	 * @param subdirectory is a sub-directory under Eclipse installation directory. 
	 * @return the Eclipse installation directory or null.
	 * 
	 */
	public static File findEclipseRoot(String subdirectory){
		
		File directory = new File(subdirectory);
		if(!directory.exists())
			return null;
		
		while(directory != null){
			if(isValidEclipse(directory.getAbsolutePath()))
				return directory;
			else{
				directory = directory.getParentFile();
			}			
		}
		return null;
	}
	
	/**
	 * 
	 * @param SDPPath represents a absolute path 
	 * @return true if SDPPath contains a SDPShared installation; false otherwise.
	 * 
	 */
	public static boolean isValidParentPluginDir(File f){
		
		final String PLUGIN_DIR_NAME = "plugins";
		
		if(PLUGIN_DIR_NAME.equalsIgnoreCase(f.getName()))
			return true;
		else
			return false;
	}
	
	
	/**
	 * 
	 * @param eclipsePath represents a absolute path 
	 * @return true if eclipsePath contains a Eclipse installation; false otherwise.
	 * 
	 */
	public static boolean isValidEclipse(String eclipsePath){
		
		final String ECLIPSE_INI_FILE = "eclipse.ini";
		
		File eclipseIniFile = new File(eclipsePath+FILE_SEPARATOR+ECLIPSE_INI_FILE);
		if(eclipseIniFile.exists())
			return true;
		else
			return false;
	}
	
	/**
	 * 
	 * @param result 
	 * @param eclipseRoot a valid Eclipse installation directory
	 * @throws IOException
	 * 
	 * All plugin directories specified in Eclipse link files under <Eclipse>/links are added to the List result.
	 * Duplicates are not added.
	 *
	 */
	private static void addEclipseLinkedPlugins(List result, File eclipseRoot) throws IOException{
		
		final String ECLIPSE_LINK_DIRECTORY = "links";
		if(!eclipseRoot.exists())
			return;
		
		File linkDirectory = new File(eclipseRoot.getCanonicalPath()+FILE_SEPARATOR+ECLIPSE_LINK_DIRECTORY);
		if(!linkDirectory.exists()){
			return;
		}
		
		FileFilter filter = new FileFilter() {
	        public boolean accept(File file) {
	            return file.getName().endsWith(".link");
	        }
	    };
		File[] linkfiles = linkDirectory.listFiles(filter);
		String line, linkDir;
		for(int i=0; i < linkfiles.length; i++){
			BufferedReader in = new BufferedReader(new FileReader(linkfiles[i]));
            if (!in.ready())
                throw new IOException();

            //all link file should contain one and only one line: path=<path>
            line = in.readLine();
            if(line != null && line.indexOf("path=")==0){
            	linkDir = line.substring(5);
            	File newPluginDir = new File(linkDir+FILE_SEPARATOR+"eclipse"+FILE_SEPARATOR+"plugins");
            	if(newPluginDir.exists() && !result.contains(newPluginDir)){
            		result.add(newPluginDir);
            	}
            }	               
            in.close();
        }
	}
	

	/**
	 * 
	 * @param eclipsePath home of Eclipse, for example: D:\eclipse\
	 * @return List contains all unique Eclipse plugin directories including those specified in link files.
	 * @throws IOException
	 */
	public static List findAllPluginDirectory(File eclipsePath)throws IOException{
		
		List result = new ArrayList(4);
		
		if(!eclipsePath.exists())
			return result;
		
		if(eclipsePath != null){				
			//add core Eclipse plugin directory
			result.add(new File(eclipsePath.getCanonicalFile()+FILE_SEPARATOR+PLUGIN_DIR));
			
			//add default third party plugin directory
			//this is only valid for Eclipse with P2 
			File dropins = new File(eclipsePath.getCanonicalFile()+FILE_SEPARATOR+"dropins"+FILE_SEPARATOR+"eclipse"+FILE_SEPARATOR+PLUGIN_DIR);
			if(dropins.exists())
				result.add(new File(eclipsePath.getCanonicalFile()+FILE_SEPARATOR+"dropins"+FILE_SEPARATOR+"eclipse"+FILE_SEPARATOR+PLUGIN_DIR));
			
			//add plugin directories in extension locations
			addEclipseLinkedPlugins(result, eclipsePath);
		}
		return result;
		
	}

}
	
