/**********************************************************************
 Copyright (c) 2005, 2010 IBM Corporation and others.
 All rights reserved. This program and the accompanying materials
 are made available under the terms of the Eclipse Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/epl-v10.html
 $Id: SetConfig.java,v 1.90 2010/11/08 14:27:27 jcayne Exp $
 
 Contributors:
     IBM Rational - initial implementation
     Randy D. Smith, Intel Corporation - Refinements for new agent
                                         controller support
 **********************************************************************/
package org.eclipse.tptp.platform.agentcontroller.config;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

import com.ibm.icu.util.ULocale;

/**
 * SetConfig.java.
 * 
 * 
 * @author Samson Wai
 * @author Paul E. Slauenwhite
 * @author Bing Xu
 * @version February 11, 2010
 * @since February 4, 2005
 */
public class SetConfig extends SetConfigSkeleton {

	public final static String IAC_PLUGIN_NAME = "org.eclipse.tptp.platform.ac";
	public final static String CONFIG_JAR_NAME = "config.jar";
	public final static String PLUGIN_DIR_NAME = "plugins";

	private static boolean _help = false;

	private static boolean _examples = false;

	private static boolean _validate = false;

	private static Boolean _show_all = Boolean.FALSE;
	
	private static boolean _default = false;
	
	/** Whether or not the config file generation is being done for the IAC */
	private static boolean _is_iac_config = false;

	/** Whether or not the config file generation is invoked by Eclipse workbench */
	private static boolean _is_invoked_by_workbench = false;
	
	private IConfigUtility configUtil = null;
	
	/*
	 * Plugins that should have an alias generated
	 */
	private final static String[] _requiredPlugins = { "org.apache.commons.logging", "org.eclipse.core.commands", "org.eclipse.core.runtime", "org.eclipse.equinox.common", "org.eclipse.osgi", "org.eclipse.emf.common", "org.eclipse.emf.ecore", "org.eclipse.emf.ecore.xmi", "org.eclipse.jface", "org.eclipse.swt", "org.eclipse.tptp.platform.models", "org.eclipse.tptp.platform.models.hierarchy", "org.hamcrest.core", "org.junit", "com.ibm.icu" };

	/*
	 * Plugins that should be in the global classpath
	 */
	private final static String[] _globalPlugins = { "org.eclipse.core.runtime", "org.eclipse.equinox.common", "org.eclipse.osgi" };

	public static void main(String[] args) {
		try{
			File dir1 = new File(".");	
			String currentDir = dir1.getCanonicalPath();			
			if (currentDir.indexOf(IAC_PLUGIN_NAME) >= 0) {
				setIsIACConfiguration(true);
			}
			setIsInvokedByWorkbench(false);
			int status = generateConfig(args,null, null);
			System.exit(status);

		} catch (IOException e) {
			Logger.out("Can't read current directory");
			System.exit(-2);
		}
	}
	
	/**
	 * Used by stand-alone AC, IAC from Eclipse and command line.
	 * @param args
	 * @param configUtil
	 * @param plugins
	 */
		public static int generateConfig(String[] args, IConfigUtility configUtil, File[] plugins)	{
		
		SetConfig sc = new SetConfig();
		
		//bug 227779
		if(configUtil != null)
			//invoked by Eclipse
			sc.configUtil = configUtil;
		else
			// invoked by command line
			sc.configUtil = new ConfigUtility();
		
		// Check the JVM that is used to run the SetConfig utility for the SetConfig component, askUser checks the JVM
		// version used for the Agents run by the AgentController.
		if (TestJvm.isMinimumVersion(ConfigUtility.getString("Config.Tool.Java.Version"), System.getProperty("java.version"))) {
			int size = 100;
			Hashtable arguments = new Hashtable(size);
			sc.populateHashtable(arguments, args);
			sc.processParameters(arguments);
			sc.init(arguments, sc.configUtil, _is_invoked_by_workbench, _is_iac_config);
			
			
			if (_help) {
				sc.printHelp();
				sc.printPluginHelp(arguments);
				return 0;
			} else if (_examples) {
				sc.printExamples();
				sc.printPluginExamples(arguments);
				return 0;
			} else if (_validate) {
				sc.validate(_show_all);
				sc.validatePlugins(_show_all, arguments);
				return 0;
			} 
			
			try {			
				sc.run(arguments, silent, overwrite);
				
				/* NM - bugzilla 173330: Call the appropriate function depending on whehter or not list of 'plugins' is null */
				if (plugins == null && !_is_iac_config)
					//invoked by stand-alone AC
					sc.generatePluginConfigurations(arguments);
				else{
					if(_is_invoked_by_workbench){
						//IAC from Eclipse workbench
						HashUtility.setValue(arguments, Constants.PLUGIN_PROPERTY_DIR, configUtil.getPluginPropertyDirectory(true).getAbsolutePath());
						sc.generatePluginConfigurations(arguments, plugins);
					} else {
						// IAC from command line
						String iacHome = sc.configFile
								.getValue(Constants.RASERVER_HOME);
						sc.generatePluginConfigurations(arguments,
								IACConfigHelper.getIACSubPlugins(iacHome));
					}
				}
				return 0;
			} catch (ConfigFileException cfe) {
				return -1;/* Error msg already printed in module, so return. */
			} catch (IOException ioe) {
				return -2;
			}// TODO
		} else {
			// The JVM version used to run SetConfig is not at least that of Config.Tool.Java.Version.
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_Version.Err") + ": " + ConfigUtility.getString("Config.Tool.Java.Version"));
			System.out.println(ConfigUtility.getString("Config.Prompt.JAVA_Version.Msg"));
			return -3;
		}
	}

	public void init(Hashtable hash, IConfigUtility configUtil,
			boolean isInvokedByWorkbench, boolean isIacConfig) {
		this.arguments = hash;
		String home = HashUtility.getValue(hash, Constants.RASERVER_HOME);
		// bug 227779
		File configDir = configUtil.getACConfigDirectory(home,
				isInvokedByWorkbench, isIacConfig);
		if (configDir.exists())
			setFileName(configDir.getAbsolutePath() + sr + "serviceconfig.xml");
		else
			setFileName(home + sr + "config" + sr + "serviceconfig.xml");
	}

	private void processParameters(Hashtable hash) {
		//
		// For both IAC and stand-alone AC, try to use the $PWD/.. directory as
		// the RASERVER_HOME if none
		// specified.
		//
		if (HashUtility.getValue(hash, Constants.RASERVER_HOME) == null) {
			File pwd = new File("..");
			try {
				HashUtility.setValue(hash, Constants.RASERVER_HOME, pwd
						.getCanonicalPath());
			} catch (IOException e) {
				// cannot resolve ".."
			}
		}
		if (HashUtility.getValue(hash, Constants.PLUGINS_HOME) == null) {
			if (!_is_iac_config)
				HashUtility.setValue(hash, Constants.PLUGINS_HOME, HashUtility
						.getValue(hash, Constants.RASERVER_HOME)
						+ sr + "plugins");
			else {
				String IACHome = HashUtility.getValue(hash,
						Constants.RASERVER_HOME);
				HashUtility.setValue(hash, Constants.PLUGINS_HOME, IACHome
						.substring(0, IACHome.indexOf(IAC_PLUGIN_NAME) - 1));
			}
		}

		if (HashUtility.exist(hash, "-help") || HashUtility.exist(hash, "-h")
				|| HashUtility.exist(hash, "-?")) {
			_help = true;
		}
		if (HashUtility.exist(hash, "-example")
				|| HashUtility.exist(hash, "-ex")) {
			_examples = true;
		}
		if (HashUtility.exist(hash, "-debug")) {
			Logger.setDebug(true);
		}
		if (HashUtility.exist(hash, "-overwrite")
				|| HashUtility.exist(hash, "-ovr")) {
			overwrite = true;
		}
		if (HashUtility.exist(hash, Constants.SILENT_FLAG)
				|| HashUtility.exist(hash, "-s")) {
			silent = true;
		}
		if (HashUtility.exist(hash, "-noverify")
				|| HashUtility.exist(hash, "-nov")) {
			verify = false;
		}

		if (HashUtility.exist(hash, "-default")) {
			_default = true;
			silent = true;
			System.out
					.println("Creating default Agent Controller configuration file.\nSecurity is turned off.  Access is set to Local.\nRun the SetConfig script to change the default settings.");
		}
		// Support --validate and --validate=show-all
		if (HashUtility.exist(hash, "--validate")) {
			_validate = true;
			if (HashUtility.getValue(hash, "--validate").equalsIgnoreCase(
					"show-all")) {
				_show_all = Boolean.TRUE;
			}
		}
	}

	public void askUser() {
		final String jvm; // 86262
		//
		// Get the required parameters
		//
		if (!isValidRaserverHome(configFile.getValue(Constants.RASERVER_HOME))) {
			if (silent) {
				return; // cannot resolve RASERVER_HOME under silent mode,
				// return
			} else {
				// Bug 87150
				String home;
				home = promptRaserverHome(configFile
						.getValue(Constants.RASERVER_HOME));
				configFile.setValue(Constants.RASERVER_HOME, home);
				configFile.setFileName(home + sr + "config" + sr
						+ "serviceconfig.xml");

				/* NM - bugzilla 173330: Make sure plugins folder is also valid */
				File pluginsFolder = new File(configFile
						.getValue(Constants.PLUGINS_HOME));
				if ((!pluginsFolder.exists()) || (!pluginsFolder.isDirectory())) {
					configFile.setValue(Constants.PLUGINS_HOME, configFile
							.getValue(Constants.RASERVER_HOME)
							+ sr + "plugins");
				}
			}
		}

		//
		// Find out if configuring a "new technology AC" release, or a standard
		// RAC release
		//
		File f = new File(
				new File(configFile.getValue(Constants.RASERVER_HOME))
						.getAbsolutePath()
						+ File.separator
						+ "bin"
						+ File.separator
						+ "ACServer"
						+ PlatformObject.exeExt);
		if (f.exists()) {
			isNewAC = true;
			f = new File(new File(configFile.getValue(Constants.RASERVER_HOME))
					.getAbsolutePath()
					+ File.separator
					+ "bin"
					+ File.separator
					+ "RAServer"
					+ PlatformObject.exeExt);
			if (f.exists()) {
				wantsBC = true;
			}
		}

		//
		// JAVA_PATH=
		//
		if (PlatformObject.name.equals("OS/400")) {
			configFile.setValue(Constants.JAVA_PATH, "/QSYS.LIB/Q5BVAJVM.PGM");
		} else {

			String defJavaHome = System.getProperty("java.home") + sr + "bin" + sr + PlatformObject.javaExecutable + PlatformObject.exeExt;

			// The SPARC JVM install includes the 64-bit executables in a separate sparcv9/ dir off the bin directory
			String osArch = System.getProperty("os.arch");
			String osName = System.getProperty("os.name");
			if(osArch != null && osArch.equalsIgnoreCase("sparcv9")) {
				defJavaHome = System.getProperty("java.home") + sr + "bin" + sr + "sparcv9" + sr + PlatformObject.javaExecutable + PlatformObject.exeExt;
			}
			
			// The Solaris x86 JVM install includes the 64-bit executables in a separate amd64/ dir off the bin directory
			if(osArch != null && osName != null && osArch.equalsIgnoreCase("amd64") 
					&& osName.equalsIgnoreCase("SunOS")) {
				defJavaHome = System.getProperty("java.home") + sr + "bin" + sr + "amd64" + sr + PlatformObject.javaExecutable + PlatformObject.exeExt;
			}
						
			if (silent) {
				
				if ((configFile.getValue(Constants.JAVA_PATH) == null) || (configFile.getValue(Constants.JAVA_PATH).equals(""))) {
					
					if (isValidJava(defJavaHome)) {
						
						configFile.setValue(Constants.JAVA_PATH, defJavaHome);
					} else {
						configFile.setValue(Constants.JAVA_PATH, "%JAVA_PATH%");
					}
				}
			} else {
				
				if (configFile.getValue(Constants.JAVA_PATH) == null) {
					configFile.setValue(Constants.JAVA_PATH, promptJavaPath(defJavaHome));
				} else {
					configFile.setValue(Constants.JAVA_PATH, promptJavaPath(configFile.getValue(Constants.JAVA_PATH)));
				}
			}
		}
		jvm = resolveJvmDll(configFile.getValue(Constants.JAVA_PATH));// 86262
		configFile.setValue("jvm", jvm);// 86262
		//
		// ALLOW=
		//
		if (silent) {
			if (configFile.getValue(Constants.ALLOW_HOST_TYPE) == null
					|| _default) {
				configFile.setValue(Constants.ALLOW_HOST_TYPE,
						Constants.ALLOW_LOCAL);
			}
		} else {
			if (configFile.getValue(Constants.ALLOW_HOST_TYPE) == null) {
				configFile.setValue(Constants.ALLOW_HOST_TYPE,
						promptAllow(Constants.ALLOW_LOCAL));
			} else {
				configFile.setValue(Constants.ALLOW_HOST_TYPE,
						promptAllow(configFile
								.getValue(Constants.ALLOW_HOST_TYPE)));
			}
		}
		//
		// HOSTS=
		//
		if (configFile.getValue(Constants.ALLOW_HOST_TYPE).toUpperCase()
				.equals(Constants.ALLOW_CUSTOM)) {
			if (silent) {
				if (configFile.getValue(Constants.HOSTS) == null) {
					configFile.setValue(Constants.HOSTS, "localhost");
				}
			} else {
				if (configFile.getValue(Constants.HOSTS) == null) {
					configFile.setValue(Constants.HOSTS,
							promptHosts("localhost"));
				} else {
					configFile.setValue(Constants.HOSTS, promptHosts(configFile
							.getValue(Constants.HOSTS)));
				}
			}
		}

		// SECURITY
		if (silent) {
			if (configFile.getValue(Constants.SECURITY) == null || _default) {
				configFile.setValue(Constants.SECURITY, "false");
			}
		} else {
			if (configFile.getValue(Constants.SECURITY) == null) {
				configFile.setValue(Constants.SECURITY,
						promptSecurity(Constants.SECURITY_OFF));
			} else {
				configFile
						.setValue(Constants.SECURITY, promptSecurity(configFile
								.getValue(Constants.SECURITY)));
			}
		}

		// USERS
		if (configFile.getValue(Constants.SECURITY).toUpperCase().equals(
				Constants.SECURITY_ON)) {
			if (silent) {
				if (configFile.getValue(Constants.USERTYPE) == null) {
					configFile.setValue(Constants.USERTYPE, Constants.USER_ALL);
				}
			} else {
				String existUserType = configFile.getValue(Constants.USERTYPE);
				String existUserList = configFile.getValue(Constants.USERLIST);
				
				configFile.setValue(Constants.USERTYPE, promptUserType(existUserType));
				if(Constants.USER_CUSTOM.equalsIgnoreCase( configFile.getValue(Constants.USERTYPE))){
					configFile.setValue(Constants.USERLIST, promptUserList(existUserList));					
				}
				
			}
		}

	}

	//
	// Retrieve the raserver home specified by the user
	//
	private String promptRaserverHome(String defaultValue) {
		String rc = null;
		System.out.println(ConfigUtility
				.getString("Config.Prompt.RASERVER_HOME.Tag"));
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.Default") + "\""
				+ defaultValue + "\"");
		System.out.print(" (");
		System.out
				.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidRaserverHome(rc));
		return rc;
	}

	//
	// Retrieve the java home specified by the user
	//
	private String promptJavaPath(String defaultValue) {
		String rc = null;
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println(ConfigUtility
					.getString("Config.Prompt.JAVA_PATH.Tag.Windows"));
		} else {
			System.out.println(ConfigUtility
					.getString("Config.Prompt.JAVA_PATH.Tag.Unix"));
		}
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.Default") + "\""
				+ defaultValue + "\"");
		System.out.print(" (");
		System.out
				.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidJava(rc));
		return rc;
	}

	//
	// Retrieve the allowed hosts specified by the user
	//
	private String promptAllow(String defaultValue) {
		String rc = null;
		System.out.println(ConfigUtility.getString("Config.Prompt.Allow.Tag"));
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.Default") + "\""
				+ defaultValue + "\"");
		System.out.print(" (");
		System.out
				.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidAllow(rc));
		return rc;
	}

	//
	// Retrieve the host names specified by the user
	//
	private String promptHosts(String defaultValue) {
		System.out.println(ConfigUtility.getString("Config.Prompt.Hosts.Tag"));
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.Default") + "\""
				+ defaultValue + "\"");
		System.out.print(" (");
		System.out
				.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.NewValue"));
		return ConfigUtility.askUser(defaultValue);
	}

	private String promptSecurity(String defaultValue) {
		String rc = null;
		System.out.println(ConfigUtility
				.getString("Config.Prompt.Security.Tag"));
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.Default") + "\""
				+ defaultValue + "\"");
		System.out.print(" (");
		System.out
				.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		do {
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.NewValue"));
			rc = ConfigUtility.askUser(defaultValue);
		} while (!isValidSecurity(rc));

		return rc;
	}
	
	private String promptUserList(String defaultUserList) {
		String userList;
		
		if(defaultUserList == null)
			defaultUserList = "";
		
		System.out.println(ConfigUtility.getString("Config.Prompt.User.List.Tag"));
		System.out.print("  "
				+ ConfigUtility.getString("Config.Prompt.Default") + "\""
				+ defaultUserList + "\"");
		System.out.print(" (");
		System.out.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
		System.out.println(")");
		System.out.print("  " + ConfigUtility.getString("Config.Prompt.NewValue"));
		userList = ConfigUtility.askUser(defaultUserList);
		return userList;
	}

	private String promptUserType(String defaultUserType) {
		
		if(defaultUserType == null)
			defaultUserType = "";
		
		String userType;
		do {
		
			System.out.println(ConfigUtility.getString("Config.Prompt.User.Type.Tag"));
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.Default") + "\""
					+ defaultUserType + "\"");
			System.out.print(" (");
			System.out
					.print(ConfigUtility.getString("Config.Prompt.DefaultAccept"));
			System.out.println(")");
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.NewValue"));
			
			
			userType = ConfigUtility.askUser(defaultUserType);			
		} while (!isValidUserType(userType));
		

		return userType;
	}

	//
	// Retrieve the Eclipse installation directory by the user
	//
	private String promptEclipsePath(String defaultValue) {
		String result = null;
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println(ConfigUtility
					.getString("Config.Prompt.ECLIPSE_PATH.Tag.Windows"));
		} else {
			System.out.println(ConfigUtility
					.getString("Config.Prompt.ECLIPSE_PATH.Tag.Unix"));
		}
		do {
			System.out.print("  "
					+ ConfigUtility.getString("Config.Prompt.NewValue"));
			result = ConfigUtility.askUser(defaultValue);
		} while (!IACConfigHelper.isValidEclipse(result));
		return result;
	}

	//
	// Check if RASREVER_HOME is valid
	//
	private boolean isValidRaserverHome(String path) {
		if (!verify) {
			return true;
		} else {
			boolean rc = false;
			if (path == null) {
				System.out.println(ConfigUtility
						.getString("Config.Prompt.InvalidRAServerPath"));
				return false;
			} else {
				File p = new File(path);
				// Make sure specified path is a directory and exist
				if (p.exists() && p.isDirectory()) {
					if (PlatformObject.name.equals("OS/400")) {
						// Check if SetConfig.sh exists
						File f = new File(p.getAbsolutePath() + File.separator
								+ "bin" + File.separator + "SetConfig.sh");
						if (f.exists()) {
							rc = true;
						} else {
							System.out.println(ConfigUtility
									.getString("Config.Prompt.FileNotFound")
									+ "SetConfig.sh");
						}
					} else {
						// Check if RAServer.exe or RAServer exists
						File f = new File(p.getAbsolutePath() + File.separator
								+ "bin" + File.separator + "RAServer"
								+ PlatformObject.exeExt);
						if (f.exists()) {
							rc = true;
						} else {
							// Check if ACServer.exe or ACServer exists
							// exclusive of RAServer
							// (could be a non-backward compatibility agent
							// controller distribution)
							f = new File(p.getAbsolutePath() + File.separator
									+ "bin" + File.separator + "ACServer"
									+ PlatformObject.exeExt);
							if (f.exists()) {
								rc = true;
							} else {
								System.out
										.println(ConfigUtility
												.getString("Config.Prompt.FileNotFound")
												+ "RAServer"
												+ PlatformObject.exeExt);
							}
						}
					}
				} else {
					System.out.println(ConfigUtility
							.getString("Config.Prompt.InvalidRAServerPath"));
				}
				return rc;
			}
		}
	}

	private boolean isValidSecurity(String rc) {
		if (!verify) {
			return true;
		} else {
			if (rc == null) {
				return false;
			} else if (rc.toLowerCase().equals(
					Constants.SECURITY_OFF.toLowerCase())
					|| rc.toLowerCase().equals(
							Constants.SECURITY_ON.toLowerCase())) {
				return true;
			} else {
				return false;
			}
		}
	}

	//
	// Check if JAVA_PATH is valid
	//
	private boolean isValidJava(String path) {
		if (!verify) {
			return true;
		} else {
			if (path == null) {
				System.out.println(ConfigUtility
						.getString("Config.Prompt.JAVA_PATH.Err"));
				return false;
			} else {
				File p = new File(path);
				// Make sure specified path is a directory and exist
				if (p.exists() && p.isFile()) {
					boolean rc = TestJvm.isSupportedJava(path, ConfigUtility
							.getString("Config.Java.Version"));
					if (!rc) {
						System.out.println(ConfigUtility
								.getString("Config.Prompt.JAVA_Version.Err")
								+ ": "
								+ ConfigUtility
										.getString("Config.Java.Version"));
					}
					return rc;
				} else {
					System.out.println(ConfigUtility
							.getString("Config.Prompt.JAVA_PATH.Err"));
					return false;
				}
			}
		}
	}
	
	//
	// Check if UserType is valid
	//
	private boolean isValidUserType(String userType) {
		if (!verify) {
			return true;
		} else {
			boolean rc = false;
			if (userType == null) {
				System.out.println(ConfigUtility.getString("Config.Prompt.UserType.Err"));
				return false;
			} else {
				if (Constants.USER_CUSTOM.equalsIgnoreCase(userType) || Constants.USER_ALL.equalsIgnoreCase(userType)) {
					rc = true;
				}
				if (!rc) {
					System.out.println(ConfigUtility.getString("Config.Prompt.UserType.Err"));
				}
				return rc;
			}
		}
	}
	//
	// Check if ALLOW is valid
	//
	private boolean isValidAllow(String allow) {
		if (!verify) {
			return true;
		} else {
			boolean rc = false;
			if (allow == null) {
				System.out.println(ConfigUtility
						.getString("Config.Prompt.Allow.Err"));
				return false;
			} else {
				if (allow.equalsIgnoreCase(Constants.ALLOW_ALL)
						|| allow.equalsIgnoreCase(Constants.ALLOW_LOCAL)
						|| allow.equalsIgnoreCase(Constants.ALLOW_CUSTOM)) {
					rc = true;
				}
				if (!rc) {
					System.out.println(ConfigUtility
							.getString("Config.Prompt.Allow.Err"));
				}
				return rc;
			}
		}
	}

	//
	// Populate the hash table using the command line arguments
	//
	private void populateHashtable(Hashtable hash, String[] args) {
		if (args != null) {
			for (int i = 0; i < args.length; i++) {
				int eq = args[i].indexOf('=');
				if (eq == -1) { // a flags, e.g. -help
					HashUtility.setValue(hash, args[i], "");
				} else { // a parameter, e.g. name=value
					String name = trim(args[i].substring(0, eq));
					String value = trim(args[i].substring(eq + 1));
					if ((name != null) && (value != null)) {
						HashUtility.setValue(hash, name, value);
					}
				}
			}
		}
	}

	//
	// Print the help menu
	//
	public void printHelp() {
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println(ConfigUtility
					.getString("Config.Cmd.Usage.Windows"));
		} else {
			System.out
					.println(ConfigUtility.getString("Config.Cmd.Usage.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Cmd.Flags"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Help.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.Help.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Silent.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.Silent.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.NoVerify.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.NoVerify.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Examples.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.Examples.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.Validate.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.Validate.Msg"));
		System.out.println(ConfigUtility
				.getString("Config.Cmd.Validate.ShowAll.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.Validate.ShowAll.Msg"));
		System.out.println();
		System.out.println(ConfigUtility.getString("Config.Cmd.Params"));
		System.out.println(ConfigUtility
				.getString("Config.Cmd.RASERVER_HOME.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.RASERVER_HOME.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.JAVA_PATH.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.JAVA_PATH.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.ALLOW.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.ALLOW.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.HOSTS.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.HOSTS.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.SECURITY.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.SECURITY.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.USERTYPE.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.USERTYPE.Msg"));
		System.out.println(ConfigUtility.getString("Config.Cmd.USERS.Tag"));
		System.out.println("\t"
				+ ConfigUtility.getString("Config.Cmd.USERS.Msg"));
	}

	//
	// Print the help menu
	//
	public void printExamples() {
		System.out.println(ConfigUtility.getString("Config.Example.Tag"));
		System.out.println(ConfigUtility.getString("Config.Example.00.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.00.Msg.Windows"));
		} else {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.00.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.01.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.01.Msg.Windows"));
		} else {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.01.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.02.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.02.Msg.Windows"));
		} else {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.02.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.03.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.03.Msg.Windows"));
		} else {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.03.Msg.Unix"));
		}
		System.out.println(ConfigUtility.getString("Config.Example.04.Tag"));
		if (PlatformObject.name.startsWith("Windows")) {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.04.Msg.Windows"));
		} else {
			System.out.println("\t"
					+ ConfigUtility.getString("Config.Example.04.Msg.Unix"));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.tptp.platform.agentcontroller.config.SetConfigSkeleton#validate
	 * (java.lang.Boolean)
	 */
	public void validate(Boolean showAll) {
		System.out
				.print("\r\nExecuting validation of agent controller installation ");
		System.out
				.println((showAll.booleanValue() ? "showing valid, invalid and questionable configuration entries"
						: "showing invalid and questionable configuration entries only")
						+ "...\r\n");
		System.out.println("Validating agent controller configuration '"
				+ new File(this.fileName) + "'");
		super.validate(showAll);
	}

	/**
	 * Validate the paths in the plugin configuration files, outputs findings to
	 * standard out, this is experimental and provisional for now but is much
	 * needed to minimize time taken for debugging issues
	 * 
	 * @param the
	 *            hashtable holding the name value command line argument pairs
	 */
	private void validatePlugins(Boolean showAll, Hashtable hash) {
		String home = HashUtility.getValue(hash, Constants.RASERVER_HOME);
		if (home != null) {
			/* NM - bugzilla 173330: Changed hardcoded plugins location */
			String pluginDirStr = HashUtility.getValue(hash,
					Constants.PLUGINS_HOME);
			File pluginDir = new File(pluginDirStr);
			if (pluginDir.exists() && pluginDir.isDirectory()) {
				File[] subDirs = pluginDir.listFiles();
				for (int i = 0; i < subDirs.length; i++) {
					String pluginName = subDirs[i].getName(); // Name of the
					// plugin,
					// including the
					// version
					// number at the
					// end
					String jarName = pluginDirStr.replace('\\', '/') + "/"
							+ pluginName + "/config.jar";
					String clsName;
					if (pluginName.lastIndexOf('_') != -1) {
						clsName = pluginName.substring(0, pluginName
								.lastIndexOf('_'))
								+ ".SetConfig";
					} else {
						clsName = pluginName + ".SetConfig";
					}
					Logger.out("Found plugin: " + pluginName);
					try {
						URL[] urls = new URL[1];
						urls[0] = new URL("file", null, -1, jarName);
						// URLClassLoader loader = new URLClassLoader(urls,
						// ClassLoader.getSystemClassLoader());
						URLClassLoader loader = URLClassLoader
								.newInstance(urls);
						Class cls = loader.loadClass(clsName);

						Object obj = cls.newInstance();

						// Set the plugin name (the directory including the
						// version)
						Method mSetPluginName = cls.getMethod("setPluginName",
								new Class[] { pluginName.getClass() });
						Logger.out("Aboout to call setPluginName()");
						mSetPluginName.invoke(obj, new Object[] { pluginName });
						Logger.out("Returned from setPluginName()");

						// Init the generator
						Method mInit = cls.getMethod("init", new Class[] { hash
								.getClass() });
						Logger.out("About to call init()");
						mInit.invoke(obj, new Object[] { hash });
						Logger.out("Returned from init()");

						// Set the class loader
						Method mSetLoad = cls.getMethod("setLoader",
								new Class[] { Class
										.forName("java.lang.ClassLoader") });
						Logger.out("About to call setLoader()");
						mSetLoad.invoke(obj, new Object[] { loader });
						Logger.out("Returned from setLoader()");

						// Call the validate() method
						System.out
								.println("Validating agent controller plug-in: "
										+ new File(pluginDir + "/" + pluginName));
						Method mValidate = cls.getMethod("validate",
								new Class[] { Boolean.class });
						Logger.out("About to call validate()");
						mValidate.invoke(obj, new Object[] { Boolean
								.valueOf(showAll.booleanValue()) });
						Logger.out("Returned from validate()");
					} catch (MalformedURLException e0) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.JarNotFound")
								+ " " + jarName);
					} catch (ClassNotFoundException e1) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.ClassNotFound")
								+ " " + clsName);
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.ClassNotFoundHint"));
					} catch (NoSuchMethodException e2) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.MethodNotFound")
								+ " run()");
					} catch (InvocationTargetException e3) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.MethodInvocationError")
										+ " run()");
					} catch (IllegalAccessException e4) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.MethodInvocationError")
										+ " run()");
					} catch (InstantiationException e5) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.ObjectInstantiationError"));
					}
				}
			} else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		} else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	/** Whether or not the config file generation is being done for the IAC */
	public static void setIsIACConfiguration(boolean b) {
		_is_iac_config = b;
	}

	//
	// Generate the main config
	//
	public void generateConfiguration() {
		Document doc;
		Element acConfig;
		Element holder;

		Element acEnv;
		Element appAliases;
		Element app;
		Element var;
		Element hosts;
		Element allow;
		Element plugin;
		Element agent;
		Element logging;
		Element processController;
		Text n;
		Element connection;
		Element transportLayer;
		Element peerMonitoringTransport;
		Element configuration;
		Element port;
		Element commandExtractor;
		Element pipeName;
		Element memName;
		Element isDataMultiplexed;
		Element processPolling;
		Element jvm;
		Element version;
		Element securityEnabled;
		Element userdef;
		Element keystore;
		String sr = PlatformObject.sr;

		//
		// Standard initializer
		//
		doc = configFile.getDoc();
		if (doc == null) {
			return;
		}
		if(configUtil == null)
			configUtil = new ConfigUtility();

		acConfig = doc.createElement(AgentControllerConfig.TAG);
		holder = configFile.getHolder();
		holder.appendChild(acConfig); // Add to the root holder

		//
		// There can be only one AgentControllerConfiguration element
		//
		if (!isNewAC) {
			AgentControllerConfig.setActiveConfiguration(acConfig, "default");
			AgentControllerConfig.setLoggingDetail(acConfig, "LOW");
			AgentControllerConfig.setLoggingLevel(acConfig, "INFORMATION");
			AgentControllerConfig.setPort(acConfig, "10002");
			AgentControllerConfig.setSecuredPort(acConfig, "10003");
			AgentControllerConfig.setVersion(acConfig, ConfigUtility.getString("Config.AgentController.Version"));
			AgentControllerConfig.setFilePort(acConfig, "10005"); // Bug 60082
			AgentControllerConfig.setProcessPolling(acConfig, "true"); // Bug
			// 59316
			AgentControllerConfig.setDataMultiplexed(acConfig, "false"); // Bug
			// 88179

			//
			// Set the version to current
			//
			AgentControllerConfig.setVersion(acConfig, ConfigUtility.getString("Config.AgentController.Version"));

			//
			// Set the Jvm location
			//
			AgentControllerConfig.setJvm(acConfig, configFile.getValue("jvm"));// 86262
		}

		//
		// Agent Controller Environment
		//
		acEnv = doc.createElement(AgentControllerEnvironment.TAG);
		AgentControllerEnvironment.setConfiguration(acEnv, "default");
		acConfig.appendChild(acEnv);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, Constants.JAVA_PATH);
		Variable.setValue(var, configFile.getValue(Constants.JAVA_PATH));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, Constants.RASERVER_HOME);
		Variable.setValue(var, configFile.getValue(Constants.RASERVER_HOME));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		if (isNewAC) {
			var = doc.createElement(Variable.TAG);
			Variable.setName(var, "TPTP_AC_HOME");
			Variable.setValue(var, "%RASERVER_HOME%");
			Variable.setPosition(var, "replace");
			acEnv.appendChild(var);
			
			//bug 227779
			var = doc.createElement(Variable.TAG);
			Variable.setName(var, "TPTP_AC_CONFIG_HOME");
			if(!isInvokedByWorkbench())
				Variable.setValue(var, "%RASERVER_HOME%"+sr+"config");
			else
				Variable.setValue(var, configUtil.getACConfigDirectory(configFile.getValue(Constants.RASERVER_HOME), true, true).getAbsolutePath());
			Variable.setPosition(var, "replace");
			acEnv.appendChild(var);
		
		}

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, Constants.PLUGINS_HOME);
		/* NM - bugzilla 173330: Changed hardcoded plugins location */
		Variable.setValue(var, configFile.getValue(Constants.PLUGINS_HOME));
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, PlatformObject.libEnv);
		Variable.setValue(var, "%RASERVER_HOME%" + sr + PlatformObject.libPath);
		Variable.setPosition(var, "prepend");
		acEnv.appendChild(var);

		/* Bug 69061 begins */
		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "SYS_TEMP_DIR");
		Variable.setValue(var, "%TEMP%"); // Bug 175696
//		Variable.setValue(var, PlatformObject.tempDir);
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);

		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "LOCAL_AGENT_TEMP_DIR"); // Bug 59544
		Variable.setValue(var, "%SYS_TEMP_DIR%");
		Variable.setPosition(var, "replace");
		acEnv.appendChild(var);
		/* Bug 69061 ends */

		/*
		 * Generate the classpath based on the plugin name
		 * 
		 */
		/* NM - bugzilla 173330: Changed hardcoded plugins location */
		String pluginHome = new String(configFile.getValue(Constants.PLUGINS_HOME));
		
		List targetSearchDirs = null;
		try{
			if(_is_iac_config && !_is_invoked_by_workbench){
				File eclipsePath = IACConfigHelper.findEclipseRoot(configFile.getValue(Constants.RASERVER_HOME));
				
				//bug 235611, for consuming product, search for parent plugin directory
				if(eclipsePath == null){
					eclipsePath = IACConfigHelper.findParentPluginDir(configFile.getValue(Constants.RASERVER_HOME));				
				}
				//end of bug 235611
				
				if(eclipsePath == null){
					//this is true when TPTP/IAC is installed outside of Eclipse in an extension location.
					String path = promptEclipsePath("");
					eclipsePath = new File(path);
				}
				targetSearchDirs = IACConfigHelper.findAllPluginDirectory(eclipsePath.getAbsoluteFile());
			}
		}catch(IOException ioe){
			Logger.out("Can't find all Eclipse plugin directories." + ioe.getMessage());
		}
		for (int i = 0; i < _requiredPlugins.length; i++) {
			// Plugin names are not translated and should always be processed using english locale.
			// bugzilla 181424
			String name = new String("CLASSPATH_" + _requiredPlugins[i].replace('.', '_').toUpperCase(ULocale.ENGLISH.toLocale()));
			var = doc.createElement(Variable.TAG);
			Variable.setName(var, name);
			Variable.setPosition(var, "append");
			
			String path = resolvePluginPath(targetSearchDirs, _requiredPlugins[i]);
			
			Variable.setValue(var, path);
			acEnv.appendChild(var);
		}

		//Required for backward compatibility:
		var = doc.createElement(Variable.TAG);
		Variable.setName(var, "CLASSPATH_ORG_APACHE_JAKARTA_COMMONS_LOGGING");
		Variable.setPosition(var, "append");
		Variable.setValue(var, "%CLASSPATH_ORG_APACHE_COMMONS_LOGGING%");
		acEnv.appendChild(var);

		/*
		 * Add the global ones to the CLASSPATH
		 * 
		 */
		for (int i = 0; i < _globalPlugins.length; i++) {
			var = doc.createElement(Variable.TAG);
			Variable.setName(var, "CLASSPATH");
			Variable.setPosition(var, "append");
			
			String path = resolvePluginPath(targetSearchDirs, _globalPlugins[i]);
		
			
			Variable.setValue(var, path);
			acEnv.appendChild(var);
		}
		
		// If the serviceconfig.xml file is being generated for the IAC..
		if(_is_iac_config && isNewAC) {

			// If security is turned on...
			if (configFile.getValue(Constants.SECURITY) != null) {
				if(configFile.getValue(Constants.SECURITY).toUpperCase().equals(Constants.SECURITY_ON)) {
					
					var = doc.createElement(Variable.TAG);
					Variable.setName(var, "CLASSPATH");
					Variable.setPosition(var, "append");
					
					// Resolve the absolute path of the jsse.jar file inside the AC plugin of eclipse
					String[] cp = configUtil.resolveAbsolutePath(pluginHome, "org.eclipse.tptp.platform.agentcontroller").split(File.pathSeparator);
					String jsseCp = null;
					for(int x = 0; x < cp.length; x++) {
						if(cp[x].endsWith("jsse.jar")) {
							jsseCp = cp[x];
						}
					}
					
					// Append to the serviceconfig.xml classpath
					if(jsseCp != null) {
						Variable.setValue(var, jsseCp);
						acEnv.appendChild(var);
					}
				}
			}
		}
		
		//
		// java.exe: Application for RAC; ApplicationAliases for AC
		//
		if (isNewAC) {
			appAliases = doc.createElement(ApplicationAliases.TAG);
			acConfig.appendChild(appAliases);

			app = doc.createElement(Application.TAG);
			Application.setConfiguration(app, "default");
			Application.setExecutable(app, "java.exe");
			Application.setPath(app, "%JAVA_PATH%");
			Application.setLocation(app, "%SYS_TEMP_DIR%");
			appAliases.appendChild(app);

			var = doc.createElement(Variable.TAG);
			Variable.setName(var, "CLASSPATH");
			Variable.setValue(var, "%CLASSPATH_ORG_ECLIPSE_EQUINOX_COMMON%");
			Variable.setPosition(var, "append");
			app.appendChild(var);

			var = doc.createElement(Variable.TAG);
			Variable.setName(var, PlatformObject.libEnv);
			Variable.setValue(var, "%RASERVER_HOME%" + sr + PlatformObject.libPath);
			Variable.setPosition(var, "prepend");
			app.appendChild(var);
		} else {
			app = doc.createElement(Application.TAG);
			Application.setConfiguration(app, "default");
			Application.setExecutable(app, "java.exe");
			Application.setPath(app, "%JAVA_PATH%");
			Application.setLocation(app, "%SYS_TEMP_DIR%");
			acConfig.appendChild(app);

			//
			// Hosts
			//
			hosts = doc.createElement(Hosts.TAG);
			Hosts.setConfiguration(hosts, "default");
			acConfig.appendChild(hosts);

			String hostType = configFile.getValue(Constants.ALLOW_HOST_TYPE);
			if (hostType != null) {
				
				allow = doc.createElement(Allow.TAG);
				Allow.setHostType(allow, hostType);
				
				if (Constants.ALLOW_CUSTOM.equalsIgnoreCase(hostType)){				
					String hostlist = configFile.getValue(Constants.HOSTS);	
					Allow.setHostList(allow, hostlist);
				}
			} else {
				allow = doc.createElement(Allow.TAG);
				Allow.setHostType(allow, Constants.ALLOW_LOCAL);				
			}
			hosts.appendChild(allow);
			
			if (configFile.getValue(Constants.SECURITY) != null) {
				if(configFile.getValue(Constants.SECURITY).toUpperCase().equals(Constants.SECURITY_ON)) {
					boolean newKeystore = false;
					if(configFile.getValue(Constants.USERS) != null) {
						String[] allowedUserList = ConfigUtility.stringToArray(configFile.getValue(Constants.USERS));
						if(allowedUserList.length > 0) {
							Element security = doc.createElement(Security.TAG);
							if(configFile.getValue("SECURITY_KEYSTORE") == null) {
								newKeystore = true;
								Security.setKeystore(security, "%RASERVER_HOME%" + sr + "security" + sr + "tptpSamplekeystore");
							}
							else {
								newKeystore = false;
								Security.setKeystore(security, configFile.getValue("SECURITY_KEYSTORE"));
							}

							if(configFile.getValue("SECURITY_KEYSTOREPASSWORD") == null) {
								Security.setKeystorePassword(security, "password");
							}
							else {
								Security.setKeystorePassword(security, configFile.getValue("SECURITY_KEYSTOREPASSWORD"));
							}
							userdef = doc.createElement(UserDefinition.TAG);
							security.appendChild(userdef);
							if(Constants.USER_ALL.equalsIgnoreCase(configFile.getValue(Constants.USERTYPE))){
								UserDefinition.setUserType(userdef, Constants.USER_ALL);
							} else {
								UserDefinition.setUserType(userdef, Constants.USER_CUSTOM);
								UserDefinition.setUserlist(userdef,configFile.getValue(Constants.USERLIST).trim().replaceAll(" ", ""));						
							}
							
							
							acConfig.appendChild(security);
						}
					}
					//
					// Only generate of there is no existing keystore
					//
					if(newKeystore) {
						String securityHome = configFile.getValue(Constants.RASERVER_HOME) + sr + "security";
						generateSecurityFiles(securityHome);
					}
				}
			}
		}

		//
		// Plugins
		//
		plugin = doc.createElement(Plugin.TAG);
		//bug 227779
		if(!isInvokedByWorkbench())
			Plugin.setPath(plugin, "%PLUGINS_HOME%");
		else
			Plugin.setPath(plugin, configUtil.getPluginPropertyDirectory(true).getAbsolutePath());
		acConfig.appendChild(plugin);

		if (isNewAC) {
			//
			// Agents
			//
			agent = doc.createElement(Agent.TAG);
			Agent.setPath(agent, "%TPTP_AC_HOME%" + sr + "agents");
			acConfig.appendChild(agent);
			
			// Jvm
			
			jvm = doc.createElement(Jvm.TAG);
			Jvm.setLocation(jvm, configFile.getValue("jvm"));
			acConfig.appendChild(jvm);

			//
			// Logging
			//
			logging = doc.createElement(Logging.TAG);
			Logging.setLevel(logging, "INFORMATION");
			Logging.setFormat(logging, "CBE");
			//bug 227779
			Logging.setDirectory(logging, "%TPTP_AC_CONFIG_HOME%");
			acConfig.appendChild(logging);

			//
			// ProcessController
			//
			processController = doc.createElement(ProcessController.TAG);
			n = doc.createTextNode(new String("%TPTP_AC_HOME%" + sr + "bin" + sr + "tptpProcessController" + PlatformObject.exeExt));
			processController.appendChild(n);
			acConfig.appendChild(processController);

			//
			// Connection
			//
			connection = doc.createElement(Connection.TAG);
			Connection.setName(connection, "Dev");
			acConfig.appendChild(connection);

			transportLayer = doc.createElement(TransportLayer.TAG);
			TransportLayer.setLoadLib(transportLayer, "socketTL");
			TransportLayer.setType(transportLayer, "TPTP_SOCKET");
			connection.appendChild(transportLayer);
			configuration = doc.createElement(Configuration.TAG);
			transportLayer.appendChild(configuration);
			port = doc.createElement(Port.TAG);
			configuration.appendChild(port);
			n = doc.createTextNode("10006");
			port.appendChild(n);
			
			Element newAcSecurityEnabled = doc.createElement(SecurityEnabled.TAG);
			configuration.appendChild(newAcSecurityEnabled);
			//
			// Security section
			//
			String sec = configFile.getValue(Constants.SECURITY);
			boolean isNewAcSecurityEnabled = false; 
 			if (sec != null && sec.toUpperCase().equals(Constants.SECURITY_ON)) {
 				n = doc.createTextNode("true");
				isNewAcSecurityEnabled = true;
 			} else {
 				n = doc.createTextNode("false");
				isNewAcSecurityEnabled = false;
 			}
			newAcSecurityEnabled.appendChild(n);
						
			// Element jsseSSLProviderLib
			if(isNewAcSecurityEnabled) {
				
				// JSSE keystore entries into SocketTL 
				
				// SslProviderLib = Jsse
				Element sslProviderLib = doc.createElement(SSLProviderLib.TAG);
				configuration.appendChild(sslProviderLib);
				n = doc.createTextNode("TptpJSSE");
				sslProviderLib.appendChild(n);
				
				// Keystore Filename
				Element jsseKeystoreFilename = doc.createElement(JSSEKeystoreFilename.TAG);
				configuration.appendChild(jsseKeystoreFilename);
				
				String jsseInstallHome = configFile.getValue(Constants.RASERVER_HOME);
				String jsseSecurityHome;
				if (jsseInstallHome == null) {
					jsseSecurityHome = new String(".." + sr + "security");
				} else {
					jsseSecurityHome = new String(jsseInstallHome + sr + "security");
				}
				n = doc.createTextNode(jsseSecurityHome + sr + "tptpSamplekeystore");
				jsseKeystoreFilename.appendChild(n);
				
				// Keystore password
				Element jsseKeystorePassword = doc.createElement(JSSEKeystorePassword.TAG);
				configuration.appendChild(jsseKeystorePassword);
				n = doc.createTextNode("password");
				jsseKeystorePassword.appendChild(n);
			}
						
			// Command Extractor
			
			commandExtractor = doc.createElement(CommandExtractor.TAG);
			transportLayer.appendChild(commandExtractor);
			n = doc.createTextNode("tptpCmdExtr");
			commandExtractor.appendChild(n);

			transportLayer = doc.createElement(TransportLayer.TAG);
			TransportLayer.setLoadLib(transportLayer, "namedPipeTL");
			TransportLayer.setType(transportLayer, "TPTP_LOG");
			connection.appendChild(transportLayer);
			configuration = doc.createElement(Configuration.TAG);
			transportLayer.appendChild(configuration);
			pipeName = doc.createElement(PipeName.TAG);
			configuration.appendChild(pipeName);
			n = doc.createTextNode("acmasterlog");
			pipeName.appendChild(n);
			commandExtractor = doc.createElement(CommandExtractor.TAG);
			transportLayer.appendChild(commandExtractor);
			n = doc.createTextNode("tptpCmdExtr");
			commandExtractor.appendChild(n);

			transportLayer = doc.createElement(TransportLayer.TAG);
			TransportLayer.setLoadLib(transportLayer, "namedPipeTL");
			TransportLayer.setType(transportLayer, "TPTP_NAMEDPIPE");
			connection.appendChild(transportLayer);
			configuration = doc.createElement(Configuration.TAG);
			transportLayer.appendChild(configuration);
			pipeName = doc.createElement(PipeName.TAG);
			configuration.appendChild(pipeName);
			n = doc.createTextNode("acmaster");
			pipeName.appendChild(n);
			commandExtractor = doc.createElement(CommandExtractor.TAG);
			transportLayer.appendChild(commandExtractor);
			n = doc.createTextNode("tptpCmdExtr");
			commandExtractor.appendChild(n);

			transportLayer = doc.createElement(TransportLayer.TAG);
			TransportLayer.setLoadLib(transportLayer, "sharedMemTL");
			TransportLayer.setType(transportLayer, "TPTP_SHAREDMEM");
			connection.appendChild(transportLayer);
			configuration = doc.createElement(Configuration.TAG);
			transportLayer.appendChild(configuration);
			memName = doc.createElement(MemName.TAG);
			configuration.appendChild(memName);
			n = doc.createTextNode("acbuffer");
			memName.appendChild(n);
			commandExtractor = doc.createElement(CommandExtractor.TAG);
			transportLayer.appendChild(commandExtractor);
			n = doc.createTextNode("tptpCmdExtr");
			commandExtractor.appendChild(n);

			if (wantsBC) {
				transportLayer = doc.createElement(TransportLayer.TAG);
				TransportLayer.setLoadLib(transportLayer, "tptpCCTL");
				TransportLayer.setType(transportLayer, "TPTP_CCTL");
				connection.appendChild(transportLayer);
				configuration = doc.createElement(Configuration.TAG);
				transportLayer.appendChild(configuration);
				port = doc.createElement(Port.TAG);
				configuration.appendChild(port);
				n = doc.createTextNode("10002");
				port.appendChild(n);
				port = doc.createElement(SecuredPort.TAG);
				configuration.appendChild(port);
				n = doc.createTextNode("10003");
				port.appendChild(n);
				port = doc.createElement(FilePort.TAG);
				configuration.appendChild(port);
				n = doc.createTextNode("10005");
				port.appendChild(n);
				isDataMultiplexed = doc.createElement(IsDataMultiplexed.TAG);
				configuration.appendChild(isDataMultiplexed);
				n = doc.createTextNode("false");
				isDataMultiplexed.appendChild(n);
				processPolling = doc.createElement(ProcessPolling.TAG);
				configuration.appendChild(processPolling);
				n = doc.createTextNode("true");
				processPolling.appendChild(n);

				version = doc.createElement(Version.TAG);
				configuration.appendChild(version);
				n = doc.createTextNode(ConfigUtility
						.getString("Config.AgentController.Version"));
				version.appendChild(n);
				
				securityEnabled = doc.createElement(SecurityEnabled.TAG);
				configuration.appendChild(securityEnabled);
				//
				// Security section
				//
				boolean isSecured = false;
				if (configFile.getValue(Constants.SECURITY) != null) {
					if (configFile.getValue(Constants.SECURITY).toUpperCase().equals(
							Constants.SECURITY_ON)) {
						isSecured = true;
						n = doc.createTextNode("true");
					} else {
						n = doc.createTextNode("false");
					}
				}
				securityEnabled.appendChild(n);

				if (isSecured) {
					String installHome = configFile.getValue(Constants.RASERVER_HOME);
					String securityHome;
					if (installHome == null) {
						securityHome = new String(".." + sr + "security");
					} else {
						securityHome = new String(installHome + sr + "security");
					}

					//
					// Users section
					//
					//243761
					userdef = doc.createElement(UserDefinition.TAG);
					String userType = configFile.getValue(Constants.USERTYPE);
					if (userType != null) {
						if (userType.toUpperCase().equals(Constants.USER_ALL)) {
							UserDefinition.setUserType(userdef, Constants.USER_ALL);
						} else {
							UserDefinition.setUserType(userdef, Constants.USER_CUSTOM);												
							UserDefinition.setUserlist(userdef, configFile.getValue(Constants.USERLIST).trim().replaceAll(" ", ""));
						}						
					} else {
						UserDefinition.setUserType(userdef, Constants.USER_ALL);	
					}
					configuration.appendChild(userdef);
					
					
					//
					// Keystore section
					//
					keystore = doc.createElement(Keystore.TAG);
					configuration.appendChild(keystore);
					n = doc.createTextNode(securityHome + sr
							+ "tptpSamplekeystore");
					keystore.appendChild(n);

					//
					// Keystore password section
					//
					keystore = doc.createElement(KeystorePassword.TAG);
					configuration.appendChild(keystore);
					n = doc.createTextNode("password");
					keystore.appendChild(n);

					generateSecurityFiles(securityHome);
				}

				//
				// Compatibility Layer Hosts
				//
				hosts = doc.createElement(Hosts.TAG);
				Hosts.setConfiguration(hosts, "default");
				configuration.appendChild(hosts);

				allow = doc.createElement(Allow.TAG);
				String hostType = configFile.getValue(Constants.ALLOW_HOST_TYPE);
				if ( hostType != null) {	
					Allow.setHostType(allow, hostType.toUpperCase());
					if (Constants.ALLOW_CUSTOM.equalsIgnoreCase(hostType)){				
						String hostlist = configFile.getValue(Constants.HOSTS);
						Allow.setHostList(allow, hostlist);
					}						
				} else {
					Allow.setHostType(allow, Constants.ALLOW_LOCAL);					
				}
								
				hosts.appendChild(allow);

				commandExtractor = doc.createElement(CommandExtractor.TAG);
				transportLayer.appendChild(commandExtractor);
				n = doc.createTextNode("tptpCmdExtr");
				commandExtractor.appendChild(n);

				transportLayer = doc.createElement(TransportLayer.TAG);
				TransportLayer.setLoadLib(transportLayer, "tptpACTL");
				TransportLayer.setType(transportLayer, "TPTP_ACTL");
				connection.appendChild(transportLayer);
				configuration = doc.createElement(Configuration.TAG);
				transportLayer.appendChild(configuration);
				commandExtractor = doc.createElement(CommandExtractor.TAG);
				transportLayer.appendChild(commandExtractor);
				n = doc.createTextNode("tptpCmdExtr");
				commandExtractor.appendChild(n);
			}

			peerMonitoringTransport = doc.createElement(PeerConnectionTransport.TAG);
			acConfig.appendChild(peerMonitoringTransport);
			if (wantsBC) {
				n = doc.createTextNode("TPTP_CCTL");
			} else {
				n = doc.createTextNode("TPTP_SOCKET");
			}
			peerMonitoringTransport.appendChild(n);
		}

		configFile.saveToFile();
	}

	//
	// Generate the plugin config files for stand-alone AC
	//
	private void generatePluginConfigurations(Hashtable hash) {
		String home = configFile.getValue(Constants.RASERVER_HOME);
		if (home != null) {
			/* NM - bugzilla 173330: Changed hardcoded plugins location */
			String pluginDirStr = configFile.getValue(Constants.PLUGINS_HOME);
			File pluginDir = new File(pluginDirStr);
			if (pluginDir.exists() && pluginDir.isDirectory()) {
				File[] subDirs = pluginDir.listFiles();
				generatePluginConfigurations(hash, subDirs);
			} else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		} else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	//
	// Generate the plugin config files for IAC
	//
	private void generatePluginConfigurations(Hashtable hash, File[] plugins) {

		for (int i = 0; i < plugins.length; i++) {
			String pluginName = plugins[i].getName(); // Name of the
														// plugin,including the
														// version number at the
														// end
			File pluginDir = new File(plugins[i].getAbsolutePath().substring(0,
					plugins[i].getAbsolutePath().indexOf(plugins[i].getName())));
			String jarName = plugins[i].getAbsolutePath() + "/config.jar";

			/*
			 * Bugzilla 184267 - NM: Check to see whether the specified
			 * pluginHome matches the location of the plugin itself
			 */
			String expectedPluginHome = HashUtility.getValue(hash,
					Constants.PLUGINS_HOME).toLowerCase();
			String actualPluginHome = pluginDir.toString().toLowerCase();
			if (expectedPluginHome.endsWith("/")
					|| expectedPluginHome.endsWith("\\"))
				expectedPluginHome = expectedPluginHome.substring(0,
						expectedPluginHome.length() - 1);
			if (actualPluginHome.endsWith("/")
					|| actualPluginHome.endsWith("\\"))
				actualPluginHome = actualPluginHome.substring(0,
						actualPluginHome.length() - 1);

			if (!actualPluginHome.equals(expectedPluginHome)) {
				HashUtility.setValue(hash, Constants.PLUGINS_HOME,
						actualPluginHome);
			}

			String clsName;
			ArrayList nlJarsVec = new ArrayList();

			if (pluginName.lastIndexOf('_') != -1) {
				clsName = pluginName.substring(0, pluginName.lastIndexOf('_'))
						+ ".SetConfig";
				Enumeration nlJars = getNLFiles(pluginDir, pluginName);
				while (nlJars.hasMoreElements()) { // low cost operation to
													// obtain the number of NL
													// jar files
					nlJarsVec.add(nlJars.nextElement().toString());
				}
			} else {
				clsName = pluginName + ".SetConfig";
			}
			Logger.out("Found plugin: " + pluginName);
			try {
				URL[] urls;
				if ((nlJarsVec != null) && (nlJarsVec.size() > 0)) { // dynamically
																		// generate
																		// URL[]
																		// to
																		// add
																		// NL
																		// jar
																		// files
																		// to
																		// classpath
					urls = new URL[1 + nlJarsVec.size()];
					urls[0] = new URL("file", null, -1, jarName);

					for (int j = 1; j <= nlJarsVec.size(); j++) {
						urls[j] = new URL("file", null, -1, nlJarsVec
								.get(j - 1).toString());
					}
				} else {
					urls = new URL[1];
					urls[0] = new URL("file", null, -1, jarName);
				}

				/*
				 * NM Bugzilla 173330 - Here we have to use the class loader
				 * that was used to load this class because of the funky
				 * classloading mechanism of Eclipse
				 */
				URLClassLoader loader = URLClassLoader.newInstance(urls, this
						.getClass().getClassLoader());
				Class cls = loader.loadClass(clsName);

				Object obj = cls.newInstance();

				// Set the plugin name (the directory including the
				// version)
				Method mSetPluginName = cls.getMethod("setPluginName",
						new Class[] { pluginName.getClass() });
				Logger.out("Aboout to call setPluginName()");
				mSetPluginName.invoke(obj, new Object[] { pluginName });
				Logger.out("Returned from setPluginName()");

				// Init the generator
				Method mInit = cls.getMethod("init", new Class[] { hash
						.getClass() });
				Logger.out("About to call init()");
				mInit.invoke(obj, new Object[] { hash });
				Logger.out("Returned from init()");

				// Set the class loader
				Method mSetLoad = cls.getMethod("setLoader",
						new Class[] { Class.forName("java.lang.ClassLoader") });
				Logger.out("About to call setLoader()");
				mSetLoad.invoke(obj, new Object[] { loader });
				Logger.out("Returned from setLoader()");

				// Invoke the run() method
				Method mRun = cls.getMethod("run", new Class[] {
						hash.getClass(), boolean.class, boolean.class });
				Logger.out("About to call run()");
				mRun.invoke(obj, new Object[] { hash, new Boolean(silent),
						new Boolean(overwrite) });
				Logger.out("Returned from run()");
			} catch (MalformedURLException e0) {
				Logger.err(ConfigUtility
						.getString("Config.Extension.JarNotFound")
						+ " " + jarName);
				Logger.err(e0.getMessage());
			} catch (ClassNotFoundException e1) {
				Logger.err(ConfigUtility
						.getString("Config.Extension.ClassNotFound")
						+ " " + clsName);
				Logger.err(ConfigUtility
						.getString("Config.Extension.ClassNotFoundHint"));
				Logger.err(e1.getMessage());
			} catch (NoSuchMethodException e2) {
				Logger.err(ConfigUtility
						.getString("Config.Extension.MethodNotFound")
						+ " run()");
				Logger.err(e2.getMessage());
			} catch (InvocationTargetException e3) {
				Logger.err(ConfigUtility
						.getString("Config.Extension.MethodInvocationError")
						+ " run()");
				Logger.err(e3.getMessage());
			} catch (IllegalAccessException e4) {
				Logger.err(ConfigUtility
						.getString("Config.Extension.MethodInvocationError")
						+ " run()");
				Logger.err(e4.getMessage());
			} catch (InstantiationException e5) {
				Logger
						.err(ConfigUtility
								.getString("Config.Extension.ObjectInstantiationError"));
				Logger.err(e5.getMessage());
			}
		}
	}

	/**
	 * Generate the files required for security support
	 * 
	 * @param securityPath
	 */
	private void generateSecurityFiles(String securityHome) {
		//
		// Create the keystore and certificate
		// $KEYTOOL -genkey -alias sample -keyalg RSA -sigalg MD5withRSA -dname
		// "CN=Sample, OU=Sample, O=Sample, C=US" -validity 3650 -keypass
		// $PASSWORD -storetype jks -keystore $KEYSTORE -storepass $PASSWORD
		// $KEYTOOL -export -alias sample -file $CERT -keystore $KEYSTORE
		// -storepass $PASSWORD
		//
		//
		File securityDir = new File(securityHome);
		if (!securityDir.exists()) {
			securityDir.mkdir();
		}

		String keytool = resolveKeytool(configFile
				.getValue(Constants.JAVA_PATH));
		String[] genCommand = { keytool, "-genkey", "-alias", "sample",
				"-keyalg", "RSA", "-sigalg", "MD5withRSA", "-dname",
				"CN=Sample, OU=Sample, O=Sample, C=US", "-validity", "3650",
				"-keypass", "password", "-storetype", "jks", "-keystore",
				securityHome + sr + "tptpSamplekeystore", "-storepass",
				"password" };

		String[] exportCommand = { keytool, "-export", "-alias", "sample",
				"-file", securityHome + sr + "tptpSample.cer", "-keystore",
				securityHome + sr + "tptpSamplekeystore", "-storepass",
				"password" };
		try {
			Process p;
			p = Runtime.getRuntime().exec(genCommand);
			p.waitFor();
			p = Runtime.getRuntime().exec(exportCommand);
			p.waitFor();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
	}

	//
	// Print the help from each plugin
	//
	private void printPluginHelp(Hashtable hash) {
		String home = HashUtility.getValue(hash, Constants.RASERVER_HOME);
		String pluginDirStr = "";

		if (home != null) {
			String raHome = HashUtility.getValue(hash, Constants.RASERVER_HOME);
			if (raHome == null || raHome.length() == 0)
				return;

			if (raHome.indexOf("org.eclipse.tptp.platform.ac") == -1)
				pluginDirStr = raHome + sr + "plugins";
			else
				pluginDirStr = raHome.substring(0, raHome
						.indexOf("org.eclipse.tptp.platform.ac") - 1);

			File pluginDir = new File(pluginDirStr);
			if (pluginDir.exists() && pluginDir.isDirectory()) {

				FileFilter f = new FileFilter() {
					public boolean accept(File file) {
						return file.isDirectory();
					}
				};
				File[] subDirs = pluginDir.listFiles(f);
				for (int i = 0; i < subDirs.length; i++) {
					String pluginName = subDirs[i].getName(); // Name of the
					// plugin,
					// including the
					// version
					// number at the
					// end
					String jarName = pluginDirStr.replace('\\', '/') + "/"
							+ pluginName + "/config.jar";
					String clsName;
					if (pluginName.lastIndexOf('_') != -1) {
						clsName = pluginName.substring(0, pluginName
								.lastIndexOf('_'))
								+ ".SetConfig";
					} else {
						clsName = pluginName + ".SetConfig";
					}
					Logger.out("Found plugin: " + pluginName);
					try {
						URL[] urls = new URL[1];
						urls[0] = new URL("file", null, -1, jarName);
						// URLClassLoader loader = new URLClassLoader(urls,
						// ClassLoader.getSystemClassLoader());
						URLClassLoader loader = URLClassLoader
								.newInstance(urls);
						Class cls = loader.loadClass(clsName);

						Object obj = cls.newInstance();

						// Set the plugin name (the directory including the
						// version)
						Method mSetPluginName = cls.getMethod("setPluginName",
								new Class[] { pluginName.getClass() });
						Logger.out("Aboout to call setPluginName()");
						mSetPluginName.invoke(obj, new Object[] { pluginName });
						Logger.out("Returned from setPluginName()");

						// Init the generator
						Method mInit = cls.getMethod("init", new Class[] { hash
								.getClass() });
						Logger.out("About to call init()");
						mInit.invoke(obj, new Object[] { hash });
						Logger.out("Returned from init()");

						// Set the class loader
						Method mSetLoad = cls.getMethod("setLoader",
								new Class[] { Class
										.forName("java.lang.ClassLoader") });
						Logger.out("About to call setLoader()");
						mSetLoad.invoke(obj, new Object[] { loader });
						Logger.out("Returned from setLoader()");

						// Call the printHelp(...) method
						Method mPrintHelp = cls.getMethod("printHelp", null);
						Logger.out("About to call printHelp()");
						mPrintHelp.invoke(obj, null);
						Logger.out("Returned from printHelp()");
					} catch (MalformedURLException e0) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.JarNotFound")
								+ " " + jarName);
					} catch (ClassNotFoundException e1) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.ClassNotFound")
								+ " " + clsName);
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.ClassNotFoundHint"));
					} catch (NoSuchMethodException e2) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.MethodNotFound")
								+ " run()");
					} catch (InvocationTargetException e3) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.MethodInvocationError")
										+ " run()");
					} catch (IllegalAccessException e4) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.MethodInvocationError")
										+ " run()");
					} catch (InstantiationException e5) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.ObjectInstantiationError"));
					}
				}
			} else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		} else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	//
	// Print the help from each plugin
	//
	private void printPluginExamples(Hashtable hash) {
		String home = HashUtility.getValue(hash, Constants.RASERVER_HOME);
		if (home != null) {
			String pluginDirStr = null;
			if(configFile != null)
				pluginDirStr = configFile.getValue(Constants.PLUGINS_HOME);
			// From printPluginHelp, try to find the plug-in directory using the RASERVER_HOME
			if(pluginDirStr == null && home.length() != 0) {
				if (home.indexOf("org.eclipse.tptp.platform.ac") == -1)
					pluginDirStr = home + sr + "plugins";
				else
					pluginDirStr = home.substring(0, home.indexOf("org.eclipse.tptp.platform.ac") - 1);
			}
			File pluginDir = new File(pluginDirStr);
			if (pluginDir.exists() && pluginDir.isDirectory()) {
				File[] subDirs = pluginDir.listFiles();
				for (int i = 0; i < subDirs.length; i++) {
					String pluginName = subDirs[i].getName(); // Name of the
					// plugin,
					// including the
					// version
					// number at the
					// end
					String jarName = pluginDirStr.replace('\\', '/') + "/"
							+ pluginName + "/config.jar";
					String clsName;
					if (pluginName.lastIndexOf('_') != -1) {
						clsName = pluginName.substring(0, pluginName
								.lastIndexOf('_'))
								+ ".SetConfig";
					} else {
						clsName = pluginName + ".SetConfig";
					}
					Logger.out("Found plugin: " + pluginName);
					try {
						URL[] urls = new URL[1];
						urls[0] = new URL("file", null, -1, jarName);
						// URLClassLoader loader = new URLClassLoader(urls,
						// ClassLoader.getSystemClassLoader());
						URLClassLoader loader = URLClassLoader
								.newInstance(urls);
						Class cls = loader.loadClass(clsName);

						Object obj = cls.newInstance();

						// Set the plugin name (the directory including the
						// version)
						Method mSetPluginName = cls.getMethod("setPluginName",
								new Class[] { pluginName.getClass() });
						Logger.out("Aboout to call setPluginName()");
						mSetPluginName.invoke(obj, new Object[] { pluginName });
						Logger.out("Returned from setPluginName()");

						// Init the generator
						Method mInit = cls.getMethod("init", new Class[] { hash
								.getClass() });
						Logger.out("About to call init()");
						mInit.invoke(obj, new Object[] { hash });
						Logger.out("Returned from init()");

						// Set the class loader
						Method mSetLoad = cls.getMethod("setLoader",
								new Class[] { Class
										.forName("java.lang.ClassLoader") });
						Logger.out("About to call setLoader()");
						mSetLoad.invoke(obj, new Object[] { loader });
						Logger.out("Returned from setLoader()");

						// Call the printExamples() method
						Method mExamples = cls.getMethod("printExamples", null);
						Logger.out("About to call printExamples()");
						mExamples.invoke(obj, null);
						Logger.out("Returned from printExamples()");
					} catch (MalformedURLException e0) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.JarNotFound")
								+ " " + jarName);
					} catch (ClassNotFoundException e1) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.ClassNotFound")
								+ " " + clsName);
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.ClassNotFoundHint"));
					} catch (NoSuchMethodException e2) {
						Logger.out(ConfigUtility
								.getString("Config.Extension.MethodNotFound")
								+ " run()");
					} catch (InvocationTargetException e3) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.MethodInvocationError")
										+ " run()");
					} catch (IllegalAccessException e4) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.MethodInvocationError")
										+ " run()");
					} catch (InstantiationException e5) {
						Logger
								.out(ConfigUtility
										.getString("Config.Extension.ObjectInstantiationError"));
					}
				}
			} else {
				Logger.out("Cannot access directory: " + pluginDirStr);
				return;
			}
		} else {
			Logger.out("RASERVER_HOME is not defined");
			return;
		}
	}

	private String resolveJvmDll(String string) {
		if (PlatformObject.name.equals("OS/400")) {
			return new String("QSYS/QJVAJNI"); // Bug 58152
		} else {
			int index;

			index = string.lastIndexOf(sr + "java" + PlatformObject.exeExt);
			if (index == -1) {
				index = string
						.lastIndexOf(sr + "javaw" + PlatformObject.exeExt);
				if (index == -1) {
					return PlatformObject.jvmLib;
				}
			}

			String str = string.substring(0, index);
			String rc = null;

			try {
				rc = ConfigUtility.getJvmLib(str);
			} catch (Exception e) {
				Logger.out(ConfigUtility
						.getString("Config.Jvm.Warning.CannotFindJvmLibrary")
						+ ": " + PlatformObject.jvmLib);
			}

			if (rc == null) {
				return PlatformObject.jvmLib;
			} else {
				return rc;
			}
		}
	}

	private String resolveKeytool(String string) {
		if (PlatformObject.name.equals("OS/400")) {
			return new String("keytool"); // no absolute path on iSeries
		} else {
			int index = string.lastIndexOf(sr + "java" + PlatformObject.exeExt);
			if (index == -1) {
				index = string
						.lastIndexOf(sr + "javaw" + PlatformObject.exeExt);
				if (index == -1) {
					return PlatformObject.keytool;
				}
			}
			String str = string.substring(0, index);
			String rc = null;

			try {
				rc = ConfigUtility.getKeytool(str);
			} catch (Exception e) {
				Logger.out(ConfigUtility
						.getString("Config.Jvm.Warning.CannotFindKeytool"));
			}

			if (rc == null) {
				return PlatformObject.keytool;
			} else {
				return rc;
			}
		}
	}

	/**
	 * 
	 * @param targetDirs
	 *            List of plugin directories to search for required plugins. For
	 *            stand-aline AC and IAC in Eclipse, this would be null. *
	 * @param pluginName
	 *            Name of plugin whose path would be resolved
	 * @return Absolute path of the plugin. Return String "null" if not found.
	 */
	private String resolvePluginPath(List targetDirs, String pluginName) {

		String path = "null";
		if (_is_iac_config && !_is_invoked_by_workbench) {
			if (targetDirs == null)
				return path;

			// need to search required plugins in all Eclipse plugin directories
			Iterator iterator = targetDirs.iterator();
			while (iterator.hasNext() && "null".equalsIgnoreCase(path)) {
				File f = (File) iterator.next();
				path = configUtil.resolveAbsolutePath(f.getAbsolutePath(),
						pluginName);
			}
		} else {
			// For stand-alone AC and IAC invoked in Eclipse
			String pluginHome = new String(configFile
					.getValue(Constants.PLUGINS_HOME));
			path = configUtil.resolveAbsolutePath(pluginHome, pluginName);
		}
		return path;
	}

	public static boolean isInvokedByWorkbench() {
		return _is_invoked_by_workbench;
	}

	public static void setIsInvokedByWorkbench(boolean _is_invoked_by_workbench) {
		SetConfig._is_invoked_by_workbench = _is_invoked_by_workbench;
	}
	
}
