/**********************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SSLFactory.java,v 1.2 2008/01/25 19:37:28 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.agentcontroller.jsse;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.UnknownHostException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;

import javax.net.ssl.KeyManager;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLServerSocketFactory;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;

public class SSLFactory {
	private static SSLFactory _instance = null;
	
	private static String _protocol = "SSL";
	 
	private SSLContext _sslContext = null;
	private boolean _sslContextInitialized = false; 
	
	private KeyManager[] _kms = new KeyManager[1];
	private TrustManager[] _tms = new TrustManager[1];

	public synchronized static SSLFactory getInstance() throws NoSuchAlgorithmException {
		if(_instance == null) {
			_instance = new SSLFactory();
		}

		return _instance;
	}

	private SSLFactory() throws NoSuchAlgorithmException {
		_sslContext = SSLContext.getInstance(_protocol);
	}

	public SSLSocket createSocket(Socket socket) {
		SSLSocketFactory factory;
		SSLSocket sslSocket = null;

		// Initialize the SSL context only once
		synchronized(_sslContext) {
			if(!_sslContextInitialized) {
				try {
					_sslContext.init(_kms, _tms, null);
					_sslContextInitialized = true;
				} catch(Exception e) {
					e.printStackTrace();
				}
			}
		}

		factory = _sslContext.getSocketFactory();
		
		// An alternative here was: factory = (SSLSocketFactory) SSLSocketFactory.getDefault();
		// This would have involved using anonymous ciphers.


		try {
			sslSocket = (SSLSocket) factory.createSocket(socket, socket.getInetAddress().getHostName(), socket.getPort(), true);
			
		} catch (IOException e) {
				//	e.printStackTrace();
		}

		return sslSocket;
	}

	public Socket getClientSocket(String host, int port) throws IOException, KeyManagementException, UnknownHostException {
		Socket socket = null;
		
		// Initialize the SSL context only once
		synchronized(_sslContext) {
			if(!_sslContextInitialized) {
				_sslContext.init(_kms, _tms, null);
				_sslContextInitialized = true;
			}
		}
		
		SSLSocketFactory factory = _sslContext.getSocketFactory();

		if(factory != null) {
			socket = factory.createSocket(host, port);
		}

		return socket;
	}

	public ServerSocket getServerSocket(int port) throws IOException, KeyManagementException {
		ServerSocket socket = null;
		
		// Initialize the SSL context only once
		synchronized(_sslContext) {
			if(!_sslContextInitialized) {
				_sslContext.init(_kms, _tms, null);
				_sslContextInitialized = true;
			}
		}
		
		SSLServerSocketFactory factory = _sslContext.getServerSocketFactory();

		if(factory != null) {
			socket = factory.createServerSocket(port);
		}

		return socket;
	}

	public void setKeyManager(KeyManager km) {
		_kms[0] = km;
	}

	public void setTrustManager(TrustManager tm) {
		_tms[0] = tm;
	}

}
