/**********************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SSLNativeBinding.java,v 1.3 2008/01/30 22:43:37 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.agentcontroller.jsse;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.Hashtable;

import javax.net.ssl.SSLSocket;

public class SSLNativeBinding {
	
	// Note: The Hashtable class is synchronized by default.
	private Hashtable /* String -> String */ _vars = new Hashtable();
	private Hashtable /* Integer -> Socket */ _sockets = new Hashtable();

	SSLFactory _factory = null;
	
	private ServerSocket _serverSocket;
	private int _socketCount = 0;
	
	// Debug 
	final static boolean DEBUG = false;
	final static Object DEBUG_SYNCH = new Object();

	
	public SSLNativeBinding() {}
	
	protected int nativeAccept() {
		int index = -1;
		Socket socket = null;

		if(DEBUG) synchWrite("Java: nativeAccept()");
		
		try {
			socket = _serverSocket.accept();
			
			if(socket == null) {
				return -1;
			}
			
			if(DEBUG) System.out.println("Connection accepted from host: " + socket.getInetAddress().getCanonicalHostName());
			index = _socketCount; 
			_socketCount++;
			_sockets.put(new Integer(index), socket);
			
		} catch (IOException e) {
			if(DEBUG) e.printStackTrace();
			return -1;
		}

		return index; // this is the connectionID
	}

	protected void nativeClose(int connectionID) {
		
		if(DEBUG) synchWrite("Java: nativeClose()");
		
		Socket socket = (Socket)_sockets.remove(new Integer(connectionID));

		if(socket == null) {
			return;
		}

		try {
			socket.close();
		} catch (IOException e) {
			// e.printStackTrace();
		}
	}

	protected String nativeGetValue(String key) {
		if(DEBUG) synchWrite("Java: nativeGet()");
		return (String)_vars.get(key);
	}

	protected void nativeHandshake(int connectionID) {
		
		if(DEBUG) synchWrite("Java: nativeHandshake()");
		
		Socket socket = (Socket)_sockets.get(new Integer(connectionID));
		SSLSocket sslSocket;
		
		if(socket == null) {
			return;
		}

		sslSocket = (SSLSocket) _factory.createSocket(socket);
		
		sslSocket.setUseClientMode(false);
		sslSocket.setNeedClientAuth(false);
		sslSocket.setWantClientAuth(false);

		// Update the socket entry in the hash
		_sockets.put(new Integer(connectionID), sslSocket);
	}

	final int NINITERR_UNABLE_TO_LISTEN = -2;
	final int NINITERR_NO_ALG = -3;
	final int NINITERR_KEYSTOREEX = -4;
	final int NINITERR_CERTEX = -5;
	final int NINITERR_UNRECOVKEY = -6;
	
	protected int nativeInit() {
		int result = 0;
		
		if(DEBUG) synchWrite("Java: nativeInit()");
		
		String ksFilename = (String)_vars.get("org.eclipse.tptp.platform.agentcontroller.jsse.ksFilename");
		String ksPassword = (String)_vars.get("org.eclipse.tptp.platform.agentcontroller.jsse.ksPassword");
		String serverPort = (String)_vars.get("org.eclipse.tptp.platform.agentcontroller.jsse.serverPort");

		try {
			_factory = SSLFactory.getInstance();

			SSLKeyStore ks = new SSLKeyStore(ksFilename, ksPassword.toCharArray());
			ks.loadKeyStore();

			SSLKeyManager km = new SSLKeyManager(ks.getKeyStore(), ksPassword.toCharArray());
			_factory.setKeyManager(km);

			SSLTrustManager tm = new SSLTrustManager(ks.getKeyStore());
			_factory.setTrustManager(tm);

			_serverSocket = new ServerSocket(Integer.parseInt(serverPort));
			
			if(_serverSocket == null) {
				result = NINITERR_UNABLE_TO_LISTEN;
			}
		} catch(NoSuchAlgorithmException nsa) {
			result = NINITERR_NO_ALG;
		} catch(KeyStoreException kse) {
			result = NINITERR_KEYSTOREEX;
		} catch(CertificateException ce) {
			result = NINITERR_CERTEX;
		} catch(UnrecoverableKeyException uke) {
			result = NINITERR_UNRECOVKEY;
		} catch(IOException e) {
			result = NINITERR_UNABLE_TO_LISTEN;
		} catch(Exception e) {
			result = NINITERR_UNABLE_TO_LISTEN;
		}		
		return result;
	}

	protected int nativeRead(int connectionID, byte[] buffer, int length) {
		
		if(DEBUG) synchWrite("Java: nativeRead()..... Connection ID: "+connectionID);
		Socket socket = (Socket)_sockets.get(new Integer(connectionID));
		
		if(socket == null) {
			if(DEBUG) {
				System.out.println("Error: Socket is null. (connectionId:"+connectionID+")");
			}
			return -1;
		}

		try {
			int returnval = socket.getInputStream().read(buffer, 0, length);
			
			if(DEBUG) {
				if(returnval > 0) {
					synchronized(DEBUG_SYNCH) {
						System.out.print("[Reading from Client: ");
						printArray(buffer, returnval);
						System.out.println("]");
					}
				}			
				
				synchWrite("nativeRead() Java return ("+returnval+")");
			}
			
			return returnval;
		} catch (Exception e) {
			// Expected exceptions here are NullPointerException and IOException.
			// e.printStackTrace(); 
		}

		return -1;
	}

	protected void nativeSetValue(String key, String value) {
		if(DEBUG) synchWrite("Java: nativeSet()");
		
		_vars.put(key, value);
	}

	protected void nativeShutdown() {
		
		if(DEBUG) synchWrite("Java: nativeShutdown()");
		
		if(_serverSocket == null) {
			return;
		}
		
		try {
			_serverSocket.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	protected void nativeWrite(int connectionID, byte[] buffer, int length) {
		if(DEBUG) synchWrite("Java: nativeWrite()");
		
		Socket socket = (Socket)_sockets.get(new Integer(connectionID));

		if(DEBUG) {
			synchronized(DEBUG_SYNCH) {
				System.out.println("-----");
				System.out.println("Connection id:" + connectionID);
				System.out.print("[Writing to Client: ");
				printArray(buffer, length);
				System.out.println("]");
				System.out.println("-----");
			}
		}
		
		if(socket == null) {
			return;
		}

		try {
			socket.getOutputStream().write(buffer, 0, length);
			socket.getOutputStream().flush();
		} catch (IOException e) {
			if(DEBUG) e.printStackTrace();
		}
	}
	
	// Debugger helper methods
	
	private static void synchWrite(String s) {
		synchronized(DEBUG_SYNCH) {
			System.out.println(s);
		}
	}
	
	private void printArray(byte[] arr, int length) {
		
		for(int x = 0; x < length; x++) {
			byte b = arr[x];
			
			if(b <= 31) {
				System.out.print("("+b+")");
			} else {
				System.out.print((char)b);
			}
			
		}
	}
}
