/*******************************************************************************
 * Copyright (c) 2005, 2006 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hoang M Nguyen, Intel - Initial API and Implementation
 *
 * $Id: TransportSupportNamedPipe.h,v 1.8 2006/11/02 18:58:10 akaylor Exp $
 *
 *******************************************************************************/ 



#ifndef TransportSupportNamedPipe_H
#define TransportSupportNamedPipe_H


#include "tptp/TPTPSupportTypes.h"


/** named pipe options for creating or opening */
enum NamedPipeOperationOptions { TPTP_PIPE_READ_ONLY, TPTP_PIPE_WRITE_ONLY, TPTP_PIPE_DUPLEX, TPTP_PIPE_NC_WRITE_ONLY } ;


#ifdef __cplusplus
extern "C" {
#endif


/**
 *********************************************************
 *
 * @brief
 *    read from the given named pipe
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
int readFromNamedPipe(HANDLE handle,
						 char *buffer,
						 int offset,
						 int length,
						 int *bytesRead) ;

/**
 *********************************************************
 *
 * @brief
 *    write to the given named pipe
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
int writeToNamedPipe(HANDLE handle,
					  char *buffer,
					  int offset,
					  int length,
					  int *bytesWritten) ;

/**
 *********************************************************
 *
 * @brief
 *    clean up the given pipe
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

int cleanPipeUp(HANDLE *pPipeHandle) ;

/**
 *********************************************************
 *
 * @brief
 *   Used by the creater of a named pipe to connect to a previously created named
 *   pipe for reading and waits for another application to open the pipe for writing. 
 *
 * @note: This is a blocking call (i.e., the call will not return until someone else
 *        has opened the pipe for writing.)
 *
 * @note: the two parameters are needed for platform-independent support
 *        (handle to be used in Windows and the name of the pipe for Linux)
 *
 * @param *pHandle - the handle of the pipe to connect to
 *
 * @param pPipeName - the full name of the pipe to connect to
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
int connectToNamedPipe(HANDLE *pHandle, const char *pPipeName) ;


/**
 *********************************************************
 *
 * @brief
 *    create a given named pipe with read only
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE createReadOnlyNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;

/**
 *********************************************************
 *
 * @brief
 *    create a given named pipe for writing
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE createWriteOnlyNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;

/**
 *********************************************************
 *
 * @brief
 *    create a given named pipe with full duplex
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE createDuplexNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;


/**
 *********************************************************
 *
 * @brief
 *    create a given named pipe with the given options
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE createGeneralNamedPipe(const char *pNameSpace, const char *pPipeName, int createOption, int bInheritHandle) ;



/**
 *********************************************************
 *
 * @brief
 *    open a given existing named pipe with read
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE  openReadOnlyNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;


/**
 *********************************************************
 *
 * @brief
 *    open a given existing named pipe for writing only
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE  openWriteOnlyNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;

/**
 *********************************************************
 *
 * @brief
 *    open a given existing named pipe for writing only
 *    but do not retry if the open fails
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE  openNonCriticalWriteOnlyNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;

/**
 *********************************************************
 *
 * @brief
 *    open a given existing named pipe with read and write
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE  openDuplexNamedPipe(const char *pNameSpace, const char *pPipeName, int bInheritHandle) ;


/**
 *********************************************************
 *
 * @brief
 *    open a given existing named pipe with the given mode
 *
 * @return
 *    positive  - Success
 *    zero or negative - Error.
 *********************************************************/
HANDLE  openGeneralNamedPipe(const char *pNameSpace, const char *pPipeName, int openOption, int bInheritHandle) ;



/**
 *********************************************************
 *
 * @brief
 *    disconnects the server end of a named pipe instance
 *       from a client process.
 *
 * @return
 *    TRUE  - Success
 *    FALSE - Error.
 *********************************************************/
int disconnectFromNamedPipe(HANDLE handle) ;


/**
 *********************************************************
 *
 * @brief
 *    completely remove the named pipe from the system
 *
 * @return
 *    0  - Success
 *    nonzero - Error.
 *********************************************************/
int destroyNamedPipe(const char *pFullPipeName) ;





#ifdef __cplusplus
}
#endif


#endif


