/*******************************************************************************
 * Copyright (c) 2005 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *  Contributors:
 *
 *    Kevin P O'Leary, Intel - Sample of FileTransfer Client
 *
 * $Id: MyFileTransferClient.cpp,v 1.8 2005/11/07 18:51:18 koleary Exp $
 *
 *
 *******************************************************************************/ 



#include <iostream>
using namespace std;

#include "tptp/client/NodeFactory.h"
#include "tptp/client/FileTransfer.h"

using namespace TPTP::Client;



/**
 *********************************************************
 *                                                       
 * @brief                                                
 *    MyFileTransferClient 
 * This is a sample client program that uses the Agent Controller 
 * to get access to the FileTransfer agent.
 *
 * The FileTransferAgent is then sent 3 messages.
 * 	getFile
 * 	deleteFile
 * 	putFile
 *
 * Usage : MyFileTransferClient			<==defaults to localhost, port 10006
 *		 : MyFileTransferClient <HostName> <Port> 
 *		 : For e.g. MyFileTransferClient
 *					MyFileTransferClient "lab-system-001" 10006
 * Expected Output:
 * 	Connected to Agent Controller message
 * 	Running getFile test
 * 	Get File X From File Y
 *
 * 	Running deleteFile test
 * 	Delete File X 
 *
 * 	Running putFile tests
 * 	Put file X To File Y
 * 	All finished
 * 	Press any key to exit...
 *
 *  External Dependencies:
 *  	1 file to test file transfers.
 *  		The content of this file does not matter, it can be text or binary.
 *  		This will be used to test the FileTransfer operations.
 *
 *  		This Sample encodes the file path as followed.
 *  		On Windows
 *  			C:\big_simple.txt
 *  		On Linux
 *  			/tmp/big_simple.txt
 * 	
 * Source Code Files:
 * 	MyFileTransferClient.cpp
 *
 *
 *********************************************************/
int main(int argc, char* argv[])
{
	char* hostName = "localhost";
	int portNum = 10006;
	INode* TargetNode = 0;
	AgentController* ACProxy = 0;
	
	//Start of generic connect to AgentController code.
	if ((argc == 2) || (argc > 3))
	{
		cout<<"Usage: MyFileTransferClient or MyFileTransferClient <HostName> <Port> "<<endl;
		cout<<"Press any key to exit..."<<endl;
		getchar();
		return -1;
	}

	// Use the host name and port number if provided on the command line.
	if (argc == 3)
	{
		hostName = argv[1];
		portNum = atoi(argv[2]);
	}
	

	// Create a Node object for the target system of choice.
	TargetNode = NodeFactory::createNode(hostName);
	if (!TargetNode)
	{
		cout<<"ERROR: Failed to create the Node object."<<endl;
		cout<<"Press any key to exit..."<<endl;
		getchar();
		return -1;
	}

	// Establish a connection with an Agent Controller on the
	// target system.  The port number specified must match
	// that defined in the configuration file for the desired
	// Agent Controller.
	ACProxy = TargetNode->connect(portNum);
	
	if (!ACProxy)
	{
		cout<<"ERROR: Unable to connect to the Agent Controller running on "<<hostName<<" at port number "<<portNum<<endl;
		cout<<"Press any key to exit..."<<endl;
		getchar();
		return -1;
	}
	
	cout<<endl<<"Connected to the Agent Controller on \""<<hostName<<"\" at port number "<<portNum<<endl;

	//End of generic connect to AgentController code.

	//Start of FileTransferAgent specific code.
	// Create a client-side representative of a FileTransfer object from which
	// a MyFileTransferAgent is derived, specifying the name of the desired agent
	// as it has been installed in the agents directory.
	FileTransfer* MyFileTransferAgent 
		= new FileTransfer("org.eclipse.tptp.FileTransferAgent");

	
	// Get a handle to the actual FileTransferAgent, storing it in the MyFileTransferAgent
	// object.  If the call is successful, the FileTransferAgent is running and
	// ready to respond to requests.
	// Note that the Agent Controller manages the life of an agent, so it will start
	// one running if needed.
	// We request controller access to the agent since we want to perform
	// actions like getFile and putFile. However, the FileTransferAgent is not checking
	// access levels on incoming requests at this time (TBD). 
	if (-1 == ACProxy->getAgent(MyFileTransferAgent, TPTP_CONTROLLER_ACCESS))
	{
		cout<<"ERROR: Unable to get agent org.eclipse.tptp.FileTransferAgent"<<endl;			
		cout<<"Press any key to exit..."<<endl;
		getchar();
		return -1;
	}

	cout<<endl<<"The FileTransferAgent Agent ID: "<<MyFileTransferAgent->getAgentID()<<endl;

	//These next variables encode the file paths for the transfered files.
#ifdef _WIN32
	char *remoteFile = "C:\\big_simple.txt";          //This is the source file for getFile and putFile. 
	char *localFile = "C:\\big_local.txt";            //This is the destination file for getFile and putFile.
							  //Deleted by deleteFile operation.
#else
	char *remoteFile = "/tmp/big_simple.txt";  	  //This is the source file for getFile and putFile.
	char *localFile = "/tmp/big_local.txt";           //This is the destination file for getFile and putFile.
							  //Deleted by deleteFile operation.
#endif

	cout <<"Running getFile test " << endl;
	cout <<"Get file " << localFile << " From File " << remoteFile << endl << endl;

	//Get the file specified in remoteFile to localFile.
	MyFileTransferAgent->getFile(localFile, remoteFile);

	cout<<"Running deleteFile test" <<endl;
	cout <<"Delete file " << localFile << endl << endl;
	MyFileTransferAgent->deleteFile(localFile);

	cout <<"Running putFile test " << endl;
	cout <<"Put file " << remoteFile << " To File " << localFile << endl;

	//Put the file specified in remoteFile to localFile.
	MyFileTransferAgent->putFile(remoteFile, localFile);
	// Wait for data to finish coming over channel.
	// Note: A protocol for recognizing end of data is TBD.
	Sleep(2000);

	// Release resources allocated by the Agent Controller
	MyFileTransferAgent->destroyDataConnection();
	MyFileTransferAgent->releaseAgent();
	ACProxy->disconnect();
	
	// Release client side resources
	delete MyFileTransferAgent;
	ACProxy->destroy();
	NodeFactory::deleteNode(TargetNode);
	
	cout<<"All finished"<<endl;
	cout<<"Press any key to exit..."<<endl;
	getchar();
	return 0;
}


