/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ossdebug.h,v 1.6 2009/01/27 19:20:01 jcayne Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

/*******************************************************************************

   Source File Name = ossdebug.h 

   Descriptive Name = Run-time OSSe debugging code

   Function:
      Defines:
         OSS_ASSERT
         OSS_VERIFY

   Dependencies:
      None

   Restrictions:
      None

   Change Activity:
   Defect Date        Who Description
   ====== =========== === ==============================================
   161486 Nov/25/2000 jpk Initial drop
   162796 Dec/19/2000 jpk Added "const" pointers to ossIsBadReadPtr and
                          ossIsBadStringPtr
   162924 Dec/19/2000 jpk Moved the function prologues inside the header

   Last Changed =    01/03/20  18:48:26

*******************************************************************************/

#ifndef OSSDEBUG_HEADER_INCLUDED
#define OSSDEBUG_HEADER_INCLUDED

#ifdef _HPUX
#include <stdlib.h>
#endif


#include "oss.h"

/******************************************************************************
   ASSERTIONS
******************************************************************************/

/******************************************************************************

   Macro Names
      OSS_ASSERT
      OSS_VERIFY

   Function
      Evaluates an expression.  When the result is false the ossAssertFailed
      function is called.  If OSS_DEBUG is not defined, OSS_ASSERT evaluates
      to nothing and the expression will not be included in the source code.
      On the other hand, the expression evaluated in OSS_VERIFY will still be
      included in the code if OSS_DEBUG is not defined but ossAssertFailed
      will *not* be called.

   Inputs
      1. expr
      The expression to evaluate.

******************************************************************************/
#if defined OSS_DEBUG
OSS_EXTERNC void OSS_API ossAssertFailed( const char *, const char *, size_t ) ;
#define OSS_ASSERT(expr) \
   (void)((expr) || (ossAssertFailed( #expr, __FILE__, __LINE__), 0))
#define OSS_VERIFY(expr)      OSS_ASSERT(expr)
#else
#define OSS_ASSERT(expr)      ((void)0)
#define OSS_VERIFY(expr)      ((void)(expr))
#endif /* OSS_DEBUG */


/******************************************************************************
   MEMORY ACCESS VALIDATION
******************************************************************************/

/******************************************************************************

   Function Name
      ossIsBadReadPtr

   Function
      Verifies that the calling process has read access to the specified
      memory range.
	
   Inputs
      1. ptr
      Address of the first byte of the memory range.
      2. size
      Number of bytes in the memory range.

   Normal Returns
      OSS_OK
         The calling process has read access to all bytes in the specified
         memory range.

   Error Returns
      OSS_ERR_FAILED
         The calling process does not have read access to all bytes in the
         specified memory range.

******************************************************************************/
OSS_EXTERNC OSSErr OSS_API ossIsBadReadPtr(
      const void * ptr,
      size_t size ) ;

/******************************************************************************

   Function Name
      ossIsBadWritePtr

   Function
      Verifies that the calling process has write access to the specified
      memory range.
	
   Inputs
      1. ptr
      Address of the first byte of the memory range.
      2. size
      Number of bytes in the memory range.

   Normal Returns
      OSS_OK
         The calling process has write access to all bytes in the specified
         memory range.

   Error Returns
      OSS_ERR_FAILED
         The calling process does not have write access to all bytes in the
         specified memory range.

******************************************************************************/
OSS_EXTERNC OSSErr OSS_API ossIsBadWritePtr(
      void * ptr,
      size_t size ) ;

/******************************************************************************

   Function Name
      ossIsBadStringPtr

   Function
      Verifies that the calling process has read access to a memory range
      contained within a string pointer.
	
   Inputs
      1. psz
      Address of a null-terminated string.
      2. length
      Specifies the maximum size, in characters, of the string. The function
      checks for read access in all bytes up to the string?s terminating null
      character or up to the number of bytes specified by this parameter,
      whichever is smaller.

   Normal Returns
      OSS_OK
         If the calling process has read access to all bytes up to the
         string?s terminating null character or up to the number of bytes
         specified by length, the return value is zero.

   Error Returns
      OSS_ERR_FAILED
         If the calling process does not have read access to all bytes up
         to the string's terminating null character or up to the number of
         bytes specified by length, the return value is nonzero.

******************************************************************************/
OSS_EXTERNC OSSErr OSS_API ossIsBadStringPtr(
      const char * psz,
      size_t length ) ;


#if defined(__linux__) && defined(__s390__)
// The following are required on Linux 390, as the xchg atomic function calls these in the case of misalignment. (See their companion in ossdebug.h)
 #ifdef __cplusplus
 extern "C" {
 #endif 

void __misaligned_u16(void);

void __misaligned_u32(void);

void __misaligned_u64(void);

#ifdef __cplusplus
 }
 #endif 

#endif
 
 
#endif

