/**********************************************************************
 * Copyright (c) 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: tptpJSSE.cpp,v 1.5 2008/02/06 12:41:00 ialelekov Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

// tptpJSSE.cpp : Defines the entry point for the DLL application.
//

#include <stdio.h>
#include <stdlib.h>

#include "tptpJSSELog.h"
#include "tptpJSSE.h"


/*
 * Interface functions for SSL provider
 */

static bool JSSE_DEBUG = false;

/*
 * This function will block until a connection is accepted
 * Parameters:
 *   none
 * Return:
 *   void*: the native socket handle for this new accepted connection
 */
void* sslAccept() {
	ssl_socket_t hSocket = NULL;
	int jSocket;

	if(JSSE_DEBUG) printf(">> sslAccept() \n");

	jSocket = nativeAccept();
	if(jSocket != -1) {
		hSocket = (ssl_socket_t)malloc(sizeof(ssl_socket_t));
		hSocket->clientSocket = jSocket;
		hSocket->secured = false;
	}

	if(JSSE_DEBUG) printf("<< sslAccept() \n");
	return hSocket;
}

/*
 * This function will close a connection specified by the connection ID
 * Parameters:
 *   int: the connection ID for the connection to be closed
 * Return:
 *   none
 */
int sslClose(void* handle) {

	if(JSSE_DEBUG) printf(">> sslClose() \n");
	
	ssl_socket_t hSocket = (ssl_socket_t)handle;
	nativeClose(hSocket->clientSocket);

	if(JSSE_DEBUG) printf("<< sslClose() \n");

	return 0;
}

/*
 *
 */
char* sslGetError(void* handle, int errcode) {
	if(JSSE_DEBUG) printf(">><< sslGetError() \n");
	return "";
}

/*
 * This function will return the SSL provide name
 * Parameters:
 *   none
 * Return:
 *   char*: the name of the SSL provider
 */
char* sslGetProviderName() {
	if(JSSE_DEBUG) printf(">><< sslGetProviderName() \n");
	return SSL_PROVIDER_NAME;
}

/*
 * This function will get the value of a name-value pair from the provider
 * Parameters:
 *   name: the name of the variable (null terminated)
 * Return:
 *   value: the value of the variable (null terminated)
 */
char* sslGetValue(const char* name) {
	if(JSSE_DEBUG) printf(">> sslGetValue() \n");
	char * r = getConfigValue(name);

	if(JSSE_DEBUG) printf("<< sslGetValue() \n");

	return r;
}

int sslHandshake(void* handle) {
	if(JSSE_DEBUG)  printf(">> sslHandshake() \n");
	int r = 0;
	ssl_socket_t hSocket = (ssl_socket_t)handle;

	r = nativeHandshake(hSocket->clientSocket);
	if(r < 0) {
		return -1;
	}
	hSocket->secured = true;
	
	if(JSSE_DEBUG) printf("<< sslHandshake() \n");
	return 0;
}

/*
 * This function will initialize the provider
 * Parameters:
 *   none
 * Return:
 *   none
 */
int sslInit(log_service_t *_logService, int port) {
	
	char * result = (char *)malloc(16 * sizeof(char));
	
	if(JSSE_DEBUG) printf(">> sslInit() \n");

	sprintf(result, "%d", port);

	setConfigValue(KEYSTORE_SERVER_PORT, result);
	
	jsseLogService = _logService;

	int r = nativeInit(_logService);

	if(JSSE_DEBUG) printf("<< sslInit() \n");

	return r;
}

/*
 * This function will read from a connection
 * Parameters:
 *   int: the connection ID
 *   char*: the buffer
 *   int: the offset of the buffer to start
 *   int: the number of bytes to read
 * Return:
 *   none
 */
int sslRead(void* handle, char* buffer, int length) {
	if(JSSE_DEBUG)  printf(">> sslRead() \n");
	int byteRead = -1;

	ssl_socket_t hSocket = (ssl_socket_t)handle;

	byteRead = nativeRead(hSocket->clientSocket, buffer, length);
	
	if(JSSE_DEBUG) printf("<< sslRead() \n");

	return byteRead;
}

/*
 * This function will shutdown the provider
 * Parameters:
 *   none
 * Return:
 *   none
 */
int sslReset() {
	if(JSSE_DEBUG) printf(">> sslReset() \n");
	int r = nativeShutdown();

	if(JSSE_DEBUG) printf("<< sslReset() \n");

	return r;
}

/*
 * This function will set the value of a name-value pair to the provider
 * Parameters:
 *   name: the name of the variable (null terminated)
 *   value: the value of the variable (null terminated)
 * Return:
 *   none
 */
void sslSetValue(const char* name, const char* value) {
	if(JSSE_DEBUG)  printf(">> sslSetValue() \n");
	setConfigValue(name, value);
	if(JSSE_DEBUG) printf("<< sslSetValue() \n");
}


/*
 * This function will write to a connection
 * Parameters:
 *   int: the connection ID
 *   char*: the buffer
 *   int: the offset of the buffer to start
 *   int: the number of bytes to write
 * Return:
 *   none
 */
int sslWrite(void* handle, char* buffer, int length) 
{
	if(JSSE_DEBUG) printf(">> sslWrite() \n");

	ssl_socket_t hSocket = (ssl_socket_t)handle;
	
	nativeWrite(hSocket->clientSocket, buffer, length);

	if(JSSE_DEBUG)  printf("<< sslWrite() \n");

	return length; // well... since write() in Java does not return the number of bytes written we need to fake the return value
}
