/*******************************************************************************
 * Copyright (c) 2005 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hoang M Nguyen, Intel - Initial API and Implementation
 *
 * $Id: NamedPipeTL.c,v 1.13 2005/09/26 22:13:37 kcallaghan Exp $
 *
 *******************************************************************************/ 



#include "tptp/NamedPipeTL.h"
#include "tptp/TransportSupport.h"
#include "NamedPipeListener.h"
#include "NamedPipeTLLog.h"

/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 createTransportListener(tptp_object* cmo, transport_layer_data_t * pTransportData, tptp_object* tlo )
{
	return (createNamedPipeListener(cmo, pTransportData, tlo)) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 destroyTransportListener( tptp_object* tlo )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		rc = destroyNamedPipeListener(tlo) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to set the function for forwarding messages
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 setProcessMessageFunc( tptp_object* tlo, tptp_object* nexto, processMessage_ptr_t func)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = setNamedPipeProcessMessageFunc(pServerData, nexto, func) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 startTransportListener( tptp_object* tlo )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = startNamedPipeListener(pServerData) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to stop the transport mechanism
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 stopTransportListener( tptp_object* tlo )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = stopNamedPipeListener(pServerData) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to terminate the given connection
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 terminateConnection(tptp_object* tlo, tptp_uint32 connectionID)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = terminateNamedPipeConnection(pServerData, connectionID) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to send to the given destination the given message
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 sendMessage( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 size, tptp_string* pCmdBlock)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = sendNamedPipeMessage(pServerData, connectionID, size, pCmdBlock) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to send data to the given destination 
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 sendData( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 size, void * pDataBlock)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = sendNamedPipeData(pServerData, connectionID, size, pDataBlock)  ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to set up the data path between source and destination
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 setIncomingDataFunc( tptp_object*   tlo, 
                                tptp_uint32    connectionID, 
						        tptp_uint32    partnerID, 
						        tptp_object*   partner,
						        sendData_ptr_t newDataFunc )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, NAMED_PIPE_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = setNamedPipeIncomingDataFunc(pServerData, connectionID, partnerID, partner, newDataFunc)   ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to terminate a given data connection
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 terminateDataConnection(tptp_object* tlo, tptp_uint32 connectionID)
{
	/* there is no difference between terminating the control or data connection */
	return ( terminateDataConnection(tlo, connectionID) ) ;
}

