/*******************************************************************************
 * Copyright (c) 2005 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hoang M Nguyen, Intel - Initial API and Implementation
 *
 * $Id: SharedMemTL.c,v 1.7 2005/08/11 06:37:21 hnguyen Exp $
 *
 *******************************************************************************/ 



#include "tptp/SharedMemTL.h"

#include "SharedMemListener.h"

/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 createTransportListener(tptp_object* cmo, transport_layer_data_t * pTransportData, tptp_object* tlo )
{
	return (createSharedMemListener(cmo, pTransportData, tlo)) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 destroyTransportListener( tptp_object* tlo )
{
	return (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID)? 
		destroySharedMemListener(tlo) : -1) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to set the function for forwarding messages
 *    (named pipe).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 setProcessMessageFunc( tptp_object* tlo, tptp_object* nexto, processMessage_ptr_t func)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = setSharedMemProcessMessageFunc(pServerData, nexto, func) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 startTransportListener(tptp_object *tlo)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = startSharedMemListener(pServerData) ;
	}

	return ( rc ) ;
}

/**
 *********************************************************
 *
 * @brief
 *    common interface to stop the transport mechanism
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 stopTransportListener(tptp_object *tlo)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = stopSharedMemListener(pServerData) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to terminate the given connection
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 terminateConnection(tptp_object *tlo, tptp_uint32 connectionID)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = terminateSharedMemConnection(pServerData, connectionID) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to send to the given destination the given message
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *    -1 - Not supported
 *********************************************************/

tptp_int32 sendMessage(tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 size, tptp_string* pCmdBlock)
{
	return ( -1 ) ;
}



/**
 *********************************************************
 *
 * @brief
 *    common interface to send data to the given destination 
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 sendData( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 size, void * pDataBlock)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = sendSharedMemData(pServerData, connectionID, size, pDataBlock) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to set up the data path between source and destination
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 setIncomingDataFunc( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 partnerID, tptp_object* partner, sendData_ptr_t newDataFunc )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SHARED_MEM_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = setSharedMemIncomingDataFunc( pServerData, connectionID, partnerID, partner, newDataFunc) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to terminate a given data connection
 *    (shared memory).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 terminateDataConnection(tptp_object* tlo, tptp_uint32 connectionID)
{
	/* there is no difference between terminating the control or data connection */
	return ( terminateDataConnection(tlo, connectionID) ) ;
}



